// *****************************************************************************
// * FILE:         tclAilVDSensor.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition: inherits from ail_tclAppInterface,
// *               state- and message-handling
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
// * 18.11.02 CM-CR/EES4-Bode
// *          Adapted to Tuareg.
//  * 28.08.09 RBEI/ECF1 - sak9kor - Trace Output related modifications bave been done
//  * 08.09.09 RBEI/ECF1 - sak9kor - Trace level has been modified
//  * 30.09.09 RBEI/ECF1 - jev1kor - Added Doxygen headers for all funcs/class
//  * 16.12.10 RBEI/ECF1 - sak9kor - Diaglog related modifications have been done
// *  10.03.11 RBEI/ECF1 - sak9kor - Initialization of new device i.e. Accelerometer
// *                       has been done
// *  04.05.11 RBEI CM-AI/PJ-CF31 - sainath - Low Voltage event handling related
// *                                          modifications have been done
//*   09.11.12 RBEI/ECF5 - sak9kor-  Fix for the issue NIKAI-231 is added
//*   10.01.13 RBEI/ECF5 - sga5kor-  Requirements of external power supply diagnosis
//*                                  of GPS Antenna for Nissan LCN2KAI implemented.
//*   06.01.17 RBEI/ECF12 - amg6kor -  Port to Gen4(x86_64)
// *****************************************************************************
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "fi_msgfw_if.h"
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#include "sensor_fi_if.h"

#define MIDW_FI_S_IMPORT_INTERFACE_FI_TYPES
#include "midw_fi_if.h"

#define CCA_S_IMPORT_INTERFACE_GENERIC
#include "cca_if.h"

#ifdef VARIANT_S_FTR_IERROR_MESSAGE
#define SENSOR_S_IMPORT_INTERFACE_IERROR_MESSAGE
#endif

#define SENSOR_S_IMPORT_INTERFACE_SERVER
#include "vds2_if.h"

#define VDS_S_IMPORT_INTERFACE_SUBSCRIBER_MANAGER
#define VDS_S_IMPORT_INTERFACE_AIL_VDSENSOR
#define VDS_S_IMPORT_INTERFACE_TRACE
#define VDS_S_IMPORT_INTERFACE_PERSISTENT_DATA
#include "vds_internal_if.h"

#include "vds_tcltracehandler.h"

static void vPrintSensoryTypeTraceMsg( tU32 u32SensorType );

#ifndef VARIANT_S_FTR_DISABLE_DIAGLOG
extern void vGetTimeStamp( tPU32 pu32TimeStamp );
#endif

/*!
* \brief The global service client manager.
*/
vds_tclServiceClientManager vds_goServiceClientManager;

// Create/destroy
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Creates a VD Sensor application object
//! \return
//!   returns the application interface
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
ail_tclAppInterface* vds_poCreateApplication
(
//! None
void
)
{
   return new tclAilVDSensor();
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Destroys a VD Sensor application object
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void vds_vDestroyApplication
(
//! (I) : pointer to application interface to be destroyed
ail_tclAppInterface* poAppl
)
{
   if ( poAppl != OSAL_NULL )
   {
      delete poAppl;
   }
}
/*!
* \brief Defines the interface for SPM Handler
*/
#ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
class vds_tclSPMHandler
{
   tU16 u16RegisterID;
   tBool bServiceAvailable;
   tBool bUpregDone;
   tBool bIsLowVoltEventOccurred;
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Will posts the message
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   void vPostMessage
   (
   //! (I) : pointer to a message to be posted
   amt_tclServiceData *poMsg
   )
   {
      if ( tclAilVDSensor::poThisAppInterface != OSAL_NULL )
      {
         tclAilVDSensor::poThisAppInterface->bPostMessage( poMsg );
      }
   }
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Sets the Amt CVM state
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   void vSetAmtCvmState
   (
   //! (I) : AMT state
   tU32 u32AmtState
   )
   {
      if ( tclAilVDSensor::poThisAppInterface != OSAL_NULL )
      {
         tclAilVDSensor::poThisAppInterface->vSetLowVoltageEvent(bIsLowVoltEventOccurred);
      }
   }

public:
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Default Constructor
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   vds_tclSPMHandler() :
   u16RegisterID( AMT_C_U16_REGID_INVALID ),
   bServiceAvailable( false ),
   bUpregDone( false ),
   bIsLowVoltEventOccurred(FALSE)
   {
   }
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //  brief
   //    Init function
   //  return
   //    None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   void vInit();
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Sets the register ID
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   void vSetRegisterID
   (
   //! (I) : Register ID to be set
   tU16 u16TheRegisterID
   )
   {
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
      "SPM register ID = %d", u16TheRegisterID );
      u16RegisterID = u16TheRegisterID;
   }
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Sets the availability of service
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   void vSetServiceAvailable
   (
   //! (I) : Flag to set the service available flag
   bool bAvail
   )
   {
      bServiceAvailable = bAvail;

      if ( bServiceAvailable && !bUpregDone )
      {
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
         "Sending CVM upreg" );
         vDoUpreg();
      }
   }
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Does upreg operation
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   void vDoUpreg
   (
   //! None
   void
   )
   {
      // send message using tclAilVDSensor::poThisAppInterface
      vds_tclSPMCvmEventUpReg oUpregMsg( CCA_C_U16_APP_SENSOR, CCA_C_U16_APP_SPM );
      oUpregMsg.vSetServiceID( CCA_C_U16_SRV_SPM );
      oUpregMsg.vSetRegisterID( u16RegisterID );
      vPostMessage( &oUpregMsg );
      bUpregDone = true;
   }
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Dispatcher SPM message
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   void vDispatchSPMMessage
   (
   //! (I) : Pointer to Service Data
   amt_tclServiceData *poMsg
   )
   {
      if ( poMsg == OSAL_NULL ) return;

      if ( poMsg->u16GetFunctionID() == VDS_C_U16_FKTID_SPM_CVMEVENT )
      {
         if ( poMsg->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_STATUS )
         {
            vds_tclSPMCvmEventStatus oStatus( poMsg );
            tU32 u32CvmStatus = oStatus.GetData1();
            tU8 u8Trcarr[6] = {0};
            u8Trcarr[0] = 06; /*ID ths is used to identify that these traces require to print a readable name insted decimal val*/
            u8Trcarr[1] = 04; /*sub id this is usd to identify to print which statment along with the redable values */
            u8Trcarr[2] = ( tU8 )( u32CvmStatus >> 24 );
            u8Trcarr[3] = ( tU8 )( u32CvmStatus >> 16 );
            u8Trcarr[4] = ( tU8 )( u32CvmStatus >> 8 );
            u8Trcarr[5] = ( tU8 )u32CvmStatus;
            vTraceMsgReadableVal( VDS_C_TRACELEVEL_COMPONENT,
            ( const tChar * )u8Trcarr, sizeof( u8Trcarr ) );

            switch ( u32CvmStatus )
            {
            case VDS_C_U32_CVM_CRITICAL_LOW_VOLTAGE_START:
               {
                  bIsLowVoltEventOccurred = FALSE;
                  vSetAmtCvmState( AMT_C_U32_LOW_VOLTAGE );
               }
               break;

            case VDS_C_U32_CVM_LOW_VOLTAGE_START:
               {
                  bIsLowVoltEventOccurred = TRUE;
                  vSetAmtCvmState( AMT_C_U32_NORMAL_VOLTAGE );
               }
               break;

            default:
               {
                  bIsLowVoltEventOccurred = FALSE;
                  vSetAmtCvmState( AMT_C_U32_NORMAL_VOLTAGE );
                  vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "Normal voltage event received by SPM");
               }
               break;
            }
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
            "SPM CvmEvent message with unexpected opcode ID %d",
            poMsg->u8GetOpCode() );
         }
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "SPM message with unexpected function ID %d",
         poMsg->u16GetFunctionID() );
      }
   }
};

vds_tclSPMHandler vds_goSPMHandler;
/*!
* \brief Defines the interface for SPM Service Listener
*/
class vds_tclSPMServiceListener : public vds_tclServiceListenerIf
{
private:
   vds_tclSPMHandler *poHandler;

public:
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Default constructor
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   vds_tclSPMServiceListener
   (
   //! (I) : Pointer to spmhandler
   vds_tclSPMHandler *poTheHandler
   )
   {
      poHandler = poTheHandler;
   }
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Default Destructor
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   virtual ~vds_tclSPMServiceListener
   (
   //! None
   )
   {
      poHandler = OSAL_NULL;
   }
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Notifies the registration
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   virtual void vNotifyRegistration
   (
   //! (I) : Service Id
   tU16 u16ServiceId,
   //! (I) : Register Id
   tU16 u16RegID
   )
   {
      if ( u16ServiceId == CCA_C_U16_SRV_SPM )
      {
         poHandler->vSetRegisterID( u16RegID );
      }
   }
   // ***************** F U N C T I O N  H E A D E R *****************************
   //
   //  DESCRIPTION:
   //
   //! \brief
   //!   Called when VD Sensor has received a service availability message
   //! \return
   //!   None
   //  HISTORY:
   // Date         |  Author              | MODIFICATION
   // ----------------------------------------------------------------------------
   //******************************************************************************
   virtual void vNotifyServiceAvailability
   (
   //! (I) : Service Id
   tU16 u16ServiceId,
   //! (I) : Register Id
   tU8 u8ServiceState
   )
   {
      if ( u16ServiceId == CCA_C_U16_SRV_SPM )
      {
         poHandler->vSetServiceAvailable(
         u8ServiceState == AMT_C_U8_SVCSTATE_AVAILABLE );
      }
   }
};
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Init function
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void vds_tclSPMHandler::vInit
(
//! None
)
{
   tBool bOk = vds_goServiceClientManager.bAddListener( new vds_tclSPMServiceListener( this ) );
   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
   "AilVDSensor:SPM handler initialized %d", bOk );
}
#endif
// Class tclAilVDSensor

tclAilVDSensor *tclAilVDSensor::poThisAppInterface = OSAL_NULL;

tCString tclAilVDSensor::coszVDSensorSemName = "VDS_VDSENSOR_SEM";
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Constructor
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclAilVDSensor::tclAilVDSensor()
: u16DiagLogRegisterID( AMT_C_U16_REGID_INVALID ),
u8DiagLogServiceState( AMT_C_U8_SVCSTATE_NOT_AVAILABLE ),
bIsbOnInitFailed(FALSE),
//u16SPMRegisterID( AMT_C_U16_REGID_INVALID ),
//u8SPMServiceState( AMT_C_U8_SVCSTATE_NOT_AVAILABLE ),
poMsgGpsIf( OSAL_NULL ),
poMsgGyroIf( OSAL_NULL ),
poMsgAccIf( OSAL_NULL),
poMsgOdometerIf( OSAL_NULL ),
poMsgAbsIf( OSAL_NULL ),
#ifdef VARIANT_S_FTR_IERROR_MESSAGE
poMsgIerrorIf( OSAL_NULL ),
#endif
poSensorMsgDispatcher( OSAL_NULL ),
hVDSensorSemaphore( OSAL_C_INVALID_HANDLE ),
poTraceHandler( OSAL_NULL ),
hTrace( OSAL_ERROR ),
//bIsLowVoltageEventTriggerred(FALSE),
bisDiaglogServiceavailable(FALSE),
bRegisterAsyncRet(FALSE)
{
   u32AppState = 0;
   poThisAppInterface = this;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Destructor
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclAilVDSensor::~tclAilVDSensor
(
//! None
)
{
   // delete all existing objects
   vCloseVDSensor();

   poThisAppInterface = NULL;
   poTraceHandler = OSAL_NULL;
   poMsgGpsIf = OSAL_NULL;
   poMsgGyroIf = OSAL_NULL;
   poMsgAccIf = OSAL_NULL;
   poMsgOdometerIf = OSAL_NULL;
   poMsgAbsIf = OSAL_NULL;
   poSensorMsgDispatcher = OSAL_NULL;
   bIsbOnInitFailed = false;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Save availability information for the DiagLog service.
//!   Notifies the service client manager about changes in service availability.
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vOnServiceState
(
//! (I) : Service Id
tU16 u16ServiceId,
//! (I) : Server Id
tU16 u16ServerId,
//! (I) : Register Id
tU16 u16RegisterId,
//! (I) : Service State
tU8 u8ServiceState,
//! (I) : Sub Id
tU16 u16SubId
)
{

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
   "AilVDSensor: vOnServiceState: u16ServiceId = %04x, "
   "u16ServerId =%04x,     u16RegisterId = %04x, "
   "u8ServiceState = %02x, u16SubId = %04x",
   u16ServiceId, u16ServerId, u16RegisterId,
   u8ServiceState, u16SubId );

   if ( u16ServiceId == CCA_C_U16_SRV_DIAGLOG )
   {
      u8DiagLogServiceState = u8ServiceState;
      if(u8DiagLogServiceState == AMT_C_U8_SVCSTATE_AVAILABLE)
      {
         bisDiaglogServiceavailable = TRUE;
      }
      /*To fix an issue NIKAI2-2385 - During Registration denied and Registration
      Invalid, service available flag is reset to FALSE*/
      else if((u8ServiceState == AMT_C_U8_SVCSTATE_REG_DENIED) || (u8ServiceState == AMT_C_U8_SVCSTATE_REG_INVALID ))
      {   
         bisDiaglogServiceavailable = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Service State %x",u8ServiceState );		
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Not expected State %x",u8ServiceState );
      }
      u16DiagLogRegisterID =  u16RegisterId;
   }

   #ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
   else if ( u16ServiceId == CCA_C_U16_SRV_SPM )
   {
      u8SPMServiceState = u8ServiceState;
   }
   #endif

   vds_goServiceClientManager.vNotifyServiceAvailability( u16ServiceId,u8ServiceState );

   #ifndef VARIANT_S_FTR_DISABLE_DIAGLOG
   if(bisDiaglogServiceavailable && ( u16ServiceId == CCA_C_U16_SRV_DIAGLOG))
   {
      vRegisterDiagLogSendNextTestResultUpReg(u16RegisterId);
   }
   #endif
}

#ifndef VARIANT_S_FTR_DISABLE_DIAGLOG
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Register for Diaglog SendNextTestResultUpreg property
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void tclAilVDSensor::vRegisterDiagLogSendNextTestResultUpReg
(
//! Diaglog service register ID
tU16 u16DiaglogRegisterID
)
{
   tBool bResult = FALSE;
   amt_tclServiceData      *poResponseMessage   = OSAL_NULL;
   midw_diaglogfi_tclMsgSendNextTestResultUpReg oSendNextTestResultUpReg;

   fi_tclVisitorMessage* poResultMessage = OSAL_NEW  fi_tclVisitorMessage( oSendNextTestResultUpReg, VDS_C_U16_SRV_SENSORS_MAJOR_VERSION );

   if( OSAL_NULL != poResultMessage )
   {
      tU32 u32TimeStamp = 0;
      vGetTimeStamp( &u32TimeStamp );

      if ( poResultMessage->bIsValid() )
      {
         // create answer-message
         poResultMessage->vInitServiceData
         (  CCA_C_U16_APP_SENSOR,                     //Source
         CCA_C_U16_APP_DIAGLOG,                    //Target
         AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,        // StreamType
         0,                                        // StreamCounter
         u16DiaglogRegisterID,                     // RegisterID
         0,                                        // nCmdCounter,
         CCA_C_U16_SRV_DIAGLOG,                    // nServiceID,
         MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT,  // nFunctionID,
         AMT_C_U8_CCAMSG_OPCODE_UPREG,             // OpCode
         0,                            // Asynchronous Completion Token (ACT)
         AMT_C_U16_DEFAULT_NULL,       // Source Sub
         AMT_C_U16_DEFAULT_NULL,       // Target Sub
         u32TimeStamp                  //TimeStamp
         );
      }

      poResponseMessage = static_cast<amt_tclServiceData *>( poResultMessage );
   }

   if (OSAL_NULL != poResponseMessage )
   {
      /*This has to be updated*/
      // send response-message
      poResponseMessage->vSetSourceAppID( CCA_C_U16_APP_SENSOR );
      poResponseMessage->vSetTargetAppID( CCA_C_U16_APP_DIAGLOG );
      poResponseMessage->vSetServiceID  ( CCA_C_U16_SRV_DIAGLOG );
      poResponseMessage->vSetTargetSubID( AMT_C_U16_DEFAULT_NULL );
      poResponseMessage->vSetCmdCounter ( 0  );
      poResponseMessage->vSetRegisterID ( u16DiaglogRegisterID  );
      poResponseMessage->vSetACT        ( 0 );
      bResult = bPostMessage( poResponseMessage );

      if(bResult == FALSE)
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Posting UPREG message of SendNextTestResult property failed");
      }

      delete poResponseMessage;
   }
}
#endif
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Simulate a toggle of DiagLog service availability. This is used to
//!   force to resend the error states to VD DiagLog.
//!   If the Diaglog service is not available, this function does nothing.
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void tclAilVDSensor::vToggleDiagLog
(
//! None
void
)
{
   if ( u8DiagLogServiceState == AMT_C_U8_SVCSTATE_AVAILABLE )
   {
      vds_goServiceClientManager.vNotifyServiceAvailability(
      CCA_C_U16_SRV_DIAGLOG, AMT_C_U8_SVCSTATE_NOT_AVAILABLE );
      vds_goServiceClientManager.vNotifyServiceAvailability(
      CCA_C_U16_SRV_DIAGLOG, AMT_C_U8_SVCSTATE_AVAILABLE );
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    If Lowvoltage Event is detected sets the internal member of tclAilVDSensor
//!    class to TRUE else FALSE. 
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
#ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
void tclAilVDSensor::vSetLowVoltageEvent
(
//! LowVoltageEvent value
tBool bLowVoltageEvent
)
{
   bIsLowVoltageEventTriggerred = bLowVoltageEvent;
}
#endif
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Handle application state changes.
//!   Inherited from class ail_tclAppInterfaceRestricted
//!   (The interface framework calls "vOnNewAppState()" whenever a
//!   new POWER-Message has been received. As default it mirrors all POWER
//!   commands to satisfy the LPM-interface. For specific application behavior,
//!   the user must overwrite this function. ) calls vSwitchFrom___ depending on
//!   u32OldAppState
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vOnNewAppState
(
//! (I) : the last confirmed application state
tU32 u32OldAppState,
//! (I) : the new requested application state
tU32 u32NewAppState
)
{

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
              "AilVDSensor:state change request received at time(ms) %u",
              OSAL_ClockGetElapsedTime() );
   //If the system config is color variant, we only 
   //initialize vdsensor but make its service unavailable
   if( bIsbOnInitFailed )
   {
      u32AppState = u32NewAppState;
      vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSOR_LOCATION, 
                                   AMT_C_U8_SVCSTATE_NOT_AVAILABLE );
      vAppStateChanged( u32NewAppState );
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                 "AilVDSensor:VDSensor service is made unavailable");
   }
   else
   {
      if ( u32OldAppState == u32NewAppState )
      {
         //Nonsense,but could happen with LPX simulator
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
         "AilVDSensor:vOnNewAppState: OldState == NewState" );
         vAppStateChanged( u32NewAppState );
      }
      else
      {

         if ( poMsgGpsIf != NULL )
         {
            poMsgGpsIf->vSetAppState( u32NewAppState );
         }

         if ( poMsgGyroIf != NULL )
         {
            poMsgGyroIf->vSetAppState( u32NewAppState );
         }
         
         if( poMsgAccIf != OSAL_NULL )
         {
            poMsgAccIf->vSetAppState( u32NewAppState );
         } 
         
         tU8 au8Trcarr[10] = {0};
         au8Trcarr[0] = 06;/*ID ths is used to identify that these traces require to print a readable name insted decimal val*/
         au8Trcarr[1] = 01;/*sub id this is usd to identify to print which statment along with the redable values */
         au8Trcarr[2] = ( tU8 )( u32OldAppState >> 24 );
         au8Trcarr[3] = ( tU8 )( u32OldAppState >> 16 );
         au8Trcarr[4] = ( tU8 )( u32OldAppState >> 8 );
         au8Trcarr[5] = ( tU8 )( u32OldAppState );
         au8Trcarr[6] = ( tU8 )( u32NewAppState >> 24 );
         au8Trcarr[7] = ( tU8 )( u32NewAppState >> 16 );
         au8Trcarr[8] = ( tU8 )( u32NewAppState >> 8 );
         au8Trcarr[9] = ( tU8 )( u32NewAppState );

         vTraceMsgReadableVal( VDS_C_TRACELEVEL_COMPONENT,
         ( const tChar * )au8Trcarr, sizeof( au8Trcarr ) );


         switch ( u32OldAppState )
         {
            
         case AMT_C_U32_STATE_INITIALIZED:
            {
               vSwitchFromInitialized( u32NewAppState );
            }
            break;
            
         case AMT_C_U32_STATE_NORMAL:
            {
               vSwitchFromNormal( u32NewAppState );
            }         
            break;
            
            #ifndef VARIANT_S_FTR_DISABLE_DIAGNOSIS			
         case AMT_C_U32_STATE_DIAGNOSIS:
            {
               vSwitchFromDiagnosis( u32NewAppState );
            }
            break;
            #endif
            
         case AMT_C_U32_STATE_RECEIVE_READY:
            {
               vSwitchFromReceiveReady( u32NewAppState );
            }
            break;
            
         case AMT_C_U32_STATE_PAUSE:
            {
               vSwitchFromPause( u32NewAppState );
            }         
            break;
            
         case AMT_C_U32_STATE_OFF:
            {
               vSwitchFromOff( u32NewAppState );
            }   
            break;
            
         case AMT_C_U32_STATE_PREPARE_DOWNLOAD:
            {
               // illegal, only transition to terminated
               // (handled in vOnApplicationClose)
            }
            break;
            
         default:
            {
               vTraceMsg( VDS_C_TRACELEVEL_ERROR,
               "vOnNewAppState:Unknown state:OldAppState %d -> NewAppState %d",
               u32OldAppState, u32NewAppState );
            }
            break;
         }
      }
   }
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Handle Service Data CCA messages
//!   Inherited from ail_tclAppInterfaceRestricted ( The interface
//!   framework calls "vOnNewMessage()" whenever a new Service
//!   Data-Message has been received. As default it releases the
//!   message buffer which belongs to "amt_tclBaseMessage*". The
//!   user has to overwrite this function to implement his own
//!   dipatcher here. Then it's his responsibility to free the
//!   message buffer. ) Handles Sensor- or SensorDiag-Message
//!   depending on ServiceId
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vOnNewMessage
(
//! (I) : pointer to the received ServiceData-Message
amt_tclBaseMessage* poMessage
)
{
   // receive amt message
   amt_tclServiceData oServiceData( poMessage );
   // get service-id
   tU16 u16ServiceId = oServiceData.u16GetServiceID();
   
   //Check Msg Validity.
   if ( poMessage->bIsValid() )
   {
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
      "vOnNewMessage:SouceAppId = 0x%x(ServiceId:0x%x FunId:0x%x Opcode:0x%x)",
      oServiceData.u16GetSourceAppID(),
      u16ServiceId,
      oServiceData.u16GetFunctionID(),
      oServiceData.u8GetOpCode() );

      // sensor- or diagnosis-message?
      switch ( u16ServiceId )
      {
         //If server is VDSensor.
      case CCA_C_U16_SRV_SENSOR_LOCATION:
         {
            vDispatchSensorMsg( &oServiceData );
         }   
         break;
         
         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         //If server is IError
      case CCA_C_U16_SRV_SENSORS_IERROR:
         {
            vDispatchSensorMsg( &oServiceData );
         }          
         break;
         #endif

         #ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
         //If server is SPM.
      case CCA_C_U16_SRV_SPM:
         {
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
            "tclAilVDSensor:vOnNewMessage:SPM Event Received");
            vds_goSPMHandler.vDispatchSPMMessage( &oServiceData );
         }
         break;
         #endif

         //If server is DIAGLOG.
      case CCA_C_U16_SRV_DIAGLOG:
         {
            // ignore method results from VD DiagLog
            #ifndef VARIANT_S_FTR_DISABLE_DIAGLOG
            /*
            *This is required to get a message from Diaglog related
            *to SendNextTestRestult Propery.
            */
            vDispatchDiaglogMsg(&oServiceData);
            #endif
         }
         break;
         
         //If server is something else.
      default:
         {
            //Error:ServiceID not supported.
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,
            "tclAilVDSensor:vOnNewMessage:Unsupported service-id:%d",
            u16ServiceId);
            //Compose error message and replay to the sender.
            amt_tclServiceDataError oErrorMsg( oServiceData, AMT_C_U16_ERROR_UNKNOWN_SVC_ID );
            bPostMessage( &oErrorMsg );
         }
         break;
      }
   }
   else
   {
      //Throw error-message
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
      "vOnNewMessage:InValid Msg Received");
   }

   //Release the message object
   oServiceData.bDelete();
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Inform the application that a client has deregistered itself from a service.
//!   Inherited from ail_tclAppInterfaceRestricted ( The interface
//!   framework calls "vOnUnregister()" every time, when a valid
//!   Service Unregister-Message has been received. This function
//!   informs the application, that all requests label with this
//!   RegisterId has got obsolete. As default no corresponding
//!   action is done. The user has to overwrite this function and
//!   to clear the RegisterId from his application
//!   references. Please note, that the framework will assign a
//!   free RegisterId next time a service is requested by some
//!   client. ) All registered notifications will be deleted
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vOnUnregister
(
//! (I) : the former requested ServiceId
tU16 u16ServiceId,
//! (I) : the obsolete RegisterId
tU16 u16RegisterId
)
{
   if ( CCA_C_U16_SRV_SENSOR_LOCATION == u16ServiceId
         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         || CCA_C_U16_SRV_SENSORS_IERROR == u16ServiceId
         #endif
         )
   {
      if ( vds_goMasterLock.bEnter() )
      {
         tclSubscriberManager *poSubscriber = tclSubscriberManager::poGetSubscriberWithRegId( NULL, u16RegisterId );
         tclSubscriberManager *poHelpSubscriber = poSubscriber;

         while ( NULL != poSubscriber )
         {
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
            "AilVDSensor:Application with AppId 0x%x and RegisterId = 0x%x unregistered",
            poSubscriber->u16GetTargetAppId(),u16RegisterId);
            // get following subscriber with RegisterId
            poHelpSubscriber = tclSubscriberManager::poGetSubscriberWithRegId( poSubscriber, u16RegisterId );
            poSubscriber->s32DeleteSubscriber();
            delete poSubscriber;
            poSubscriber = poHelpSubscriber;
         }

         vds_goMasterLock.vLeave();
      }
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Initialize the application.
//!   Inherited from class ail_tclAppInterfaceRestricted ( The
//!   interface framework calls "bOnInit()" before starting any
//!   communication action.  As default this function always
//!   returns TRUE. The user has to implement all his local
//!   application initialization ( creating required threads,
//!   allocate application memory, ... ). If returns TRUE,
//!   initializing could be successfully performed and framework
//!   will register application at the LocalPower Management, else,
//!   the application would end itself. )
//!   Initializes Sensor-Threads, Interfaces to persistent data,
//!   Interface to report-memory, ...
//! \return
//!   - \c  TRUE : application could be successfully initialized\n
//!   - \c  FALSE : application could be successfully not initialized\n
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclAilVDSensor::bOnInit
(
//! None
)
{

   tBool bSuccess = TRUE;
   tU32 u32SensorTypes;

   //Open Trace Device.
   bOpenTraceDevice( &hTrace );

   if( !vds_goMasterLock.bInit() )
   {
      bSuccess = FALSE;
   }

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,"VDS2: started");

   //Get required data from registry.
   tclSystemInformation::vInit();

   u32SensorTypes = tclSystemInformation::rInfo.rSystem.u32SensorTypes;

   //Print sensors which are available in readable format.
   vPrintSensoryTypeTraceMsg( u32SensorTypes );


   //Create Semaphore.
   if( OSAL_ERROR == OSAL_s32SemaphoreCreate(coszVDSensorSemName,&hVDSensorSemaphore,1) )
   {
      bSuccess = FALSE;
   }

   //Initialize GPS message interface.
   if( NULL == poMsgGpsIf )
   {
      poMsgGpsIf = new tclMsgGpsIf();

      if( poMsgGpsIf == NULL )
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Unable to create tclMsgGpsIf object");
      }
      else if( VDS_E_NO_ERROR 
            != 
            poMsgGpsIf->s32Init( tclAilVDSensor::s32SendSubscriberMessage) )
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AilVDSensor:tclMsgGpsIf Init Failed" );
      }
      else
      {
         //Flush GNSS SCC buffer if only GNSS is available in the project.
         if( u32SensorTypes == ( VDS_C_U32_SENSOR_GNSS_FEATURE | VDS_C_U32_SENSOR_GPS ) )
         {
            poMsgGpsIf->vSetGpsThreadFlags( enbFlushGnssBuf,true );
         }
      }
   }

   //Initialize odo message interface.
   if( (NULL == poMsgOdometerIf) &&
         ((u32SensorTypes & VDS_C_U32_SENSOR_ODO) != 0) )
   {
      //Initalize odometer-interface.
      poMsgOdometerIf = new tclMsgOdometerIf();

      if( poMsgOdometerIf == NULL )
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "AilVDSensor:Unable to create tclMsgOdometerIf object");
      }
      else if( (tS32)VDS_E_NO_ERROR 
               != poMsgOdometerIf->s32Init( tclAilVDSensor::s32SendSubscriberMessage) )
      {
         bIsbOnInitFailed = TRUE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "AilVDSensor:tclMsgOdometerIf Init Failed");
      }
   }

   //Initialize gyro message interface.
   if( ( NULL == poMsgGyroIf ) &&
         (( u32SensorTypes & VDS_C_U32_SENSOR_GYRO ) != 0) )
   {
      poMsgGyroIf = new tclMsgGyroIf();

      if( poMsgGyroIf == NULL )
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "AilVDSensor:Unable to create tclMsgGyroIf object");
      }
      else if( (tS32)VDS_E_NO_ERROR 
               != poMsgGyroIf->s32Init( tclAilVDSensor::s32SendSubscriberMessage) ) 
      {
         bIsbOnInitFailed = TRUE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "AilVDSensor:tclMsgGyroIf Init Failed");
      }
   }

   //Initialize Acc message interface.
   if( (NULL == poMsgAccIf) &&
         ( (u32SensorTypes & VDS_C_U32_SENSOR_ACC) != 0) )
   {
      poMsgAccIf = new tclMsgAccIf();

      if( poMsgAccIf == NULL )
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Unable to create tclMsgAccIf object");
      }
      else if( (tS32)VDS_E_NO_ERROR 
               != poMsgAccIf->s32Init( tclAilVDSensor::s32SendSubscriberMessage) )
      {  
         bIsbOnInitFailed = TRUE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                       "AilVDSensor:tclMsgAccIf Init Failed");
      }
   }

   //Initialize ABS message interface.
   //Initialize ABS only if Odo is not initialized.
   if( ( (u32SensorTypes & VDS_C_U32_SENSOR_ABS) != 0) &&
         ( (u32SensorTypes & VDS_C_U32_SENSOR_ODO) != 0) )
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
      "AilVDSensor:Both Odo,ABS available!Skipping ABS Init!\
               This may cause reset!Check registry!");
   }
   else if( ( NULL == poMsgOdometerIf ) &&
         ( NULL == poMsgAbsIf ) &&
         ( (u32SensorTypes & VDS_C_U32_SENSOR_ABS) != 0 ) )
   {
      poMsgAbsIf = new vds_tclMsgAbsIf();

      if ( poMsgAbsIf == OSAL_NULL )
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Unable to create vds_tclMsgAbsIf object");
      }
      else if( (tS32)VDS_E_NO_ERROR 
               != poMsgAbsIf->s32Init( tclAilVDSensor::s32SendSubscriberMessage) )
      {
         bIsbOnInitFailed = TRUE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:vds_tclMsgAbsIf Init Failed");
      } 
   }

   #ifdef VARIANT_S_FTR_IERROR_MESSAGE
   if ( NULL == poMsgIerrorIf )
   {
      // initialize ierror-interface
      poMsgIerrorIf = new tclMsgIerrorIf();

      if ( poMsgIerrorIf == NULL )
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Unable to create tclMsgIerrorIf object");
      }
      else if( ( tS32 )VDS_E_NO_ERROR 
            != 
            poMsgIerrorIf->s32Init( tclAilVDSensor::s32SendSubscriberMessage ) )
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AilVDSensor:tclMsgIerrorIf Init Failed" );
      }
   }
   #endif //VARIANT_S_FTR_IERROR_MESSAGE

   //Initialize trace handler.
   poTraceHandler = new vds_tclTraceHandler( poMsgGpsIf );

   if ( NULL == poTraceHandler )
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
      "tclAilVDSensor::TraceHandler init Failed" );
   }
   else
   {
      poTraceHandler->vRegisterToTtfis( hTrace );
   }

   //Initialize message dispatcher.
   if( NULL == poSensorMsgDispatcher )
   {
      poSensorMsgDispatcher = new tclSensorMsgDispatcher();

      if( poSensorMsgDispatcher == NULL )
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Unable to create tclSensorMsgDispatcher object");
      }
      else if( VDS_E_NO_ERROR != poSensorMsgDispatcher->s32Init( poMsgGpsIf,
               poMsgGyroIf,
               poMsgAccIf,
               poMsgOdometerIf,
               poMsgAbsIf
               #ifdef VARIANT_S_FTR_IERROR_MESSAGE
               ,poMsgIerrorIf
               #endif 
               ))
      {
         bSuccess = FALSE;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:tclSensorMsgDispatcher Init Failed");
      }
   }

   //Initialize subscriber manager.
   if( VDS_E_NO_ERROR != tclSubscriberManager::s32Init() )
   {
      bSuccess = FALSE;
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
      "AilVDSensor:tclSubscriberManager Init Failed" );
   }

   
   if ( bSuccess )
   {
      if( bIsbOnInitFailed )
      {
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "tclAilVDSensor:bOnInit:vdsensor service should be made unavailable");
      }
      else
      {
   
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                    "tclAilVDSensor:bOnInit:VDSensor initialized successfully");
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
      "tclAilVDSensor:bOnInit:VDSensor initialization failed");
      // delete all created objects
      vCloseVDSensor();
   }

   #ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
   vds_goSPMHandler.vInit();
   #endif

   return bSuccess;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Tells the application to close itself.
//!   Inherited from ail_tclAppInterfaceRestricted.  This function
//!   is called, when the application shall be immediatley closed.
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vOnApplicationClose
(
//! None
)
{
   // delete all existing objects
   vCloseVDSensor();

   if ( NULL == poTraceHandler )
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
      "AilVDSensor::vOnApplicationClose -> poTraceHandler invalid" );
   }
   else
   {
      poTraceHandler->vUnregisterToTtfis( hTrace );
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "AilVDSensor:vUnregisterToTtfis() done" );
   }
   vCloseTraceDevice( &hTrace );
   vApplicationCloseAcknowledge( AIL_C_U8_APP_END_SUCCESSFUL );
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Query supported service version.
//!   The server application has to override this method to report the
//!   actual version number to a given service id
//! \return
//!   - \c  TRUE : server supports requested service\n
//!   - \c  FALSE : server does not support requested service\n
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclAilVDSensor::bGetServiceVersion
(
//! (I) : service id
tU16 u16MyServiceID,
//! (O) : major version number
tU16 &rfu16MajorVersion,
//! (O) : minor version number
tU16 &rfu16MinorVersion,
//! (O) : patch version number
tU16 &rfu16PatchVersion
)
{
   tBool bServiceIsValid = TRUE;

   switch ( u16MyServiceID )
   {
   case CCA_C_U16_SRV_SENSOR_LOCATION:
      {
         rfu16MajorVersion = VDS_C_U16_SRV_SENSORS_MAJOR_VERSION;
         rfu16MinorVersion = VDS_C_U16_SRV_SENSORS_MINOR_VERSION;
         rfu16PatchVersion = VDS_C_U16_SRV_SENSORS_PATCH_VERSION;
      }
      break;

      #ifdef VARIANT_S_FTR_IERROR_MESSAGE
   case CCA_C_U16_SRV_SENSORS_IERROR:
      {
         rfu16MajorVersion = VDS_C_U16_SRV_IERROR_MAJOR_VERSION;
         rfu16MinorVersion = VDS_C_U16_SRV_IERROR_MINOR_VERSION;
         rfu16PatchVersion = VDS_C_U16_SRV_IERROR_PATCH_VERSION;
      }
      break;
      #endif

   default:
      {
         rfu16MajorVersion = AMT_C_U16_SERVICE_VERSION_INVALID;
         rfu16MinorVersion = AMT_C_U16_SERVICE_VERSION_INVALID;
         rfu16PatchVersion = AMT_C_U16_SERVICE_VERSION_INVALID;
         bServiceIsValid = FALSE;
      }
      break;
   }

   return bServiceIsValid;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Return application state of VD Sensor
//! \return
//!   application state (AMT-states)
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tU32 tclAilVDSensor::u32GetAppState ()
{
   return u32AppState;
}

// ********************* F U N C T I O N H E A D E R **************************
//  DESCRIPTION:
//
//! \brief
//!   Starts all the sensor threads i.e activates the threads which are
//!   already created at the time of VDSensor initialization and these 
//!   threads collects data from the respective OSAL drivers.
//!   
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_THREADCREATE_ERROR : Thread Create Error
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclAilVDSensor::s32StartSensorThreads
(
//! None
tVoid
)
{
   tS32 s32RetVal = (tS32)VDS_E_NO_ERROR;

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
   "AilVDSensor:Starting sensor threads");

   //Start GPS Thread
   if ( NULL != poMsgGpsIf )
   {
      s32RetVal = poMsgGpsIf->s32StartThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = (tS32)VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Starting GPS thread failed");
      }
   }

   // start gyro-thread
   if ( NULL != poMsgGyroIf )
   {
      s32RetVal = poMsgGyroIf->s32StartThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Starting gyro thread failed");
      }
   }

   //Start Accelerometer Thread
   if ( OSAL_NULL != poMsgAccIf )
   {
      s32RetVal = poMsgAccIf->s32StartThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = (tS32)VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Starting accelerometer thread failed");         
      }
   }

   // start odometer-thread
   if ( NULL != poMsgOdometerIf )
   {
      s32RetVal = poMsgOdometerIf->s32StartThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Starting odometer thread failed");
      }
   }

   // start ABS thread
   if ( NULL != poMsgAbsIf )
   {
      s32RetVal =  poMsgAbsIf->s32StartThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = (tS32)VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
         "AilVDSensor:Starting ABS thread failed.");
      }
   }

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Stop sensor threads.
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_THREADCREATE_ERROR : Thread Create Error
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclAilVDSensor::s32StopSensorThreads
(
//! None
)
{
   tS32 s32RetVal = (tS32)VDS_E_NO_ERROR;

   // stop gps-thread
   if ( NULL != poMsgGpsIf )
   {
      s32RetVal = poMsgGpsIf->s32StopThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Stopping GPS thread failed");
      }
   }

   //Stop Gyro Thread
   if ( NULL != poMsgGyroIf )
   {
      s32RetVal = poMsgGyroIf->s32StopThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Stopping gyro thread failed");
      }
   }

   //stop Acc-thread
   if ( OSAL_NULL != poMsgAccIf )
   {
      s32RetVal = poMsgAccIf->s32StopThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Stopping accelerometer thread failed");
      }
   }

   // stop odometer-thread
   if ( NULL != poMsgOdometerIf )
   {
      s32RetVal = poMsgOdometerIf->s32StopThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Stopping odometer thread failed");
      }
   }

   //Stop ABS Thread
   if( NULL != poMsgAbsIf )
   {
      s32RetVal = poMsgAbsIf->s32StopThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Stopping ABS thread failed");
      }
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Delete sensor threads.
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_THREADCREATE_ERROR : Thread Create Error
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclAilVDSensor::s32DeleteSensorThreads
(
//! None
)
{
   tS32 s32RetVal = ( tS32 )VDS_E_NO_ERROR;

   //Delete GNSS Thread
   if ( NULL != poMsgGpsIf )
   {
      s32RetVal = poMsgGpsIf->s32DeleteThread();
      if( s32RetVal != (tS32)VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Deleting GPS thread failed");
      }
   }

   //Delete Gyro Thread
   if ( NULL != poMsgGyroIf )
   {
      s32RetVal = poMsgGyroIf->s32DeleteThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Deleting gyro thread failed");
      }
   }

   //Delete Accelerometer Thread
   if ( OSAL_NULL != poMsgAccIf )
   {
      s32RetVal = poMsgAccIf->s32DeleteThread();
      if ( s32RetVal != ( tS32 )VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Deleting accelerometer thread failed");
      }
   }

   //Delete Odometer Thread
   if ( NULL != poMsgOdometerIf )
   {
      s32RetVal = poMsgOdometerIf->s32DeleteThread();

      if ( s32RetVal != (tS32)VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Deleting odometer thread failed");
      }
   }

   //Delete ABS Thread
   if ( NULL != poMsgAbsIf )
   {
      s32RetVal = poMsgAbsIf->s32DeleteThread();
      if ( s32RetVal != (tS32)VDS_E_NO_ERROR )
      {
         s32RetVal = ( tS32 )VDS_E_THREADCREATE_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Deleting ABS tread failed");
      }
   }

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Sends message to subscribers
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_AILVDSENSOR_NOT_INITIALIZED : AilVdSensor not initialized
//!   - \c VDS_E_AIL_STATE_NOT_NORMAL : AIL state is not normal
//!   - \c VDS_E_AIL_ALLOC_ERROR : AIL Allocation error
//!   - \c VDS_E_INVALID_PARAMETER : Invalid Parameter
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclAilVDSensor::s32SendSubscriberMessage
(
//! (O) : pointer to Service Data
amt_tclServiceData *poServiceData,
//! (I) : pointer to Subscriber Data
tclSubscriberManager *poSubscriber
)
{
   tS32 s32RetVal = ( tS32 )VDS_E_NO_ERROR;

   if ( NULL != poServiceData )
   {
      if ( poServiceData->bIsValid() )
      {
         // send only in normal/diag-mode (could be removed, all callers check state, too)
         tU32 u32CurrentAppState = poThisAppInterface->u32GetAppState();

         if ( AMT_C_U32_STATE_NORMAL == u32CurrentAppState ||
               AMT_C_U32_STATE_DIAGNOSIS == u32CurrentAppState )
         {
            poServiceData->vSetRegisterID(  poSubscriber->u16GetRegisterId() );
            poServiceData->vSetServiceID(   poSubscriber->u16GetServiceId() );
            poServiceData->vSetTargetSubID( poSubscriber->u16GetMessageSubId() );
            poServiceData->vSetCmdCounter(  poSubscriber->u16GetCommandCntr() );
            poServiceData->vSetACT(         poSubscriber->u8GetACT() );
            // lock only needed in sensor-thread-context
            tBool bLocked = FALSE;

            if ( poSubscriber->bGetMessageLockNeeded() )
            {
               if ( poThisAppInterface->bCritSectionBegin() )
               {
                  bLocked = TRUE;
               }
            }
            else
            {
               bLocked = TRUE; // not needed -> is already locked
            }

            if( bLocked )
            {
               //post message
               poThisAppInterface->bPostMessage( poServiceData );

               if ( poSubscriber->bGetMessageLockNeeded() )
               {
                  poThisAppInterface->vCritSectionEnd();
               }
            }
            else
            {
               vTraceMsg( VDS_C_TRACELEVEL_ERROR,
               "AilVDSensor:Couldn't send message,lock failed" );
               poServiceData->bDelete();
               s32RetVal = ( tS32 )VDS_E_AILVDSENSOR_NOT_INITIALIZED;
            }
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
            "AilVDSensor:Couldn't send message,state not normal" );
            poServiceData->bDelete();
            s32RetVal = ( tS32 )VDS_E_AIL_STATE_NOT_NORMAL;
         }
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "AilVDSensor:Couldn't send message,service-data invalid" );
         s32RetVal = (tS32)VDS_E_AIL_ALLOC_ERROR;
      }
   }
   else
   {
      s32RetVal = (tS32)VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
      "AilVDSensor:Couldn't send message,invalid parameters" );
   }

   return s32RetVal;
}

#if defined(DIAGLOG_POLLING_HACK)
static tU32 gmsTimeToTry = 0;
#endif
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Send message (and delete if failed)
//! \return
//!   - \c  TRUE : Message Post Success\n
//!   - \c  FALSE : Message Post Failed\n
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclAilVDSensor::bPostMessage
(
//! (I) : Pointer to Service Data
amt_tclServiceData *poServiceData
)
{
   tBool bRetVal = FALSE;

   if ( NULL != poServiceData )
   {
      if ( poServiceData->bIsValid() )
      {
         ail_tenCommunicationError enCommError;

         if ( AIL_EN_N_NO_ERROR == ( enCommError = enPostMessage( poServiceData ) ) )
         {
            bRetVal = TRUE;
         }
         else
         {
            tU16  u16TargetAppId = poServiceData->u16GetTargetAppID();
            tU16  u16ServiceId = poServiceData->u16GetServiceID();
            tU16  u16RegisterId = poServiceData->u16GetRegisterID();
            tU16  u16SourceAppId = poServiceData->u16GetSourceAppID();
            tU16  u16SourceSubId = poServiceData->u16GetSourceSubID();
            tU8   u8MsgOpCode = poServiceData->u8GetOpCode();

            vTraceMsg(VDS_C_TRACELEVEL_ERROR,
            "u16TargetAppId = %d, u16ServiceId = %d, "
            "u16RegisterId = %d, u16SourceAppId = %d, "
            "u16SourceSubId = %d, u8MsgOpCode = %d",
            u16TargetAppId,u16ServiceId,u16RegisterId,
            u16SourceAppId,u16SourceSubId,u8MsgOpCode );

            //delete message
            vTraceMsg( VDS_C_TRACELEVEL_ERROR, "tclAilVDSensor: enPostMessage failed (0x%04X)", ( tU32 )enCommError );
            poServiceData->bDelete();
         }
      }
   }

   #if defined(DIAGLOG_POLLING_HACK)
   if ( u16DiagLogRegisterID == AMT_C_U16_REGID_INVALID )
   {
      tU32 msCurrentTime = OSAL_ClockGetElapsedTime( );

      if ( msCurrentTime > gmsTimeToTry )
      {
         gmsTimeToTry = msCurrentTime + 1000;
         vRegisterDiagLog( );
      }
   }
   #endif

   return bRetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   close file-handles, free memory, close semaphores, ...
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vCloseVDSensor
(
//! None
)
{

   tclSubscriberManager::s32Close();

   //Close sensor message dispatcher
   if ( NULL != poSensorMsgDispatcher )
   {
      poSensorMsgDispatcher->s32Close();
      delete poSensorMsgDispatcher;
      poSensorMsgDispatcher = NULL;
   }

   #ifdef VARIANT_S_FTR_IERROR_MESSAGE
   //Close Ierror interface
   if ( NULL != poMsgIerrorIf )
   {
      delete poMsgIerrorIf;
      poMsgIerrorIf = NULL;
   }
   #endif // VARIANT_S_FTR_IERROR_MESSAGE

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
   "AilVDSensor:Stop sensor worker threads");

   //Close ABS interface
   if ( NULL != poMsgAbsIf )
   {
      //if thread is running,stop thread
      if ( poMsgAbsIf->s32StopThread() != ( tS32 )VDS_E_NO_ERROR )
      {
         poMsgAbsIf->s32DeleteThread();
      }
   }

   //Close odometer-interface
   if ( NULL != poMsgOdometerIf )
   {
      //if thread is running, stop thread
      if ( poMsgOdometerIf->s32StopThread() != ( tS32 )VDS_E_NO_ERROR )
      {
         poMsgOdometerIf->s32DeleteThread();
      }
   }

   //Close gyro-interface
   if ( NULL != poMsgGyroIf )
   {
      //if thread is running, stop thread
      if ( poMsgGyroIf->s32StopThread() != ( tS32 )VDS_E_NO_ERROR )
      {
         poMsgGyroIf->s32DeleteThread();
      }
   }

   //Stop and delete the threads
   if ( OSAL_NULL != poMsgAccIf )
   {
      //if thread is running, stop thread
      if ( poMsgAccIf->s32StopThread() != ( tS32 )VDS_E_NO_ERROR )
      {
         poMsgAccIf->s32DeleteThread();
      }
   }

   //Close gps-interface
   if ( NULL != poMsgGpsIf )
   {
      //if thread is running, stop thread
      if ( poMsgGpsIf->s32StopThread() != ( tS32 )VDS_E_NO_ERROR )
      {
         poMsgGpsIf->s32DeleteThread();
      }
   }

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,"AilVDSensor:Delete sensor message IFs" );

   //Close ABS interface
   if ( NULL != poMsgAbsIf )
   {
      delete poMsgAbsIf;
      poMsgAbsIf = NULL;
   }

   //Close Odometer interface
   if ( NULL != poMsgOdometerIf )
   {
      delete poMsgOdometerIf;
      poMsgOdometerIf = NULL;
   }

   //Close Gyro interface
   if ( NULL != poMsgGyroIf )
   {
      delete poMsgGyroIf;
      poMsgGyroIf = NULL;
   }

   //Close Accelerometer interface
   if(OSAL_NULL != poMsgAccIf)
   {
      delete poMsgAccIf;
      poMsgAccIf = OSAL_NULL;
   }

   //Close GPS interface
   if ( NULL != poMsgGpsIf )
   {
      delete poMsgGpsIf;
      poMsgGpsIf = NULL;
   }

   //Delete semaphore
   if ( OSAL_C_INVALID_HANDLE != hVDSensorSemaphore )
   {
      OSAL_s32SemaphoreClose( hVDSensorSemaphore );
      OSAL_s32SemaphoreDelete( coszVDSensorSemName );
      hVDSensorSemaphore = OSAL_C_INVALID_HANDLE;
   }

   vds_goMasterLock.bShutdown();

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,"AilVDSensor:VDSensor closed!" );

}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Handle incoming CCA Service Data messages.
//!   handles following message-opcodes\n
//!   Get: collect data from Msg*If and send it back\n
//!   Set: set data in Msg*If and send status-msg back\n
//! Check Subscriber :
//!   UpReg: create SubscriberElement
//!   RelUpReg: delete SubscriberElement
//!   MethodStart: calulate result, check for subscriber
//!   MethodAbort: stops calculation
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vDispatchSensorMsg
(
//! (I) : pointer to message with ServiceId 'Sensor'
amt_tclServiceData* poServiceData
)
{
   //pointer to response-message
   amt_tclServiceData  *poResponseMessage  = NULL;

   /* pointer to error-message (in the following code,
   * the allocation of an error-msg is not checked,
   * 'cause we can't do any more, when this error occurs)
   */
   amt_tclServiceDataError *poErrorMessage = NULL;
   tU16 u16SourceAppId = 0;
   tU16 u16TargetAppId = 0;

   //param-check
   if ( NULL != poServiceData )
   {
      u16SourceAppId = poServiceData->u16GetTargetAppID();
      u16TargetAppId = poServiceData->u16GetSourceAppID();

      tBool bSubscribersLocked = FALSE;

      if ( vds_goMasterLock.bEnter() )
      {
         bSubscribersLocked = TRUE; // else no subscriber exists
      }
      
      if ( bCritSectionBegin() )
      {
         // MsgDispatcher initialized?
         if ( NULL != poSensorMsgDispatcher )
         {
            if ( ( tS32 )VDS_E_NO_ERROR == poSensorMsgDispatcher->s32DispatchSensorMsg( poServiceData, &poResponseMessage, &poErrorMessage ) )
            {
               // create of sensor-msg already checked, maybe a methodstart returns no
               //  answer (sent later).
               /*// response-message available?
               if ( NULL == poResponseMessage )
                  poErrorMessage = new amt_tclServiceDataError( poServiceData, VDS_E_DISPATCH_ERROR );*/
            }
            else
            {
               // if poErrorMessage==NULL, there may be an alloc-error or a methodabort fails,
               //  cause method already finished.
               /*if ( NULL == poErrorMessage )
                  poErrorMessage = new amt_tclServiceDataError( poServiceData, VDS_E_DISPATCH_ERROR );*/
            }
         } // end if MsgDispatcher != NULL
         else
         {
            // MsgDispatcher not initialized, send error-message
            poErrorMessage = new amt_tclServiceDataError(
            poServiceData, ( tU16 )VDS_E_AILVDSENSOR_NOT_INITIALIZED );
         }

         if (  NULL == poErrorMessage
               && NULL != poResponseMessage )
         {
            // send response-message
            poResponseMessage->vSetSourceAppID( u16SourceAppId );
            poResponseMessage->vSetTargetAppID( u16TargetAppId );
            poResponseMessage->vSetServiceID  ( poServiceData->u16GetServiceID()   );
            poResponseMessage->vSetTargetSubID( poServiceData->u16GetSourceSubID() );
            poResponseMessage->vSetCmdCounter ( poServiceData->u16GetCmdCounter()  );
            poResponseMessage->vSetRegisterID ( poServiceData->u16GetRegisterID()  );
            poResponseMessage->vSetACT        ( poServiceData->u8GetACT()          );
            bPostMessage( poResponseMessage );
            delete poResponseMessage;
         }

         vCritSectionEnd();
      }
      else // if bCritSectionBegin
      {
         poErrorMessage = new amt_tclServiceDataError( poServiceData,(tU16)VDS_E_AILVDSENSOR_NOT_INITIALIZED );
      }

      if ( bSubscribersLocked )
      {
         vds_goMasterLock.vLeave();
      }

      if( NULL != poErrorMessage )
      {
         // send error-message
         poErrorMessage->vSetSourceAppID( u16SourceAppId );
         poErrorMessage->vSetTargetAppID( u16TargetAppId );
         poErrorMessage->vSetServiceID  ( poServiceData->u16GetServiceID()   );
         poErrorMessage->vSetTargetSubID( poServiceData->u16GetSourceSubID() );
         poErrorMessage->vSetCmdCounter ( poServiceData->u16GetCmdCounter()  );
         poErrorMessage->vSetRegisterID ( poServiceData->u16GetRegisterID()  );
         poErrorMessage->vSetACT        ( poServiceData->u8GetACT()          );
         try
         {
            bPostMessage( poErrorMessage );
         }
         catch(std::bad_alloc& ba)
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,"exception bad_alloc caught: %s",ba.what());
         }
         delete poErrorMessage;
      }
   } // end if poServiceData != NULL

   // else poServiceData==NULL: keine Ahnung wohin Fehlermeldung geschickt werden sollte!
}


#ifndef VARIANT_S_FTR_DISABLE_DIAGLOG
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Dispatch the message received from Diaglog (Server)
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vDispatchDiaglogMsg
(
//! (I) : pointer to message with ServiceId 'Sensor'
amt_tclServiceData *poServiceData
)
{
   tU16 u16FunctionId = poServiceData->u16GetFunctionID();

   // param-check
   if ( NULL != poServiceData )
   {
      /*if ( vds_goMasterLock.bEnter() )
         bSubscribersLocked = TRUE; // else no subscriber exists*/

      if( bCritSectionBegin() )
      {

         if( (NULL != poSensorMsgDispatcher) && (u16FunctionId == MIDW_DIAGLOGFI_C_U16_SENDNEXTTESTRESULT) )
         {
            poSensorMsgDispatcher->vDispatchDiaglogMsg ( poServiceData);
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
            "tclAilVDSensor:Invalid diaglog Function ID or invalid sensormsgdisptacher" );
         }
         
         vCritSectionEnd();
      }

      /*if ( bSubscribersLocked )
         vds_goMasterLock.vLeave();*/
   }
}
#endif

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Register Service for Diag Log
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void tclAilVDSensor::vRegisterDiagLog
(
//! None
void
)
{

   /*To avoid multiple Diaglog service registration.
   If VDSensor already register for the service Diaglog, then it will not 
   register for the service once again*/
   if((u16DiagLogRegisterID == AMT_C_U16_REGID_INVALID)&&(bRegisterAsyncRet == FALSE))
   {
      bRegisterAsyncRet = bRegisterAsync( CCA_C_U16_SRV_DIAGLOG, 1, 0, 0, CCA_C_U16_APP_DIAGLOG );
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Register Service for SPM.
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
#ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
void tclAilVDSensor::vRegisterSPM
(
//! None
void
)
{
   tBool bSuccess = bRegisterAsync( CCA_C_U16_SRV_SPM, 1, 0, 0, CCA_C_U16_APP_SPM );

   if ( FALSE == bSuccess )
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
      "tclAilVDSensor:Unable to trigger async registration at SPM service" );
   }
}
#endif
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Confirmation call to registration to a service from framework.
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vOnAsyncRegisterConf (tU16 u16RegisterId, tU16 /*u16ServerAppId*/, tU16 u16ServiceId, tU16 /*u16TargetSubId*/)
{
   switch(u16ServiceId)
   {
   case CCA_C_U16_SRV_DIAGLOG:
      {
         u16DiagLogRegisterID = u16RegisterId;
         if( u16DiagLogRegisterID == AMT_C_U16_REGID_INVALID )
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
            "tclAilVDSensor:Unable to register at Diaglog service" );
            bRegisterAsyncRet = FALSE;
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
            "tclAilVDSensor:Registered to diaglog with RegiterID = %d",
            u16DiagLogRegisterID );

            vds_goServiceClientManager.vNotifyRegistration( CCA_C_U16_SRV_DIAGLOG,
            u16DiagLogRegisterID );
            bisDiaglogServiceavailable = TRUE;  
         }
      }
      break;

      #ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
   case CCA_C_U16_SRV_SPM:
      {
         u16SPMRegisterID = u16RegisterId;
         if( u16SPMRegisterID == AMT_C_U16_REGID_INVALID )
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
            "tclAilVDSensor:Unable to register at SPM service" );
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
            "tclAilVDSensor:Registered to SPM with RegisterID = %d",
            u16SPMRegisterID );
            vds_goServiceClientManager.vNotifyRegistration( CCA_C_U16_SRV_SPM,u16SPMRegisterID );
         }
      }
      break;
      #endif

   default:
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         "tclAilVDSensor:ServiceId invalid/not expected");
      }
      break;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   handles transition from state initialized to given state
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vSwitchFromInitialized
(
//! (I) : switch to this application-state
tU32 u32NewAppState
)
{
   tBool bAck = FALSE;

   switch ( u32NewAppState )
   {
      // state: normal
   case AMT_C_U32_STATE_NORMAL:
      {

         // set interfaces to normal
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL );

         if ( ( tS32 )VDS_E_NO_ERROR == s32StartSensorThreads() )
         {

            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                       "tclAilVDSensor:all threads active at time(ms) %u",
                       OSAL_ClockGetElapsedTime() );
            if( poMsgGpsIf != NULL )
            {
               poMsgGpsIf->vSetGpsThreadEvent( VDS_GPS_EVENT_START );
            }

            #ifndef VARIANT_S_FTR_DISABLE_DIAGLOG
            vRegisterDiagLog();
            #endif

            #ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
            vRegisterSPM();
            #endif

            //Set the flag in tclGpsThread indicating the need to update
            //service availability. Once thread receives first record,
            //vdsensor's service availability status will be updated to spm.
            if( poMsgGpsIf )
               poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );

            #ifdef VARIANT_S_FTR_IERROR_MESSAGE
            vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_AVAILABLE );
            #endif
            
            bAck = TRUE;
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,
            "tclAilVDSensor:Error in vSwitchFromInitialized, couldn't start sensor-threads" );
         }
         break;
      }

      // state: diagnosis
      #ifndef VARIANT_S_FTR_DISABLE_DIAGNOSIS      
   case AMT_C_U32_STATE_DIAGNOSIS:
      {
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL_DIAG );

         if ( ( tS32 )VDS_E_NO_ERROR == s32StartSensorThreads() )
         {
            if( poMsgGpsIf != NULL )
            {
               poMsgGpsIf->vSetGpsThreadEvent( VDS_GPS_EVENT_START );
            }

            vRegisterDiagLog();

            #ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
            vRegisterSPM();
            #endif

            //Set the flag in tclGpsThread indicating the need to update
            //service availability. Once thread receives first record,
            //vdsensor's service availability status will be updated to spm.
            if( poMsgGpsIf )
               poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );

            #ifdef VARIANT_S_FTR_IERROR_MESSAGE
            vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_AVAILABLE );
            #endif
            bAck = TRUE;
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR, 
            "AilVDSensor:Error in vSwitchFromInitialized, couldn't start sensor-threads" );
         }
         break;
      }
      #endif

      // state: receive-ready
   case AMT_C_U32_STATE_RECEIVE_READY:
      // fall through
      // state: pause
   case AMT_C_U32_STATE_PAUSE:
      // fall through
      // state: off
   case AMT_C_U32_STATE_OFF:
      {
         // set interfaces to pending
         vSetInterfaceState( VDS_C_S32_STATE_PENDING );

         if ( ( tS32 )VDS_E_NO_ERROR != s32StartSensorThreads() )
         {
            vTraceMsg( 
            VDS_C_TRACELEVEL_ERROR, 
            "AilVDSensor:Error in vSwitchFromInitialized, couldn't start sensor-threads" );
         }

         bAck = TRUE;
         break;
      }
   default:
      {
         break;
      }
   }

   if ( bAck )
   {
      // store internal state
      u32AppState = u32NewAppState;
      /* give always an answer to the LPM */
      vAppStateChanged( u32AppState );
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                "AilVDSensor:FromInit:State change acknowledged at time(ms) %u",
                OSAL_ClockGetElapsedTime() );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL, 
                 "AilVDSensor:FromInit:Can't acknowledge state change" );
   }

}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   handles transition from state normal to given state
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vSwitchFromNormal
(
//! (I) : switch to this application-state
tU32 u32NewAppState
)
{
   tBool bAck = FALSE;

   switch ( u32NewAppState )
   {
      #ifndef VARIANT_S_FTR_DISABLE_DIAGNOSIS   
      // state: diagnosis
   case AMT_C_U32_STATE_DIAGNOSIS:
      {
         //Set the flag in tclGpsThread indicating the need to update
         //service availability. Once thread receives first record,
         //vdsensor's service availability status will be updated to spm.
         if( poMsgGpsIf )
            poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );

         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_AVAILABLE );
         #endif
         //set interfaces to normal+diag
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL_DIAG );
         tclSubscriberManager::vResetSubscribers();
         bAck = TRUE;
         break;
      }
      #endif
      
      // state: receive-ready
   case AMT_C_U32_STATE_RECEIVE_READY:
      // fall through
      // state: pause
   case AMT_C_U32_STATE_PAUSE:
      // fall through
      // state: off
   case AMT_C_U32_STATE_OFF:
      {
         vTraceMsg( VDS_C_TRACELEVEL_SYSTEM_MIN, "vSwitchFromNormal( AMT_C_U32_STATE_OFF )" );

         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSOR_LOCATION, AMT_C_U8_SVCSTATE_NOT_AVAILABLE );
         
         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_NOT_AVAILABLE );
         #endif
         
         // set interfaces to pending
         vSetInterfaceState( VDS_C_S32_STATE_PENDING );

         tclSubscriberManager::vResetSubscribers();

         if ( poMsgGpsIf != NULL )
         {
           poMsgGpsIf->vSetGpsThreadEvent( VDS_GPS_EVENT_STOP );
         }

         bAck = TRUE;
         break;
      }
   default:
      {
         break;
      }
   }

   if ( bAck )
   {
      // store internal state
      u32AppState = u32NewAppState;
      /* give always an answer to the LPM */
      vAppStateChanged( u32AppState );
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                 "AilVDSensor:FromNormal:State change acknowledged at time(ms) %u",
                 OSAL_ClockGetElapsedTime() );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                 "AilVDSensor:FromNormal:Can't acknowledge state change" );
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   handles transition from state diagnosis to given state
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vSwitchFromDiagnosis
(
//! (I) : switch to this application-state
tU32 u32NewAppState
)
{
   tBool bAck = FALSE;

   switch ( u32NewAppState )
   {
      //state: normal
   case AMT_C_U32_STATE_NORMAL:
      {
         //set interfaces to normal
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL );

         //Set the flag in tclGpsThread indicating the need to update
         //service availability. Once thread receives first record,
         //vdsensor's service availability status will be updated to spm.
         if( poMsgGpsIf )
            poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );
         
         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR, AMT_C_U8_SVCSTATE_AVAILABLE );
         #endif
         
         bAck = TRUE;
         break;
      }

      // state: receive-ready
   case AMT_C_U32_STATE_RECEIVE_READY:
      // fall through
      // state: pause
   case AMT_C_U32_STATE_PAUSE:
      // fall through
      // state: off
   case AMT_C_U32_STATE_OFF:
      {
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSOR_LOCATION, AMT_C_U8_SVCSTATE_NOT_AVAILABLE );
         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_NOT_AVAILABLE );
         // set interfaces to pending
         vSetInterfaceState( VDS_C_S32_STATE_PENDING );
         #endif

         bAck = TRUE;
         break;
      }
   default:
      {
         break;
      }
   }

   if ( bAck )
   {
      // store internal state
      u32AppState = u32NewAppState;
      /* give always an answer to the LPM */
      vAppStateChanged( u32AppState );
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                 "AilVDSensor:FromDiagnosis:State change acknowledged at time(ms) %u",
                 OSAL_ClockGetElapsedTime() );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                 "AilVDSensor:FromDiagnosis:Can't acknowledge state change" );
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   handles transition from state ReceiveReady to given state
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vSwitchFromReceiveReady
(
//! (I) : switch to this application-state
tU32 u32NewAppState
)
{
   tBool bAck = FALSE;

   switch ( u32NewAppState )
   {
      //state: normal
   case AMT_C_U32_STATE_NORMAL:
      {
         if( poMsgGpsIf != NULL )
         {
            poMsgGpsIf->vSetGpsThreadEvent( VDS_GPS_EVENT_START );
         }

         // set interfaces to normal
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL );

         //Set the flag in tclGpsThread indicating the need to update
         //service availability. Once thread receives first record,
         //vdsensor's service availability status will be updated to spm.
         if( poMsgGpsIf )
            poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );
         
         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR, AMT_C_U8_SVCSTATE_AVAILABLE );
         #endif

         bAck = TRUE;
         break;
      }

      #ifndef VARIANT_S_FTR_DISABLE_DIAGNOSIS	  
      // state: diagnosis
   case AMT_C_U32_STATE_DIAGNOSIS:
      {
         if ( poMsgGpsIf != NULL )
         {
            poMsgGpsIf->vSetGpsThreadEvent( VDS_GPS_EVENT_START );
         }

         // set interfaces to normal+diag
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL_DIAG );

         //Set the flag in tclGpsThread indicating the need to update
         //service availability. Once thread receives first record,
         //vdsensor's service availability status will be updated to spm.
         if( poMsgGpsIf )
            poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );
         
         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_AVAILABLE );
         #endif

         bAck = TRUE;
         break;
      }
      #endif

      // state: pause
   case AMT_C_U32_STATE_PAUSE:
      // fall through
      // state: off
   case AMT_C_U32_STATE_OFF:
      {
         bAck = TRUE;
         break;
      }
   default:
      {
         break;
      }
   }

   if ( bAck )
   {
      // store internal state
      u32AppState = u32NewAppState;
      /* give always an answer to the LPM */
      vAppStateChanged( u32AppState );
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                 "AilVDSensor:FromReceiveReady:State change acknowledged at time(ms) %u",
                 OSAL_ClockGetElapsedTime() );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                 "AilVDSensor:FromReceiveReady:Can't acknowledge state change");
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   handles transition from state pause to given state
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vSwitchFromPause
(
//! (I) :switch to this application-state
tU32 u32NewAppState
)
{
   tBool bAck = FALSE;

   switch ( u32NewAppState )
   {
      // state: normal
   case AMT_C_U32_STATE_NORMAL:
      {
         if( poMsgGpsIf != NULL )
         {
            poMsgGpsIf->vSetGpsThreadEvent( VDS_GPS_EVENT_START );
         }

         // set interfaces to normal
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL );

         //Set the flag in tclGpsThread indicating the need to update
         //service availability. Once thread receives first record,
         //vdsensor's service availability status will be updated to spm.
         if( poMsgGpsIf )
            poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );

         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_AVAILABLE );
         #endif
         
         bAck = TRUE;
         break;
      }

      #ifndef VARIANT_S_FTR_DISABLE_DIAGNOSIS
      // state: diagnosis
   case AMT_C_U32_STATE_DIAGNOSIS:
      {
         if ( poMsgGpsIf != NULL )
         {
            poMsgGpsIf->vSetGpsThreadEvent( VDS_GPS_EVENT_START );
         }

         // set interfaces to normal+diag
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL_DIAG );

         //Set the flag in tclGpsThread indicating the need to update
         //service availability. Once thread receives first record,
         //vdsensor's service availability status will be updated to spm.
         if( poMsgGpsIf )
            poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );

         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_AVAILABLE );
         #endif

         bAck = TRUE;
         break;
      }
      #endif

      // state: receive-ready
   case AMT_C_U32_STATE_RECEIVE_READY:
      // fall through
      // state: off
   case AMT_C_U32_STATE_OFF:
      {
         bAck = TRUE;
         break;
      }
   default:
      {
         break;
      }
   }

   if( bAck )
   {
      // store internal state
      u32AppState = u32NewAppState;
      /* give always an answer to the LPM */
      vAppStateChanged( u32AppState );
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                 "AilVDSensor:FromPause:State change acknowledged at time(ms) %u",
                 OSAL_ClockGetElapsedTime() );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                 "AilVDSensor:FromPause:Can't acknowledge state change" );
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   handles transition from state off to given state
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vSwitchFromOff
(
//! (I) :switch to this application-state
tU32 u32NewAppState
)
{
   tBool bAck = FALSE;

   switch ( u32NewAppState )
   {
      //state: normal
   case AMT_C_U32_STATE_NORMAL:
      {
         // set interfaces to normal
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL );

         if( poMsgGpsIf != NULL )
         {
            poMsgGpsIf->vSetGpsThreadEvent( VDS_GPS_EVENT_START );
         }

         //Set the flag in tclGpsThread indicating the need to update
         //service availability. Once thread receives first record,
         //vdsensor's service availability status will be updated to spm.
         if( poMsgGpsIf )
            poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );

         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_AVAILABLE );
         #endif

         bAck = TRUE;
         break;
      }
      
      #ifndef VARIANT_S_FTR_DISABLE_DIAGNOSIS	  
      // state: diagnosis
   case AMT_C_U32_STATE_DIAGNOSIS:
      {
         if( poMsgGpsIf != NULL )
         {
            poMsgGpsIf->vSetGpsThreadEvent( VDS_GPS_EVENT_START );
         }

         // set interfaces to normal+diag
         vSetInterfaceState( VDS_C_S32_STATE_NORMAL_DIAG );

         //Set the flag in tclGpsThread indicating the need to update
         //service availability. Once thread receives first record,
         //vdsensor's service availability status will be updated to spm.
         if( poMsgGpsIf )
            poMsgGpsIf->vSetGpsThreadFlags( enbInformVdsServiceAvail,true );

         #ifdef VARIANT_S_FTR_IERROR_MESSAGE
         vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSORS_IERROR,AMT_C_U8_SVCSTATE_AVAILABLE );
         #endif

         bAck = TRUE;
         break;
      }
      #endif

      // state: prepare-download
   case AMT_C_U32_STATE_PREPARE_DOWNLOAD:
      // fall through
      // state: receive-ready
   case AMT_C_U32_STATE_RECEIVE_READY:
      // fall through
      // state: pause
   case AMT_C_U32_STATE_PAUSE:
      {
         bAck = TRUE;
         break;
      }
   default:
      {
         break;
      }
   }

   if ( bAck )
   {
      // store internal state
      u32AppState = u32NewAppState;
      /* give always an answer to the LPM */
      vAppStateChanged( u32AppState );
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                 "AilVDSensor:FromOff:State change acknowledged at time(ms) %u",
                 OSAL_ClockGetElapsedTime() );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                 "AilVDSensor:FromOff:Can't acknowledge state change" );
   }

}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   sets new interface-state (normal or pending)
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vSetInterfaceState
(
//! (I) :new state (VDS_C_S32_STATE_NORMAL or VDS_C_S32_STATE_PENDING)
tS32 s32NewState
)
{
   if( NULL != poMsgGyroIf )
   {
      poMsgGyroIf->vSetState( s32NewState );
   }

   if( OSAL_NULL != poMsgAccIf )
   {
      poMsgAccIf->vSetState( s32NewState );
   }

   if( NULL != poMsgOdometerIf )
   {
      poMsgOdometerIf->vSetState( s32NewState );
   }

   if( NULL != poMsgGpsIf )
   {
      poMsgGpsIf->vSetState( s32NewState );
   }

   #ifdef VARIANT_S_FTR_IERROR_MESSAGE
   if( NULL != poMsgIerrorIf )
   {
      poMsgIerrorIf->vSetState( s32NewState );
   }
   #endif //VARIANT_S_FTR_IERROR_MESSAGE
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   locks access to critical sections.only when TRUE is returned, access
//!   is allowed
//! \return
//!   - \c  TRUE : access allowed
//!   - \c  FALSE : error, access is not allowed
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclAilVDSensor::bCritSectionBegin
(
//! None
)
{
   tBool bLocked = FALSE;

   if ( OSAL_C_INVALID_HANDLE != hVDSensorSemaphore )
   {
      if( OSAL_ERROR != OSAL_s32SemaphoreWait(hVDSensorSemaphore, OSAL_C_U32_INFINITE) )
      {
         bLocked = TRUE;
      }
   }

   return bLocked;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   unlocks access to critical sections.
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAilVDSensor::vCritSectionEnd
(
//! None
tVoid
)
{
   if ( OSAL_C_INVALID_HANDLE != hVDSensorSemaphore )
   {
      OSAL_s32SemaphorePost( hVDSensorSemaphore );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
      "AilVDSensor:VDSensor-Semaphore invalid");
   }
}

//!****************************************************************************
//! Functions which doesn't belongs to any class( both local and global scope )
//!****************************************************************************

//!***************** F U N C T I O N  H E A D E R *****************************
//! \brief   Sets services of vdsensor available.This function is called from
//!          VDS_GPS thread in the class tclGpsThread.
//! \param   void.
//! \return  void.
//! \author  Sanjay G (RBEI/ECF5)    \date    25.Apr.2016
//!****************************************************************************
tVoid vUpdateVdsServiceAvailability( tVoid )
{
   //Inform service availability to spm.
   tclAilVDSensor::poThisAppInterface->
           vServiceAvailabilityChanged( CCA_C_U16_SRV_SENSOR_LOCATION,
                                        AMT_C_U8_SVCSTATE_AVAILABLE   );
   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
              "AilVDSensor:service available at time(ms) %d",
              OSAL_ClockGetElapsedTime() );
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   This Function frames a TTFis buffer based on the sensor type.
//!   This buffer contains the names of actvie sensor devices
//! \return
//!   None
//  HISTORY:
// Date          |  Author                | MODIFICATION
// ----------------------------------------------------------------------------
//  13.08.2009   | sak9kor (RBEI/ECF1)    | XML generated FI has been adapted
//                                          instead manual generated FI
//******************************************************************************
static void vPrintSensoryTypeTraceMsg
(
//! (I) Activated Sensor type
tU32 u32SensorType
)
{
   tChar cTraceBuf[200] = {0};
   /*
   *If any new sensor devices adds in vds_types.h, please include
   *the new sensor in below mentioned array.
   */
   tCString pVDSdevices[] = 
   {
      "GPS",
      "GYRO",
      "ODO",
      "ABS",
      "ODO_FAKE",
      "ACC",
      "GNSS"
   };

   ( tVoid )OSAL_szStringConcat( cTraceBuf, "Sensor Types = " );

   //Here the intention is to print only GPS or GNSS.
   if( ((u32SensorType >> 0)&(0x01)) && 
         (u32SensorType >> 6)&(0x01) )
   {
      ( tVoid )OSAL_szStringConcat( cTraceBuf, pVDSdevices[6] );
      ( tVoid )OSAL_szStringConcat( cTraceBuf, " " );
   }
   else if( (u32SensorType >> 0)&(0x01) )
   {
      ( tVoid )OSAL_szStringConcat( cTraceBuf, pVDSdevices[0] );
      ( tVoid )OSAL_szStringConcat( cTraceBuf, " " );   
   }
   else
   {
      //Nothing.
   }

   //Print remaining(other than GPS and GNSS) sensor information.
   for ( tU32 i = 1; i < ( VDS_C_U32_ACTIVE_SENSORS - 1 ); i++ )
   {
      if ( ( u32SensorType >> i ) & ( 0x01 ) )
      {
         ( tVoid )OSAL_szStringConcat( cTraceBuf, pVDSdevices[i] );
         ( tVoid )OSAL_szStringConcat( cTraceBuf, " " );
      }
   }

   vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,"%s",cTraceBuf );
}


// Local Variables: ***
// Use 3 Spaces ***
// End: ***
