// *****************************************************************************
// * FILE:         tclAccThread.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition:Opens the accelerometer Device and
// *               provides an interface to read the accelerometer raw data
// * AUTHOR:       RBEI/ECF1-Sainath Kalpuri
// * COPYRIGHT:    (c) 2011 Robert Bosch
// * HISTORY:
// * 24.02.15 fsj2hi removed dead code, get hwinfo via ioctrl
// * 06.01.17 amg6kor Port to Gen4(x86_64)
// *****************************************************************************
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define  SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#include "sensor_fi_if.h"

#define VDS_S_IMPORT_INTERFACE_SENSOR_THREADS
#define VDS_S_IMPORT_INTERFACE_SENSOR_DEFINES
#define VDS_S_IMPORT_INTERFACE_TRACE
#define VDS_S_IMPORT_INTERFACE_MESSAGE_INTERFACES
#define VDS_S_IMPORT_INTERFACE_PERSISTENT_DATA
#include "vds_internal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define CCA_S_IMPORT_INTERFACE_GENERIC // CCA_C_U16_APP_SENSOR
#include "cca_if.h"
#define SCD_S_IMPORT_INTERFACE_GENERIC // scd_s32GetThreadConfiguration
#include "scd_if.h"

#include "regkeys.h"    // THREADNAME_C_STRING_SENSOR_VDS_ACC
#include "vds_hwinfo.h"

#define MIN_CYCLE_TIME (10 * 1000000) // 10ms = 100Hz
// Class tclAccThread
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Constructor for tclAccThread class
//! \return
//!   None
//  HISTORY:
// Date         |  Author                          | MODIFICATION
// ----------------------------------------------------------------------------
// 11-03-2011      RBEI/ECF1 Sainath K             Initial version
//******************************************************************************
tclAccThread::tclAccThread()
   :parAccData( OSAL_NULL ),
    u32NumOfAccData( 0 ),
    par3dAccStorage( OSAL_NULL ),
    pfs32Add3dAccList( OSAL_NULL ),
    poMsgAccIf( OSAL_NULL ),
    u32AccSensorCycleTimeInNS(50000000)
{

}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Destructor for tclAccThread class
//! \return
//!   None
//  HISTORY:
// Date         |  Author                          | MODIFICATION
// ----------------------------------------------------------------------------
// 11-03-2011      RBEI/ECF1 Sainath K             Initial version
//******************************************************************************
tclAccThread::~tclAccThread()
{
   tclAccThread::vThreadClose();

   parAccData = OSAL_NULL;
   par3dAccStorage = OSAL_NULL;
   poMsgAccIf = OSAL_NULL;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!     Opens the Accelerometer Device and gets the handle which can be used to
//!     get the accelerometer data
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCTHREAD_DEVICE_NOT_FOUND :   Device not found
//!   - \c VDS_E_ACCTHREAD_ALLOC_ERROR :  Allocation error
//!   - \c VDS_E_ACCTHREAD_THREADCREATE_FAILED : Thread create failed
//  HISTORY:
// Date         |  Author                          | MODIFICATION
// ----------------------------------------------------------------------------
// 11-03-2011      RBEI/ECF1 Sainath K             Initial version
// 24-05-2011      Sainath K (RBEI CM-AI/PJ-CF31)  Acc Thread has been created
//                                                 which is required to read the
//                                                 acc data and update the ringbuffer.
//******************************************************************************
tS32 tclAccThread::s32ThreadInit
(
   //! (I) : Pointer to Accmsgif parameter
   tclMsgAccIf    *poMsgAccIfParam,
   //! (I) : this is a pointer to the function, where sensor-data will be
   //! stored.  this function has to be called, when sensor-data is read.
   tS32( *pfs32Add3dAccListParam )( const sensor_fi_tcl_3dAccData*, tU32 )
   )
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( VDS_C_S32_THREAD_UNINITIALIZED == s32GetInternalState() )
   {
      this->poMsgAccIf = poMsgAccIfParam;
      this->pfs32Add3dAccList = pfs32Add3dAccListParam;

      if ( OSAL_NULL == pfs32Add3dAccList )
      {
         s32RetVal = VDS_E_INVALID_PARAMETER;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AccThread:Init:Invalid param" );
      }

      // open Acc-device
      SensorFd = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READONLY ) ;

      if(SensorFd == OSAL_ERROR)
      {
         s32RetVal = VDS_E_ACCTHREAD_DEVICE_NOT_FOUND;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "AccThread:Init:Device open failed error = 0x%x",OSAL_u32ErrorCode() );
      }
      // configure acc-device
    
      // Getcycle time ioctrl  (as used for gyro)
      if ( VDS_E_NO_ERROR == s32RetVal )
      {
         tS32 s32AccCycleTime = 0;
         tS32 s32AccCycleRetVal = OSAL_s32IOControl( SensorFd,
                                                     OSAL_C_S32_IOCTRL_ACC_GETCYCLETIME,
                                                     (tLong)&s32AccCycleTime );

         //check return value of OSAL_s32IOControl and cycle time value, must be
         //!= 0, because of division with it in tclMsgAccIf
         
         if( ( OSAL_ERROR != s32AccCycleRetVal ) && 
              ( s32AccCycleTime >= MIN_CYCLE_TIME ) ) //lint !e845 !e774 PQM_authorized_multi_549 PQM_authorized_multi_550 */
         { 
            u32AccSensorCycleTimeInNS = (tU32)s32AccCycleTime;
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
            "AccThread:OSAL_C_S32_IOCTRL_ACC_GETCYCLETIME SUCCESS s32AccCycleRetVal - %d :s32AccCycleTime - %d"
            ,s32AccCycleRetVal,s32AccCycleTime);
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                       "AccThread:Unable to get acc cycletime, use init default value %d",
                       u32AccSensorCycleTimeInNS );
         }
      }
 
      tclSystemInformation::rInfo.rAcc.r3dAccHWInfo.SampleRate =
         (1000000000u / u32AccSensorCycleTimeInNS);
      
      OSAL_trIOCtrlHwInfo rIOCtrlHwInfo = {0};

      if ( OSAL_ERROR ==
           OSAL_s32IOControl( SensorFd,
                              OSAL_C_S32_IOCTRL_ACC_GET_HW_INFO,
                              (tLong)&rIOCtrlHwInfo ))

      {
          vTraceMsg( VDS_C_TRACELEVEL_FATAL, "Error while calling OSAL_C_S32_IOCTRL_ACC_GET_HW_INFO");
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                    "sucess OSAL_C_S32_IOCTRL_ACC_GET_HW_INFO");

         vVdsCopyOsalToFiHwInfo(
            rIOCtrlHwInfo,
            tclSystemInformation::rInfo.rAcc.r3dAccHWInfo.MountAngles,
            tclSystemInformation::rInfo.rAcc.r3dAccHWInfo.RAxes,
            tclSystemInformation::rInfo.rAcc.r3dAccHWInfo.SAxes,
            tclSystemInformation::rInfo.rAcc.r3dAccHWInfo.TAxes );

         vTraceMsg( VDS_C_TRACELEVEL_DATA,"f32DriftOffset= %f",rIOCtrlHwInfo.rRAxes.f32DriftOffset);

      }
      
      u32NumOfAccData = tclSystemInformation::rInfo.rAcc.u16BlockSize;

      parAccData = new OSAL_trIOCtrlAccData[ u32NumOfAccData ];
      // storage for sending data to Acc-if
      par3dAccStorage = new sensor_fi_tcl_3dAccData[ u32NumOfAccData ];

      // if error occured...
      if ( OSAL_NULL == par3dAccStorage )
      {
         s32RetVal = VDS_E_ACCTHREAD_ALLOC_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AccThread:Init:Alloc error" );
      }

      // *** create thread-attribute ***
      OSAL_trThreadAttribute rThreadAttribute = {0};
      rThreadAttribute.szName  =   const_cast<tString>(THREADNAME_C_STRING_SENSOR_VDS_ACC);
      rThreadAttribute.pfEntry =   ( OSAL_tpfThreadEntry ) tclAccThread::vThreadMainWrapper;
      rThreadAttribute.pvArg   = ( tPVoid ) this;

      // get thread-configuration
      if ( OSAL_ERROR
           ==
           scd_s32GetThreadConfiguration( CCA_C_U16_APP_SENSOR,
                                          THREADNAME_C_STRING_SENSOR_VDS_ACC,
                                          &rThreadAttribute.u32Priority,
                                          &rThreadAttribute.s32StackSize ) )
      {
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AccThread:Init:Can't get thread config" );
         s32RetVal = VDS_E_ACCTHREAD_THREADCREATE_FAILED;
      }
      // *** create sensor-thread ***
      if ( s32RetVal == VDS_E_NO_ERROR )
      {
         if ( OSAL_ERROR
              ==
              ( SensorThreadId = OSAL_ThreadCreate( &rThreadAttribute ) ) )
         {
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AccThread:Init:Thread create failed" );
            s32RetVal = VDS_E_ACCTHREAD_THREADCREATE_FAILED;
         }
      }
      // does error occured?
      if ( VDS_E_NO_ERROR == s32RetVal )
      {
         vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
      }
      else
      {
         vThreadClose();
      }
   }
   else
   {
      s32RetVal = VDS_E_ACCTHREAD_NOT_UNINITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_FATAL, "AccThread:Init:Already initialized" );
   }
   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   inherited from class tclSensorThread.when thread is started, set state RUNNING.
//!   receive acc-data from acc-device when internal state is set to STOP,
//!   leave loop and set state INITIALIZED.
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAccThread::vThreadMain
(
   //! void
   tPVoid
   )
{
   tU32 u32TempNumOfAccData = 0;

   OSAL_trIOCtrlAccData *parTempAccData = OSAL_NULL;
   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTempAccData);

   tU32 u32ErrorCount;
   if ( s32GetInternalState() == VDS_C_S32_THREAD_INITIALIZED
        &&
        OSAL_ERROR != SensorFd
        &&
        0 < u32NumOfAccData
        &&
        OSAL_NULL != parAccData )
   {
      // thread is activated
      vSetInternalState( VDS_C_S32_THREAD_RUNNING );

      // read available data from acc-device
      if ( OSAL_ERROR
           ==
           OSAL_s32IOControl( SensorFd,
                              OSAL_C_S32_IOCTRL_ACC_GETCNT,
                              ( tLong ) &u32TempNumOfAccData ) )
      {
         // read u32NumOfAccData datas (=VDS_C_U16_ACCTHREAD_BLOCKREADSIZE)
         u32TempNumOfAccData = u32NumOfAccData;
      }

      // at the first time, there would be more acc-data than u32NumOfAccData
      if ( u32TempNumOfAccData > u32NumOfAccData )
      {

         parTempAccData =
            new OSAL_trIOCtrlAccData[ u32TempNumOfAccData ];

         if ( OSAL_NULL == parTempAccData )
         {
            // creation of temp buffer failed, so use parAccData-buffer
            parTempAccData = parAccData;
            u32TempNumOfAccData = u32NumOfAccData;
         }
      }
      else
      {
         // buffer is big enough, use it
         parTempAccData = parAccData;
      }
      // read and store acc data
      s32ReadAccData( parTempAccData, u32TempNumOfAccData );

      // delete temp-buffer
      if ( parTempAccData != parAccData )
      {
         delete [] parTempAccData;
         parTempAccData = OSAL_NULL;
      }

      u32ErrorCount = 0;

      do
      {
         // consistence check
         if ( OSAL_ERROR != SensorFd
              &&
              OSAL_NULL != parAccData )
         {
            tS32 s32Ret = s32ReadAccData( parAccData, u32NumOfAccData );

            if ( s32Ret == VDS_E_ACCTHREAD_IOREAD_ERROR )
            {

            }
            else
            {
               u32ErrorCount = 0;
            }
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "AccThread: IO or Memory uninit." );

            vSetInternalState( VDS_C_S32_THREAD_ERROR );
         }
      }
      while ( s32GetInternalState() == VDS_C_S32_THREAD_RUNNING );

      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "AccThread:Exit main Loop!");

      // set new state
      if ( s32GetInternalState() == VDS_C_S32_THREAD_STOP )
      {
         vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
      }
      else
      {
         // close handles, ...
         vThreadClose();
         vSetInternalState( VDS_C_S32_THREAD_UNINITIALIZED );
      }
   }
   else
   {
      /* the below mentioned code is added to resolve the issue where VDSensor
        is taking more than 5 seconds to change the state INITIALIZED to NORMAL
        in Non China Variants.  This probel is occuring because when ACC thread
        is being activated from tclAilVDSesnor at the end it will call
        s32ThreadStart function in tclSensorThread.cpp and this function will
        check whether the ACC thread state is running,if it is not in RUNNING
        state then this function will check for the state as RUNNING with 50ms
        timer for multiple times which is equivalent to 5 seconds. To resolve
        the above mentioned problem,the thread sets its state as RUNNING though
        the ACC driver open is failed .*/

      vSetInternalState( VDS_C_S32_THREAD_RUNNING );
      for(;s32GetInternalState() == VDS_C_S32_THREAD_RUNNING;)
      {
         if(OSAL_s32ThreadWait( 100 )!= OSAL_OK)
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "tclAccThread: thread wait failed" );
         }
      }
      if ( s32GetInternalState() == VDS_C_S32_THREAD_STOP )
      {
         vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
      }
      else
      {
         // close handles, ...
         vThreadClose();
         vSetInternalState( VDS_C_S32_THREAD_UNINITIALIZED );
      }
   }

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTempAccData);

   OSAL_vThreadExit();

   // else no error-handling needed, state is uninitialized
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   inherited from class tclSensorThread.closes handles, deletes semaphores,
//!   deletes allocated memory
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAccThread::vThreadClose
(
   //! void
   )
{
   // if acc-data exists, delete
   if ( OSAL_NULL != parAccData )
   {
      delete [] parAccData;
      parAccData = OSAL_NULL;
   }

   // if acc-storage exists, delete
   if ( OSAL_NULL != par3dAccStorage )
   {
      delete [] par3dAccStorage;
      par3dAccStorage = OSAL_NULL;
   }


   u32NumOfAccData = 0;
   // semaphore and device will be deleted in tclSensorThread
   // reset function-pointer
   pfs32Add3dAccList = OSAL_NULL;
   tclSensorThread::vThreadClose();
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   reads acc-data from acc-device,
//! \return
//!   Number of Elements read
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCTHREAD_STORAGE_ERROR     : Storage error
//!   - \c VDS_E_ACCTHREAD_IOREAD_ERROR :  IO Read error
//!   - \c VDS_E_ACCTHREAD_NOT_INITIALIZED : Not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclAccThread::s32ReadAccData
(
   //! (O) : buffer for received ACC-data
   //OSAL_trIOCtrlAccData *parIOCtrlAccData,
   tVoid *parIOCtrlAccData,
   //! (I) : max. number of acc-data to be stored in buffer
   tU32 u32NumOfElem
   )
{
//   tU32   ulIdx;
   tS32 s32NumOfElemRead = 0;

   OSAL_trIOCtrlAccData *pAccData =
      (OSAL_trIOCtrlAccData *)parIOCtrlAccData;

   // param-check
   if ( OSAL_NULL != pAccData
        &&
        0 < u32NumOfElem
        &&
        10000 > u32NumOfElem )
   {
      // consistence-check
      if ( OSAL_ERROR != SensorFd )
      {
         tS32 s32AccDataRead =
            OSAL_s32IORead( SensorFd,
                            ( tPS8 )pAccData,
                            u32NumOfElem * (tU32) sizeof( OSAL_trIOCtrlAccData ) );

         s32NumOfElemRead =
            s32AccDataRead / ( tS32 )sizeof( OSAL_trIOCtrlAccData );



         if ( ( s32NumOfElemRead > 0 )
              &&
              ( ( tU32 )s32NumOfElemRead <= u32NumOfElem ) )
         {
            if ( VDS_E_NO_ERROR
                 !=
                 s32StoreAccData( (tVoid *)pAccData,
                                  ( tU32 )s32NumOfElemRead ) )
            {
               s32NumOfElemRead = VDS_E_ACCTHREAD_STORAGE_ERROR;
            }
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR, "AccThread: IORead failed" );
            s32NumOfElemRead = VDS_E_ACCTHREAD_IOREAD_ERROR;
         }
      }
      else
      {
         s32NumOfElemRead = VDS_E_ACCTHREAD_NOT_INITIALIZED;
      }
   }
   else
   {
      s32NumOfElemRead = VDS_E_INVALID_PARAMETER;
   }

   return s32NumOfElemRead;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   static wrapper for main-loop of instance
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclAccThread::vThreadMainWrapper
(
   //! (I) :  pointer to instance
   tPVoid pvThreadArg
   )
{
   // param-check
   if ( OSAL_NULL != pvThreadArg )
   {
      tclAccThread *poAccThread = static_cast<tclAccThread *>( pvThreadArg );
      // Can not be NULL
      poAccThread->vThreadMain( OSAL_NULL );
   }
   else
      vTraceMsg( VDS_C_TRACELEVEL_ERROR, "AccThread: ThreadArgument NULL" );
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   returns acc-sensor-cycle-time\n
//! \return
//!      Sensor cycle time
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tU32 tclAccThread::u32GetAccCycleTime
(
   //! None
   tVoid
   )
{
   return u32AccSensorCycleTimeInNS;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   sends received acc-data to tclMsgAccIf
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCTHREAD_ALLOC_ERROR :   Allocation error
//!   - \c VDS_E_ACCTHREAD_STORAGE_ERROR :  Storage Error
//!   - \c VDS_E_ACCTHREAD_NOT_INITIALIZED : Not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclAccThread::s32StoreAccData
(
   //! (I) : buffer with received acc-data
   tVoid *parIOCtrlAccData,
   //! (I) : number of acc-data in buffer
   tU32 u32NumOfElem
   )
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
   sensor_fi_tcl_3dAccData *parTemp3dAccStorage = OSAL_NULL;
   tU32 u32AccIndex = 0;

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTemp3dAccStorage);

   OSAL_trIOCtrlAccData *pAccData = (OSAL_trIOCtrlAccData *)parIOCtrlAccData;
   // param-check
   if ( OSAL_NULL != pAccData
        &&
        u32NumOfElem   > 0 )
   {
      // consistence check
      if ( OSAL_NULL != pfs32Add3dAccList
           &&
           OSAL_NULL != par3dAccStorage

         )
      {
         // if too much data, generate temp storage (expected for the first time)
         if ( u32NumOfElem > u32NumOfAccData )
         {

            parTemp3dAccStorage = new sensor_fi_tcl_3dAccData[u32NumOfElem];

            if ( OSAL_NULL == parTemp3dAccStorage )
            {
               vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                          "AccThread: s32Store3dAccData: alloc error" );

               s32RetVal = VDS_E_ACCTHREAD_ALLOC_ERROR;
            }


         }
         else
         {
            parTemp3dAccStorage = par3dAccStorage;
         }
         if (OSAL_NULL != parTemp3dAccStorage)
         {
            // copy acc-device-format to acc-vdsensor-format
            for ( u32AccIndex = 0; u32AccIndex < u32NumOfElem; u32AccIndex++ )
            {
               // copy first 3d data plain to the 3d Buffer
               parTemp3dAccStorage[u32AccIndex].TimeStamp3dAcc =
                  pAccData[u32AccIndex].u32TimeStamp;

               parTemp3dAccStorage[u32AccIndex].RVal =
                  pAccData[u32AccIndex].u16Acc_x;

               parTemp3dAccStorage[u32AccIndex].SVal =
                  pAccData[u32AccIndex].u16Acc_y;

               parTemp3dAccStorage[u32AccIndex].TVal =
                  pAccData[u32AccIndex].u16Acc_z;

               if ( pAccData[ u32AccIndex ].u16ErrorCounter > 0 )
               {
                  parTemp3dAccStorage[u32AccIndex].RStatus.enType =
                     sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR;

                  parTemp3dAccStorage[u32AccIndex].SStatus.enType =
                     sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR;

                  parTemp3dAccStorage[u32AccIndex].TStatus.enType =
                     sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR;
               }
               else
               {
                  parTemp3dAccStorage[u32AccIndex].RStatus.enType =
                     sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_NORMAL;

                  parTemp3dAccStorage[u32AccIndex].SStatus.enType =
                     sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_NORMAL;

                  parTemp3dAccStorage[u32AccIndex].TStatus.enType =
                     sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_NORMAL;
               }
            }
            // send acc-data
            if ( VDS_E_NO_ERROR
                 !=
                 pfs32Add3dAccList( parTemp3dAccStorage, u32NumOfElem ) )
            {
               s32RetVal = VDS_E_ACCTHREAD_STORAGE_ERROR;
            }

            // delete 3D temp buffer
            if ( parTemp3dAccStorage != par3dAccStorage )
            {
               delete [] parTemp3dAccStorage;
               parTemp3dAccStorage = OSAL_NULL;
            }
         }
      }
      else
      {
         s32RetVal = VDS_E_ACCTHREAD_NOT_INITIALIZED;
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTemp3dAccStorage);

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//! \brief
//!       Start Acc self test.
//! \return
//!   - \c TRUE  : Self test finished successfully.
//!   - \c FALSE : Unable to start self test.
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclAccThread::bStartSelftest
(
   //! None
   tS32 *ps32SelfTestResult
   )
{
   tBool bRetVal = FALSE;
   tS32 s32SelfTestResult = -1;

   if( OSAL_OK == OSAL_s32IOControl( SensorFd,
                                     OSAL_C_S32_IOCTRL_ACC_SELF_TEST,
                                     (tLong)&s32SelfTestResult ) )
   {
      bRetVal = TRUE;
      *ps32SelfTestResult = s32SelfTestResult;
   }

   return bRetVal;
}
//EOF
