#include <iostream>
#include <fstream>

#include "gtest/gtest.h"

#include "ShellCmd.h"

TEST(test1, simpleexample){
    EXPECT_EQ(1, 1);
}

TEST(Cmd, FullPathEcho){
    ShellCmd cmd = ShellCmd("/bin/echo");
    int res;

    cmd.addArgs(std::string("World"));

    res = cmd.exec();
    EXPECT_EQ(res, 0);
}

TEST(Cmd, SimpleEcho){
    ShellCmd cmd = ShellCmd("echo");
    int res;

    cmd.addArgs(std::string("World"));

    res = cmd.exec();
    EXPECT_EQ(res, 0);
}

TEST(Cmd, EchoWithLongParamList){
    ShellCmd cmd = ShellCmd("echo");
    int res, i;

    for(i = 0; i < 100; i++){
        cmd.addArgs(std::string("test"));
    }

    res = cmd.exec();
    EXPECT_EQ(res, 0);
}

TEST(Cmd, EchoWithLongParamString){
    ShellCmd cmd = ShellCmd("echo");
    int res;

    cmd.addArgs(std::string("test -l 10 asdf --help -p /path/for/test"));

    res = cmd.exec();
    EXPECT_EQ(res, 0);
}

TEST(Cmd, FullPathLs){
    ShellCmd cmd = ShellCmd("/bin/ls");
    int res;

    cmd.addArgs(std::string("-l -a"));

    res = cmd.exec();
    EXPECT_EQ(res, 0);
}

TEST(Cmd, SimpleLs){
    ShellCmd cmd = ShellCmd("ls");
    int res;

    res = cmd.exec();
    EXPECT_EQ(res, 0);
}

TEST(Cmd, SimpleEchoFileRedirect){
    ShellCmd cmd = ShellCmd("/bin/echo");
    std::ifstream outputfile;
    int res;
    std::string content;

    cmd.addArgs(std::string("World"));
    cmd.setOutputFile(std::string("/tmp/echo.log"));

    res = cmd.exec();
    EXPECT_EQ(res, 0);

    outputfile.open("/tmp/echo.log", std::ios::in);

    EXPECT_EQ(outputfile.is_open(), true);

    std::getline(outputfile, content);

    EXPECT_STREQ(content.c_str(), "World");
}

TEST(Cmd, ErrmemRedirect){
    ShellCmd cmd = ShellCmd("/bin/echo");
    std::ifstream outputfile;
    int res;
    std::string content;

    cmd.addArgs(std::string("World"));
    cmd.setOutputFile(std::string("/dev/errmem"));

    res = cmd.exec();
    EXPECT_EQ(res, 0);
}

TEST(Cmd, SimpleEchoPipeRedirect){
    ShellCmd cmd = ShellCmd("/bin/echo");
    int res;
    std::string content;

    cmd.addArgs(std::string("World"));
    cmd.setOutputPipe(true);

    res = cmd.exec();
    EXPECT_EQ(res, 0);

    content = cmd.getCommandOutput();

    EXPECT_STREQ(content.c_str(), "World\n");
}

TEST(Cmd, EchoFailingRedirect){
    ShellCmd cmd = ShellCmd("echo");
    int res;

    cmd.setOutputFile(std::string("/no/way/this/path/exists/i/dare/you"));

    res = cmd.exec();
    /* See man ls returns 1 if a small problem occured */
    EXPECT_EQ(res, 0);
}

TEST(Cmd, Inexistant){
    ShellCmd cmd = ShellCmd("fakecommand");
    int res;

    res = cmd.exec();
    EXPECT_EQ(res, 127);
}

TEST(Cmd, NotExecutable){
    ShellCmd cmd = ShellCmd("/dev/null");
    int res;

    res = cmd.exec();
    EXPECT_EQ(res, 126);
}

TEST(Cmd, ListInexistentFolder){
    ShellCmd cmd = ShellCmd("ls");
    int res;

    cmd.addArgs("-l");
    cmd.addArgs("/no/way/this/path/exists/i/dare/you");

    res = cmd.exec();
    /* See man ls returns 1 if a small problem occured */
    EXPECT_EQ(res, 1);
}

TEST(Cmd, BackgroundTask){
    ShellCmd cmd = ShellCmd("cat");
    ShellCmd ps("ps");
    int res;
    pid_t pid;

    cmd.setBackgroundTask(true);

    res = cmd.exec();
    EXPECT_EQ(res, 0);

    pid = cmd.getRunningTaskPid();

    ps.addArgs(std::to_string(pid));
    res = ps.exec();

    EXPECT_EQ(res, 0);

    kill(pid, SIGKILL);
}

TEST(Cmd, GetFullCommandLine){
    ShellCmd cmd = ShellCmd("ls");
    std::string expected = "ls: -l /no/way/this/path/exists/i/dare/you";

    cmd.addArgs("-l");
    cmd.addArgs("/no/way/this/path/exists/i/dare/you");

    /* See man ls returns 1 if a small problem occured */
    EXPECT_EQ(expected, cmd.getFullCommandLine());
}

TEST(Sanitize, isSignal){
    for (int sig = SIGHUP; sig <= 64; sig++){
        EXPECT_TRUE(isSignal(std::to_string(sig)));
    }

    EXPECT_TRUE(isSignal(std::to_string(SIGSYS+1)));
    EXPECT_FALSE(isSignal(std::to_string(-1)));
    EXPECT_FALSE(isSignal(std::string("flsdakfjaö")));
    EXPECT_FALSE(isPid(std::string("143flsdakfjaö")));
}

TEST(Sanitize, isValidPid){

    EXPECT_TRUE(isPid(std::to_string(1)));
    EXPECT_TRUE(isPid(std::to_string(32768)));

    EXPECT_FALSE(isPid(std::to_string(-1)));
    EXPECT_FALSE(isPid(std::string("flsdakfjaö")));
    EXPECT_FALSE(isPid(std::string("143flsdakfjaö")));
}

TEST(Sanitize, isServiceName){

    EXPECT_TRUE(isServiceName(std::string("startup.service")));
    EXPECT_TRUE(isServiceName(std::string("startup_test.service")));
    EXPECT_TRUE(isServiceName(std::string("startup-test.service")));
    EXPECT_TRUE(isServiceName(std::string("startup")));
    EXPECT_FALSE(isServiceName(std::string("startup not allowed")));
    EXPECT_FALSE(isServiceName(std::string("sta/r/tup")));
    EXPECT_FALSE(isServiceName(std::string("star?tup")));
}

TEST(Sanitize, isPath){

    EXPECT_TRUE(isPath(std::string("/valid/path.xml")));
    EXPECT_TRUE(isPath(std::string("/target/dir")));
    EXPECT_TRUE(isPath(std::string("/numbers1/are/fine0")));
    EXPECT_TRUE(isPath(std::string("singlefile")));
    EXPECT_TRUE(isPath(std::string("single_file.txt")));
    EXPECT_TRUE(isPath(std::string("single-file.txt")));

    EXPECT_FALSE(isPath(std::string("$^*()&=")));
}

TEST(Sanitize, isTaskName){

    EXPECT_TRUE(isPath(std::string("/valid/path.xml")));
    EXPECT_TRUE(isPath(std::string("/target/dir")));
    EXPECT_TRUE(isPath(std::string("/numbers1/are/fine0")));
    EXPECT_TRUE(isPath(std::string("singlefile")));
    EXPECT_TRUE(isPath(std::string("single_file.txt")));
    EXPECT_TRUE(isPath(std::string("single-file.txt")));

    EXPECT_FALSE(isPath(std::string("$^*()&=")));
    /* Make sure injection command no longer works */
    EXPECT_FALSE(isPath(std::string("bash\";touch${IFS}/tmp/test.log;echo\"")));
}

TEST(Sanitize, stripNonASCII) {
    std::string goodStr("/just/a/path");
    std::string badStr("\xFF\xF1\xF2\xF3");

    stripNonASCII(goodStr);
    stripNonASCII(badStr);

    EXPECT_STREQ("/just/a/path", goodStr.c_str());
    EXPECT_STREQ("", badStr.c_str());
}

TEST(Sanitize, splitString) {
    std::string goodStr("arg1 arg2");
    std::string arg1, arg2;

    splitStringAtSpace(goodStr, arg1, arg2);

    EXPECT_STREQ("arg1", arg1.c_str());
    EXPECT_STREQ("arg2", arg2.c_str());
}
