#include <iostream>
#include <fstream>
#include <string>
#include <unistd.h>

#include <errno.h>
#include "gtest/gtest.h"
#include "gmock/gmock.h"

#include "RootDaemonHelper.h"
#include "RootDaemonTestUtils.h"
#include "ShellCmd.h"

typedef enum
{
   TRIGGER_TIME_TRACKER,
   GET_RUNNING_THREADS,
   GET_THREAD_INFO,
   GET_PROCESS_INFO_BY_NAME,
   GET_PROCESS_INFO_BY_ID,
   SEND_SIGNAL_TO_PROCESS_VIA_SYSTEMD,
   SEND_SIGNAL_TO_PROCESS,
   SEND_SIGNAL_TO_ALL_PROCESSES,
   GET_PROCESS_HIGH_WATER_MARKS,
   UNMOUNT_PARTITIONS,
   GET_CURRENT_PROCESSES,
   KILL_PROCESS,
   SYSTEMCTL_START_UNIT,
   SYSTEMCTL_STOP_UNIT,
   START_PROCESS_VIA_GDB,
   NUMBER_OF_LCM_ROOT_COMMANDS // Last entry
} tenRootDaemonCommands;

TEST(LcmClientTest, GetCurrentProcesses){
    CmdData data;
    int cmdnum, res;
    std::string args("");
    std::string errmem_dump;
    std::string pattern("PID");
    std::size_t found;

    const char *clientName = "lcm";
    cmdnum = GET_CURRENT_PROCESSES;

    res = erase_errmem();
    EXPECT_EQ(res, 0);

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);

    errmem_dump = dump_errmem();

    found = errmem_dump.find(pattern);
    EXPECT_NE(found, std::string::npos);
}

TEST(LcmClientTest, GetRunningThreads){
    CmdData data;
    int cmdnum, res;
    std::string logfile("/tmp/_spm_task_info.log");
    std::string args("");

    if (file_exists(logfile)) {
        unlink(logfile.c_str());
    }

    const char *clientName = "lcm";
    cmdnum = GET_RUNNING_THREADS;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
    EXPECT_TRUE(file_exists(logfile));
}

TEST(LcmClientTest, GetThreadInfo){
    CmdData data;
    int cmdnum, res;
    pthread_t thread_id;
    std::string logfile("/tmp/_spm_task_info.log");
    std::string args("");

    if (file_exists(logfile)) {
        unlink(logfile.c_str());
    }

    args += std::to_string(thread_id);

    const char *clientName = "lcm";
    cmdnum = GET_THREAD_INFO;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
    EXPECT_TRUE(file_exists(logfile));
}

TEST(LcmClientTest, GetProcessInfoByName){
    CmdData data;
    int cmdnum, res;
    std::string logfile("/tmp/_spm_task_info.log");
    std::string args("rootdaemon_gtest");

    if (file_exists(logfile)) {
        unlink(logfile.c_str());
    }

    const char *clientName = "lcm";
    cmdnum = GET_PROCESS_INFO_BY_NAME;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
    EXPECT_TRUE(file_exists(logfile));
}

TEST(LcmClientTest, GetProcessInfoById){
    CmdData data;
    int cmdnum, res;
    int pid;
    std::string logfile("/tmp/_spm_task_info.log");
    std::string args("");

    if (file_exists(logfile)) {
        unlink(logfile.c_str());
    }

    pid = getpid();

    args += std::to_string(pid);

    const char *clientName = "lcm";
    cmdnum = GET_PROCESS_INFO_BY_ID;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
    EXPECT_TRUE(file_exists(logfile));
}


TEST(LcmClientTest, DISABLED_GetProcessHighWaterMarks){
    CmdData data;
    int cmdnum, res;
    std::string logfile("/tmp/_spm_hwm_info.log");
    std::string args("");

    if (file_exists(logfile)) {
        unlink(logfile.c_str());
    }

    const char *clientName = "lcm";
    cmdnum = GET_PROCESS_HIGH_WATER_MARKS;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
    EXPECT_TRUE(file_exists(logfile));
}


TEST(LcmClientTest, SystemctlStartUnit){
    CmdData data;
    int cmdnum, res;
    std::string args("rootdaemon_gtest.service");

    const char *clientName = "lcm";
    cmdnum = SYSTEMCTL_START_UNIT;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
}

TEST(LcmClientTest, SystemctlStopUnit){
    CmdData data;
    int cmdnum, res;
    std::string args("rootdaemon_gtest.service");

    const char *clientName = "lcm";
    cmdnum = SYSTEMCTL_STOP_UNIT;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
}

TEST(LcmClientTest, StartProcessViaGdb){
    CmdData data;
    int cmdnum, res;
    std::string args("");

    const char *clientName = "lcm";
    cmdnum = START_PROCESS_VIA_GDB;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
}

TEST(LcmClientTest, DISABLED_KillProcess){
    CmdData data;
    ShellCmd ps("ps");
    ShellCmd cmd("cat");
    int cmdnum, pid, status, res;
    std::string args("");

    cmd.setBackgroundTask(true);
    res = cmd.exec();
    EXPECT_EQ(res, 0);

    pid = cmd.getRunningTaskPid();

    const char *clientName = "lcm";
    cmdnum = KILL_PROCESS;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);

    ps.addArgs("-p");
    ps.addArgs(std::to_string(pid));
    res = ps.exec();
    EXPECT_NE(res, 0);

    status = wait_for_process(pid);
    EXPECT_EQ(WSTOPSIG(status), SIGTERM);
}

TEST(LcmClientTest, DISABLED_UnmountPartitions){
    CmdData data;
    int cmdnum, res;
    std::string args("");

    const char *clientName = "lcm";
    cmdnum = UNMOUNT_PARTITIONS;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
}

TEST(LcmClientTest, TriggerTimeTrackerAll){
    CmdData data;
    int cmdnum, res;
    std::string logfile("/tmp/_spm_top.log");
    std::string args("");

    if (file_exists(logfile)) {
        unlink(logfile.c_str());
    }

    const char *clientName = "lcm";
    cmdnum = TRIGGER_TIME_TRACKER;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
    EXPECT_TRUE(file_exists(logfile));
}

TEST(LcmClientTest, TriggerTimeTrackerProcesses){
    CmdData data;
    int cmdnum, res;
    std::string args("processes");

    const char *clientName = "lcm";
    cmdnum = TRIGGER_TIME_TRACKER;

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
}


TEST(LcmClientTest, SendSignalToProcess){
    CmdData data;
    int cmdnum, pid, status, res;
    std::string args("");

    pid = fork_loop();

    ASSERT_NE(pid, -1);

    const char *clientName = "lcm";
    cmdnum = SEND_SIGNAL_TO_PROCESS;

    args += std::to_string(SIGKILL);
    args += std::string(" ");
    args += std::to_string(pid);

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);

    status = wait_for_process(pid);
    EXPECT_TRUE(WIFSIGNALED(status));
    EXPECT_EQ(WTERMSIG(status), SIGKILL);
}

TEST(LcmClientTest, SendSignalToProcessViaSystemd){
    CmdData data;
    int cmdnum, res;
    ShellCmd cmd("systemctl");
    std::string args("");

    cmd.addArgs("start rootdaemon_gtest");
    res = cmd.exec();

    EXPECT_EQ(res, 0);

    const char *clientName = "lcm";
    cmdnum = SEND_SIGNAL_TO_PROCESS_VIA_SYSTEMD;

    args += std::string("9 rootdaemon_gtest");

    data = RootDaemonHelper::performRootOp(clientName,
                                          cmdnum,
                                          args);

    res = std::stoi(data.message);
    EXPECT_EQ(data.errorNo, ERR_NONE);
    EXPECT_EQ(res, 0);
}
