/*
 * ShellCmd.h
 *
 * This defines the standard API for setting up a call to a shell command
 * in a safe manner.
 *
 *  Created on May 7 2020
 *      Author: Maxime Chemin
 */

#ifndef _SHELL_CMD_H_
#define _SHELL_CMD_H_

#include <iostream>
#include <vector>

/**
 * Standard Shell errors that a shell would return. This is used to imitate
 * the standard shell behavior.
 */

typedef enum
{
    CMD_NOT_EXECUTABLE = 126,
    CMD_NOT_FOUND = 127
} ShellExitCodes;

/**
 * When opening a pipe you always open a read and write channel. This is a
 * helper enum to make it clear inside the code which one we are using.
 */
typedef enum
{
    READ_PIPE = 0,
    WRITE_PIPE = 1,
    MAX_PIPE
} PipeChannels;

/**
 * Helper class to execute command with behavior similar to a shell
 *
 * The goal of this class is to be safer than the similar system() call by
 * default. The approach behind the class will be to rely on the exec functions
 * to implement a similar environment to system() with some limitations.
 */

class ShellCmd {
private:
    std::string progname;
    std::vector<std::string> arguments;

    bool outputToPipe;
    bool isBackgroundTask;

    std::string outputpath;
    int pipeLink[MAX_PIPE];
    pid_t cmdPid;

    static char * const default_env[];

    /**
     * Wait for the child process
     *
     * @param pid wait for the child process with this pid
     * @return int status of the child process
     */
    int waitForChild(int pid);

    /**
     * Redirect current process stdout and stderr to file
     *
     * @param string path to file where output needs to be saved
     */
    void redirectOutputToFile(const std::string &filepath);

    /**
     * Redirect current processes stdout and stderr to a pipe
     *
     */
    void redirectOutputToPipe();

public:
    ShellCmd(std::string program);
    ~ShellCmd();

    /**
     * Add all arguments inside provided string
     *
     * @param string Arguments to be added
     * @throws bad_alloc if not enough memory available to create the command
     * line array.
     */
    void addArgs(std::string args);

    /**
     * Set the name of the output file
     *
     * This will activate redirection of standard and error output to the file.
     *
     * @param string name of file (full path recommended)
     */
    void setOutputFile(std::string filepath);

    /**
     * Activates sending the output of the exec command to a pipe.
     *
     * This will redirect the output of the exec command to a pipe. This allows
     * the calling program to retrieve the output directly without requiring a
     * file.
     *
     * @param bool true if the output should be written to a pipe, false to
     * just send it to standard output as before.
     */
    void setOutputPipe(bool state);

    /**
     * Sets the command as a background task, meaning the command will be run
     * and the current process will not wait for it to finish.
     *
     * @param bool true if the task should be run in background and false
     * otherwise.
     */
    void setBackgroundTask(bool state);

    /**
     * Get full command-line
     *
     * String format would follow this format: PROGNAME: ARG1 ARG2 ...
     *
     * @return string full command line including program name and arguments
     */
    std::string getFullCommandLine();

    /**
     * Get pid of the currently running task (only makes sense if the task runs
     * in the background)
     *
     * @return pid_t Pid of the currently executing task
     */
    pid_t getRunningTaskPid();

    /**
     * Get previously run command output
     *
     * If you have activated the redirection to a pipe you can retrieve the
     * command output with this function.
     *
     * @throws bad_alloc If not enough memory available to copy the program output.
     * @return string Output from the previous command
     *
     */
    std::string getCommandOutput();

    /**
     * Execute command with all previously specific arguments
     */
    int exec();
};

/**
 * Takes a string that contains a number and checks if it is a valid signal
 * number.
 *
 * @param String representing a signal number
 * @return true if it is a valid signal and false otherwise
 */

bool isSignal(const std::string &number);

/**
 * Takes a string and checks if it follows the correct format for a systemd
 * service name.
 *
 * @param string name of a systemd name
 * @return true if the syntax of the service name is valid and false otherwise
 */
bool isServiceName(const std::string &service_name);

/**
 * Takes a string representing a number and checks that it is potential pid.
 *
 * @param string String representing a pid number
 * @return true if it is a potential pid number and false otherwise
 */
bool isPid(const std::string &process_id);

/**
 * Takes a string representing a path to a file and checks if it is valid
 *
 * @param string Path to a file
 * @return true if it is a valid path syntax
 */

bool isPath(const std::string &path);

/**
 * Takes a string representing a task name and checks if it is valid
 *
 * @param string Task name
 * @return true if it is a valid task name syntax
 */

bool isTaskName(const std::string &task_name);

/**
 * Takes a string and removes any non ASCII characters from it
 *
 * @param string reference to string to be stripped
 */

void stripNonASCII(std::string &str);

/**
 * Split string in two at first space
 *
 * @param string String to be split
 * @param string reference to string that will hold the first chunk
 * @param string reference to string that will hold the second chunk
 */
void splitStringAtSpace(const std::string &strIn, std::string &first, std::string &sec);

#endif
