/*************************************************************************************************
 FILE:           scd_ssc_main.cpp
 PROJECT:        GM3Gen MY17 - Security
 SW-COMPONENT:   SCD SSC - SmartCard System Security Controller
 DESCRIPTION:    It is a file that has the main function - the start of the code execution
 AUTHOR:         sva1fh
 COPYRIGHT:      RBNA

 HISTORY:
 Date            | Author                | Modification
 05.10.2015      | sva1fh                | initial version

 ************************************************************************************************/
#include "scd_ssc_main.h"

/*************************************************************************************************
	FUNCTION:         main
	PARAMETERS:       int - number of arguments, array of strings - debug arguments
	RETURN VALUE:     0 - OK, anything else - NOT OK
	DESCRIPTION:      this function does the followings:

				1. It initializes four objects on heap memory
				    - scd_ssc_tclClientHandler_ALD is an ALD client that communicates with ALD
				      on D-Bus by implementing methods for that purpose.
				    - scd_ssc_tclSmartcardHandler is a CodeMeter client that communicates with
				      CodeMeter or a smartcard.
				    - scd_ssc_tclProperty is or behaves like a container or a storage unit or
				      a pool of data. In other words, it contains constant data and also data
				      generated at run-time
                    - scd_ssc_tclUtility is class that performs a work on a data such as:
                      composing a sequence of data (for example. response), testing the data
                      for consistency/correctness and similar functionality

                2. it implements the logic that enables unlocking or locking a HMIM device

                3. During the development and testing phase a few tests can be enabled - will be
                   removed completely later

	HISTORY:
	Date         | Author                | Modification
    05.10.2015   | sva1fh                | initial version

*************************************************************************************************/
int main(int argc, char *argv[])
{
	if(argc > 2 && (strcmp(logging, argv[1]) == 0) && strcmp(term, argv[2]) == 0)
	{
		logging_enabled = true;
		term_enabled = true;
	}
	if((argc == 2) && (strcmp(logging, argv[1]) == 0)){logging_enabled = true;}
	if((argc == 2) && (strcmp(term, argv[1]) == 0)){term_enabled = true;}

	/* initialize the objects on heap                                                           */
	scd_ssc_tclLogger  *m_poLogger = new scd_ssc_tclLogger(logging_enabled, term_enabled);
	if(m_poLogger == NULL){ std::cerr << "\nm_poCLogger is NULL\n" << std::endl;
		exit(1);
	}

	scd_ssc_tclProperty *m_poProperty = new scd_ssc_tclProperty();
	if(m_poProperty == NULL){ std::cerr << "\nm_poProperty is NULL\n"  << std::endl;
		exit(1);
	}

	scd_ssc_tclUtility  *m_poUtility = new scd_ssc_tclUtility();
	if(m_poUtility == NULL){ std::cerr << "\nm_poUtility is NULL\n"  << std::endl;
		exit(1);
	}

	scd_ssc_tclSmartcardHandler *m_poSmartcardHandler = new scd_ssc_tclSmartcardHandler();
	if(m_poSmartcardHandler == NULL){ std::cerr << "\nm_poSmartcardHandler is NULL\n"  << std::endl;
		exit(1);
	}

	scd_ssc_tclClientHandler_ALD  *m_poClientHandlerALD = new scd_ssc_tclClientHandler_ALD();
	if(m_poClientHandlerALD == NULL){ std::cerr << "\nm_poClientHandlerALD is NULL\n" << std::endl;
		exit(1);
	}

	/*                              WORKING TEST FUNCTIONS                                      */
	//scd_ssc_tclTest test;
	//m_poLogger->log("running scd_ssc test functions");
	//test.run_smartcard_discovery_test(m_poSmartcardHandler, m_poLogger);
	//test.run_ald_challenge_test(m_poClientHandlerALD, m_poProperty, m_poLogger);
	//test.run_ald_get_status_test(m_poClientHandlerALD, m_poProperty, m_poLogger);
	/*                           NOT WORKING TEST FUNCTIONS                                     */
	//test.run_codemeter_initialization_test(m_poSmartcardHandler, m_poProperty, m_poClientHandlerALD, m_poLogger);
	//test.run_ald_smartcard_challenge_response_test(m_poClientHandlerALD, m_poProperty, m_poUtility,
	//				m_poSmartcardHandler, m_poLogger);

	/*                               END OF TEST FUNCTIONS                                      */

	/* -----------------------------------  IMPLEMENTATION  ----------------------------------- */
	/* are we running on DLinitRAMFS or regular Linux                                           */
	if(m_poUtility->getMode(m_poProperty))
	{
		m_poLogger->log("running on DLinitRAMFS");
		/* if we are in here then we are in DLinitRAMFS                                         */
		if(unlockIt(m_poClientHandlerALD, m_poProperty, m_poSmartcardHandler, m_poLogger))
			m_poLogger->log("scd_ssc_main -> main() - the system is unlocked");
		else
			m_poLogger->log("error in scd_ssc_main -> main() - unlocking not performed");
	}
	else /*                      regular Linux mode of operation                                */
	{
		char *current_ch = NULL;
		m_poLogger->log("running on Linux");
		/* test the system lock level - if the system is unlocked - then lock it                */
		if(m_poClientHandlerALD->getStatus(m_poProperty, m_poLogger))
		{	/* before locking or unlocking get the current lock level                           */
			current_ch = (char*)m_poProperty->getCurrentLockLevel();
			std::string current_str(current_ch);
			m_poLogger->log("scd_ssc_main -> main() - the current lock level: " + current_str);
			int current = atoi(current_ch);
			int max_unlocked = m_poProperty->getMaxUnlockedLevel();
			/* lock it if current unlock level is gt 0 and current is le max_unlocked (50)      */
			//if((current > m_poProperty->getLockedLevel()) && (current <= m_poProperty->getMaxUnlockedLevel()))
			/* for testing only - current gt 50                                                 */
			if(current >  max_unlocked)
			{
				if(lockIt(m_poClientHandlerALD, m_poProperty, m_poLogger))
				{	/* get the lock level after it is locked                                    */
					current_ch = (char*)m_poProperty->getNewLockLevel();
					std::string current_str(current_ch);
					m_poLogger->log("scd_ssc_main -> main() - the system is locked, level: " + current_str);
				}
				else
					m_poLogger->log("error in scd_ssc_main -> main()  - locking not performed");
			}
			/* otherwise unlock it                                                              */
			else
			{
				if(unlockIt(m_poClientHandlerALD, m_poProperty, m_poSmartcardHandler, m_poLogger))
				{	/* get the current lock level after it is unlocked                          */
					current_ch = (char*)m_poProperty->getNewLockLevel();
					std::string current_str(current_ch);
					m_poLogger->log("scd_ssc_main -> main() - the system is unlocked, level: " + current_str);

				}
				else
					m_poLogger->log("error in scd_ssc_main -> main() - unlocking not performed");
		    }
		}
		else /* lock it if the current unlock level unknown                                     */
		{
			if(lockIt(m_poClientHandlerALD, m_poProperty, m_poLogger))
			{
				current_ch = (char*)m_poProperty->getNewLockLevel();
				std::string current_str(current_ch);
				m_poLogger->log("scd_ssc_main -> main() - current unlock level unknown - "
											"        the system is locked, level: " + current_str);
			}
			else
				m_poLogger->log("error in scd_ssc_main -> main() - current unlock level unknown - "
						"        locking not performed");
		}
	}
	delete(m_poClientHandlerALD);
	delete(m_poSmartcardHandler);
	delete(m_poUtility);
	delete(m_poProperty);
	delete(m_poLogger);

	return 0;
}   /*                             END OF MAIN                                                  */



/*************************************************************************************************
	FUNCTION: lockIt
	PARAMETERS: scd_ssc_tclClientHandler_ALD, scd_ssc_tclProperty
	RETURN VALUE: bool, true - success, false - failed
	DESCRIPTION: this function locks the system - unlock level 0

	HISTORY:
    Date        | Author                | Modification
    10.07.2015  | sva1fh                | initial version

*************************************************************************************************/
bool lockIt(scd_ssc_tclClientHandler_ALD * m_poClientHandlerALD, scd_ssc_tclProperty *m_poProperty,
     scd_ssc_tclLogger  *m_poLogger)
{
	bool status = false;
	status = m_poClientHandlerALD->lockDevice(m_poProperty, m_poLogger);

	if(status)
		m_poLogger->log("scd_ssc_main -> lockIt() - succeeded");
	else
		m_poLogger->log("scd_ssc_main -> lockIt() - failed");

	return status;
}

/*************************************************************************************************
	FUNCTION: unlockIt
	PARAMETERS: scd_ssc_tclClientHandler_ALD, scd_ssc_tclProperty, scd_ssc_tclSmartcardHandler
	RETURN VALUE:  bool, true - success, false - failed
	DESCRIPTION: this function unlocks the system

	HISTORY:
    Date        | Author                | Modification
    10.07.2015  | sva1fh                | initial version

*************************************************************************************************/
bool unlockIt(scd_ssc_tclClientHandler_ALD * m_poClientHandlerALD, scd_ssc_tclProperty *m_poProperty,
		scd_ssc_tclSmartcardHandler *m_poSmartcardHandler, scd_ssc_tclLogger  *m_poLogger)
{
	bool status = false;
	/* get challenge (96 bytes) from ALD and test it                                            */
	if(m_poClientHandlerALD->challenge(m_poProperty, m_poLogger)){
		m_poLogger->log("scd_ssc_main -> unlockIt() -> challenge() - succeeded");
		/* set serial, lock level, public key, sign it, and generated response                  */
		if(m_poSmartcardHandler->initialize(m_poProperty, m_poLogger)){
			m_poLogger->log("scd_ssc_main -> unlockIt() -> initialize() - succeeded");
			/* send response (1400 bytes) to ALD  and test it                                   */
			if(m_poClientHandlerALD->response(m_poProperty, m_poLogger)){
				/* if we get here then the system is unlocked                                   */
				status = true;
				m_poLogger->log("scd_ssc_main -> unlockIt() -> response() - succeeded");
			}else{m_poLogger->log("scd_ssc_main -> unlockIt() -> response() - failed");}
		}else{m_poLogger->log("scd_ssc_main -> unlockIt() -> initialize() - failed");}
	}else{m_poLogger->log("scd_ssc_main -> unlockIt() -> challenge() - failed");}

	return status;
}



