/**
 * CCodeMeter Class
 * ----------------------------------------------------------------------------
 * Author:	Dietmar Wegner
 *			Mourad Goumrhar
 *			Mirko Lochau
 *			Michael Reinhold
 *
 * This class encapsulates all CodeMeter related functionality. It is able to
 * establish a connection to a specific CmContainer, read the required data
 * and use cryptography functions of the CmContainer for signature generation.
 * For completeness, PKCS v1.5 padding is implemented in this class as well.
 *
 * Features to be implemented:
 * - UnitCounter
 * - Expiration Data
 *
 * Version History
 * ----------------------------------------------------------------------------
 * 06.11.2014	Initial Version
 * 11.11.2014	Refactoring done, commamnd line binary product introduced
 * 05.08.2015   added, setProductFirmCode, a member function to this class - by sva1fh
 * 
 */


#include "CCodeMeter.h"

#include <string.h>
#include <stdlib.h>
#include <cstdio>
#include <iostream>
#include <string>

/**
 * Constructor of CCodeMeter Class
 * 
 * return void
 */

CCodeMeter::CCodeMeter(CMULONG firmcode, CMULONG prodcode)

{
	// initialize with zero...
	memset(&m_cmaccess, 0, sizeof(CMACCESS2));
	memset(&m_cmboxinfo, 0, sizeof(CMBOXINFO));
	memset(&m_publicKey, 0, sizeof(m_publicKey));
	memset(&m_expirationTime, 0, sizeof(m_expirationTime));

	m_hcmse			= 0;
	m_nBoxes		= 0;
	m_SerialNumber	= 0;
	m_MajorVersion	= 0;
	m_MinorVersion	= 0;
	m_UnitCounter	= 0;
	m_name			= "";
	m_nMode			= 0;

	m_firmCode		= firmcode;
	m_prodCode		= prodcode;

	CmStaticLibOnInit();
}

/**
 * Destructor of CCodeMeter class
 *
 * return void
 */
CCodeMeter::~CCodeMeter()
{
	if (m_hcmse) 
	{
		CmRelease(m_hcmse);
		CmStaticLibOnExit();
	}
	else
	{

	}
}

void CCodeMeter::setProductFirmCode(unsigned int prodcode, unsigned int firmcode)
{
	this->m_firmCode = firmcode;
	this->m_prodCode = prodcode;
}

/**
 * Initialize the CmContainer
 * Establish a connection, check if exactly one CmContainer matching to the given parameter
 * is connected and read the required data from the CmContainer.
 *l
 * return int - 0 for successl
 */
int CCodeMeter::initialize(scd_ssc_tclLogger  *m_poLogger)
{
	scd_ssc_tclUtility  poUtility;
	// set parameters to access the CmContainer
	//m_cmaccess.mflCtrl			= CM_ACCESS_NOUSERLIMIT;	// needs no license for each CmAccess() call. The License Quantity will not be decreased in this mode.
	//m_cmaccess.mflCtrl = CM_ACCESS_SUBSYSTEM;
	m_cmaccess.mulFirmCode		= m_firmCode;				// set the firmcode we got from CAPL
	m_cmaccess.mulProductCode	= m_prodCode;				// set the product code we got from CAPL
	m_poLogger->log("entering CCodeMeter -> initialize() ...");

	// Access local subsystem CM_ACCESS_LOCAL

	m_hcmse = CmAccess2(CM_ACCESS_LOCAL, &m_cmaccess);

	if (0 == m_hcmse)
	{
		std::string product(poUtility.longToString(m_cmaccess.mulProductCode));
		std::string firm(poUtility.longToString(m_cmaccess.mulFirmCode));
		std::string error(poUtility.intToString(CmGetLastErrorCode()));

		m_poLogger->log("CCodeMeter -> initialize() -> CmAccess2() - failed - CmConatiner with productCode: " +
				         product + " and firmCode: " + firm + " doesn't exist - error code: " + error);
		return CmGetLastErrorCode();
	}
	else
	{

		// !! assuming that we have exactly one CmContainer connected now !!
		// get SerialNumber and Major/Minor Version from the one CmContainer ...
		if (0 == CmGetInfo(m_hcmse, CM_GEI_BOXINFO, &m_cmboxinfo, sizeof(CMBOXINFO)))
		{
			std::string error(poUtility.intToString(CmGetLastErrorCode()));
			m_poLogger->log("CCodeMeter -> initialize() -> CmGetInfo(CM_GEI_BOXINFO) - failed - error code: " + error);
			return CmGetLastErrorCode();
		}
		m_SerialNumber		= m_cmboxinfo.mulSerialNumber;
		m_MajorVersion		= m_cmboxinfo.mbMajorVersion;
		m_MinorVersion		= m_cmboxinfo.mbMinorVersion;
		
		// get SmartCard PublicKey from ExtProtected data #0 and #1
		int iNum = CmGetInfo(m_hcmse,CM_GEI_ENTRYDATA,NULL,0);	// get number of bytes...
		int iCount = iNum / sizeof(CMENTRYDATA);				// calculate number of entries so determine how large the CMENTRYDATA object needs to be
		CMENTRYDATA *cmentrydata = new CMENTRYDATA[iCount];		// create a new CMENTRYDATA object that holds all the data
	
		iNum = CmGetInfo(m_hcmse,CM_GEI_ENTRYDATA,cmentrydata,iCount*sizeof(CMENTRYDATA));
		if (0 == iNum) // error occured
		{
			std::string error(poUtility.intToString(CmGetLastErrorCode()));
			m_poLogger->log("CCodeMeter -> initialize() -> CmGetInfo(CM_GEI_ENTRYDATA) - failed - error code: " + error);
			return CmGetLastErrorCode();
		}

		for (int i = 0; i < iCount; i++)
		{
			int extType = cmentrydata[i].mflCtrl >> 16;		// what kind of ExtProtectedData is it?
			int len = cmentrydata[i].mcbData;
			switch(cmentrydata[i].mflCtrl & CM_GF_PIO_MASK)
			{
				case CM_GF_EXTPROTDATA:
					if (extType == 0)
					{	// Copy public key from Extended Protected Data #0 to m_chal
						memcpy(&m_publicKey[0],cmentrydata[i].mabData,len);
					}
					if (extType == 1)
					{	// Copy public key from Extended Protected Data #1 to m_chal
						memcpy(&m_publicKey[128],cmentrydata[i].mabData,len);	//TODO: check the size of the public key!
					}
					break;
			}
		}

		CMBOXINFO	cmBoxInfo[5];				// We expect just one box (CmContainer) to be connected!
		const int	cbBoxEntry=100;			// why is this limited?
		CMBOXENTRY2	cmBoxEntry[cbBoxEntry];	//

		CmRelease(m_hcmse);

		memset(&m_cmaccess, 0, sizeof(CMACCESS2));
		m_cmaccess.mflCtrl = CM_ACCESS_SUBSYSTEM;
		m_cmaccess.mulFirmCode		= m_firmCode;				// set the firmcode we got from CAPL
		m_cmaccess.mulProductCode	= m_prodCode;				// set the product code we got from CAPL

		// Access local subsystem CM_ACCESS_LOCAL

		m_hcmse = CmAccess2(CM_ACCESS_LOCAL, &m_cmaccess);

		// All entries of all connected CmContainers available at m_hcmse will be returned --> just ONE expected!
		int nNumEntries = CmGetBoxContents2(m_hcmse,CM_GBC_ALLENTRIES,m_firmCode,&cmBoxInfo[0],cmBoxEntry,cbBoxEntry);

		// only one box is expected to be connected!
		if((0 == nNumEntries) || (cbBoxEntry < nNumEntries))
		{
			// From WiBu API Guide: No entry was found and an error code is set or Number of retrieved elements (cbBoxEntry) is smaller than the number of found entries
			// From WiBu API Guide: Depending on flCtrl for example CMERROR_BOX_NOT_FOUND or CMERROR_ENTRY_NOT_FOUND 
			//#ifdef _DEBUG
			//	m_log->error("Error in CmGetBoxContents(CM_GBC_ALLENTRIES): %d", CmGetLastErrorCode());
			//	m_log->error("Error in CmGetBoxContents(CM_GBC_ALLENTRIES): nNumEntries: %d ", nNumEntries);
			//#endif
			std::string error(poUtility.intToString(CmGetLastErrorCode()));
			m_poLogger->log("CCodeMeter -> initialize() -> CmGetBoxContents2() - failed - error code: " + error);
			return -1;
		}

		for (int i = 0; i < 5; i++)
		{
			if (cmBoxEntry[i].mulFirmCode == m_firmCode)
			{
				if (cmBoxEntry[i].mulProductCode == m_prodCode)
				{
					m_UnitCounter=cmBoxEntry[i].mulUnitCounter;

					// read out the Firm Text field ("Robert Bosch Gmbh Car Multimedia")
					for (int j=0; j < CM_MAX_STRING_LEN; j++)
					{
						sFirmItemText[j]=(char)cmBoxEntry[i].mausFirmItemText[j];
					}
					
					m_name = std::string(sFirmItemText);

					memset(&m_expirationTime,0,sizeof(CMTIME));
					m_expirationTime=cmBoxEntry[i].mcmExpirationTime;
				}
			}
		}
	}
	CmRelease(m_hcmse);
	m_poLogger->log("exiting CCodeMeter -> initialize()");
	return 0;
}


/**
 * Set the seed that we got from CAPL...
 *
 * return void
 */
int CCodeMeter::sign(unsigned char *data, unsigned int length, unsigned char *signature, scd_ssc_tclLogger  *m_poLogger)
{
	scd_ssc_tclUtility  poUtility;
	m_poLogger->log("entering CCodeMeter -> sign() ... ");

	memset(&m_cmaccess, 0, sizeof(CMACCESS2));
	m_cmaccess.mflCtrl = CM_ACCESS_NOUSERLIMIT;	// needs no license for each CmAccess() call. The License Quantity will not be decreased in this mode.
	m_cmaccess.mulFirmCode		= m_firmCode;				// set the firmcode we got from CAPL
	m_cmaccess.mulProductCode	= m_prodCode;				// set the product code we got from CAPL

	// Access local subsystem CM_ACCESS_LOCAL

	m_hcmse = CmAccess2(CM_ACCESS_LOCAL, &m_cmaccess);
	//FILE * out;
	// Step 1: calculate SHA256 Hash
	CmCalculateDigest(data,length,m_sha256Hash,CM_DIGEST_LEN);

	//#ifdef _DEBUG
		// store the SHA256 Hash that has a length of 32 bytes
		//out=fopen("1_SHA256.dat","wb");
		//fwrite(m_sha256Hash,1,CM_DIGEST_LEN,out);
		//fclose(out);
	//#endif

	// Step 2: Add padding since SHA256 Hashes are just 32 bytes long and we need 256 bytes
	unsigned char hashWithPadding[256];
	addPadding(CM_DIGEST_LEN, m_sha256Hash, &hashWithPadding[0]);

	//#ifdef _DEBUG
		// store the SHA256 Hash with padding now -> 256 bytes length
		//out=fopen("2_SHA256_with_padding.dat","wb");
		//fwrite(&hashWithPadding[0],1,256,out);
		//fclose(out);
	//#endif

	// Step 3: Send the 256 bytes (Hash+padding) to the CmContainer for RSA 2048 signature generation
	CMCRYPT2 cmCrypt;	// create CmCrypt object
	memset(&cmCrypt, 0, sizeof(cmCrypt));	// initialize with 0
	
	// copy hashWithPadding to the signature return parameter ... which is then encrypted by CmCrypt2 later on
	memcpy(&signature[0], &hashWithPadding[0], SIGNATURE_SIZE);

	// set encryption parameters
	cmCrypt.mcmBaseCrypt.mflCtrl |= CM_CRYPT_SECRETDATA;	// use the key in the secret data
	cmCrypt.mcmBaseCrypt.mflCtrl |= CM_CRYPT_RSA;			// use RSA
	cmCrypt.mcmBaseCrypt.mulKeyExtType = 130;				// start with secret data #130
	//cmCrypt.mcmBaseCrypt.mulEncryptionCodeOptions = 1;
	cmCrypt.mcmBaseCrypt.mulEncryptionCodeOptions |= CM_CRYPT_UCCHECK;
	cmCrypt.mcmBaseCrypt.mulEncryptionCodeOptions |= CM_CRYPT_ATCHECK;
	cmCrypt.mcmBaseCrypt.mulEncryptionCodeOptions |= CM_CRYPT_ETCHECK;
	cmCrypt.mcmBaseCrypt.mulEncryptionCodeOptions |= CM_CRYPT_SAUNLIMITED;
//	cmCrypt.mcmBaseCrypt.mulEncryptionCodeOptions |= CM_CRYPT_FACDECREMENT;   //decrement counter
	
	int nSignLength=16;
	
	unsigned char sign[nSignLength];
	memset(sign, 0, nSignLength);
	memcpy(&sign[0], &hashWithPadding[0], nSignLength);
	
	reverseArray(sign,nSignLength, m_poLogger);
	
	//out=fopen("3_Hash_reversed.dat","wb");
	//fwrite(&sign[0],1,nSignLength,out);
	//fclose(out);
		
	if (0 == CmCrypt2(m_hcmse, CM_CRYPT_DIRECT_ENC, &cmCrypt, sign, nSignLength))
	{
		std::string error(poUtility.intToString(CmGetLastErrorCode()));
		m_poLogger->log("CCodeMeter -> sign() -> CmCrypt2() - failed - error code: " + error);
		m_poLogger->log("exiting CCodeMeter -> sign()");
		return CmGetLastErrorCode();
	}
	else
	{
		//out=fopen("4_signature.dat","wb");
		//fwrite(&sign[0],1,nSignLength,out);
		//fclose(out);
		// signature successfully calculated
		//CmStaticLibOnExit();
		reverseArray(sign,nSignLength, m_poLogger);
		
		//out=fopen("5_signature_reversed.dat","wb");
		//fwrite(&sign[0],1,nSignLength,out);
		//fclose(out);
		
		memcpy(&signature[0], &sign[0], nSignLength);
		m_poLogger->log("exiting CCodeMeter -> sign()");
		return 0;
	}
}

/**
 * Change endiannes
 * params:
 *		data: data buffer to reverse (in)
 *		length: buffer size (in)
 * return void
*/
void CCodeMeter::reverseArray(unsigned char *data, unsigned int length, scd_ssc_tclLogger  *m_poLogger)
{
	m_poLogger->log("entering CCodeMeter -> reverseArray() ... ");
	unsigned char *start, *end;

	for ( start = (unsigned char *)data, end = start + length - 1; start < end; --end, ++start )
	{
		unsigned char tmp = *start;
		*start = *end;
		*end = tmp;
	}
	m_poLogger->log("exiting CCodeMeter -> reverseArray()");
}

/*
 * Add PKCS v1.5 Padding to SHA 256 hash
 * return void
 */
void CCodeMeter::addPadding(int hashlen, unsigned char *hash, unsigned char *sig)
{
	unsigned char *p;
	size_t nb_pad;

	// required for PKCS v1.5 Padding
	static const unsigned char oid[9] = {0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01}; // sha256

	p = sig;
	nb_pad = SIGNATURE_SIZE - 3 - 10 - sizeof(oid) - hashlen;

	*p++ = 0;
	*p++ = 1;
	memset( p, 0xFF, nb_pad );
	p += nb_pad;
	*p++ = 0;

	*p++ = 0x30;
	*p++ = (unsigned char) ( 0x08 + sizeof(oid) + hashlen ); 
	*p++ = 0x30; 
	*p++ = (unsigned char) ( 0x04 + sizeof(oid) ); 
	*p++ = 0x06; 
	*p++ = sizeof(oid);
	
	memcpy(p, oid, sizeof(oid));
	p += sizeof(oid);

	*p++ = 0x05;
	*p++ = 0x00;
	*p++ = 0x04;
	*p++ = hashlen;
	memcpy( p, hash, hashlen );
}


















