/**
* \file sdc_op_common.h
*
* \brief libSDC: Definition of functions etc. required by convenience and
* non convenience operation API - <b>should not be included directly!</b>
*
* Application should not include this header directly!
*
* \author Christoph Gellner (cgellner@de.adit-jv.com)
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

/**
 * \ingroup generic
 * \defgroup operations Cryptographic operations
 * Cryptographic operations
 */

/**
 * \ingroup operations
 * \defgroup operations_encdec Encrypt/Decrypt
 * API to perform encryption and decryption without authentication
 */
/**
 * \ingroup operations
 * \defgroup operations_signver Sign/Verify
 * API to perform authentication
 */
/**
 * \ingroup operations
 * \defgroup operations_wrapunwrap Wrap/Unwrap
 * API to perform encryption and decryption with authentication
 */
/**
 * \ingroup operations_wrapunwrap
 * \defgroup operations_wrapunwrap_common Common
 * Common functions (e.g. to get the default setting for algorithm and block mode)
 */

#ifndef __SDC_LIB_OP_COMMON_H_
#define __SDC_LIB_OP_COMMON_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include "sdc.h"


/* Definitions types and defaults */

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Define to select architecture and type dependent default IV length
 */
#define SDC_IV_USE_DEFAULT SIZE_MAX

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Define to select architecture and type dependent default signature length
 */
#define SDC_SIG_USE_DEFAULT SIZE_MAX

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Specification of the algorithm and block mode to be used for wrap or unwrap
 */
typedef struct sdc_wrap_unwrap_type sdc_wrap_unwrap_type_t;

/**
 * \ingroup operations_wrapunwrap_common
 * \brief libSDC algorithms available for wrap/unwrap
 *
 * In case some architecture supports a new type it will be appended to the list.\n
 * <b>Not all algorithms need to be available for each architecture!</b>
 */
typedef enum {
	SDC_WRAP_ALG_INVALID,
	SDC_WRAP_ALG_FIRST,						/**< helper to iterate over algorithms */
    SDC_WRAP_ALG_AES = SDC_WRAP_ALG_FIRST,	/**< Algorithm AES */
    SDC_WRAP_ALG_END						/**< helper to iterate over algorithms */
} sdc_wrap_unwrap_alg_t;

/**
 * \ingroup operations_wrapunwrap_common
 * \brief libSDC block modes available for wrap/unwrap
 *
 * In case some architecture supports a new type it will be appended to the list.\n
 * <b>Not all block modes need to be available for each architecture!</b>
 */
typedef enum {
	SDC_WRAP_BLK_INVALID,
	SDC_WRAP_BLK_FIRST,						/**< helper to iterate over block modes */
    SDC_WRAP_BLK_CCM = SDC_WRAP_BLK_FIRST,	/**< Block mode CCM */
    SDC_WRAP_BLK_END						/**< helper to iterate over block modes */
} sdc_wrap_unwrap_blk_t;

/**
 * \ingroup operations_wrapunwrap_common
 * \brief libSDC padding types to align unaligned data to the length required
 * by the algorithms/block modes
 */
typedef enum {
	SDC_PADDING_NONE,	/**< The algorithm/block mode does not require alignment */
	SDC_PADDING_ZERO,	/**< Add zeros to the end of data until it is aligned */
	SDC_PADDING_PKCS7	/**< The algorithm/block mode requires PKCS7 conform padding */
} sdc_padding_t;

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Description of relevant parameters for a \ref sdc_wrap_unwrap_type_t
 *
 * \todo document
 * \todo add missing fields
 */
typedef struct sdc_wrap_unwrap_desc {
	sdc_keylen_bmsk_t keylen_supported; /**< Bitmask to list the key length supported by the algorithm */
	sdc_padding_t padding;  /**< Padding required by this algorithm in order to align data */
	size_t max_total_len;   /**< Total length of data to be wrapped must not exceed this value */
    size_t max_block_len;   /**< Max length of data the non convenience wrap functions can wrap at once */
    size_t block_alignment_last; /**< Alignment of the final block processed */
    size_t block_alignment_not_last; /**< Each other block has to be aligned to this value */
    size_t iv_min;          /**< Minimal valid length for the IV */
    size_t iv_max;          /**< Maximal valid length for the IV */
    size_t iv_mod;          /**< If > 1 the result of IV length modulo this value has to be 0 */
    size_t iv_default;      /**< Default IV length */
    size_t sig_min;         /**< Minimal valid length for the signature */
    size_t sig_max;         /**< Maximal valid length for the signature */
    size_t sig_mod;         /**< If > 1 the result of signature length modulo this value has to be 0 */
    size_t sig_default;     /**< Default signature length */
} sdc_wrap_unwrap_desc_t;


/* Functions */

/**
 * \ingroup operations_wrapunwrap_common
 * \brief fill the \ref sdc_wrap_unwrap_desc_t descriptor with relevant parameters
 *
 * \todo document
 * currently these types and functions are not required externally (no benefit)
 */
sdc_error_t sdc_wrap_unwrap_desc_fill (
    const sdc_wrap_unwrap_type_t *type,
    sdc_wrap_unwrap_desc_t *desc,
    size_t total_data_len);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get descriptor of the architecture dependent default algorithm/blockmode for wrap and unwrap
 *
 * \return pointer to constant memory block defining the default \ref sdc_wrap_unwrap_type_t descriptor of the current architecture
 */
const sdc_wrap_unwrap_type_t *sdc_wrap_unwrap_get_default(void);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Allocate \ref sdc_wrap_unwrap_type_t
 *
 * The result is initialized with the architecture dependent default.
 * After usage it needs to be freed using \ref sdc_wrap_unwrap_type_free.\n
 * The content can be modified using \ref sdc_wrap_unwrap_type_set_algorithm
 * and \ref sdc_wrap_unwrap_type_set_block_mode.
 *
 * \param type pointer to return the allocated type
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - out of memory
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_wrap_unwrap_type_alloc(sdc_wrap_unwrap_type_t **type);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Free \ref sdc_wrap_unwrap_type_t
 *
 * \param type \ref sdc_wrap_unwrap_type_t type to free
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_wrap_unwrap_type_free(sdc_wrap_unwrap_type_t *type);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Set algorithm of \ref sdc_wrap_unwrap_type_t
 *
 * Set the algorithm within \ref sdc_wrap_unwrap_type_t.
 * <b>This function needs to be called before \ref sdc_wrap_unwrap_type_set_block_mode</b>
 * When the algorithm is set to a valid value the block mode is changed
 * to the architecture dependent default block mode in combination with this
 * algorithm.
 *
 * \param type \ref sdc_wrap_unwrap_type_t type
 * \param alg \ref sdc_wrap_unwrap_alg_t to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid or algorithm invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_wrap_unwrap_type_set_algorithm(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t alg);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Set block mode of \ref sdc_wrap_unwrap_type_t
 *
 * Set the block mode within \ref sdc_wrap_unwrap_type_t.
 * <b>This function needs to be called after \ref sdc_wrap_unwrap_type_set_algorithm</b>
 * In case the block mode is not available in combination with the already
 * selected algorithm SDC_ALG_MODE_INVALID is returned.
 *
 * \param type \ref sdc_wrap_unwrap_type_t type
 * \param blk \ref sdc_wrap_unwrap_blk_t to set
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid or block mode invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_wrap_unwrap_type_set_block_mode(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t blk);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get algorithm of \ref sdc_wrap_unwrap_type_t
 *
 * \param type \ref sdc_wrap_unwrap_type_t type
 * \param alg pointer to return the algorithm
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - alg pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_wrap_unwrap_type_get_algorithm(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t *alg);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get block mode of \ref sdc_wrap_unwrap_type_t
 *
 * \param type \ref sdc_wrap_unwrap_type_t type
 * \param blk pointer to return the block mode
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_ALG_MODE_INVALID - type invalid invalid
 * \return \ref SDC_INVALID_PARAMETER - blk pointer invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_wrap_unwrap_type_get_block_mode(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t *blk);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get name of algorithm

 * \param alg \ref sdc_wrap_unwrap_alg_t algorithm
 *
 * \return name or NULL if invalid algorithm
 */
const char* sdc_wrap_unwrap_algorithm_name (sdc_wrap_unwrap_alg_t alg);

/**
 * \ingroup operations_wrapunwrap_common
 * \brief Get name of block mode

 * \param blk \ref sdc_wrap_unwrap_blk_t block mode
 *
 * \return name or NULL if invalid block mode
 */
const char* sdc_wrap_unwrap_block_mode_name (sdc_wrap_unwrap_blk_t blk);

#ifdef __cplusplus
}
#endif

#endif
