/**
* \file sdc.h
*
* \brief libSDC: Definition of generic functions and types (e.g. error codes) - <b>should not be included directly!</b>
*
* Application should not include this header directly!
*
* \author Christoph Gellner (cgellner@de.adit-jv.com)
*
* \copyright (c) 2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

/**
 * \defgroup generic Generic
 */

/**
 * \ingroup generic
 * \defgroup generictype Generic types / enums
 */

/**
 * \ingroup generic
 * \defgroup keymanagement Key management
 * Functions to generate, insert, import or remove keys
 */

/**
 * \defgroup helper Helper functions
 */

#ifndef __SDC_LIB_H_
#define __SDC_LIB_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>
#include <stdbool.h>
#include <stdint.h>

/* Definitions types and defaults */

/**
 * \ingroup keymanagement_keystore
 * \brief KID used to flag an invalid key storage key
 */
#define SDC_FLAG_INVALID_KID             0

/**
 * \ingroup keymanagement_keystore
 * \brief Type for key identifier
 */
typedef uint32_t sdc_key_id_t;




/**
 * \ingroup generictype
 * \brief libSDC error codes
 */
typedef enum {
    SDC_OK = 0,                     /**< SDC_OK - no error */

    SDC_NO_MEM,                     /**< SDC_NO_MEM - no memory available */
    SDC_INVALID_VERSION,            /**< SDC_INVALID_VERSION - Mismatch of kernel and user-space version detected
                                         only used by version check functions like sdc_kernel_version_verify which is not
                                         available for normal applications using libSDC */

    SDC_CONFIG_FAILED,              /**< SDC_CONFIG_FAILED - failed to read SDC config
                                         most likely the credentials of the libSDC config are wrong or
                                         a config entry is simply invalid */
    SDC_CONFIG_MISSING,             /**< SDC_CONFIG_MISSING - no such SDC config entry */

    SDC_INVALID_PARAMETER,          /**< SDC_INVALID_PARAMETER - unspecifided parameters is invalid */
    SDC_KEY_INVALID,                /**< SDC_KEY_INVALID - the given key is invalid */
    SDC_ALG_MODE_INVALID,           /**< SDC_ALG_MODE_INVALID - the given algorithm or mode is invalid */
    SDC_IV_INVALID,                 /**< SDC_IV_INVALID - the given IV is invalid */
    SDC_IN_DATA_INVALID,            /**< SDC_IN_DATA_INVALID - the given in data is invalid */
    SDC_OUT_DATA_INVALID,           /**< SDC_OUT_DATA_INVALID - the given out data is invalid */
    SDC_AAD_DATA_INVALID,           /**< SDC_AAD_DATA_INVALID - the given additional authenticated data (aad) is invalid */
    SDC_SIG_DATA_INVALID,           /**< SDC_SIG_DATA_INVALID - the given signature data is invalid */
    SDC_SESSION_INVALID,            /**< SDC_SESSION_INVALID - the given session is invalid */
    SDC_PERM_INVALID,               /**< SDC_PERM_INVALID - the given permissions are invalid */
    SDC_FORMATTED_DATA_INVALID,     /**< SDC_FORMATTED_DATA_INVALID - an issue with the given formatted data has been detected */
    SDC_KEYLEN_INVALID,             /**< SDC_KEYLEN_INVALID - the given key-length is invalid */

    SDC_DAEMON_COMMUNICATION_ERROR, /**< SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
                                         most likely some permissions are missing or the daemon has not been started */
    SDC_DAEMON_REQUEST_REJECTED,    /**< SDC_DAEMON_REQUEST_REJECTED - the request has been rejected by the daemon */
    SDC_DAEMON_INVALID_RESPONSE,    /**< SDC_DAEMON_INVALID_RESPONSE - checking response of daemon failed */
    SDC_DAEMON_TIMEOUT_ERROR,             /**< SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time */
    SDC_KEY_STORAGE_ALREADY_PRESENT, /**< SDC_KEY_STORAGE_ALREADY_PRESENT - Key-storage does already exist */
    SDC_KEY_STORAGE_ACCESS_FAILED,  /**< SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access the key storage */
    SDC_KEY_STORAGE_LOCK_FAILED,    /**< SDC_KEY_STORAGE_LOCK_FAILED - daemon failed to lock the key storage */
    SDC_KEY_CONTAINER_IMPORT_FAILED,/**< SDC_KEY_CONTAINER_IMPORT_FAILED - daemon failed to import a key container */

    SDC_KID_NOT_AVAILABLE,          /**< SDC_KID_NOT_AVAILABLE - the KID does not exist (e.g. for load or remove) */
    SDC_KID_EXISTS,                 /**< SDC_KID_EXISTS - could not install key as the KID is already used */

    SDC_KEY_UNSET,                  /**< SDC_KEY_UNSET - operation fails as no key has been assigned to a session */
    SDC_KEY_LOCKED,                 /**< SDC_KEY_LOCKED - operation aborted as key or key storage is currently locked - try again later */
    SDC_KEY_NOT_READY,              /**< SDC_KEY_NOT_READY - operation aborted as key is not ready to be used - try again later */
    SDC_KEY_TAMPERED,               /**< SDC_KEY_TAMPERED - operation aborted as the key seems to be tampered */

    SDC_ACCESS_DENIED,              /**< SDC_ACCESS_DENIED - the process has not sufficient permissions to perform the requested operation */
    SDC_OP_NOT_SUPPORTED,           /**< SDC_OP_NOT_SUPPORTED - Operation not supported
                                         most likely the key does not fit to the operation e.g. DES and 256bit key */
    SDC_AUTHENTICATION_FAILED,      /**< SDC_AUTHENTICATION_FAILED - authentication of data failed */

    SDC_OPEN_DEVICENODE_FAILED,     /**< SDC_OPEN_DEVICENODE_FAILED - failed open the device node
                                         most likely some permissions are missing */
    SDC_DEVICENODE_IOTCTL_FAILED,   /**< SDC_DEVICENODE_IOTCTL_FAILED - the ioctl failed for some unspecified reason */
    SDC_DEVICENODE_INVALID,         /**< SDC_DEVICENODE_INVALID - ioctl returned EBADF - device seems to be closed */

    SDC_NOT_SUPPORTED,              /**< SDC_NOT_SUPPORTED - the function called is currently not supported for the current architecture */
    SDC_UNKNOWN_ERROR,              /**< SDC_UNKNOWN_ERROR - ? */
} sdc_error_t;


/**
 * \ingroup generictype
 * \brief libSDC supported key lengths
 *
 * These values represent individual bits in the key length bitmask
 */
typedef enum {
	SDC_KEYLEN_BMSK_INVALID = 0, /**< Invalid key length */
	SDC_KEYLEN_BMSK_FIRST 	= 1, /**< helper to iterate over keylength */
	SDC_KEYLEN_BMSK_64bit 	= SDC_KEYLEN_BMSK_FIRST,		/**< Key length 64bit = 8bytes */
	SDC_KEYLEN_BMSK_96bit 	= SDC_KEYLEN_BMSK_64bit<<1,		/**< Key length 96bit = 12bytes */
	SDC_KEYLEN_BMSK_128bit 	= SDC_KEYLEN_BMSK_96bit<<1,		/**< Key length 128bit = 16bytes */
	SDC_KEYLEN_BMSK_192bit 	= SDC_KEYLEN_BMSK_128bit<<1,	/**< Key length 192bit = 24bytes */
	SDC_KEYLEN_BMSK_256bit 	= SDC_KEYLEN_BMSK_192bit<<1,	/**< Key length 256bit = 32bytes */
	SDC_KEYLEN_BMSK_384bit 	= SDC_KEYLEN_BMSK_256bit<<1,	/**< Key length 384bit = 48bytes */
	SDC_KEYLEN_BMSK_512bit 	= SDC_KEYLEN_BMSK_384bit<<1,	/**< Key length 512bit = 64bytes */
	SDC_KEYLEN_BMSK_END
} sdc_keylen_bmsk_val_t;

/**
 * \ingroup generictype
 * \brief libSDC bitmask to return the supported key lengths
 *
 * Depending on the supported keylength one or multiple bits are set.
 * Keylength values according to \ref sdc_keylen_bmsk_val_t
 */
typedef unsigned int sdc_keylen_bmsk_t;


/* Functions */

/**
 * \ingroup helper
 * \brief Get a string message describing the given error code.
 *
 * \param err \ref sdc_error_t error code
 * \return error message
 */
const char* sdc_get_error_string (sdc_error_t err);



/**
 * \ingroup helper
 * \brief Free a plain buffer which might contain confidential data.
 *
 * In contrast to ::sdc_cipher_buffer_free this function will first
 * write all 0 to the buffer.
 *
 * \param plain_data    pointer to buffer to be freed
 * \param plain_len     length of the buffer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - data NULL or len 0
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_plain_buffer_free (uint8_t *plain_data, const size_t plain_len);

/**
 * \ingroup helper
 * \brief Free a buffer which contains encrypted or wrapped data.
 *
 * As it is uncritical to leave encrypted or wrapped data in memory this
 * function will only call free.
 *
 * \param cipher_data    pointer to buffer to be freed
 * \param cipher_len     length of the buffer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - data NULL or len 0
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_cipher_buffer_free (uint8_t *cipher_data, const size_t cipher_len);

/**
 * \ingroup helper
 * \brief Determine the bits of the given key-length value
 *
 * \param keylen_val    \ref sdc_keylen_bmsk_val_t value
 * \param keylen_bits   pointer to return the bits
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_bits
 * \return \ref SDC_KEYLEN_INVALID - the \p keylen_val value is not a element of \ref sdc_keylen_bmsk_val_t
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_keylen_bmsk_val_bits(sdc_keylen_bmsk_val_t keylen_val, unsigned int *keylen_bits);

/**
 * \ingroup helper
 * \brief Determine the bytes of the given key-length value
 *
 * \param keylen_val    \ref sdc_keylen_bmsk_val_t value
 * \param keylen_bytes  pointer to return the bytes
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_bytes
 * \return \ref SDC_KEYLEN_INVALID - the \p keylen_val value is not a element of \ref sdc_keylen_bmsk_val_t
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_keylen_bmsk_val_bytes(sdc_keylen_bmsk_val_t keylen_val, unsigned int *keylen_bytes);


/**
 * \ingroup helper
 * \brief Using \ref sdc_keylen_bmsk_val_first and \ref sdc_keylen_bmsk_val_next it is
 * possible to iterate over the key length given in \p keylen_bmsk one by one
 *
 * This function is just a shortcut to initialize \p keylen_val to SDC_KEYLEN_BMSK_INVALID.
 * Afterwards it will internally call \ref sdc_keylen_bmsk_val_next.
 * Please refer to \ref sdc_keylen_bmsk_val_next for more details.
 *
 * \param keylen_bmsk    \ref sdc_keylen_bmsk_t value with all bits set
 * \param keylen_val   	\ref sdc_keylen_bmsk_val_t to return the first key-length
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_val
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_keylen_bmsk_val_first(sdc_keylen_bmsk_t keylen_bmsk, sdc_keylen_bmsk_val_t *keylen_val);

/**
 * \ingroup helper
 * \brief Using \ref sdc_keylen_bmsk_val_first and \ref sdc_keylen_bmsk_val_next it is
 * possible to iterate over the key length given in \p keylen_bmsk one by one
 *
 * While iterating \p keylen_val has to be the value returned in the previous call.
 * For initialization either use \ref sdc_keylen_bmsk_val_first or set
 * \p keylen_val to SDC_KEYLEN_BMSK_INVALID.
 *
 * In case no other key-length is found \p keylen_val will be set to SDC_KEYLEN_BMSK_INVALID.
 * Note : SDC_OK is returned in this case. Calling sdc_keylen_bmsk_val_next again, will result
 * in a loop.
 *
 * \param keylen_bmsk    \ref sdc_keylen_bmsk_t value with all bits set
 * \param keylen_val   	\ref sdc_keylen_bmsk_val_t to return the next key-length - use the last result when calling
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_val
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_keylen_bmsk_val_next(sdc_keylen_bmsk_t keylen_bmsk, sdc_keylen_bmsk_val_t *keylen_val);

/**
 * \ingroup helper
 * \brief Function to return the key lengths supported by the architecture
 *
 * Please note: That not all algorithms and block modes need to support these values
 * for processing. But the system has to at least support key management for these
 * key lengths.
 *
 * \param arch_keylen_bmsk	\ref sdc_keylen_bmsk_t pointer to return the supported key lengths
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p arch_keylen_bmsk
 * \return otherwise an unexpected error occurred that should be handled by default
 */
sdc_error_t sdc_keylen_bmsks(sdc_keylen_bmsk_t *arch_keylen_bmsk);


#ifdef __cplusplus
}
#endif

#endif
