/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        ecm_utils.c

   \brief       Utilities module of Embedded Certificate Manger

   $Rev: 830 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-09-12 16:11:18 +0200 (Di, 12. Sep 2017) $
****************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

#include "../inc/ecm.h"
#include "../inc/ecm_cm_main.h"
#include "../inc/ecm_utils.h"

/* linux headers */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <signal.h>
#include <sys/time.h>
#include <pthread.h>
#include <errno.h>
#include <dirent.h>
#include <arpa/inet.h>
#include <dbus/dbus.h>

/* curl Header */
#include <curl/curl.h>
#include <curl/easy.h>

/* ASN.1 headers */
#include "../../cmp/asn1-2005-multi-file-os-asn1c/PKIMessage.h"
#include "../../cmp/inc/asn1_utils.h"

/* common headers */
#include "../../common/inc/esc_debug.h"
#include "../../common/inc/esc_utils.h"
#include "../../common/inc/esc_asn1_utils.h"
#include "../../common/inc/base64.h"

/* cryto header */
#include <crypto.h>

/* CMP header */
#include "../../cmp/inc/cmp.h"
#include "../../cmp/inc/cmp_utils.h"
#include "../../cmp/asn1-2005-multi-file-os-asn1c/AuthorityInfoAccessSyntax.h"


#ifdef ECM_PKCS10
    #include <pkcs10.h>
    #include <pkcs10_utils.h>
#endif

#include "crl_utils.h"

/* ECM header */
#include "../inc/ecm_cm_main.h"
#include "../inc/ecm_version.h"

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif

#include "ecm_utils.h"
/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

static const STR_LOOKUP_t formatContainerLut[] = {
    { ID_FORMAT_CONTAINER_PKCS1, "PKCS#1" },
    { ID_FORMAT_CONTAINER_PKCS8, "PKCS#8" },
    { ID_FORMAT_CONTAINER_DER,   "DER"    },
    { ID_FORMAT_CONTAINER_PEM,   "PEM"    },
};

static UINT32 formatContainerLutSize =
    sizeof ( formatContainerLut ) / sizeof ( STR_LOOKUP_t );

static const STR_LOOKUP_t boolLut[] = {
    { TRUE, "true" },
    { FALSE, "false" },
};

static UINT32 boolLutSize =
    sizeof ( boolLut ) / sizeof ( STR_LOOKUP_t );

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

static char *ecmFileId2FileName (
    FILE_ID_e fId,
    ECM_PARAMS_t *ecm )
{
    char *fileName;

    if ( !ecm ) {
        return NULL;
    }
    switch ( fId ) {
        case ( file_id_pkcs1PrivKey ):
            fileName = ecm->str.devKey;
            break;
        case ( file_id_nv_data ):
            fileName = ecm->str.nvData;
            break;
        default:
            dbgPrint ( DBG_ERR, DBG_FLS "Error: Invalid file id '%d'\n", DBG_FL, fId );
            fileName = NULL;
            break;
    }

    return ( fileName );
}

static FILE *openFp (
    BOOL quite,
    FILE_ID_e fileId,
    char *name,
    ECM_PARAMS_t *ecm,
    char *flag )
{
    BOOL failed = FALSE;
    FILE *fp = NULL;
    char *fileName;

    if ( ((fileId == file_id_none) && !name ) || (!flag)) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        if ( fileId == file_id_none ) {
            fileName = name;
        } else {
            fileName = ecmFileId2FileName( fileId, ecm );
        }

        if ( !fileName ) {
            failed = TRUE;
        }
    }

    ifNotFailed {

        if (( fp = fopen ( fileName, flag )) == NULL ) {
            if ( !quite ) {
                dbgPrint ( DBG_ERR, DBG_FLS "Error: Failed to open '%s' (%s)\n",
                    DBG_FL, fileName, strerror ( errno ) );
            }
        }
    }

    return ( fp );
}

ECM_RET_t bool2EcmRet ( 
    BOOL failed )
{
    return ( ( failed == TRUE ) ? EcmRetFailed : EcmRetOk );
}


// ToDo MR move adding linebreaks directly to PEM creation to avoid this overhead
char *ecmAddPemNl (
    const UINT8 *data )
{
    size_t resultlen = 0;
    size_t newsize = 0;
    char *result_with_linebreak = NULL;
    int j = 0;
    int i;

    if ( !data ) {
        return ( NULL );
    }

    resultlen = strlen((char *)data);
    newsize = resultlen+resultlen/64;

    if ( escCheckSizetForOv( newsize ) == TRUE ) {
        return ( NULL );
    }
    result_with_linebreak = malloc(newsize+1);

    for(i = 0; i < resultlen+1; i++){
        if (i%64 == 0 && i != 0){
            result_with_linebreak[j] = '\n';
            result_with_linebreak[j+1] = data[i];
            j = j+2;
        } else {
            result_with_linebreak[j] = data[i];
            j = j+1;
        }
    }
    return ( result_with_linebreak );
}

size_t ecmCatFileName (
    BOOL sep,
    char **dst,
    char *pre,
    char *suf )
{
    size_t pathLen = 0, remSize = 0, totalLen = 0;
    BOOL addSep = FALSE;
    BOOL failed = FALSE;

    if ( !dst || !pre || !suf ) {
        failed = TRUE;
    }

    ifNotFailed {

        pathLen = strlen ( pre );

        if ( sep ) {
            if ( pre[pathLen] != '/' ) {
                addSep = TRUE;
                pathLen += 1;
            }
        }

        remSize = pathLen + strlen ( suf ) + 1;
        failed = escMalloc ((void **)dst, remSize );
    }
    ifNotFailed {

        totalLen = remSize;

        remSize -= 1;

        strncpy ( *dst, pre, remSize );
        remSize -= pathLen;
        if ( addSep == TRUE ) {
            strncat ( *dst, "/", remSize );
        }
        strncat ( *dst, suf, remSize );
    }

    return ( totalLen );
}

BOOL ecmDeleteDir (
    char *path )
{
    BOOL failed = FALSE;
    int num = 0, i;
    struct dirent **namelist = NULL, **list;
    char *fileName = NULL;
    char *absFileName = NULL;

    dbgIn ( __func__ );

    /* does directory still exists? Delete content first! */
    if ( access ( path, F_OK ) == 0 ) {

        num = scandir ( path, &namelist, 0, alphasort );
        list = namelist;

        for ( i = 0; i < num; i++, list++ ) {

            fileName = (*list)->d_name;

            /* skip current directory, parent directory and certificate files */
            if ( ( strncmp ( fileName, ".", strlen (".")) == 0 ) ||
                 ( strncmp ( fileName, "..", strlen (".." )) == 0 ) ) {

                continue;

            } else {

                ecmCatFileName( TRUE, &absFileName, path, fileName );
                if ( remove ( absFileName ) < 0 ) {
                    dbgPrint ( DBG_ERR, DBG_FLS "Failed to remove '%s' (%s)\n",
                        DBG_FL, fileName, strerror ( errno ));
                    failed = TRUE;
                    break;
                }
                escFreePtr( (UINT8**) &absFileName );
            }
        }

        if ( rmdir ( path ) < 0 ) {
            dbgPrint ( DBG_ERR, DBG_FLS "Failed to remove '%s' (%s)\n",
                DBG_FL, path, strerror ( errno ));
            failed = TRUE;
        }
    }

    escFreeDirent( (void **)namelist, num );
    escFreePtr( (UINT8**) &absFileName );

    dbgOut ( __func__ );
    return ( failed );
}

BOOL loadFromDiskToDER (
    SDC_PARAMS_s *sdc,
    char *fileName,
    UINT8** derData,
    UINT32* derLength)
{
    if(!fileName){
        *derData = NULL;
        *derLength = 0;
        return TRUE;
    }

    // loadFile from Disk
    UINT8* byteStream = NULL;
    UINT32 streamSize = 0;
    if(loadFromFileToBuffer( sdc, fileName, &byteStream, &streamSize)
            || !byteStream
            || streamSize < 1)
        return TRUE;

    // find Format - if PEM convert first to DER
    if(checkForPEM((char*)byteStream)){
        UINT8* der_Data = NULL;
        UINT32 der_DataSize = 0;
        if(decodePEM(byteStream, streamSize, &der_Data, &der_DataSize) ||
                !der_Data || der_DataSize < 1){

            free(byteStream);
            if ( der_Data ) {
                escFreePtr((UINT8**) &der_Data);
            }
            dbgPrint(DBG_ERR, DBG_FLS "Could not decode PEM Data\n", DBG_FL );
            return TRUE;
        }
        free (byteStream);
        byteStream = der_Data;
        streamSize = der_DataSize;
    }

    *derData = byteStream;
    *derLength = streamSize;

    return FALSE;
}


static ECM_RET_t writeToDisc (
    FILE_ID_e fileId,
    char *fileName,
    ECM_PARAMS_t *ecm,
    const UINT32 formatid,
    int datatype,
    UINT8 *data,
    UINT32 dataSize )
{
    ECM_RET_t retVal = EcmRetOk;
    FILE *fp = NULL;
    size_t ret;
    UINT8 *wrData = NULL;
    UINT32 wrDataSize = 0;
    char *start_tag = NULL;
    char *end_tag = NULL;
    char *base64Enc = NULL;
    char *tmp = NULL;
    char *tmp2 = NULL;
    char *ptr;
    UINT32 len;
    size_t base64EncSize = 0;
    BOOL failed = FALSE;

    dbgIn ( __func__ );

    if ( data == NULL ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        retVal = EcmRetFailed;
    }

    ifReturnOk {
        fp = openFp ( FALSE, fileId, fileName, ecm, "w" );

        if ( !fp ) {
            retVal = EcmRetFailed;
        }
    }

    ifReturnOk {

        if ( formatid == ID_FORMAT_CONTAINER_PEM ) {

            switch ( datatype ) {
                case ( DATA_TYPE_KEY ):
                    start_tag = PEM_KEY_START_TAG;
                    end_tag = PEM_KEY_END_TAG;
                    break;
                case ( DATA_TYPE_CERT ):
                    start_tag = PEM_CERT_START_TAG;
                    end_tag = PEM_CERT_END_TAG;
                    break;
                default:
                    dbgPrint ( DBG_ERR, DBG_FLS "Invalid type %d\n", DBG_FL, datatype );
                    retVal = EcmRetFailed;
                    break;
            }

            ifReturnOk {

                /* Do the base 64 encoding */
                tmp = escBase64Encode(data, dataSize, (UINT32 *)&base64EncSize );

                failed = escCheckSizetForOv( base64EncSize );
                retVal = bool2EcmRet( failed );
            }

            ifReturnOk {

                /* Add line breaks */
                failed = escMalloc ((void **)&tmp2, base64EncSize + 1 );
                retVal = bool2EcmRet( failed );
            }

            ifReturnOk {

                memcpy ( tmp2, tmp, base64EncSize );
                escFreePtr( ( UINT8 **) &tmp );
                tmp = ecmAddPemNl( (UINT8 *)tmp2 );

                /* Add start and end tags */
                base64EncSize = strlen (start_tag ) +
                    strlen (tmp ) +
                    strlen( end_tag );

                failed = escCheckSizetForOv( base64EncSize );
                retVal = bool2EcmRet( failed );
            }

            ifReturnOk {

                failed = escMalloc ((void **)&base64Enc, base64EncSize + 1);
                retVal = bool2EcmRet( failed );
            }

            ifReturnOk {
                ptr = base64Enc;
                len = strlen ( start_tag );
                memcpy ( ptr, start_tag, len );
                ptr += len;
                len = strlen ( tmp );
                memcpy ( ptr, tmp, len );
                ptr += len;
                len = strlen ( end_tag );
                memcpy ( ptr, end_tag, len );

                /* Set pointer and size */
                wrDataSize = base64EncSize;
                wrData = (UINT8 *) base64Enc;
            }

        /* No PEM is used, just set pointer and size */
        } else {
            wrData = data;
            wrDataSize = dataSize;
        }
    }

    ifReturnOk {

        ret = fwrite ( wrData, 1, wrDataSize, fp );

        if ( ret != wrDataSize ) {
            dbgPrint ( DBG_ERR, DBG_FLS "writing to disk failed\n", DBG_FL );
            retVal = EcmRetFailed;
        }
    }

    if ( fp ) {
        fclose ( fp );
    }

    if ( wrData != data ) {
        escFreePtr( (UINT8 ** ) &wrData );
    }
    escFreePtr( (UINT8 **) &tmp2 );
    escFreePtr( (UINT8 **) &tmp );

    dbgOut ( __func__ );

    return ( retVal );
}

ECM_RET_t loadNvParams (
    ECM_PARAMS_t *ecm )
{
    ECM_RET_t retVal = EcmRetOk;
    UINT8 *data = NULL;
    UINT32 size = 0;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        retVal = EcmRetFailed;
    }

    /* read NV data from disc */
    ifReturnOk {
        ecm->sdc.execSdcOp = FALSE;
        retVal = bool2EcmRet( loadFromDiskToDER(
            &ecm->sdc, ecmFileId2FileName(file_id_nv_data, ecm), &data, &size));

        if ( size != sizeof (ecm->nv.flags)) {
            retVal = EcmRetFailed;
        }
    }

    ifReturnOk {
        memcpy ( &ecm->nv.flags, data, size );
    }

    escFreePtr( &data );

    dbgOut ( __func__ );

    return ( retVal );
}

ECM_RET_t ecmSetNvFlag (
    ECM_PARAMS_t *ecm,
    UINT32 flag )
{
    ECM_RET_t retVal = EcmRetOk;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL );
        retVal = EcmRetFailed;
    }

    ifReturnOk {
        ecm->nv.flags |= flag;

        UINT32 val = ecm->nv.flags;
        val = htonl(val);

        retVal = writeToDisc ( file_id_nv_data, NULL, ecm, 0, 0,
//            (UINT8 *)&ecm->nv.flags, sizeof ( ecm->nv.flags));
            (UINT8 *)&val, sizeof ( ecm->nv.flags));
    }

    dbgOut ( __func__ );

    return ( retVal );
}

ECM_RET_t ecmClearNvFlag (
    ECM_PARAMS_t *ecm,
    UINT32 flag )
{
    ECM_RET_t retVal = EcmRetOk;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL );
        retVal = EcmRetFailed;
    }

    ifReturnOk {
        ecm->nv.flags &=~ flag;

        retVal = writeToDisc ( file_id_nv_data, NULL, ecm, 0, 0,
            (UINT8 *)&ecm->nv.flags, sizeof ( ecm->nv.flags));
    }

    dbgOut ( __func__ );

    return ( retVal );
}

BOOL handleConfigFileLine (
    char *nameBuf,
    char *linePtr,
    ECM_PARAMS_t *ecm )
{
    BOOL failed = FALSE;

    /* sanity checks */
    if ( !ecm || !nameBuf || !linePtr ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        /* process all parameters */
        if ( strncmp ( nameBuf, "debug_level", strlen ("debug_level" )) == 0 ) {

            ecm->logLevel = strtol ( linePtr, NULL, 0 );

        } else if ( strncmp ( nameBuf, "cert_key_and_file_store_path", strlen ( "cert_key_and_file_store_path" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.fileStorePath );

        } else if ( strncmp ( nameBuf, "alt_key_path", strlen ( "alt_key_path" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.keyPath );

        } else if ( strncmp ( nameBuf, "log_file_name", strlen ( "log_file_name" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.logFile);

        } else if ( strncmp ( nameBuf, "uid_file", strlen ( "uid_file" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.uidFile );

        } else if ( strncmp ( nameBuf, "allowed_issuer_CN", strlen ("allow_issuer_CN")) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->cmpData.issuerCN );

            if (strncmp(ecm->cmpData.issuerCN, "", strlen ( "" )) == 0)
            {
               ecm->cmpData.verifyIssuer = FALSE;
            } else {
                ecm->cmpData.verifyIssuer = TRUE;
            }

        } else if ( strncmp ( nameBuf, "cert_format", strlen ("cert_format")) == 0 ) {

            failed = escString2IdLookup( formatContainerLut,
                formatContainerLutSize, linePtr, &ecm->certFormat );

        } else if ( strncmp ( nameBuf, "key_format", strlen ( "key_format" )) == 0 ) {

            failed = escString2IdLookup( formatContainerLut,
                formatContainerLutSize, linePtr, &ecm->keyFormat );

        } else if ( strncmp ( nameBuf, "key_size", strlen ( "key_size" )) == 0 ) {

            ecm->keySize = strtol (linePtr, NULL, 0 );
            ecm->cmpData.keySize = ecm->keySize;

            /* accept key sizes of 2048 and 4096 bit only */
            if (( ecm->keySize != 2048 ) && ( ecm->keySize != 4096 ) && (ecm->keySize != 1024)) {
                failed = TRUE;
            }

        } else if ( strncmp ( nameBuf, "generate_key_at_startup", strlen ( "generate_key_at_startup" )) == 0 ) {

            UINT32 tmp;
            failed = escString2IdLookup( boolLut, boolLutSize, linePtr, &tmp );

            ifNotFailed {
                ecm->genDevKeyAtStartup = (tmp == 1) ? TRUE : FALSE;
            }
            //failed = escString2IdLookup( boolLut, boolLutSize, linePtr, &ecm->genDevKeyAtStartup );
		} else if ( strncmp ( nameBuf, "disable_device_certificate_check", strlen ( "disable_device_certificate_check" )) == 0 ) {

            UINT32 tmp;
            failed = escString2IdLookup( boolLut, boolLutSize, linePtr, &tmp );

            ifNotFailed {
                ecm->disableDeviceCertificateCheck = (tmp == 1) ? TRUE : FALSE;
            } 
		} else if ( strncmp ( nameBuf, "disable_CRL_check", strlen ( "disable_CRL_check" )) == 0 ) {

            UINT32 tmp;
            failed = escString2IdLookup( boolLut, boolLutSize, linePtr, &tmp );

            ifNotFailed {
                ecm->disableCRLcheck = (tmp == 1) ? TRUE : FALSE;
            }
			
        } else if ( strncmp ( nameBuf, "key_container", strlen ( "key_container" )) == 0 ) {

            failed = escString2IdLookup( formatContainerLut,
                formatContainerLutSize, linePtr, &ecm->keyContainer );

        } else if ( strncmp ( nameBuf, "root_crt_file", strlen ( "root_crt_file" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.rootCrt );

        } else if ( strncmp ( nameBuf, "crl_path", strlen ("crl_path" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.crlPath );

        } else if ( strncmp ( nameBuf, "crl_url", strlen ( "crl_uri")) == 0 ) {

            failed = escCsvList2StringList ( linePtr, &ecm->str.crlURL );

        } else if ( strncmp ( nameBuf, "device_cert_URI", strlen ( "device_cert_URI" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.deviceCertURI );

        } else if ( strncmp ( nameBuf, "key_name", strlen ( "key_name" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.devKeyName );

        } else if ( strncmp ( nameBuf, "device_cert_name", strlen ( "device_cert_name" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.deviceCertName );

        } else if ( strncmp ( nameBuf, "csr_file_path", strlen ( "csr_file_path" )) == 0 ) {

            failed = copyParameterValue ( linePtr, &ecm->str.csrFileName );

        } else if ( strncmp ( nameBuf, "sdc_uin_wrap_key_id", strlen ( "sdc_uin_wrap_key_id" )) == 0 ) {

            ecm->sdc.uinWrapKeyId = strtol (linePtr, NULL, 0 );

        } else if ( strncmp ( nameBuf, "sdc_dev_key_wrap_key_id", strlen ( "sdc_dev_key_wrap_key_id" )) == 0 ) {

            ecm->sdc.devKeyWrapKeyId = strtol (linePtr, NULL, 0 );

        } else if ( strncmp ( nameBuf, "sdc_cert_wrap_key_id", strlen ( "sdc_cert_wrap_key_id" )) == 0 ) {

            ecm->sdc.certWrapKeyId = strtol (linePtr, NULL, 0 );

        } else {
            dbgPrint ( DBG_INFO, "Unknown parameter (%s)\n", nameBuf );
        }
    }

    return ( failed );
}

BOOL setDefaultOptions (
    ECM_PARAMS_t *ecm )
{
    BOOL failed = FALSE;

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        if ( 0 == ecm->sdc.uinWrapKeyId ) {
            ecm->sdc.uinWrapKeyId = SDC_UIN_WRAP_KEY_ID;
        }

        if ( 0 == ecm->sdc.devKeyWrapKeyId ) {
            ecm->sdc.devKeyWrapKeyId = SDC_DEV_KEY_WRAP_KEY_ID;
        }

        if ( 0 == ecm->sdc.certWrapKeyId ) {
            ecm->sdc.certWrapKeyId = SDC_CERT_WRAP_KEY_ID;
        }
    }

    return ( failed );
}

BOOL checkMandatoryOptions (
    ECM_PARAMS_t *ecm )
{
    BOOL failed = FALSE;

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        if ( !ecm->str.rootCrt ) {
            dbgPrint ( DBG_ERR, DBG_FLS "No root Certificate Path defined\n", DBG_FL );
            failed = TRUE;
        }

        if ( !ecm->str.csrFileName ) {
            dbgPrint ( DBG_ERR, DBG_FLS "No CSR file path defined\n", DBG_FL );
            failed = TRUE;
        }
    }

    return ( failed );
}

void freeEcmData (
    ECM_PARAMS_t *ecm )
{
    char *ptr;

    ptr = ecm->cmpData.uniqueIdSrc;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.fileStorePath;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.keyPath;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.devKey;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.devKeyName;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.deviceCertName;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.csrFileName;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.logFile;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.crlPath;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.deviceCertURI;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->str.uidFile;
    escFreePtr( (UINT8 **)&ptr );

    ptr = ecm->uniqueID;
    escFreePtr( (UINT8 **)&ptr );

    ptr = (char*) ecm->pkcs1PubKey;
    escFreePtr( (UINT8 **)&ptr );

    escFreeStringList( &ecm->str.crlURL );

    freeCertList(&ecm->certList);

    ptr = ecm->str.rootCrt;
    escFreePtr ( (UINT8 **)&ptr );

    freeCmpData( &ecm->cmpData );

    freeCRL(ecm->crList);
    ecm->crList = NULL;

    freeCRLHashes(ecm->crListHash);
    ecm->crListHash = NULL;

    ptr = ecm->str.nvData;
    escFreePtr( (UINT8 **)&ptr );

#ifdef ECM_PKCS10
    freePkcs10Data( &ecm->pkcs10Data );
#endif

#ifdef CONFIG_ECM_USE_SDC

    sdc_close_session(ecm->sdc.session);
    (void) sdc_wrap_unwrap_type_free(ecm->sdc.type);

#endif /* CONFIG_ECM_USE_SDC */
}

BOOL downloadDeviceCertificate2 (
    char *uin,
    char *uribase,
    UINT8 **data,
    UINT32 *dataSize )
{
    BOOL failed = FALSE;
    size_t s = 0;
    char *uri = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !uribase || !uin || !data || !dataSize ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL );
        return ( TRUE );
    }

    s = strlen(uribase) + strlen(uin);
    failed = escCheckSizetForOv( s );

    ifNotFailed {
        failed = escMalloc ((void **)&uri, s + 1 );
    }

    ifNotFailed {

        strncpy(uri, uribase, strlen(uribase));
        strncat(uri, uin, strlen(uin));

        failed = downloadFromURItoBuffer(uri, (char**) data, dataSize);
    }

    ifNotFailed {

        if (checkForPEMCertificate((char*) *data)) {
            UINT8* derData;
            UINT32 derDataSize;

            if (!decodePEM( *data, *dataSize, &derData, &derDataSize)) {
                free(*data);
                *data = derData;
                *dataSize = derDataSize;
            } else {
                failed = TRUE;
            }

        } else {

            /* Its not a PEM encode certificate but it should!
             * Try if its a printable string, e.g. HTTP error codes
             * (if DER encoded, data starts with 0x30, 0x80 and strlen returns 1 */
            if ( strlen ( (char * ) *data ) > 1 ) {
                dbgPrint( DBG_ERR, DBG_FLS "No certificate received: %s\n", DBG_FL, (char *) *data );
                failed = TRUE;
            }
        }
    }

    escFreePtr((UINT8 **)&uri);

    dbgOut ( __func__ );

    return ( failed );
}

size_t static write_data(void *ptr, size_t size, size_t nmemb, void *userp){
    size_t realsize = size * nmemb;
    BufferStruct_t *mem = (BufferStruct_t *) userp;

    mem->buf = realloc(mem->buf, mem->size + realsize + 1);
    if (mem->buf == NULL) {
        /* out of memory! */
        dbgPrint(DBG_ERR, DBG_FLS "not enough memory (realloc returned NULL)\n", DBG_FL );
        return 0;
    }

    memcpy(&(mem->buf[mem->size]), ptr, realsize);
    mem->size += realsize;
    mem->buf[mem->size] = 0;

    return realsize;
}

BOOL downloadFromURItoBuffer(char* uri, char** buf, UINT32* size){
    BOOL failed = FALSE;
    CURL *curl = NULL;
    CURLcode res;
    curl = curl_easy_init();
    if (curl) {
        BufferStruct_t data;
        data.buf = malloc(1);
        data.size = 0;

        curl_easy_setopt(curl, CURLOPT_URL, uri);
        curl_easy_setopt(curl, CURLOPT_TIMEOUT, 30);
        curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_data);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&data );
        res = curl_easy_perform(curl);
        curl_easy_cleanup(curl);
        if(res != CURLE_OK){

            dbgPrint ( DBG_ERR, DBG_FLS "curl_easy_perform() failed: %s\n",
                DBG_FL, curl_easy_strerror(res));

            size = 0;
            *buf = NULL;
            return failed = TRUE;
        }
        *buf = data.buf;
        *size = data.size;
    } else {
        size = 0;
        *buf = NULL;
        failed = TRUE;
    }
    return failed;
}

BOOL downloadFromURItoDisk(char* uri, char* dest){
    if(!uri || !dest || 3 > strlen(uri) || 3 > strlen(dest))
        return TRUE;

    BOOL failed = FALSE;
    CURL *curl = NULL;
    CURLcode res;
    FILE *crlFile = NULL;

    curl = curl_easy_init();
    if(curl){
        curl_easy_setopt(curl, CURLOPT_URL, uri);
        curl_easy_setopt(curl, CURLOPT_TIMEOUT, 30);
        curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);

        /* Open File to write CRL to */
        crlFile = fopen(dest, "w");

        if ( crlFile ) {

            curl_easy_setopt(curl, CURLOPT_WRITEDATA, crlFile);

            /* Perform the request, res will get the return code */
            res = curl_easy_perform(curl);
            /* Check for errors */
            if(res != CURLE_OK){
                dbgPrint ( DBG_ERR, DBG_FLS "curl_easy_perform() failed: %s\n",
                    DBG_FL, curl_easy_strerror(res));
              failed = TRUE;
            }

            /* always cleanup */
            curl_easy_cleanup(curl);
        }

        if ( crlFile ) {
            failed |= fclose(crlFile);
        }
    }
    else
        failed = TRUE;

    return failed;
}

CERT_LIST_s *findCertListLastItem (
    CERT_LIST_s *lst )
{
    BOOL failed = FALSE;
    CERT_LIST_s *ret = NULL, *ptr = lst;

    dbgIn ( __func__ );

    if ( !lst ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        while ( ptr ) {
            ret = ptr;
            ptr = ptr->next;
        }
    }

    dbgOut ( __func__ );

    return ( ret );
}

BOOL isRootCert(CERT_LIST_s* crtEntry){
    if(!crtEntry)
        return FALSE;
    if(strncmp(crtEntry->issuerCn, crtEntry->subjectCn, strlen ( crtEntry->issuerCn )))
        return FALSE;
    else
        return TRUE;
}

int downloadCertificateChain(Certificate_t *crt, CERT_LIST_s** certList){
    if(!crt || !certList)
        return -1;
    int count = 0;
    UINT8* issuerCert = NULL;
    UINT32 size = 0;
    BOOL failed = FALSE;
    Certificate_t* tempCrt;

    CERT_LIST_s* certEntry = NULL;
    CERT_LIST_s* certListStart = NULL;
    CERT_LIST_s* newCLEntry = NULL;

    while(!failed && !downloadIssuerCertificate(crt, &issuerCert, &size)){
        count ++;
        failed = alloc((void**) &tempCrt, sizeof(Certificate_t));
        if(!failed)
            failed = asn1Decode(
                issuerCert, size, &asn_DEF_Certificate, tempCrt, FALSE);

        ifNotFailed {
            crt = tempCrt;
            failed = escMalloc ((void **)&newCLEntry, sizeof(CERT_LIST_s));
        }

        ifNotFailed {
            newCLEntry->certPtr = issuerCert;
            newCLEntry->certSize = size;

            failed = cmpExtractCrtParamsFromTBSCertificate (
                &crt->tbsCertificate, newCLEntry );
        }

        ifNotFailed {
            newCLEntry->crt = tempCrt;

            //check if current Certificate is a root Cert and discard in that case
            // otherwise append to given List
            if(!isRootCert(newCLEntry)){
                // add CRL URI
                newCLEntry->crlUri = getCRLURIfromCertificate(tempCrt);
                if(!certEntry){
                    certListStart = newCLEntry;
                    certEntry = newCLEntry;
                }
                else {
                    certEntry->next = newCLEntry;
                    certEntry = certEntry->next;
                }
            }
            else{
                freeCertList(&newCLEntry);
                failed = TRUE;
            }
        }
    }
//    ASN_STRUCT_FREE_CONTENTS_ONLY(asn_DEF_Certificate, &temp);
    if(*certList == NULL){
        // empty List --> replace
        *certList = certListStart;
    } else {
        // not empty List --> append
        CERT_LIST_s* temp = findCertListLastItem(*certList);
        temp->next = certListStart;
    }
    return count;
}


BOOL downloadIssuerCertificate(
    Certificate_t *crt,
    UINT8** issuerCrt,
    UINT32* issuerCrtSize)
{
    BOOL failed = FALSE;

    if(!crt || !issuerCrt || !issuerCrtSize ) {
        return TRUE;
    }

    /* get AuthorityInfo Extension --> XCN_OID_AUTHORITY_INFO_ACCESS (1.3.6.1.5.5.6.1.1) */
    unsigned int authInfoOid[9] = {1,3,6,1,5,5,7,1,1};
    Extension_t** extArray = NULL;
    unsigned int i = 0;
    BOOL match = FALSE;
    Certificate_t *tmpCrt = NULL;

    if ( NULL == crt->tbsCertificate.extensions ) {
        return TRUE;
    }
    extArray = (crt->tbsCertificate.extensions->list.array);

    while(!match && i < crt->tbsCertificate.extensions->list.count){
        OBJECT_IDENTIFIER_t *oid = &extArray[i]->extnID;
        unsigned int fixed_arcs[9];
        unsigned int *arcs = fixed_arcs;
        int arc_type_size = sizeof(fixed_arcs[0]);  // sizeof(long)
        int arc_slots = sizeof(fixed_arcs) / sizeof(fixed_arcs[0]); // 10
        int count;  // Real number of arcs.

        count = OBJECT_IDENTIFIER_get_arcs(oid, arcs, arc_type_size, arc_slots);
        if(count != arc_slots){
            /* if count of OID Args do not match try the next OID */
            ++i;
            continue;
        }

        /* Check if OIDs match */
        match = TRUE;
        unsigned int j = 0;
        while(match && j < 9){
            if(authInfoOid[j] != arcs[j])
                match = FALSE;
            ++j;
        }
        ++i;
    }

    if(!match){
        dbgPrint(DBG_INFO, "Could not find \"Authority Info Access OID\"\n");
        return failed = TRUE;
    }

    // get Authority Info as extension value
    UINT8* p_aiVal = extArray[i-1]->extnValue.buf;
    UINT32 bufSize = extArray[i-1]->extnValue.size;


//       find id-ad 2 (issuers uri) --> 1.3.6.1.5.5.7.48.2 in extension value buffer
//       reuse authInfo - change form ...1.1 to ...48.2
    authInfoOid[7] = 48;
    authInfoOid[8] = 2;
    UINT32 oidSize = 9;
    i=0;
    match = FALSE;
    while (!match && i < bufSize-oidSize){
        OBJECT_IDENTIFIER_t oid;
        oid.size=oidSize;
        oid.buf = p_aiVal+i;
        unsigned int fixed_arcs[oidSize];
        unsigned int *arcs = fixed_arcs;
        int arc_type_size = sizeof(fixed_arcs[0]);  // sizeof(long)
        int arc_slots = sizeof(fixed_arcs) / sizeof(fixed_arcs[0]); // 9
        int count;  // Real number of arcs.

        count = OBJECT_IDENTIFIER_get_arcs(&oid, arcs, arc_type_size, arc_slots);

        if(count != arc_slots){
        	i++;
        	continue;
        }
        match = matchUIntArrays(arcs, authInfoOid, oidSize);
        if(match){
        	dbgPrint(DBG_INFO, "Found caIssuers-URI OID\n");
        }

        ++i;
    }

    if(!match){
        dbgPrint(DBG_INFO, "Could not find 'Issuers - URI OID'\n");
        return failed = TRUE;
    }

    // issuer URI starts beyond the OID
    char* issuerUri = (char*) p_aiVal+i+oidSize;
    dbgPrint(DBG_INFO, "caIssuers - URI: %s\n", issuerUri);

    /* Download Certificate to Buffer */
    char* data = NULL;
    UINT32 size = 0;
    failed = downloadFromURItoBuffer(issuerUri, &data, &size);

    ifNotFailed {
        /* If Certificate in PEM Formate -> convert to DER */
        if(checkForPEMCertificate(data)){
            char* derData = NULL;
            unsigned int derSize=0;
            failed = decodePEM((UINT8*)data, size, (UINT8**) &derData, &derSize);
            if(failed){
                free(data);
                free(derData);
                dbgPrint(DBG_INFO, "Could not decode PEM to DER");
                return failed = TRUE;
            }

            /* free pem Certificate */
            free(data);
            data = derData;
            size = derSize;
        }
    }

    ifNotFailed {

        /* we shall also try to decode the data to a certificate */
        failed = alloc((void**)&tmpCrt, sizeof(Certificate_t));
    }

    ifNotFailed {
        failed = asn1Decode( (UINT8 *) data, size, &asn_DEF_Certificate, tmpCrt, FALSE);
    }

    ifNotFailed {
        /* return Issuer Certificate */
        *issuerCrt = (UINT8*) data;
        *issuerCrtSize = size;
    } else {

        *issuerCrt = NULL;
        *issuerCrtSize = 0;
        escFreePtr ( (UINT8 **) &data );
    }

    if ( tmpCrt ) {
        ASN_STRUCT_FREE(asn_DEF_Certificate, tmpCrt);
    }

    return failed;
}

BOOL checkForPEM(char* data){
    if(!data)
        return FALSE;
    /* If certificate, crl, etc. starts with "-----BEGIN " it is most likely PEM coded (rfc1421) */
    if(strncmp((char*)data,"-----BEGIN ", 11) == 0){
        return TRUE;
    }
    else return FALSE;
}

BOOL checkForPEMCertificate(char* data){
    if(!data)
        return FALSE;
    /* If certificate, crl, etc. starts with "-----BEGIN " it is most likely PEM coded (rfc1421) */
    if(strncmp((char*)data,"-----BEGIN CERTIFICATE", strlen("-----BEGIN CERTIFICATE")) == 0){
        return TRUE;
    }
    else return FALSE;
}

BOOL checkForPEMRSAKey(char* data){
    if(!data)
        return FALSE;
    if(strncmp((char*)data,"-----BEGIN RSA PRIVATE KEY-----", strlen("-----BEGIN RSA PRIVATE KEY-----")) == 0){
        return TRUE;
    }
    else return FALSE;
}

BOOL checkForPEMPKCS8Key(char* data){
    if(!data)
        return FALSE;
    if(strncmp((char*)data,"-----BEGIN PRIVATE KEY-----", strlen("-----BEGIN PRIVATE KEY-----")) == 0){
        return TRUE;
    }
    else return FALSE;
}


BOOL matchUIntArrays(unsigned int* array1, unsigned int* array2, unsigned int length){
    if(!array1 || !array2)
        return FALSE;
    BOOL match = TRUE;
    if(length == 0)
        return match = TRUE;
    if(length > 0 && array1[0] == array2[0]){
        match = TRUE;
        match = matchUIntArrays(array1+1, array2+1, length-1);
    }
    else
        match = FALSE;

    return match;
}

#ifdef UNIT_TESTING
/* de-activate unit testing re-directs of malloc() and free() here,
 * since SDC also uses malloc() which however we cannot re-direct and
 * makes the unit test failing */
#undef malloc
#undef free
#endif /* UNIT_TESTING */
BOOL
writeByteStreamToFile(
    SDC_PARAMS_s *sdc,
    const char* fileName,
    UINT8* byteStream,
    UINT32 length)
{
    BOOL failed = FALSE;
    FILE *fp = NULL;
    UINT8 *bsEnc = NULL;
    UINT32 bsEncSize = 0;

    /* sanity checks */
    if ( !sdc || !fileName || !byteStream ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameters!\n", DBG_FL );
        return ( TRUE );
    }

    fp = fopen( fileName, "w" );
#ifdef CONFIG_ECM_USE_SDC

    /* Do SDC encryption if necessary */
    if ( sdc->execSdcOp == TRUE ) {

        sdc->err = sdc_wrap_formatted( sdc->session, sdc->type, byteStream, length, &bsEnc, &bsEncSize );
        failed = checkSdcRet( sdc->err );

    } else {

        /* Just copy otherwise */
        bsEnc = malloc ( length );
        memcpy ( bsEnc, byteStream, length );
        bsEncSize = length;
    }

#else /* CONFIG_ECM_USE_SDC */

    /* No SDC, just copy */
    bsEnc = malloc ( length );
    memcpy ( bsEnc, byteStream, length );
    bsEncSize = length;

#endif /* CONFIG_ECM_USE_SDC */

    if ( fp != NULL ) {
        if ( bsEncSize != fwrite( bsEnc, 1, bsEncSize, fp ) ) {
            failed = TRUE;
			} else {
            sync();
        }
    } else {
        failed = TRUE;
    }
    if ( fp ) {
        fclose( fp );
    }

    if ( bsEnc ) {
        free ( bsEnc );
    }

    return failed;
}

#ifdef UNIT_TESTING
#define free(ptr) _test_free(ptr, __FILE__, __LINE__)
#define malloc(num) _test_malloc(num, __FILE__, __LINE__)
#endif /* UNIT_TESTING */

BOOL
loadFromFileToBuffer(
    SDC_PARAMS_s *sdc, const char* const path, UINT8** byteStream, UINT32* length)
{
    BOOL failed = FALSE;
    FILE *fp = NULL;

    UINT8 *bsRaw = NULL;
    UINT32 bsRawSize = 0;
    long size = 0;

    /* open File */
	if( (fp = fopen(path, "r")) == NULL){
		dbgPrint ( DBG_ERR, DBG_FLS "Error: Failed to open '%s' (%s)\n",
							DBG_FL, path, strerror ( errno ) );
		*byteStream = NULL;
		*length = 0;
		return failed = TRUE;
	}

    /* get File Size */
    ifNotFailed {
        if ( fseek ( fp, 0 , SEEK_END ) != 0 ) {
            dbgPrint ( DBG_ERR, DBG_FLS "Failed to read file size\n", DBG_FL );
            failed = TRUE;
        } else {
            size = ftell(fp);
            if ( size < 0 ) {
                failed = TRUE;
            } else {
                bsRawSize = size;
            }
            rewind ( fp );
        }
        failed |= (bsRawSize > 0) ? FALSE : TRUE;
    }

    /* allocate Buffer */
    ifNotFailed{
    	bsRaw = malloc (bsRawSize);
        failed = (bsRaw == NULL) ? TRUE : FALSE;
    }

    /* read File */
    ifNotFailed{
        size_t ret = 0;
        ret = fread ( bsRaw, 1, bsRawSize, fp );
        if ( ret != bsRawSize ) {
            dbgPrint ( DBG_ERR, DBG_FLS "reading from disk failed\n", DBG_FL );
            free (bsRaw);
            bsRaw = NULL;
            bsRawSize = 0;
            failed = TRUE;
        }
    }

#ifdef CONFIG_ECM_USE_SDC

    ifNotFailed {
        /* Do SDC decryption if necessary */
        if ( sdc->execSdcOp == TRUE ) {

            sdc->err = sdc_unwrap_formatted( sdc->session, sdc->type, bsRaw, bsRawSize, byteStream, length );
            failed = checkSdcRet( sdc->err );

        } else {

            /* Just copy otherwise */
            *byteStream = malloc ( bsRawSize );
            memcpy ( *byteStream, bsRaw, bsRawSize );
            *length = bsRawSize;
        }
    }

#else /* CONFIG_ECM_USE_SDC */

    /* Copy otherwise */
    *byteStream = malloc ( bsRawSize );
    memcpy ( *byteStream, bsRaw, bsRawSize );
    *length = bsRawSize;

#endif /* CONFIG_ECM_USE_SDC */

    if ( bsRaw ) {
        free ( bsRaw );
    }

    /* Close File */
    if(fp != NULL) {
        fclose(fp);
    }

    return failed;
}

BOOL extractValidity (
    Time_t *validity,
    ull_time_t *value )
{
    BOOL failed = FALSE;

    dbgIn (__func__ );

    /* sanity checks */
    if ( !validity || !value ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        switch ( validity->present ) {

            case ( Time_PR_utcTime ):
                *value = asn_UT2time ( &validity->choice.utcTime, NULL, 0 );
                break;

            case ( Time_PR_generalTime ):
                *value = asn_GT2time ( &validity->choice.generalTime, NULL, 0 );
                break;

            default:
                dbgPrint ( DBG_ERR, DBG_FLS "Unsupported time format\n", DBG_FL);
                failed = TRUE;
                break;
        }
    }

    dbgOut (__func__ );

    return ( failed );
}


BOOL checkForTimeValidity(
    CERT_LIST_s* certList,
    STATUS_s *stat,
    ull_time_t time)
{
    BOOL failed = FALSE;
    Certificate_t* cert = NULL;
    if(time == 0 || !certList ){
        //no time set yet
        return TRUE;
    }
    while(certList){
        // if no Certificate_t -> decode
        if ( !certList->crt ) {
            dbgPrint ( DBG_INFO, "Decode device certificate\n" );
            failed = alloc( (void **) &cert, sizeof ( Certificate_t ));
            ifNotFailed{
                failed = asn1Decode ( certList->certPtr, certList->certSize,
                    &asn_DEF_Certificate, cert, FALSE );
            }
        } else{
            cert = certList->crt;
        }

        if(failed){
            if ( !certList->crt ) {
                ASN_STRUCT_FREE(asn_DEF_Certificate, cert);
            }
            return TRUE;
        }

        // get Validity Time
        ull_time_t validUntil = 0;
        ull_time_t validAfter = 0;

        failed = extractValidity(
            &cert->tbsCertificate.validity.notAfter, &validUntil );
        failed |= extractValidity(
            &cert->tbsCertificate.validity.notBefore, &validAfter );

        if(failed){
            if ( !certList->crt ) {
                ASN_STRUCT_FREE(asn_DEF_Certificate, cert);
            }
            return TRUE;
        }

        if(validUntil < time){
            dbgPrint ( DBG_ERR, DBG_FLS "Certificate has expired\n", DBG_FL );
            if ( !certList->crt ) {
                ASN_STRUCT_FREE(asn_DEF_Certificate, cert);
            }
            ecmCmSetStatus( stat, EXT_IF_ERR_CERT_EXPIRED, certList->subjectCn, NULL );
            return TRUE;
        }

        /* 2017-06-19: New requirement ECM shall never check for
         * Effective date or Valid from of CRL and Crt from ECM. */

        if ( !certList->crt ) {
            ASN_STRUCT_FREE(asn_DEF_Certificate, cert);
        }
        certList->validTimestamps = TRUE;
        certList = certList->next;
    }
    return failed;
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/

