/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
 \file        ecm_cm_EventHandler.c

 \brief       ECM CM Event Handler

 $Rev: 832 $
 */
/***************************************************************************
 $Author: mlange $
 $Date: 2017-09-13 10:07:21 +0200 (Mi, 13. Sep 2017) $
 ****************************************************************************/

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include <stddef.h>
#include <unistd.h>
#include <pthread.h>
#include <dbus/dbus.h>
#include <signal.h>

#include "../inc/ecm.h"
#include "../inc/ecm_cm_tools.h"
#include "../inc/ecm_cm_main.h"
#include "../../cmp/inc/asn1_utils.h"
#include "../inc/ecm_utils.h"
#include "../../common/inc/esc_common.h"
#include "../../common/inc/esc_debug.h"
#include "../../common/inc/esc_utils.h"
#include "../../common/inc/esc_asn1_utils.h"
#include "../../cycurlib/lib/inc/pkcs1_v15.h"
#include "pkcs10_utils.h"
#include "../../cmp/inc/cmp_utils.h"
#include <crl_utils.h>
#include <base64.h>
#include "../inc/ecm_cm_EventHandler.h"

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

typedef struct {
    DBusConnection* conn;
    DBusError err;
    DBusPendingCall* pending;
    DBusMessage* req;
    DBusMessage* rep;
    DBusMessageIter args;
} DBUS_PARAMS_s;

DBUS_PARAMS_s dbus;

static void* eventHandlerLoop(void* arg);

static void ecmCmReplyToCheckCert(
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus );

static void ecmCmReplyToPushCert(
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus );

static void ecmCmReplyToPushCrl(
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus );

static void ecmCmReplyToGetCrlNextUpd (
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus );

static void
ecmCmReplyToGetCsr(
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus );

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

BOOL ecmCmDbusInit()
{
    BOOL failed = FALSE;
    int ret;

    dbgIn ( __func__ );

    dbus_error_init(&dbus.err);

    /* connect to DBus */
    dbus.conn = dbus_bus_get(DBUS_BUS_SESSION, &dbus.err);

    if (dbus_error_is_set(&dbus.err)) {
        dbgPrint( DBG_ERR, "DBus 'dbus_bus_get()' failed (%s)\n", dbus.err.message);
        failed = TRUE;
    }

    ifNotFailed {
        if (NULL == dbus.conn) {
            dbgPrint( DBG_ERR, "DBus connection failed\n");
            failed = TRUE;
        }
    }

    ifNotFailed {

        /* register out name on DBus */
        ret = dbus_bus_request_name(
            dbus.conn, "com.escrypt.ecm", DBUS_NAME_FLAG_REPLACE_EXISTING, &dbus.err);

        if (dbus_error_is_set(&dbus.err)) {
            dbgPrint( DBG_ERR, "DBus 'dbus_bus_request_name() failed (%s)\n", dbus.err.message);
            failed = TRUE;
        }
    }

    ifNotFailed {
        if (DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER != ret) {
            dbgPrint( DBG_ERR, "DBus not primary owner (%d)\n", ret);
            failed = TRUE;
        }
    }

    /* add rule for for signals we want to see (online trigger) */
    ifNotFailed {

        dbus_bus_add_match(dbus.conn, "type='signal',interface='org.bosch.connectivity.fc_connect'", &dbus.err);
        dbus_connection_flush(dbus.conn);
        if (dbus_error_is_set(&dbus.err)) {
           dbgPrint(DBG_ERR, "DBus 'dbus_bus_add_match() failed (%s)\n", dbus.err.message);
           failed = TRUE;
        }
    }

    ifFailed {
        dbus_error_free(&dbus.err);
        if ( dbus.conn ) {
            dbus_connection_unref(dbus.conn);
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

pthread_t startEventHandler(
    ECM_PARAMS_t* ecm)
{
    BOOL failed = FALSE;
    pthread_t tid = 0;
    pthread_attr_t t_attr;
    int tmp = 0;

    EventHandlerData* evh = calloc(1, sizeof(EventHandlerData));

    exitCMEventHandler = FALSE;
    ecm->certConfigValid = FALSE;

    /* set pointer to ECM global structure */
    evh->ecm = ecm;

    /* create DBus thread */
    ifNotFailed {
        pthread_attr_init(&t_attr);
        tmp = pthread_create(&tid, &t_attr, eventHandlerLoop, (void *) evh);
    }

    if (!tmp){
        return tid;
    } else {
        return 0;
    }
}

ECMCM_DBUS_ERROR_t ecmCmPushTimeToStm (
    STC_INTERFACE_e i,
    ull_time_t time )
{
    ECMCM_DBUS_ERROR_t retVal = DBUS_SUCCESS;
    dbus_uint64_t dbusParam;
    DBusMessage* req = NULL;
    DBusMessage* rep = NULL;
    DBusPendingCall* pending = NULL;

    dbgIn ( __func__ );

    dbusParam = (dbus_uint64_t) time;

    switch ( i ) {
        case ( stc_interface_none ):
        case ( stc_interface_max ):
        default:
            retVal = DBUS_ARGS_INVALID;
            break;

        case ( stc_interface_set_time ):

            // create a new method call and check for errors
            req = dbus_message_new_method_call(
                "FC_STC.FC_STCApp", // target for the method call
                "/org/bosch/stc/fc_stc", // object to call on
                "org.bosch.stc.stcservices", // interface to call on
                "SetTime"); // method name
            break;

        case ( stc_interface_provide_cea_time ):
            // create a new method call and check for errors
            req = dbus_message_new_method_call(
                "FC_STC.FC_STCApp", // target for the method call
                "/org/bosch/stc/fc_stc", // object to call on
                "org.bosch.stc.stcservices", // interface to call on
                "ProvideCEAtime"); // method name

            break;
    }

    if ( DBUS_SUCCESS == retVal ) {
        if (NULL == req) {
            dbgPrint ( DBG_ERR, "DBus 'dbus_message_new_method_call()' failed\n");
            retVal = DBUS_MESSAGE_CREATION_FAILED;
        }
    }

    /* append time */
    if ( DBUS_SUCCESS == retVal ) {

        if (!dbus_message_append_args(req, DBUS_TYPE_UINT64, &dbusParam, DBUS_TYPE_INVALID)) {
            dbgPrint ( DBG_ERR, "DBus 'dbus_message_append_args()' failed\n");
            retVal = DBUS_APPEND_ARGS_FAILED;
        }
    }

    if ( DBUS_SUCCESS == retVal ) {

        // send message and get a handle for a reply
        if (!dbus_connection_send_with_reply (dbus.conn, req, &pending, -1)) { // -1 is default timeout
            dbgPrint ( DBG_ERR, "DBus 'dbus_connection_send_with_reply()' failed\n");
            retVal = DBUS_SEND_MESSAGE_WITH_REPLY_FAILED;
        }
    }

    if ( DBUS_SUCCESS == retVal ) {
        if (NULL == pending) {
            dbgPrint ( DBG_ERR, "DBus pending pointer is NULL\n");
            retVal = DBUS_PENDING_IS_NULL;
        }
    }

    if ( DBUS_SUCCESS == retVal ) {

        /* send message on bus */
        dbus_connection_flush(dbus.conn);

        /* block until we receive a reply */
        dbus_pending_call_block(pending);

        /* get the reply message */
        rep = dbus_pending_call_steal_reply(pending);
        if (NULL == rep) {
            dbgPrint ( DBG_ERR, "DBus reply message is NULL\n");
            retVal = DBUS_REPLY_IS_NULL;
        }
    }

    if ( DBUS_SUCCESS == retVal ) {

        /* free the pending message handle */
        dbus_pending_call_unref(pending);
    }

    dbus_error_free(&dbus.err);

    /* free request message */
    if ( req ) {
        dbus_message_unref(req);
    }

    /* free reply message */
    if ( rep ) {
        dbus_message_unref(rep);
    }

    dbgOut ( __func__ );

    return (retVal);
}

/* STC ... Secure Time Module */
ECMCM_DBUS_ERROR_t ecmCmGetTimeFromStm (
    ull_time_t *currTime,
    BOOL quiet )
{
    ECMCM_DBUS_ERROR_t retVal = DBUS_SUCCESS;
    dbus_uint64_t dbusRet;

    DBusMessage* req = NULL;
    DBusMessage* rep = NULL;
    DBusPendingCall* pending = NULL;

    dbgIn(__func__);

    /* sanity checks */
    if ( !currTime ) {
        if ( FALSE == quiet ) {
            dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        }
        retVal = DBUS_ARGS_INVALID;
    }

    if ( DBUS_SUCCESS == retVal ) {

        // create a new method call and check for errors
        req = dbus_message_new_method_call(
            "FC_STC.FC_STCApp", // target for the method call
            "/org/bosch/stc/fc_stc", // object to call on
            "org.bosch.stc.stcservices", // interface to call on
            "GetTime"); // method name

        if (NULL == req) {
            if ( FALSE == quiet ) {
                dbgPrint ( DBG_ERR, "DBus 'dbus_message_new_method_call()' failed\n");
            }
            retVal = DBUS_MESSAGE_CREATION_FAILED;
        }
    }

    if ( DBUS_SUCCESS == retVal ) {

        // send message and get a handle for a reply
        if (!dbus_connection_send_with_reply (dbus.conn, req, &pending, -1)) { // -1 is default timeout
            if ( FALSE == quiet ) {
                dbgPrint ( DBG_ERR, "DBus 'dbus_connection_send_with_reply()' failed\n");
            }
            retVal = DBUS_SEND_MESSAGE_WITH_REPLY_FAILED;
        }
    }

    if ( DBUS_SUCCESS == retVal ) {
        if (NULL == pending) {
            if ( FALSE == quiet ) {
                dbgPrint ( DBG_ERR, "DBus pending pointer is NULL\n");
            }
            retVal = DBUS_PENDING_IS_NULL;
        }
    }

    if ( DBUS_SUCCESS == retVal ) {

        /* send message on bus */
        dbus_connection_flush(dbus.conn);

        /* block until we receive a reply */
        dbus_pending_call_block(pending);

        /* get the reply message */
        rep = dbus_pending_call_steal_reply(pending);
        if (NULL == rep) {
            if ( FALSE == quiet ) {
                dbgPrint ( DBG_ERR, "DBus reply message is NULL\n");
            }
            retVal = DBUS_REPLY_IS_NULL;
        }
    }

    if ( DBUS_SUCCESS == retVal ) {

        /* free the pending message handle */
        dbus_pending_call_unref(pending);

        /* read message arguments */
        if(FALSE == dbus_message_get_args(rep, &dbus.err, DBUS_TYPE_UINT64, &dbusRet, DBUS_TYPE_INVALID))
        {
            if ( FALSE == quiet ) {
                dbgPrint ( DBG_ERR, "DBus read reply args failed\n");
            }
            retVal = DBUS_READ_REPLY_ARGS_FAILED;
        }
    }

    if ( DBUS_SUCCESS == retVal ) {
        dbgPrint (DBG_INFO, "Time stamp received from STC %llu (dbus_uint64_t)\n", dbusRet );
        *currTime = dbusRet;
        dbgPrint (DBG_INFO, "Time stamp received from STC %llu (ull_time_t)\n", *currTime );
    } else {
        dbus_error_free(&dbus.err);
    }

    /* free request message */
    if ( req ) {
        dbus_message_unref(req);
    }

    /* free reply message */
    if ( rep ) {
        dbus_message_unref(rep);
    }

    dbgOut(__func__);

    return ( retVal );
}

BOOL ecmCmResolveErrorsByDownload (
    ECM_PARAMS_t *ecm )
{
    BOOL failed = FALSE;
    Certificate_t* cert = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ecm ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        switch ( ecm->status.err ) {

            case (EXT_IF_ERR_CERT_INVALID ):
            case (EXT_IF_ERR_CERT_EXPIRED ):
            case (EXT_IF_ERR_CERT_REVOKED ):
            case ( EXT_IF_ERR_CERT_NOT_PRESENT ):

                /* which certificate has the error? */
                if ( strncmp (ecm->status.str, ecm->pkcs10Data.deviceUniqueId, strlen ( ecm->pkcs10Data.deviceUniqueId)) == 0 )
                {
                    /* device certificate */
                    UINT8 *data = NULL;
                    UINT32 dataSize = 0;

                    failed = downloadDeviceCertificate2 ( ecm->pkcs10Data.deviceUniqueId, ecm->pkcs10Data.devCertUri, &data, &dataSize );

                    failed = alloc((void**)&cert, sizeof(Certificate_t));
                    if(failed){
                        free(data);
                        return -3;
                    }

                    failed = asn1Decode(
                        data, dataSize, &asn_DEF_Certificate, cert, FALSE);


                    if(failed){
                        ASN_STRUCT_FREE(asn_DEF_Certificate, cert);
                        free(data);
                        return -3;
                    }

                    // check if Certificate is really the Device Certificate
                    if(!ecm->pkcs10Data.pkcs1PubKey) {
                        return -1;
                    }

                    /* use cmpDetermineCrtType()? */
                    if( memcmp (
                            cert->tbsCertificate.subjectPublicKeyInfo.subjectPublicKey.buf,
                            ecm->pkcs10Data.pkcs1PubKey,
                            cert->tbsCertificate.subjectPublicKeyInfo.subjectPublicKey.size )
                            != 0 )
                    {
                        failed = TRUE;
                        ASN_STRUCT_FREE(asn_DEF_Certificate, cert);
                        free(data);
                        dbgPrint(DBG_ERR, "Downloaded Device Certificate does not match our key\n");
                        // not recoverable
                        return -1;
                    }

                    // Device Certificate belongs to this device --> append to CERT_LIST
                    CERT_LIST_s* devCertEntry = calloc(1, sizeof(CERT_LIST_s));
                    if ( NULL != devCertEntry ) {
                        devCertEntry->certPtr = data;
                        devCertEntry->certSize = dataSize;
                        devCertEntry->crlUri = getCRLURIfromCertificate(cert);
                        devCertEntry->crt = cert;
                        devCertEntry->certId = cert_id_device;
                        devCertEntry->devChain = TRUE;
                        devCertEntry->store = TRUE;
                        failed = cmpExtractCrtParamsFromTBSCertificate (
                            &cert->tbsCertificate, devCertEntry );

                        ifNotFailed {
                            CERT_LIST_s* tempCertList = ecm->certList;
                            while(tempCertList->next){
                                tempCertList = tempCertList->next;
                            }
                            tempCertList->next = devCertEntry;
                        }
                    }

                } else {

                    /* other certificate */
                    UINT8 *data = NULL;
                    UINT32 dataSize = 0;
                    Certificate_t *cert = NULL;

                    CERT_LIST_s *tmp;
                    tmp = checkCertListForCertByIssuerCN ( ecm->certList, ecm->status.str );

                    if ( NULL == tmp) {

                        /* we do not have a certificate that is issued by the one we need to retieve
                         * this means that we cannot determine the download URL and
                         * cannot recover from this situation even if we are online */

                        /* keep status and break */
                        dbgPrint (DBG_INFO, "No certificate available that contains download URL of certificate 'issuer CN=%s'\n", ecm->status.str);
                        break;
                    }

                    /* do we need to decode the certificate first? */
                    if ( NULL == tmp->crt ) {
                        failed = alloc ((void **)&tmp->crt, sizeof(Certificate_t));
                        ifNotFailed {
                            failed = asn1Decode( tmp->certPtr, tmp->certSize, &asn_DEF_Certificate, tmp->crt, FALSE );
                        }
                    }

                    ifNotFailed {
                        failed = downloadIssuerCertificate(tmp->crt, &data, &dataSize );
                    }

                    ifNotFailed {
                        failed = alloc((void**)&cert, sizeof(Certificate_t));
                    }

                    ifNotFailed {
                        failed = asn1Decode( data, dataSize, &asn_DEF_Certificate, cert, FALSE);
                    }

                    if(failed){
                        if ( cert ) {
                            ASN_STRUCT_FREE(asn_DEF_Certificate, cert);
                        }
                        escFreePtr((UINT8 **)&data);
                        return -3;
                    }

                    // Device Certificate belongs to this device --> append to CERT_LIST
                    CERT_LIST_s* devCertEntry = calloc(1, sizeof(CERT_LIST_s));
                    devCertEntry->certPtr = data;
                    devCertEntry->certSize = dataSize;
                    devCertEntry->crlUri = getCRLURIfromCertificate(cert);
                    devCertEntry->crt = cert;
                    devCertEntry->store = TRUE;
                    failed = cmpExtractCrtParamsFromTBSCertificate (
                        &cert->tbsCertificate, devCertEntry );

                    /* check if we downloaded a root certificate */
                    ifNotFailed {
                        if ( strncmp ( devCertEntry->issuerCn, devCertEntry->subjectCn,
                            strlen ( devCertEntry->subjectCn )) == 0 ) {

                            dbgPrint (DBG_INFO,
                                "Root certificate downloaded 'subject CN=%s'. This happens if previous processed certificates are not part of the certificate chain the local root certificate belongs to.\n", devCertEntry->subjectCn);
                            failed = TRUE;
                        }
                    }

                    ifNotFailed {
                        CERT_LIST_s* tempCertList = ecm->certList;
                        while(tempCertList->next){
                            tempCertList = tempCertList->next;
                        }
                        tempCertList->next = devCertEntry;

                        failed = cmpDetermineCrtTypesAndHierarchy2( ecm->certList, ecm->pkcs10Data.pkcs1PubKey, ecm->pkcs10Data.pkcs1PubKeySize );
                    }
                }

                break;

            /* CRL is
             * - missing
             * - has an invalid signature
             * - is outdated,
             * try to download again or reply with appropriate error */
            case ( EXT_IF_ERR_CRL_NOT_PRESENT ):
            case ( EXT_IF_ERR_CRL_INVALID ):
            case ( EXT_IF_ERR_CRL_EXPIRED ):
            {
                CR_LIST_t* crList = NULL;
                int dlret = 0;

                ull_time_t crtime = ecm->currentTime;
                ull_time_t createTime = 0, nextUpdTime = 0;

                /* shall we download a particular CRL, all CRLs or do we have the download URL at all?
                 * This is signaled by ecm->status.uri:
                 * - == NULL             ... URL not available
                 * - == "all"            ... download all
                 * - == any other string ... download the particular one from the given uri */
                if ( ecm->status.url == NULL ) {

                    dbgPrint( DBG_INFO, "CRL URI for download not available.\n");

                } else {

                    if ( strncmp ( ecm->status.url, "all", strlen ( "n/a")) == 0 ) {

                        /* free local list first */
                        freeCRL ( ecm->crList );

                        /* internally pushes CRL time to STC */
                        dlret = loadCRLandUpdateTime( ecm->certList, &ecm->crList, &crtime, &ecm->crListHash, &ecm->status);

                    } else {

                        dlret = addCRLfromURI(ecm->status.url, &crList, ecm->certList, &createTime, &nextUpdTime, &ecm->crListHash, &ecm->status );

                        if ( dlret >= 0 ) {

                            /* in case new revoked certificates where received */

                            if ( crList ) {
                                /* remove list entries from the current issuer */
                                removeCrListEntryByIssuerCn(crList->issuerCN, &ecm->crList );

                                /* append received list */
                                CR_LIST_t **ptr = &ecm->crList;
                                while ( *ptr ) {
                                    ptr = &(*ptr)->next;
                                }

                                *ptr = crList;
                            }

                            /* push CRL time to STC */
                            ecmCmPushTimeToStm( stc_interface_provide_cea_time, createTime );
                        }
                    }
                }
                break;
            }

            case ( EXT_IF_ERR_UIN_FILE_NOT_AVAILABLE ):
                /* cannot recover this error in online mode */
                failed = TRUE;
                break;

            default:
                break;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL ecmCmSendDbusReply (
    STATUS_s *stat,
    DBUS_PARAMS_s *dbus,
    DBUS_REPLY_e type,
    void *val )
{
    BOOL failed = FALSE;
    DBusMessage* reply = NULL;
    dbus_uint32_t serial = 0;
    DBusError err;
    char *nullStr = "null";

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !stat || !dbus ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        dbus_error_init(&err);

        reply = dbus_message_new_method_return(dbus->req);

        /* append error code */
        if ( !dbus_message_append_args ( reply, DBUS_TYPE_UINT32, &stat->err, DBUS_TYPE_INVALID )) {
            failed = TRUE;
        }
    }

    /* append error string if available */
    ifNotFailed {
        if ( stat->str ) {
            if ( !dbus_message_append_args ( reply, DBUS_TYPE_STRING, &stat->str, DBUS_TYPE_INVALID )) {
                failed = TRUE;
            }
        } else {
            /* "null" otherwise */
            if ( !dbus_message_append_args ( reply, DBUS_TYPE_STRING, &nullStr, DBUS_TYPE_INVALID )) {
                failed = TRUE;
            }
        }
    }

    /* append reply type specific arguments */
    ifNotFailed {
        if ( type == dbusRep_getCrlNextUpd ) {

            dbus_uint64_t nextUpd = 0;

            if ( val ) {
                nextUpd = *((ull_time_t *) val );
            }

            if ( !dbus_message_append_args ( reply, DBUS_TYPE_UINT64, &nextUpd, DBUS_TYPE_INVALID )) {
                failed = TRUE;
            }
        }

        if ( type == dbusRep_getCsr ) {

            DBUS_GET_CSR_REPLY_s *r;
            DBUS_GET_CSR_REPLY_s rep;

            if ( !val ) {
                memset ( &rep, 0x0, sizeof ( rep ));
                r = &rep;

            } else {

                r = (DBUS_GET_CSR_REPLY_s *) val;
            }

            if ( !dbus_message_append_args ( reply, DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE,
                    &r->csr, r->csrLen, DBUS_TYPE_INVALID)) {
                failed = TRUE;
            }
        }
    }

    ifNotFailed {

        // send the reply && flush the connection
        if (!dbus_connection_send(dbus->conn, reply, &serial)) {
            dbgPrint(DBG_ERR, "Warning: DBUS reply to Call failed\n");
            failed = TRUE;
        }
    }

    ifNotFailed {
        dbus_connection_flush(dbus->conn);
    }

    // free the reply
    if ( reply ) {
        dbus_message_unref(reply);
    }

    dbgOut ( __func__ );

    return ( failed );
}

BOOL ecmCmCheckForCsrAndReply (
    ECM_PARAMS_t *ecm,
    DBUS_REPLY_e r )
{
    BOOL replied = FALSE;

    STATUS_s stat;
    stat.str = "null";

    /* process DBus message only if CSR was created */
    if ( FALSE == ecm->pkcs10Data.csrRequested ) {
        dbgPrint( DBG_INFO, "Cannot handle DBus request since CSR was not requested yet\n");

        /* send reply */
        stat.err = EXT_IF_ERR_CSR_NOT_REQUESTED_YET;

        ecmCmSendDbusReply ( &stat, &dbus, r, NULL );

        replied = TRUE;
    }

    return ( replied );
}

BOOL ecmCmCheckForKeyAndReply (
    ECM_PARAMS_t *ecm,
    BOOL keyExpected,
    DBUS_REPLY_e r )
{
    BOOL replied = FALSE;

    STATUS_s stat;
    stat.str = "null";

    if ( TRUE == keyExpected ) {

        /* process DBus message only if device key is available */
        if ( NULL == ecm->pkcs10Data.keyPair ) {
            dbgPrint( DBG_INFO, "Cannot handle DBus request since device key is not available\n");

            /* send reply */
            stat.err = EXT_IF_ERR_PRIV_KEY_MISSING;

            ecmCmSendDbusReply ( &stat, &dbus, r, NULL );

            replied = TRUE;
        }

    } else {

        /* process DBus message only if device key is not available */
        if ( NULL != ecm->pkcs10Data.keyPair ) {
            dbgPrint( DBG_INFO, "Cannot handle DBus request since device key is already created\n");

            /* send reply */
            stat.err = EXT_IF_ERR_PRIV_KEY_ALREADY_GENERATED;

            ecmCmSendDbusReply ( &stat, &dbus, r, NULL );

            replied = TRUE;
        }
    }

    return ( replied );
}

static void* eventHandlerLoop(
    void* arg)
{
    static int retVal = 0;
    void* pretVal = &retVal;

    EventHandlerData *evh = (EventHandlerData*) arg;

    dbgPrint( DBG_INFO, "Started Event Handler Thread\n");

    while(!exitCMEventHandler){

        /* try reading DBus message */
        dbus_connection_read_write(dbus.conn, 0);
        dbus.req = dbus_connection_pop_message(dbus.conn);

        /* something received? */
        if ( dbus.req != NULL ) {

            // check for online Trigger and set Flag
            // (DBus - listen for Signal SigClientRouteConnectionStatus in FC_Connect with arg ClientOnlineState)
            // check if the message is a signal from the correct interface and with the correct name
            if (dbus_message_is_signal(
                    dbus.req,
                    "org.bosch.connectivity.fc_connect",
                    "SigClientRouteConnectionStatus"))
            {

                /* process DBus message only if device key is available */
                if ( ecmCmCheckForKeyAndReply( evh->ecm, TRUE, dbusRep_generic )) {
                    continue;
                }

                /* process DBus message only if CSR was created */
                if ( ecmCmCheckForCsrAndReply(evh->ecm, dbusRep_generic )) {
                    continue;
                }

                /* read DBus parameters */
                if (!dbus_message_iter_init(dbus.req, &dbus.args)) {
                    dbgPrint(DBG_ERR, "Invalid DBus online trigger (no argument), drop message\n");
                } else if (DBUS_TYPE_BOOLEAN != dbus_message_iter_get_arg_type(&dbus.args)) {
                    dbgPrint(DBG_ERR, "Invalid DBus online trigger (invalid argument), drop message\n");
                } else {
                    BOOL online = FALSE;
                    dbus_message_iter_get_basic(&dbus.args, &online);

                    dbgPrint(DBG_INFO, "DBus fc_connect trigger received (%s)\n", (( online == TRUE) ? "online" : "offline" ));

                    if ( online ) {

                        BOOL failed = FALSE;

                        /* online trigger */
                        evh->ecm->networkAvailable = TRUE;

                        /* we are online now, do we need to do something?
                         * Also consider missing EC CRL as an error in this case */
                        failed = ecmCmCheckForErrors( evh->ecm, FALSE );

                        STATUS_s lastStat;
                        memset ( &lastStat, 0x0, sizeof ( STATUS_s ));

                        /* yes we have errors to resolve.
                         * Try as long as the error code changes,
                         * if we get stuck, give up */
                        while (( evh->ecm->status.err != EXT_IF_ERR_NO_ERROR ) &&
                            ((evh->ecm->status.err != lastStat.err) || (strncmp(evh->ecm->status.str, lastStat.str, strlen (lastStat.str)) != 0 )))
                        {
                            lastStat.err = evh->ecm->status.err;
                            escFreePtr( (UINT8 **)&lastStat.str);
                            if ( evh->ecm->status.str ) {
                                lastStat.str = strdup(evh->ecm->status.str);
                            }

                            if ( evh->ecm->status.url ) {
                                lastStat.url = strdup(evh->ecm->status.url);
                            }

                            failed = ecmCmResolveErrorsByDownload( evh->ecm );
                            failed = ecmCmCheckForErrors( evh->ecm, FALSE );
                        }

                        escFreePtr( (UINT8 **)&lastStat.str);

                        ifFailed {
                            dbgPrint ( DBG_INFO, "Could not resolve error even if we are online.\n");
                        }

                        /* do we need to store certificates or CRLs? */
                        ifNotFailed {
                            failed = ecmCmStoreCrls( evh->ecm );
                        }
                        ifNotFailed {
                            failed = ecmCmStoreCerts( evh->ecm );
                        }

                        /* remove invalid CRLs that are still available in our local list */
                        removeInvalidCrlHashEntries( evh->ecm );

                    } else {
                        /* offline trigger */
                        evh->ecm->networkAvailable = FALSE;
                    }
                }
            }

            /* check for key generation request */
            if (dbus_message_is_method_call(dbus.req, "com.escrypt.ecm", "genDevKey")){

                /* process DBus message only if device key is not available */
                if ( ecmCmCheckForKeyAndReply( evh->ecm, FALSE, dbusRep_generic )) {
                    continue;
                }

                /* reply first and generate key afterwards */
               dbgPrint( DBG_INFO, "Start generating device key\n");

                /* send reply */
                STATUS_s stat;
                stat.err = EXT_IF_ERR_PRIV_KEY_GEN_STARTED;
                stat.str = "null";
                ull_time_t t = 0;

                ecmCmSendDbusReply ( &stat, &dbus, dbusRep_generic, (void *)&t );

                /* generate key here */

                EscRandom_ContextT randCtx;
                memset(&randCtx, 0, sizeof(EscRandom_ContextT));

                if(createKeyFnc( evh->ecm )){
                    dbgPrint(DBG_ERR, "Could not create Key.\n");
                } else{
                    // Try to load Key again
                    if(loadKeyCM( evh->ecm )){
                        dbgPrint(DBG_ERR, "Could not load Key.\n");
                    }
                }

                continue;
            }

            /* PKCS10 generation request */
            if (dbus_message_is_method_call(dbus.req, "com.escrypt.ecm", "getCSR")){

                dbgPrint( DBG_INFO, "DBus 'getCSR' message received\n");

                /* process DBus message only if device key is available */
                if ( ecmCmCheckForKeyAndReply ( evh->ecm, TRUE, dbusRep_getCsr )) {
                    continue;
                }

                ecmCmReplyToGetCsr ( evh->ecm, &dbus );
            }

            /* Retrieve CRL time stamp for nextUpdate */
            if (dbus_message_is_method_call(dbus.req, "com.escrypt.ecm", "getCrlNextUpd")){

                dbgPrint( DBG_INFO, "DBus 'getCrlNextUpd' message received\n");

                /* process DBus message only if device key is available */
                if ( ecmCmCheckForKeyAndReply( evh->ecm, TRUE, dbusRep_getCrlNextUpd )) {
                    continue;
                }

                /* process DBus message only if CSR was created */
                if ( ecmCmCheckForCsrAndReply(evh->ecm, dbusRep_getCrlNextUpd )) {
                    continue;
                }

                /* process request */
                ecmCmReplyToGetCrlNextUpd( evh->ecm, &dbus );
            }

            /* CertCheck Request */
            if (dbus_message_is_method_call(dbus.req, "com.escrypt.ecm", "checkCert")){

                dbgPrint( DBG_INFO, "DBus 'checkCert' message received\n");

                /* process DBus message only if device key is available */
                if ( ecmCmCheckForKeyAndReply( evh->ecm, TRUE, dbusRep_generic )) {
                    continue;
                }

                /* process DBus message only if CSR was created */
                if ( ecmCmCheckForCsrAndReply(evh->ecm, dbusRep_generic  )) {
                    continue;
                }

                /* check for errors first - to detect outdated certificates / CRLs */
                ecmCmCheckForErrors( evh->ecm, TRUE );

                /* are we online? */
                if ( evh->ecm->networkAvailable == TRUE) {
                    
                    STATUS_s lastStat;
                    memset (&lastStat, 0x0, sizeof(STATUS_s));

                    /* yes, do we have an error to resolve? */
                    while (( evh->ecm->status.err != EXT_IF_ERR_NO_ERROR ) && (evh->ecm->status.err != lastStat.err) ) {
                        lastStat.err = evh->ecm->status.err;
                        ecmCmResolveErrorsByDownload( evh->ecm );
                        ecmCmCheckForErrors( evh->ecm, TRUE );

                        /* break if error cannot be fixed (old error == new error)??? */
                    }
                }

                /* is there still an error? */
                if ( evh->ecm->status.err != EXT_IF_ERR_NO_ERROR ) {
                    /* yes, we are in an error state, reply accordingly */
                    ecmCmSendDbusReply( &evh->ecm->status, &dbus, dbusRep_generic, NULL );
                } else {
                    /* process certificate do checks and resolve errors, if any */
                    ecmCmReplyToCheckCert( evh->ecm, &dbus );
                }
            }

            /* Check for external PushCert request */
            if ( dbus_message_is_method_call ( dbus.req, "com.escrypt.ecm", "pushCert")) {

                dbgPrint(DBG_INFO, "DBus 'pushCert' message received\n");

                /* process DBus message only if device key is available */
                if ( ecmCmCheckForKeyAndReply( evh->ecm, TRUE, dbusRep_generic )) {
                    continue;
                }

                /* process DBus message only if CSR was created */
                if ( ecmCmCheckForCsrAndReply(evh->ecm, dbusRep_generic  )) {
                    continue;
                }

                /* do not check for errors first,
                 * since pushCert() / pushCRL() is used to resolve errors */

                ecmCmReplyToPushCert( evh->ecm, &dbus );
            }

            /* Check for external PushCRL request */
            if ( dbus_message_is_method_call ( dbus.req, "com.escrypt.ecm", "pushCrl")) {

                dbgPrint(DBG_INFO, "DBus 'pushCRL' message received\n");

                /* process DBus message only if device key is available */
                if ( ecmCmCheckForKeyAndReply( evh->ecm, TRUE, dbusRep_generic )) {
                    continue;
                }

                /* process DBus message only if CSR was created */
                if ( ecmCmCheckForCsrAndReply(evh->ecm, dbusRep_generic  )) {
                    continue;
                }

                /* do not check for errors first,
                 * since pushCert() / pushCRL() is used to resolve errors */

                ecmCmReplyToPushCrl( evh->ecm, &dbus );
            }

            if ( dbus.req ) {
                dbus_message_unref(dbus.req);
            }
        }

        usleep(100000);
    }

    // free EventHandler Data
    free(evh);

    /* close DBus connection */
    dbus_connection_unref(dbus.conn);
    dbgPrint( DBG_INFO, "DBus Service stopped\n");

    return ( pretVal);
}

static BOOL ecmCmCompleteCertChain (
    CERT_LIST_s *crtList,
    CERT_LIST_s *locCrtList,
    STATUS_s *stat,
    BOOL online )
{
    BOOL failed = FALSE;

    dbgIn ( __func__ );

    CERT_LIST_s *issuer;
    CERT_LIST_s *currCrt;
    UINT8 *dlCrtBin = NULL;
    UINT32 dlCrtSize = 0;
    Certificate_t *dlCrt = NULL;

    /* sanity checks */
    if ( !crtList || !locCrtList || !stat ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        currCrt = crtList;

        /* continue until we resolved all dependencies, do not set status */
        while ( checkCertListForCompleteChain ( crtList, NULL, FALSE )) {

            issuer = checkCertListForCertBySubjectCN (locCrtList, currCrt->issuerCn );

            if ( issuer ) {

                /* found issuer at local list, copy certificate */
                dbgPrint ( DBG_INFO, "Issuer certificate 'CN=%s' found in local list\n", currCrt->issuerCn );
                copyCertListItem ( issuer, &currCrt->next );

            } else {

                if ( FALSE == online ) {
                    /* we are not online, return with error code */
                    ecmCmSetStatus( stat, EXT_IF_ERR_CERT_NOT_PRESENT, currCrt->issuerCn, NULL );
                    break;
                }

                /* issuer not available at local list, download */
                dbgPrint ( DBG_INFO, "Need to download issuer certificate 'subject CN=%s'\n", currCrt->issuerCn );
                failed = downloadIssuerCertificate(currCrt->crt, &dlCrtBin, &dlCrtSize);

                ifNotFailed {
                    failed = alloc((void**) &dlCrt, sizeof(Certificate_t));
                }

                if(!failed) {
                    failed = asn1Decode( dlCrtBin, dlCrtSize, &asn_DEF_Certificate, dlCrt, FALSE);
                }

                ifNotFailed {

                    currCrt->next = calloc(1, sizeof(CERT_LIST_s));
                    if ( NULL == currCrt->next ) {
                        failed = TRUE;
                    }
                }
                ifNotFailed {
                    currCrt->next->certPtr = calloc ( 1, dlCrtSize );
                    if ( NULL == currCrt->next->certPtr ) {
                        failed = TRUE;
                    }
                }
                ifNotFailed {
                    memcpy( currCrt->next->certPtr, dlCrtBin, dlCrtSize);
                    currCrt->next->certSize = dlCrtSize;
                    currCrt->next->crlUri = getCRLURIfromCertificate(dlCrt);
                    currCrt->next->crt = dlCrt;

                    failed = cmpExtractCrtParamsFromTBSCertificate (
                        &dlCrt->tbsCertificate, currCrt->next );
                }

                /* check if we downloaded a root certificate */
                ifNotFailed {
                    if ( strncmp ( currCrt->next->issuerCn, currCrt->next->subjectCn,
                        strlen ( currCrt->next->subjectCn )) == 0 ) {

                        dbgPrint (DBG_INFO,
                            "Root certificate downloaded 'subject CN=%s'. "\
                            "This happens if previous processed certificates "\
                            "are not part of the certificate chain the "\
                            "local root certificate belongs to.\n", currCrt->next->subjectCn);
                        failed = TRUE;
                    }
                }
            }

            ifNotFailed {

                currCrt = currCrt->next;

            } else {

                dbgPrint ( DBG_INFO, "Download of issuer certificate 'subject CN=%s' failed!\n", currCrt->issuerCn );
                ecmCmSetStatus( stat, EXT_IF_ERR_CERT_NOT_PRESENT, currCrt->issuerCn, NULL );
                if ( currCrt->next ) {
                    freeCertList ( &currCrt->next );
                }
                escFreePtr( (UINT8**) &dlCrtBin );
                break;
            }
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

/* returns TRUE if CRL is avialable! */
BOOL ecmCmIsCrlAvailable (
    CERT_LIST_s *crt,
    CRL_HASHES_t *crlHashes,
    BOOL *available )
{
    BOOL failed = FALSE;

    CRL_HASHES_t *crl;

    dbgIn( __func__ );

    /* sanity checks */
    if ( !crt || !available) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        crl = crlHashes;
        *available = FALSE;

        while ( crl ) {
            if ( strncmp ( crt->issuerCn, crl->issuerCN, strlen (crl->issuerCN)) == 0 ) {
                *available = TRUE;
                break;
            }
            crl = crl ->next;
        }
    }

    dbgOut( __func__ );

    return ( failed );
}

BOOL ecmCmExtractCertFromDbusMsg (
    DBUS_PARAMS_s *dbus,
    CERT_LIST_s **certChain,
    STATUS_s *stat )
{
    BOOL failed = FALSE;
    UINT8 *data = NULL;
    UINT32 datasize = 0;
    UINT8 *derData = NULL;
    UINT32 derDataSize = 0;

    Certificate_t* cert = NULL;

    dbgIn ( __func__);

    /* sanity checks */
    if ( !dbus || !certChain || !stat ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        return ( TRUE );
    }

    ifNotFailed {

        dbus_error_init(&dbus->err);

        // read the arguments
        if (!dbus_message_get_args(dbus->req, &dbus->err, DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE,
                &data, &datasize, DBUS_TYPE_INVALID)){
            dbgPrint(DBG_ERR, "DBus 'dbus_message_get_args()' failed\n");
            failed = TRUE;
        }
    }

    ifNotFailed {
        if(datasize == 0){
            dbgPrint(DBG_ERR, "DBus invalid data size\n");
            failed = TRUE;
        } else {
            dbgPrint(DBG_INFO, "DBus %d bytes received\n", datasize);
        }
    }

    if(!failed){


        // Convert msg to Certificate_t
        failed = alloc((void**)&cert, sizeof(Certificate_t));
        if(!failed){

            /* Did we receive a PEM encoded certificate? */
            if ( checkForPEMCertificate( (char *)data) == TRUE ) {

                failed = decodePEM( data, datasize, &derData, &derDataSize);

            } else {
                /* No, assume DER and just adjust pointer */
                derData = data;
                derDataSize = datasize;
            }
        }

        if ( !failed ) {
            /* decode certificate */
            failed = asn1Decode(derData, derDataSize, &asn_DEF_Certificate, cert, FALSE);
        }

        if(!failed){
            // append tbc Cert
            *certChain = calloc(1, sizeof(CERT_LIST_s));
            if ( NULL == *certChain ) {
                failed = TRUE;
            }
        }

        ifNotFailed {
            (*certChain)->certPtr = calloc(1, derDataSize);
            if ( NULL == (*certChain)->certPtr ) {
                failed = TRUE;
            }
        }
        ifNotFailed {
            memcpy((*certChain)->certPtr, derData, derDataSize);
            (*certChain)->certSize = derDataSize;
            (*certChain)->crt = cert;
            (*certChain)->crlUri = getCRLURIfromCertificate(cert);
            (*certChain)->next = NULL;
            failed = cmpExtractCrtParamsFromTBSCertificate(&cert->tbsCertificate, (*certChain));
        }
    }

    if(derData != data) {
        free(derData);
    }

    ifFailed {
        ecmCmSetStatus(stat, EXT_IF_ERR_INTERNAL_ERROR, NULL, NULL );
    }

    dbgOut ( __func__);

    return ( failed );
}

void ecmCmReplyToCheckCert (
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus )
{
    BOOL failed = FALSE;
    BOOL available = FALSE;
    CERT_LIST_s* certChain = NULL;
    int ret;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ecm || !dbus ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        return;
    }

    ifNotFailed {
        failed = ecmCmExtractCertFromDbusMsg ( dbus, &certChain, &ecm->status );
    }

    ifNotFailed {

        /* un-mark cert for storing */
        certChain->store = FALSE;

        /* complete certificate chain of given certificate
           by either using certificates from our local list or
           by downloading them if possible */
        failed = ecmCmCompleteCertChain ( certChain, ecm->certList, &ecm->status, ecm->networkAvailable );
    }
    ifNotFailed {
        failed = cmpDetermineCrtTypesAndHierarchy2(certChain, ecm->pkcs1PubKey, ecm->pkcs1PubKeySize );
    }

    /* do certificate checks */
    ifNotFailed {
        failed = checkCertListForCompleteChain(certChain, &ecm->status, TRUE);
    }
    ifNotFailed {
        failed = cmpVerifyCertSignatures(certChain, &ecm->status, TRUE, TRUE );
    }
    ifNotFailed {
        failed = checkForTimeValidity(certChain, &ecm->status, ecm->currentTime);
    }

    /* copy all downloaded certificates to local certificate list */
    ifNotFailed {

        CERT_LIST_s *tmp = NULL;
        CERT_LIST_s **localListEnd = NULL;
        CERT_LIST_s *crtLst = certChain;
        /* skip certificate that was requested for checking,
         * since we do not want to keep it */
        crtLst = crtLst->next;

        localListEnd = &ecm->certList;

        while ( crtLst ) {

            while ( *localListEnd ) {
                localListEnd = &((*localListEnd)->next);
            }

            tmp = NULL;

            /* is cert from new list already available in local list? */
            tmp = checkCertListForCertBySubjectCN(ecm->certList, crtLst->subjectCn );
            if ( NULL == tmp ) {
                /* no, copy it */
                copyCertListItem ( crtLst, localListEnd );
            }
            crtLst = crtLst->next;
        }
    }

    /* do CRL checks */
    ifNotFailed {
        failed = ecmCmIsCrlAvailable( certChain, ecm->crListHash, &available );
        if ( (FALSE == available) && (TRUE != ecm->disableCRLcheck)  ) {

            if ( ecm->networkAvailable == TRUE ) {
                ull_time_t createTime = 0, nextUpdTime = 0;

                ret = addCRLfromURI( certChain->crlUri, &ecm->crList, ecm->certList, &createTime, &nextUpdTime, &ecm->crListHash, &ecm->status );
                if ( ret < 0 ) {
                    /* CLR download failed */
                    ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CRL_NOT_PRESENT, certChain->issuerCn, NULL );
                    failed = TRUE;
                } else {
                    /* download succeeded, store CRL */
                    failed = ecmCmStoreCrls( ecm );
                }
            } else {
                ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CRL_NOT_PRESENT, certChain->issuerCn, NULL );
                failed = TRUE;
            }
        } else if ( TRUE == ecm->disableCRLcheck ) {
			dbgPrint ( DBG_INFO, "CRL not available but CRL check disabled so ignoring\n");
			failed = FALSE;
		}
    }

    // Check if the CRL check is not disabled then check if cert is part of CRL (if CRL is not empty)
    if(!failed && ecm->crList && (FALSE == ecm->disableCRLcheck)){
        failed = verifyCertificateAgainstCRL(certChain->crt, ecm->crList);
    }

    ifNotFailed {

        /* at this point every new downloaded certificate was copied to
         * our local certificate list, store them */

        ecmCmStoreCerts( ecm );
    }

    freeCertList(&certChain);

    if(failed){
        ecmCmSendDbusReply( &ecm->status, dbus, dbusRep_generic, NULL );

    } else{

        dbgPrint(DBG_INFO, "External certificate verification succeeded\n");
        STATUS_s stat;
        stat.err = EXT_IF_ERR_CHECK_CERT_SUCCESS;
        stat.str = "null";

        ecmCmSendDbusReply ( &stat, dbus, dbusRep_generic, NULL );
    }

    dbgOut ( __func__ );
}

void ecmCmReplyToPushCert (
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus )
{
    BOOL failed = FALSE;
    BOOL replace = FALSE;
    CERT_LIST_s *certChain = NULL;

    /* sanity checks */
    if ( !ecm || !dbus ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        return;
    }

    ifNotFailed {
        dbus_error_init(&dbus->err);

        /* extract certificate from dbus message */
        failed = ecmCmExtractCertFromDbusMsg ( dbus, &certChain, &ecm->status );
    }

    /* did we receive a root certificate? */
    ifNotFailed {
        if ( strncmp ( certChain->issuerCn, certChain->subjectCn, strlen ( certChain->issuerCn)) == 0 ) {

            dbgPrint ( DBG_INFO, "Certificate 'subject CN=%s' is a root certificate which is not allowed to be pushed\n", certChain->subjectCn );
            ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CERT_ROOT_NOT_ALLOWED, certChain->subjectCn, NULL );
            failed = TRUE;
        }
    }

    /* do we already have this certificate? */
    ifNotFailed {

        CERT_LIST_s *tmp = NULL;
        //cmpGetCrtFromListBySbjOrIssuerOid( oid_subject, oid_commonName, certChain->subjectCn, ecm->certList, &tmp, TRUE);
        tmp = checkCertListForCertBySubjectCN(ecm->certList, certChain->subjectCn );

        if ( tmp ) {

            /* certificate with same subject CN already exists, compare data */
            if ( memcmp ( certChain->certPtr, tmp->certPtr, certChain->certSize) == 0 ) {

                /* We already have this certificate but
                 * are there other errors to report instead of EXT_IF_ERR_CERT_ALREADY_PRESENT,
                 * e.g. a missing issuer certificate? */

                if ( ecmCmCheckForErrors( ecm, TRUE )) {

                    /* cert chain is not complete,
                     * report error from ecmCmIdentifyCerts() */

                } else {

                    /* no other errors and the certificate is completely the same,
                     * nothing to do but set error code */
                    dbgPrint ( DBG_INFO, "Certificate 'subject CN=%s' is already available\n", certChain->subjectCn );
                    ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CERT_ALREADY_PRESENT, certChain->subjectCn, NULL );
                }

                /* operation failed in both cases above */
                failed = TRUE;

            } else {
                /* its different, which one is newer?
                 * also process if they have same age (the current one may be invalid) */
                if ( certChain->validNotBefore >= tmp->validNotBefore ) {

                    /* the received one is newer, process it */
                    replace = TRUE;

                } else {
                    /* the one we already have is newer, nothing to do but set error code */
                    dbgPrint ( DBG_INFO, "Certificate 'subject CN=%s' is already available\n", certChain->subjectCn );
                    ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CERT_REPLACED, certChain->subjectCn, NULL );
                    failed = TRUE;
                }
            }
        }
    }

    /* add certificate to certificate list,
     * replacing is done by removing the current on from the local list and
     * adding the new one at list end */
    ifNotFailed {
        if ( replace == TRUE ) {
            failed = ecmCmRemoveCertFromList ( certChain->subjectCn, ecm->certList );
        }
    }

    ifNotFailed {

        CERT_LIST_s **lst = &ecm->certList;
        while ( *lst ) {
            lst = &(*lst)->next;
        }

        copyCertListItem( certChain, lst );

        /* determine certificate type */
        failed = cmpDetermineCrtTypesAndHierarchy2 ( ecm->certList, ecm->pkcs1PubKey, ecm->pkcs1PubKeySize );
    }

    /* do re-check of entire setup */
    ifNotFailed {
        failed = ecmCmCheckForErrors( ecm, TRUE );
    }

    /* are we online? */
    if ((failed == TRUE) && (ecm->networkAvailable == TRUE)) {

        STATUS_s lastStat;
        memset ( &lastStat, 0x0, sizeof ( STATUS_s));

        /* It cannot be handled online if:
         * - someone tries to push a root cert
         * - an internal error occurred
         */
        if (( ecm->status.err != EXT_IF_ERR_CERT_ROOT_NOT_ALLOWED ) &&
            ( ecm->status.err != EXT_IF_ERR_INTERNAL_ERROR ) &&
            ( ecm->status.err != EXT_IF_ERR_CERT_ALREADY_PRESENT ))
        {
            /* yes, do we have an error to resolve? */
           while (( ecm->status.err != EXT_IF_ERR_NO_ERROR ) &&
                ((ecm->status.err != lastStat.err) || (strncmp(ecm->status.str, lastStat.str, strlen (lastStat.str)) != 0 )))
            {
                lastStat.err = ecm->status.err;
                escFreePtr( (UINT8 **)&lastStat.str);
                if ( ecm->status.str ) {
                    lastStat.str = strdup(ecm->status.str);
                }

                if ( ecm->status.url ) {
                    lastStat.url = strdup(ecm->status.url);
                }

                ecmCmResolveErrorsByDownload( ecm );
                failed = ecmCmCheckForErrors( ecm, TRUE );
            }
        }

        ifFailed {
            dbgPrint ( DBG_INFO, "Could not resolve error even if we are online.\n");
        }
   }

    /* store certificate(s) and CRLs if necessary */
    ifNotFailed {
        failed = ecmCmStoreCerts( ecm );
    }
    ifNotFailed {
        failed = ecmCmStoreCrls( ecm );
    }

    /* send reply */
    if(failed){
        ecmCmSendDbusReply( &ecm->status, dbus, dbusRep_generic, NULL );
    } else{
        dbgPrint(DBG_INFO, "Certificate push succeeded\n");
        STATUS_s stat;
        stat.err = EXT_IF_ERR_PUSH_CERT_SUCCESS;
        stat.str = "null";

        ecmCmSendDbusReply ( &stat, dbus, dbusRep_generic, NULL );
    }

    // according to DBUS Spec data does not need to be freed, as it points to DBusMessage
}

BOOL ecmCmGetCrlHashByIssuerCn (
    char *issuer,
    CRL_HASHES_t *crlHashes,
    CRL_HASHES_t **ptr )
{
    BOOL failed = FALSE;
    CRL_HASHES_t *hashPtr = crlHashes;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !issuer || !ptr ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        while ( hashPtr ) {
			if ( hashPtr->issuerCN ) {
            if ( strncmp (issuer, hashPtr->issuerCN, strlen ( issuer)) == 0 ) {
                *ptr = hashPtr;
                break;
            }
			}
            hashPtr = hashPtr->next;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

void ecmCmReplyToGetCrlNextUpd (
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus )
{
    char *str = NULL;
    BOOL failed = FALSE;
    STATUS_s stat;
    ull_time_t nextUpd = 0;
    CRL_HASHES_t *crlPtr = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ecm || !dbus ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        return;
    }

    ifNotFailed {
        /* extract CRL from dbus message */
        dbus_error_init(&dbus->err);

        // read the arguments
        if (!dbus_message_get_args(dbus->req, &dbus->err, DBUS_TYPE_STRING, &str, DBUS_TYPE_INVALID)){
            dbgPrint(DBG_ERR, "DBus 'dbus_message_get_args()' failed\n");
            failed = TRUE;
        }
    }

    ifNotFailed {
        /* check the argument and lookup CRL */
        if (( strlen (str) > strlen (ROOT_STR)) ||
            ( strlen (str) < strlen (IC_STR)))
        {
            dbgPrint(DBG_ERR, "String with invalid length received\n");
            failed = TRUE;

        } else if ( strncmp ( str, ROOT_STR, strlen (ROOT_STR)) == 0 ) {

            /* root crl requested */
            failed = getCrlFromListByLocalId(ecm->crListHash, crl_id_root, &crlPtr );

        } else if ( strncmp ( str, IC_STR, strlen ( IC_STR )) == 0 ) {

            /* IC crl requested */
            failed = getCrlFromListByLocalId(ecm->crListHash, crl_id_ic, &crlPtr );

        } else {
            dbgPrint(DBG_ERR, "Invalid string '%s' received\n", str);
            failed = TRUE;
        }
    }

    /* prepare response */
    ifNotFailed {
        dbgPrint(DBG_INFO, "'nextUpdate' value requested for '%s' CRL\n", str);
        nextUpd = crlPtr->nextUpdate;

        stat.err = EXT_IF_ERR_GET_CRL_NEXT_UPD_SUCCESS;
        stat.str = str;
        ecmCmSendDbusReply( &stat, dbus, dbusRep_getCrlNextUpd, (void *)&nextUpd );

    } else {

        stat.err = EXT_IF_ERR_CRL_NOT_PRESENT;
        stat.str = str;
        ecmCmSendDbusReply( &stat, dbus, dbusRep_getCrlNextUpd, (void *)&nextUpd );
    }

    dbgOut ( __func__ );

    return;
}

CR_LIST_t * ecmCmCheckForCrListEntry (
    CR_LIST_t *lst,
    char *issuerCn,
    CertificateSerialNumber_t sn )
{
    CR_LIST_t *item = NULL;
    int size = 0;
    BOOL found = FALSE;
    BOOL failed = FALSE;

    /* sanity checks */
    if ( !lst || !issuerCn ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed  {
        while ( lst ) {

            /* check issuer CN */
            if ( strncmp ( lst->issuerCN, issuerCn, strlen ( issuerCn)) == 0 ) {

                /* issuer CN matches, check for serial number size */
                if ( lst->userID.size == sn.size ) {

                    /* serial number size matches, check number */
                    //if ( memcmp (lst->userID.buf, sn.buf, sn.size ) == 0 ) {
                    size = sn.size-1;
                    found = TRUE;
                    while ( size >= 0 ) {

                        if ( lst->userID.buf[size] != sn.buf[size] ) {
                            found = FALSE;
                            break;
                        }

                        size--;
                    }
                }
            }

            if ( TRUE == found ) {
                item = lst;
                break;
            }

            lst = lst->next;
        }
    }

    return ( item );
}

void ecmCmReplyToPushCrl (
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus )
{
    BOOL failed = FALSE;
    UINT8 *data = NULL;
    UINT32 datasize = 0;
    UINT8 *derData = NULL;
    UINT32 derDataSize = 0;
    ull_time_t crlCreationTime = 0;
    ull_time_t crlNextUpdTime = 0;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !ecm || !dbus ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        return;
    }

    /* remove invalid CRLs that are still available in out local list */
    ifNotFailed {
        failed = removeInvalidCrlHashEntries( ecm );
    }

    ifNotFailed {

        /* extract CRL from dbus message */
        dbus_error_init(&dbus->err);

        // read the arguments
        if (!dbus_message_get_args(dbus->req, &dbus->err, DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE,
                &data, &datasize, DBUS_TYPE_INVALID)){
            dbgPrint(DBG_ERR, "DBus 'dbus_message_get_args()' failed\n");
            failed = TRUE;
        }
    }

    ifNotFailed {
        if(datasize == 0){
            dbgPrint(DBG_ERR, "DBus invalid data size\n");
            failed = TRUE;
        } else {
            dbgPrint(DBG_INFO, "DBus %d bytes received\n", datasize);
        }
    }

    /* Decode CRL */
    CertificateList_t *cl = NULL;

    ifNotFailed {
        failed = alloc((void *) &cl, sizeof(CertificateList_t));
    }

    ifNotFailed {
        /* Did we receive a PEM encoded CRL? */
        if ( checkForPEM( (char *)data) == TRUE ) {

            failed = decodePEM( data, datasize, &derData, &derDataSize);

        } else {
            /* No, assume DER and just adjust pointer */
            derData = data;
            derDataSize = datasize;
        }
    }

    ifNotFailed {
        failed = asn1Decode(derData, derDataSize, &asn_DEF_CertificateList, cl, FALSE);
        ifFailed{
            ecmCmSetStatus(&ecm->status, EXT_IF_ERR_INTERNAL_ERROR, NULL, NULL );
        }
    }

    char *crlIssuerCN = NULL;

    /* get CRL issuer */
    ifNotFailed {
        failed = cmpGetObjFromRDNSequence( &(cl->tbsCertList.issuer.choice.rdnSequence), oid_commonName, TRUE, &crlIssuerCN);
        ifFailed{
            ecmCmSetStatus(&ecm->status, EXT_IF_ERR_INTERNAL_ERROR, NULL, NULL );
        }
    }

    /* extract CRL creation time */
    ifNotFailed {

        dbgPrint (DBG_INFO, "Processing CRL issued by 'CN=%s'\n", crlIssuerCN);

        crlCreationTime = cmpExtractAsn1Time( cl->tbsCertList.thisUpdate );
        if ( cl->tbsCertList.nextUpdate ){
            crlNextUpdTime = cmpExtractAsn1Time( *(cl->tbsCertList.nextUpdate) );
        }
    }

    UINT8 *digest = NULL;
    BOOL upd = TRUE;

    ifNotFailed {
        failed = hashCRL(cl, &digest);
    }

    /* check / verify CRL */
    ifNotFailed {

        CRL_HASHES_t *tmp = NULL;

        failed = ecmCmGetCrlHashByIssuerCn( crlIssuerCN, ecm->crListHash, &tmp );

        if ( tmp ) {
            /* a CRL of the same issuer is already there,
             * is this one newer and has the digest changed? */
            if (( tmp->thisUpdate < crlCreationTime)) {

                /* TODO: is the CRL digest calculation necessary anymore?
                 * This was introduced to prevent CRLs being stored after
                 * every download but having only the time stamps changed.
                 * Since the requirement to download CRLs on every power-up
                 * has changed (time comes now from the secure time module),
                 * CRLs are now downloaded after they have expired and
                 * we always need to store the CRL to keep track of
                 * the expiration time, we just always save the CRL
                 * if it is more recent and even if the content did not change */


                /* it is newer, but has the digest changed? */
                if ( memcmp ( tmp->digest, digest, EscSha1_DIGEST_LEN ) == 0 ) {

                    /* No, digest is the same */
                    dbgPrint( DBG_INFO, "CRL 'issuer CN=%s' is newer than local one but content has not changed.\n", crlIssuerCN );

                    /* Note: We do not set the upd flag to false
                     * to force the new CLR to be stored,
                     * even if the content did not change
                     * we want to have the latest validity times of the CRL stored
                    upd = FALSE;
                     */

                } else {
                    /* yes, digest changed */
                    dbgPrint( DBG_INFO, "CRL 'issuer CN=%s' is newer than local one and content has changed. Update local one...\n", crlIssuerCN );
                }
            } else {

                /* the received CLR is not newer, nothing to do */
                dbgPrint( DBG_INFO, "CRL 'issuer CN=%s' is not newer than local one.\n", crlIssuerCN );
                upd = FALSE;
                ecmCmSetStatus( &ecm->status, EXT_IF_ERR_CRL_ALREADY_PRESENT, crlIssuerCN, NULL );
                failed = TRUE;
            }
        }
    }

    CRL_HASHES_t **crlHashesPtr = NULL;
    CR_LIST_t **crListPtr = NULL;
    CR_LIST_t *thisCrList = NULL;
    CR_LIST_t **thisCrListPtr = NULL;
    char *crlIssuerCn = NULL;
    size_t s = 0;

    /* add revoked certificates to CR list */
    if ( (FALSE == failed) && (TRUE == upd)) {
        int i = 0;
        if(cl->tbsCertList.revokedCertificates){

            failed = cmpGetObjFromRDNSequence( &(cl->tbsCertList.issuer.choice.rdnSequence), oid_commonName, TRUE, &crlIssuerCn);

            /* remove all CRL entries of the current issuer
             * NOTE: removing and re-adding of all CRL entries
             *       of the current issuer is much faster for large lists
             *       like EC-CRL than checking the local list if
             *       entry already exists,
             *       maybe list sorting and binary search can be used instead */
            ifNotFailed {
                removeCrListEntryByIssuerCn( crlIssuerCn, &ecm->crList );

                /* fast forward to end of crlist */
                crListPtr = &ecm->crList;

                while ( *crListPtr ) {
                    crListPtr = &(*crListPtr)->next;
                }

                thisCrListPtr = &thisCrList;

                dbgPrint (DBG_INFO, "Processing %d CRL entries\n", cl->tbsCertList.revokedCertificates->list.count);
            }

            while (!failed && i < cl->tbsCertList.revokedCertificates->list.count) {

                *thisCrListPtr = calloc ( 1, sizeof (CR_LIST_t ));
                if ( NULL == *thisCrListPtr ) {
                    failed = TRUE;
                }

                ifNotFailed {
                    memset ( *thisCrListPtr, 0x0, sizeof ( CR_LIST_t ));
                    failed = escInt2SizetAndCheckForOv( cl->tbsCertList.revokedCertificates->list.array[i]->userCertificate.size, &s );
                }
                ifNotFailed {
                    (*thisCrListPtr)->userID.buf = calloc(1, s );
                    if (!(*thisCrListPtr)->userID.buf) {
                        failed = TRUE;
                    }
                }

                ifNotFailed {
                    (*thisCrListPtr)->userID.size = s;
                    memcpy((*thisCrListPtr)->userID.buf, cl->tbsCertList.revokedCertificates->list.array[i]->userCertificate.buf, s );
                    (*thisCrListPtr)->issuerCN = strdup(crlIssuerCn);
                }

                thisCrListPtr = &(*thisCrListPtr)->next;
                i++;
            }

            ifNotFailed {
                *crListPtr = thisCrList;
                dbgPrint (DBG_INFO, "%d revoked entries added\n", i);
            }
        }
    }

    /* create CRL hashes list entry */
    if ( (FALSE == failed) && (TRUE == upd)) {
        crlHashesPtr = &ecm->crListHash;

        /* fast forward to end of CRL hashes list */
        while ( *crlHashesPtr) {
            crlHashesPtr = &(*crlHashesPtr)->next;
        }

        if ( NULL == *crlHashesPtr ) {
            *crlHashesPtr = calloc ( 1, sizeof ( CRL_HASHES_t ));
            if ( NULL == *crlHashesPtr ) {
                failed = TRUE;
            }
        }

        ifNotFailed {

            (*crlHashesPtr)->issuerCN = crlIssuerCN;
            (*crlHashesPtr)->cl = cl;
            (*crlHashesPtr)->certList = malloc ( derDataSize );
            if ( NULL == (*crlHashesPtr)->certList ) {
                failed = TRUE;
            }
        }

        ifNotFailed {
            memcpy ( (*crlHashesPtr)->certList, derData, derDataSize);
            (*crlHashesPtr)->certListLen = derDataSize;
            (*crlHashesPtr)->thisUpdate = crlCreationTime;
            (*crlHashesPtr)->nextUpdate = crlNextUpdTime;
            (*crlHashesPtr)->digest = digest;
            (*crlHashesPtr)->store = TRUE;

            failed = determineCrlTypeByIssuerCn ( &ecm->crListHash, ecm->certList, &ecm->status );
        }
    }

    if ( derData != data ) {
        free(derData);
    }

    /* do re-check of entire setup */
    ifNotFailed {
        failed = ecmCmCheckForErrors( ecm, TRUE );
    }

    /* are we online? */
    if ((failed == TRUE) && (ecm->networkAvailable == TRUE)) {

        STATUS_s lastStat;
        memset ( &lastStat, 0x0, sizeof ( STATUS_s ));

        /* It cannot be handled online if:
         * - an internal error occurred
         */
        if (( ecm->status.err != EXT_IF_ERR_INTERNAL_ERROR ) &&
            ( ecm->status.err != EXT_IF_ERR_CRL_ALREADY_PRESENT ))
        {
            /* we are online, do we have an error to resolve? Try as long as the error code changes,
             * if we get stuck, give up */
            while (( ecm->status.err != EXT_IF_ERR_NO_ERROR) && (ecm->status.err != lastStat.err )) {
                lastStat.err = ecm->status.err;
                ecmCmResolveErrorsByDownload( ecm );
                failed = ecmCmCheckForErrors( ecm, TRUE );
            }
        }

        ifFailed {
            dbgPrint ( DBG_INFO, "Could not resolve error even if we are online.\n");
        }
    }

    /* store certificate(s) and CRLs if necessary and push CRL creation time to STC */
    ifNotFailed {
        failed = ecmCmStoreCerts( ecm );
    }
    ifNotFailed {
        failed = ecmCmStoreCrls( ecm );
        ecmCmPushTimeToStm( stc_interface_set_time, crlCreationTime );
    }

    /* send reply */
    if(failed){
        ecmCmSendDbusReply( &ecm->status, dbus, dbusRep_generic, NULL );
    } else{
        dbgPrint(DBG_INFO, "CRL push succeeded\n");
        STATUS_s stat;
        stat.err = EXT_IF_ERR_PUSH_CRL_SUCCESS;
        stat.str = "null";

        ecmCmSendDbusReply ( &stat, dbus, dbusRep_generic, NULL );
    }

    // according to DBUS Spec data does not need to be freed, as it points to DBusMessage

    escFreePtr( (UINT8 **)&crlIssuerCn );

    /* we cannot free cl here since it might be used later on even if the current call failed,
     * e.g. pushCRL without having the CLR issuer certificate available!
     * If the whole CRL hashes entry becomes invalid, cl will be freed anyhow.
     */

    dbgOut ( __func__ );
}

void ecmCmReplyToGetCsr (
    ECM_PARAMS_t *ecm,
    DBUS_PARAMS_s *dbus )
{
    DBusMessageIter args;
    UINT8* csrBuf = NULL;
    UINT32 csrSize = MAX_ENC_PKCS10_MESSAGE_LEN;
    char* param = "";
    BOOL failed = FALSE;

    /* sanity checks */
    if ( !ecm || !dbus ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        return;
    }

    // read the arguments
    if (!dbus_message_iter_init(dbus->req, &args)) {
        dbgPrint(DBG_INFO, "DBus Message has no arguments - using default UIN\n");
    } else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) {
        dbgPrint(DBG_INFO, "DBus Message has unexpected Type\n");
    } else {
        dbus_message_iter_get_basic(&args, &param);
    }

    dbgPrint( DBG_INFO,
        "DBus-PKCS10 Method called with Parameter: (%s)\n", param);

    /* shall we use the UIN provided as Dbus parameter? */
    if((strlen(param) != 0 )) {

        ecm->pkcs10Data.setUniqueIdAsCn = TRUE;
        if(ecm->pkcs10Data.deviceUniqueId){
            /* backup default UIN */
            escFreePtr( (UINT8**) &ecm->pkcs10Data.deviceUniqueIdBak );
            ecm->pkcs10Data.deviceUniqueIdBak = strdup (ecm->pkcs10Data.deviceUniqueId );

            free(ecm->pkcs10Data.deviceUniqueId);
        }

        failed = escCheckSizetForOv( strlen( param ));
        ifNotFailed {
            ecm->pkcs10Data.deviceUniqueId = calloc(1, strlen(param)+1);
            if ( NULL == ecm->pkcs10Data.deviceUniqueId ) {
                failed = TRUE;
            }
        }
        ifNotFailed {
            strncpy(ecm->pkcs10Data.deviceUniqueId, param, strlen ( param ));
        } else {

            dbgPrint(DBG_INFO, "Failed to extract UIN from DBUS request\n");

            STATUS_s stat;
            stat.err = EXT_IF_ERR_INTERNAL_ERROR;
            stat.str = "null";

            ecmCmSendDbusReply ( &stat, dbus, dbusRep_getCsr, NULL );
            return;
        }
    }

    /* else use default params, but check if it is available */
    if(ecm->pkcs10Data.deviceUniqueId == NULL){

        ecm->uniqueID = getUID( ecm );
        if(!ecm->uniqueID){

            dbgPrint(DBG_INFO, "UIN file is not available\n");

            STATUS_s stat;
            stat.err = EXT_IF_ERR_UIN_FILE_NOT_AVAILABLE;
            stat.str = "null";

            ecmCmSendDbusReply ( &stat, dbus, dbusRep_getCsr, NULL );
            return;
        }
    }

    /* generate CSR */
    createPKCS10CSR(&ecm->pkcs10Data, &csrBuf, &csrSize);

    /* send reply */
    dbgPrint(DBG_INFO, "PKCS#10 CSR generation succeeded\n");

    STATUS_s stat;
    stat.err = EXT_IF_ERR_CSR_GENERATION_SUCCESS;
    stat.str = ecm->pkcs10Data.deviceUniqueId;

    DBUS_GET_CSR_REPLY_s r;
    memset ( &r, 0x0, sizeof ( r ));
    r.csr = csrBuf;
    r.csrLen = csrSize;

    ecmCmSendDbusReply ( &stat, dbus, dbusRep_getCsr, (void*)&r );

    free(csrBuf);
    csrBuf= NULL;

    /* restore backup of default UIN - if necessary */
    if ( ecm->pkcs10Data.deviceUniqueIdBak ) {
        escFreePtr( (UINT8 **) &ecm->pkcs10Data.deviceUniqueId );
        ecm->pkcs10Data.deviceUniqueId = strdup ( ecm->pkcs10Data.deviceUniqueIdBak );
        escFreePtr( (UINT8 **) &ecm->pkcs10Data.deviceUniqueIdBak );
    }
}

void stopDBusService()
{
    exitCMEventHandler = TRUE;
}


/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
