/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        ecm.h

   \brief       Embedded Certificate Manager main include file

   ***

   $Rev: 809 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-24 16:55:24 +0200 (Do, 24. Aug 2017) $
****************************************************************************/

#ifndef __ECM_H__
#define __ECM_H__

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/



#include "../../common/inc/esc_common.h"
#include "../../cmp/inc/cmp.h"
#include "../../crl/inc/crl_utils.h"

#include "ecm_cm_sdc_utils.h"

#include "../../pkcs10/inc/pkcs10.h"

#include "../../crypto/inc/crypto.h"

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

#ifdef CONFIG_ECM_USE_SDC
/** Default ID of SDC key that is used for un/wrap the UIN if not set in config file*/
#define SDC_UIN_WRAP_KEY_ID     0x2
/** Default ID of SDC key that is used for un/wrap the device key if not set in config file*/
#define SDC_DEV_KEY_WRAP_KEY_ID 0x2
/** Default ID of SDC key that is used for un/wrap certificates if not set in config file*/
#define SDC_CERT_WRAP_KEY_ID    0x2
/** Suffix that is appended to SDC wrapped file names */
#define SDC_WRAP_SUFFIX ".wrap"
#endif /* CONFIG_ECM_USE_SDC */


/** Short form for faster checking if return code is equal EcmRetOk */
#define ifReturnOk if ( retVal == EcmRetOk )
/** Short form for faster checking if return code is not equal EcmRetOk */
#define ifReturnNotOk if ( retVal != EcmRetOk )

/** ID for PKCS#1 key container */
#define ID_FORMAT_CONTAINER_PKCS1 0x00000001
/** ID for PKCS#8 key container */
#define ID_FORMAT_CONTAINER_PKCS8 0x00000002
/** ID for DER container format */
#define ID_FORMAT_CONTAINER_DER   0x00000003
/** ID for PEM container format */
#define ID_FORMAT_CONTAINER_PEM   0x00000004

/** DATA TYPE Key for writeToDisc  */
#define DATA_TYPE_KEY   1
/** DATA TYPE certificate for writeToDisc */
#define DATA_TYPE_CERT   2

/** PEM key start tag */
#define PEM_KEY_START_TAG       "-----BEGIN RSA PRIVATE KEY-----\n";
/** PEM key end tag */
#define PEM_KEY_END_TAG         "\n-----END RSA PRIVATE KEY-----";
/** PEM key start tag */
#define PEM_KEY_PKCS8_START_TAG "-----BEGIN PRIVATE KEY-----\n";
/** PEM key end tag */
#define PEM_KEY_PKCS8_END_TAG   "\n-----END PRIVATE KEY-----";
/** PEM certificate start tag */
#define PEM_CERT_START_TAG      "-----BEGIN CERTIFICATE-----\n";
/** PEM certificate end tag */
#define PEM_CERT_END_TAG        "\n-----END CERTIFICATE-----";

/** Name of the device key file der*/
# define KEY_FILE_NAME         "RCC-privkey"
/** Name of the previous device key file der */
# define PREV_KEY_FILE_NAME    "_RCC-privkey"

/** File extention for PEM files */
#define FILE_EXT_PEM ".pem"
/** File extention for DER files */
#define FILE_EXT_DER ".der"
/** File extension for certificates */
#define FILE_EXT_CRT ".crt"
/** File extension for CRLs */
#define FILE_EXT_CRL ".crl"

/** PKCS#8 file name identifier */
#define PKCS8_FILE_IDENTIFIER "-p8"

/** Certificate file identifier string - used to identify files in CFKS directory */
#define CERT_FILE_IDENTIFIER "-cert"

/** Union representing different Key Pair Sizes */
typedef union{
    /** RSA 2048bit Key Pair */
	EscCrypto_RsaKeyPairT keyPair2048;
	/** RSA 4096bit Key Pair */
	EscCrypto4096_RsaKeyPairT keyPair4096;
	/** RSA 1024bit Key Pair */
	EscCrypto1024_RsaKeyPairT keyPair1024;
}KeyPairT;

/** Name of the NV data file */
#define NV_DATA_FILE_NAME "nvdata.bin"

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/


/** Holds configuration file information */
typedef struct {
    /** Flag to signal if this config file is used */
    BOOL used;
    /** Pointer to the configuration file name */
    char *name;
} CONF_FILES_t;


/** Exit codes */
typedef enum {
    /** Device registration failed */
    DeviceRegistrationFailed = 2,
    /** Certificate renewal failed */
    CertificateRenewalFailed,
    /** Key generation failed */
    KeyGenerationFailed,
    /** Server response invalid. Message content invalid or timeout */
    ServerResponseInvalid,
    /** LRA common error */
    LraCommonError,
    /** CMP message invalid */
    CmpMessageInvalid,
    /** CMP error message received */
    CmpErrorMessageReceived,
    /** Certificate signature error */
    CertSignatureError,
    /** Certificate storage error */
    CertStorageError,
    /** Certificate check failed */
    CertCheckError,
    /** Certificate confirmation failed */
    CertificateConfirmationFailed,
} ECM_EXIT_CODES_t;

/** Enumeration of return codes */
typedef enum {
    /** Return code OK */
    EcmRetOk,
    /** Return code failed */
    EcmRetFailed,
    /** Return code out of memory */
    EcmRetOutOfMemory,
    /** Return code retry */
    EcmRetRetry,
    /** Return code do TLS re-handshake */
    EcmRetDoTlsReHandshake,
    /** Return code device certificate missing */
    EcmRetDeviceCrtMissing
} ECM_RET_t;

/** Structure containing all parameters stored in NV memory */
typedef struct {
    /** NV flags */
    UINT32 flags;
} NV_PARAMS_s;


/** Structure containing all path and file name strings */
typedef struct {
    /** Pointer to certificate and key store path name */
    char *fileStorePath;
    /** Pointer alternative key store location */
    char *keyPath;
    /** Pointer to log file name */
    char *logFile;
    /** Pointer to UID file name */
    char *uidFile;
    /** Pointer to device key file path */
    char *devKey;
    /** Pointer to device key name */
    char *devKeyName;
    /** Pointer to NV data file name */
    char *nvData;
    /** Pointer to root certificate file name */
    char *rootCrt;
    /** Pointer to CRL file name */
    char *crlPath;
    /** Pointer to CRL-URL List */
    STRING_LIST_s *crlURL;
    /** URI of Device Certificate Download **/
    char* deviceCertURI;
    /** Base file name for device certificate */
    char *deviceCertName;
    /** path to file where CSR is stored */
    char *csrFileName;
} FILE_PATH_NAME_STRINGS_s;

/** Overall ECM parameters structure */
typedef struct {
    /** Debug mask */
    unsigned int dbgMask;
    /** Log level */
    UINT32 logLevel;
    /** Key container ID */
    UINT32 keyContainer;
    /** Key format ID */
    UINT32 keyFormat;
    /** Key size ID */
    UINT32 keySize;
    /** Certificate format ID */
    UINT32 certFormat;
    /** CMP parameters */
    CMP_DATA_s cmpData;
    /** PKCS10 parameters **/
    PKCS10_DATA_s pkcs10Data;
    /** NV parameter structure */
    NV_PARAMS_s nv;
    /** Path and file name structure */
    FILE_PATH_NAME_STRINGS_s str;
    /** Unique Identifier e.g. Serial Nr */
    char* uniqueID;
    /** Pointer to certificate list */
    CERT_LIST_s *certList;
    /** Pointer to certificate revokation list */
    CR_LIST_t *crList;
    /** Flag if latest CRL is available */
    BOOL latestCRLavailable;
    /** Hash of current CR List */
    CRL_HASHES_t* crListHash;
    /** Pointer to PKCS#1 Public Key */
    UINT8* pkcs1PubKey;
    /** Size of the plain PKCS#1 Public Key */
    UINT32 pkcs1PubKeySize;
    /** Event Handler Thread ID */
    pthread_t cm_EventHandler;
    /** Time of the latest "secure" Time */
    ull_time_t currentTime;
    /** Parameter of Secure Data Container */
    SDC_PARAMS_s sdc;
    /** Parameter to signal availability of a successful loaded CRL */
    BOOL localCrlLoaded;
    /** current error state */
    STATUS_s status;
    /** Flag to singal network available */
    BOOL networkAvailable;
    /** Device key pair structure */
    KeyPairT keyPair;
    /** Random context */
    EscRandom_ContextT randCtx;
    /** Flag to signal if Certs can be verified */
    volatile BOOL certConfigValid;
    /** Flag to signal if device key shall be generated at startup */
    BOOL genDevKeyAtStartup;
	/** Flag to signal if the certificates to be checked against the CRL file **/
	BOOL disableCRLcheck;
	/** Flag to signal if the device certificate presence needs to be checked **/
	BOOL disableDeviceCertificateCheck;
} ECM_PARAMS_t;

/** Count of possible configuration file locations */
#define CONF_FILE_LIST_SIZE 3

/** ECM Configuration Parameters */
typedef struct {
    /** Flag to signal if Process shall not be daemonized */
    BOOL noDaemonFlag;
    /** Flag to signal if Output shall be omitted */
    BOOL quietFlag;
    /** Files List of Configuration Files */
    CONF_FILES_t confFiles[CONF_FILE_LIST_SIZE];
    /** Serial Number */
    char* serial;
} ECM_CONFIG_t;

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. FUNCTION PROTOTYPES                                                  *
 ***************************************************************************/

/***************************************************************************
 * 6. MACRO FUNCTIONS                                                      *
 ***************************************************************************/

/***************************************************************************
 * 7. END                                                                  *
 ***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* __ECM_H__ */
