/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        rsa.c

   \brief       RSA encryption and decryption

   $Rev: 937 $
 */
/***************************************************************************/
/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/
#include "../inc/rsa.h"
#include <stdlib.h>
/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** Length of single multiplication results in words */
#define EscRsa_RSA_SIZE_WORDS_LONG ( ( 2U * EscRsa_RSA_SIZE_WORDS ) )

#ifdef EscRsa_ENABLE_GENRSA
/** Number of small primes that fit into a UINT8 (used for RSA key pair generation) */
#    define EscRsa_NUM_PRIMES 53U
#endif

/***************************************************************
 Distance between p and q: ceil(2^0.1 << (EscRsa_RSA_BASE - 1))
 ***************************************************************/
#ifdef EscRsa_ENABLE_GENRSA
#    ifdef Esc_HAS_INT64
#        define EscRsa_PQ_LOWER_BOUND   2301615986U
#    else
#        define EscRsa_PQ_LOWER_BOUND   35120U
#    endif
#endif

/***************************************************************
Extracts the high part of FWORD w and returns it as a FWORD
 ***************************************************************/
#define EscRsa_HI_FWORD( w ) ( ( ( w ) >> EscRsa_RSA_BASE ) & EscRsa_MAX_VAL )

/***************************************************************
Extracts the high part of FWORD w and returns it as a HWORD
 ***************************************************************/
#define EscRsa_HI_HWORD( w ) ( (EscRsa_HWORD)( ( ( w ) >> EscRsa_RSA_BASE ) & EscRsa_MAX_VAL ) )

/***************************************************************
Extracts the low part of FWORD w and returns it as a FWORD
 ***************************************************************/
#define EscRsa_LO_FWORD( w ) ( ( w ) & EscRsa_MAX_VAL )

/***************************************************************
Extracts the low part of FWORD w and returns it as a HWORD
 ***************************************************************/
#define EscRsa_LO_HWORD( w ) ( (EscRsa_HWORD)( ( w ) & EscRsa_MAX_VAL ) )

/***************************************************************
return min(x,y)
 ***************************************************************/
#define EscRsa_MIN( x, y ) ( ( ( x ) < ( y ) ) ? ( x ) : ( y ) )

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/* Multiplication result field element. */
typedef struct {
    EscRsa_HWORD words[ EscRsa_RSA_SIZE_WORDS_LONG ];
    /* No carry needed */
} EscRsa_FieldElementLongT;

typedef struct {
    /** Data to calculate c = x^e mod m */

    /** Multiplication result */
    EscRsa_FieldElementLongT* c;

    /** Modulus */
    const EscRsa_FieldElementT* m;

    /** x */
    EscRsa_FieldElementT* x;
} EscRsa_MultiplyDataT;

#ifdef EscRsa_USE_MONTGOM_MUL
typedef struct {
    /** Modulus */
    EscRsa_FieldElementT mod;
    /** negative inverse of mod[0] */
    EscRsa_HWORD m;
} EscRsa_MontGomElementT;
#endif

/* Field element functions */
#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
static void
EscRsaFe_ToBytesBE(
    UINT8 dest[],
    const EscRsa_FieldElementT* source );

#endif

#if ( !defined( EscRsa_USE_MONTGOM_MUL ) ) || defined( EscRsa_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa_USE_SHORT_EXPONENT ) || defined( EscRsa_ENABLE_GENRSA ) || defined( EscRsa_ENABLE_CRT )
static void
EscRsaFe_ToLongFe(
    EscRsa_FieldElementLongT* dst,
    const EscRsa_FieldElementT* src );
#endif

static void
EscRsaFe_FromLongFe(
    EscRsa_FieldElementT* dst,
    const EscRsa_FieldElementLongT* src );

#if defined( EscRsa_USE_SLIDING_WINDOW ) || defined( EscRsa_USE_MONTGOM_MUL ) || defined( EscRsa_ENABLE_GENRSA ) || defined ( EscRsa_ENABLE_CRT )
static void
EscRsaFe_Assign(
    EscRsa_FieldElementT* dst,
    const EscRsa_FieldElementT* src );
#endif

static BOOL
EscRsaFe_GreaterThan(
    const EscRsa_FieldElementT* a,
    const EscRsa_FieldElementT* b );

#ifdef EscRsa_USE_MONTGOM_MUL
static void
EscRsaFe_MontGom(
    EscRsa_FieldElementT* x,
    const EscRsa_FieldElementT* y,
    const EscRsa_MontGomElementT* mg );

static void
EscRsa_MontGom_Init(
    const EscRsa_MultiplyDataT* pm,
    EscRsa_MontGomElementT* mg );

#endif

static void
EscRsaFe_ModularReduction(
    const EscRsa_MultiplyDataT* pm );

static void
EscRsaFe_MultiplyClassically(
    const EscRsa_MultiplyDataT* pm );

#if ( !defined( EscRsa_USE_MONTGOM_MUL ) ) || defined( EscRsa_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa_USE_SHORT_EXPONENT ) || defined( EscRsa_ENABLE_GENRSA )
static void
EscRsaFe_SquareClassically(
    const EscRsa_MultiplyDataT* pm );

static void
EscRsaFe_Square(
    const EscRsa_MultiplyDataT* pm );
#endif

static void
EscRsaFe_Multiply(
    const EscRsa_MultiplyDataT* pm );

static BOOL
EscRsaFe_IsBitSet(
    const EscRsa_FieldElementT* e,
    UINT32 thebit );

static void
EscRsaFe_BigPow(
    const EscRsa_MultiplyDataT* pm,
    const EscRsa_FieldElementT* e );

#if defined( EscRsa_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa_USE_SHORT_EXPONENT ) || defined( EscRsa_ENABLE_STACK_SAVING_INTERFACE )
#    ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
static BOOL
EscRsa_ShortPow(
    const UINT8 message[],
    const UINT8 modulus[],
    const UINT32 exponent,
    UINT8 result[] );
#    else

static BOOL
EscRsa_ShortPow(
    EscRsa_FieldElementT* message,
    const EscRsa_FieldElementT* modulus,
    const UINT32 exponent );
#    endif
#endif

#if defined ( EscRsa_ENABLE_GENRSA ) || defined ( EscRsa_ENABLE_CRT )
static SINT8
EscRsaFe_AbsoluteCompare(
    const EscRsa_FieldElementT* a,
    const EscRsa_FieldElementT* b );

static void
EscRsaFe_Subtract(
    EscRsa_FieldElementT* a,
    const EscRsa_FieldElementT* b );
#endif

#ifdef EscRsa_ENABLE_GENRSA
static void
EscRsaFe_ShiftRight(
    EscRsa_FieldElementT* a );

static void
EscRsaFe_SignedShiftRight(
    EscRsa_FieldElementT* a,
    EscRsa_HWORD* aSign );

static BOOL
EscRsaFe_IsZero(
    const EscRsa_FieldElementT* a );

static BOOL
EscRsaFe_IsOne(
    const EscRsa_FieldElementT* a );

static void
EscRsaFe_SignedAdd(
    EscRsa_FieldElementT* a,
    EscRsa_HWORD* aSign,
    const EscRsa_FieldElementT* b,
    const EscRsa_HWORD bSign );

static void
EscRsaFe_SignedSubtract(
    EscRsa_FieldElementT* a,
    EscRsa_HWORD* aSign,
    const EscRsa_FieldElementT* b,
    const EscRsa_HWORD bSign );

static void
EscRsaFe_AddWord(
    EscRsa_FieldElementT* c,
    const EscRsa_HWORD a );

static EscRsa_HWORD
EscRsaFe_ModWord(
    const EscRsa_FieldElementT* a,
    EscRsa_HWORD w );

static BOOL
EscRsaFe_IsPositive(
    EscRsa_HWORD sign,
    const EscRsa_FieldElementT* fe);

static BOOL
EscRsaFe_ModInv(
    const EscRsa_FieldElementT* x,
    const EscRsa_FieldElementT* y,
    EscRsa_FieldElementT* result );

static BOOL
EscRsa_MillerRabin(
    EscRandom_ContextT* randCtx,
    const EscRsa_FieldElementT* n,
    const UINT32 iter,
    BOOL* prime );

static BOOL
EscRsa_GeneratePrime(
    EscRandom_ContextT* randCtx,
    EscRsa_FieldElementT* n );
#endif

#ifdef EscRsa_ENABLE_CRT
static void
EscRsaFe_Add(
    EscRsa_FieldElementT* a,
    const EscRsa_FieldElementT* b );
#endif

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

#if ( !defined( EscRsa_ENABLE_STACK_SAVING_INTERFACE ) ) || defined( EscRsa_ENABLE_GENRSA ) || defined( EscRsa_ENABLE_CRT )
/***************************************************************
 * Converts a byte array into a word array (big endian)        *
 ***************************************************************/
void
EscRsaFe_FromBytesBE(
    EscRsa_FieldElementT* dest,
    const UINT8 source[],
    UINT16 len )
{
    UINT16 i;
    EscRsa_HWORD* dstArray;

    dstArray = dest->words;

    for ( i = 0U; i < ( len / EscRsa_WORD_SIZE ); i++ ) {
        EscRsa_HWORD d;
        d = ( (EscRsa_HWORD)source[ ( len - 1U ) - ( i * EscRsa_WORD_SIZE ) ] ) |
            ( (EscRsa_HWORD)( ( (EscRsa_HWORD)source[ ( len - 1U ) - ( ( i * EscRsa_WORD_SIZE ) + 1U ) ] ) << 8 ) );

#    if EscRsa_WORD_SIZE == 4U
        d |= ( (EscRsa_HWORD)( ( (EscRsa_HWORD)source[ ( len - 1U ) - ( ( i * EscRsa_WORD_SIZE ) + 2U ) ] ) << 16 ) ) |
            ( (EscRsa_HWORD)( ( (EscRsa_HWORD)source[ ( len - 1U ) - ( ( i * EscRsa_WORD_SIZE ) + 3U ) ] ) << 24 ) );
#    endif

        /*lint -save -esym(960,17.4) dstArray is an array*/
        dstArray[ i ] = d;
        /*lint -restore dstArray is an array*/
    }

    /*     This padding is not used in ECM as the len is always = KeyBytes
    *     (orig: i < EscRsa_RSA_SIZE_WORDS)
    *     This is a fix to use this function with different Key Sizes, but it avoids
    *     the padding here
    */
    for ( i = ( len / EscRsa_WORD_SIZE ); i < ( len / EscRsa_WORD_SIZE ); i++ ) {
        /*lint -e960 dstArray is an array*/
        dstArray[ i ] = 0U;
        /*lint +e960 dstArray is an array*/
    }
}

#endif

/***************************************************************
 * Converts a word array into a byte array (big endian)        *
 ***************************************************************/
#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
static void
EscRsaFe_ToBytesBE(
    UINT8 dest[],
    const EscRsa_FieldElementT* source )
{
    UINT16 i;
    EscRsa_HWORD v;

    for ( i = 0U; i < EscRsa_KEY_BYTES; i++ ) {
        v = source->words[ i >> ( EscRsa_WORD_SIZE / 2U ) ];
        v >>= ( (UINT16)( ( i & ( EscRsa_WORD_SIZE - 1U ) ) << 3 ) ) & 0xFFU;
        dest[ ( EscRsa_KEY_BYTES - 1U ) - i ] = (UINT8)v;
    }
}

#endif

/************************************************************************
 Assigns element src to long element dst
 dst := src
************************************************************************/
#if ( !defined( EscRsa_USE_MONTGOM_MUL ) ) || defined( EscRsa_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa_USE_SHORT_EXPONENT ) || defined( EscRsa_ENABLE_GENRSA ) || defined( EscRsa_ENABLE_CRT )
static void
EscRsaFe_ToLongFe(
    EscRsa_FieldElementLongT* dst,
    const EscRsa_FieldElementT* src )
{
    UINT16 i;

    /* copy src */
    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        dst->words[ i ] = src->words[ i ];
    }

    /* fill remaining values with zero */
    for ( i = EscRsa_RSA_SIZE_WORDS; i < EscRsa_RSA_SIZE_WORDS_LONG; i++ ) {
        dst->words[ i ] = 0U;
    }
}

#endif
/************************************************************************
 Assigns long element src to element dst
 dst := src
************************************************************************/
static void
EscRsaFe_FromLongFe(
    EscRsa_FieldElementT* dst,
    const EscRsa_FieldElementLongT* src )
{
    UINT16 i;

    /** The value fits into dst */
    Esc_ASSERT( src->words[ EscRsa_RSA_SIZE_WORDS ] == 0U );

    /* copy src */
    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        dst->words[ i ] = src->words[ i ];
    }
}

/************************************************************************
 Assigns element src to element dst
 dst := src
************************************************************************/
#if defined( EscRsa_USE_SLIDING_WINDOW ) || defined( EscRsa_USE_MONTGOM_MUL ) || defined( EscRsa_ENABLE_GENRSA ) || defined ( EscRsa_ENABLE_CRT )
static void
EscRsaFe_Assign(
    EscRsa_FieldElementT* dst,
    const EscRsa_FieldElementT* src )
{
    UINT16 i;
    /* copy src */
    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        dst->words[ i ] = src->words[ i ];
    }
}

#endif

/************************************************************************
Main reduction loop (HAC, Algorithm 14.20)
c := c mod m
m must be != 0
************************************************************************/
static void
EscRsaFe_ModularReduction(
    const EscRsa_MultiplyDataT* pm )
{
    /*lint --e{661} --e{662} It is confirmed that the array out of bounds warning is a false positive */
    SINT16 mwords_msb;
    SINT16 i;
    UINT16 j;
    EscRsa_HWORD q;
    EscRsa_HWORD carry;
    EscRsa_FWORD ciAndCmin1;
    EscRsa_HWORD* cwords;
    const EscRsa_HWORD* mwords;
    BOOL finished;

    cwords = pm->c->words;
    mwords = pm->m->words;

    mwords_msb = (SINT16)EscRsa_RSA_SIZE_WORDS - 1;
    /*lint -save -esym(960,17.4) cwords, mwords are arrays */
    while ( ( mwords_msb >= 0 ) && ( mwords[ mwords_msb ] == 0U ) ) {
        mwords_msb--;
    }
    Esc_ASSERT( mwords_msb >= 0 );

    for ( i = ( (SINT16)EscRsa_RSA_SIZE_WORDS_LONG - 1 ); i > mwords_msb; i-- ) {
        /* if x_i == m_k-1 then q=b-1 else q=(x_i*b + x_i-1) div m_k-1 */
        /* Calc (x_i*b + x_i-1) already, we need it later too */
        ciAndCmin1 = ( (EscRsa_FWORD)cwords[ i ] << EscRsa_RSA_BASE ) | ( (EscRsa_FWORD)cwords[ i - 1 ] );

        if ( cwords[ i ] == mwords[ mwords_msb ] ) {
            q = (EscRsa_HWORD)EscRsa_MAX_VAL;
        } else {
            q = (EscRsa_HWORD)( ( ciAndCmin1 ) / (EscRsa_FWORD)mwords[  mwords_msb ] );
        }

        /* while q*(m[k-1] * b + m[k-2]) > x[i]*b^2 + x[i-1] *b + x[i-2] do q--; */
        finished = FALSE;
        while ( finished == FALSE ) {
            EscRsa_FWORD qcHi;
            EscRsa_FWORD qcLo;

            EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

            qcHi = (EscRsa_FWORD) q * (EscRsa_FWORD) mwords[ mwords_msb ];
            if ( mwords_msb > 0 ) {
                qcLo = (EscRsa_FWORD) q * (EscRsa_FWORD) mwords[ mwords_msb - 1 ];
            } else {
                qcLo = 0U;
            }
            qcHi += EscRsa_HI_FWORD( qcLo );

            /* qcHi contains the upper two bytes of q*(m[k-1] * b + m[k-2]),
               ciAndCmin1 the upper two bytes of x[i]*b^2 + x[i-1] */
            if ( qcHi < ciAndCmin1 ) {
                /* q*(m[k-1] * b + m[k-2]) < x[i]*b^2 + x[i-1] *b + x[i-2] */
                finished = TRUE;
            } else if ( ( qcHi == ciAndCmin1 ) && ( EscRsa_LO_HWORD( qcLo ) <= cwords[ i - 2 ] ) ) {
                /* q*(m[k-1] * b + m[k-2]) <= x[i]*b^2 + x[i-1] *b + x[i-2] */
                finished = TRUE;
            } else {
                q--;
            }
        }

        /* x = x-q*m*b^(i-k) */
        carry = 0U;
        for ( j = 0U; j <= (UINT16)mwords_msb; j++ ) {
            EscRsa_FWORD xhi;
            UINT16 cIdx;

            EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

            xhi = ( (EscRsa_FWORD)q * (EscRsa_FWORD)mwords[ j ] ) + (EscRsa_FWORD)carry;
            carry = EscRsa_HI_HWORD( xhi );
            cIdx = (UINT16)( ( ( (UINT16) i - (UINT16) mwords_msb ) - 1U ) + j );
            xhi = (EscRsa_FWORD)cwords[ cIdx ] - EscRsa_LO_FWORD( xhi );
            cwords[ cIdx ] = EscRsa_LO_HWORD( xhi );

            if ( EscRsa_HI_FWORD( xhi ) != 0U ) {
                carry++;
            }
        }

        /* if (x<0) then x=x+m*b^(i-k) */
        if ( cwords[ i ] < carry ) {
            carry = 0U;

            for ( j = 0U; j <= (UINT16) mwords_msb; j++ ) {
                EscRsa_FWORD xhi;
                UINT16 cIdx;

                EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

                cIdx = (UINT16)( ( ( (UINT16) i - (UINT16) mwords_msb ) - 1U ) + j );
                xhi = (EscRsa_FWORD)cwords[ cIdx ] + ( (EscRsa_FWORD)mwords[ j ] ) + (EscRsa_FWORD)carry;
                carry = EscRsa_HI_HWORD( xhi );
                cwords[ cIdx ] = EscRsa_LO_HWORD( xhi );
            }
        }
        cwords[ i ] = 0U;
        /*lint -restore cwords is an array; out of bounds pointer */
    }
}

/*********************************
  Multiplies two field elements c := c * x.
  We use the integer multiplication algorithm from Menzenes, et. al.
  "Elliptic Curve Cryptography". Algorithm 2.9.
  The algorithm is endianness independent.
 *********************************/
static void
EscRsaFe_MultiplyClassically(
    const EscRsa_MultiplyDataT* pm )
{
    EscRsa_HWORD i;
    EscRsa_HWORD j;
    EscRsa_HWORD min;
    EscRsa_HWORD temp;
    EscRsa_FWORD t;
    EscRsa_HWORD b_0, a_0;
    EscRsa_FWORD uv_t;
    EscRsa_FWORD uv;

    EscRsa_FieldElementT inC;
    EscRsa_HWORD* awords;
    EscRsa_HWORD* bwords;
    EscRsa_HWORD* cwords;

    awords = inC.words;
    bwords = pm->x->words;
    cwords = pm->c->words;

    uv_t = 0U;
    t = 0U;

    /* 1.) make a copy of c, clear c. inC:= c; c := 0 */
    Esc_ASSERT( cwords[ EscRsa_RSA_SIZE_WORDS ] == 0U );    /* single precision */
    EscRsaFe_FromLongFe( &inC, pm->c );

    for ( i = 0U; i < ( 2U * EscRsa_RSA_SIZE_WORDS ); i++ ) {
        b_0 = EscRsa_MIN( ( EscRsa_RSA_SIZE_WORDS - 1U ), i );
        a_0 = i - b_0;
        min = EscRsa_MIN( ( EscRsa_RSA_SIZE_WORDS - a_0 ), ( b_0 + 1U ) );
        for ( j = 0U; j < min; j++ ) {
            /* a[a_o +j] * b[b_o - j] */
            /*lint -save -esym(960,17.4) awords and bwords are arrays*/
            uv = (EscRsa_FWORD)awords[ a_0 + j ] * (EscRsa_FWORD)bwords[ b_0 - j ];
            /*lint -restore awords and bwords are arrays*/
            /* sum on uv_t */
            uv_t += uv;
            /* if overflow save carry to t */
            if ( uv_t < uv ) {
                t++;
            }
        }

        /* c[i] = v, v = u, u = t, t = 0 */
        /*lint -save -esym(960,17.4) cwords is an array*/
        cwords[ i ] = EscRsa_LO_HWORD( uv_t );
        /*lint -restore cwords is an array*/
        temp = EscRsa_HI_HWORD( uv_t );
        uv_t = temp;
        uv_t |= t << EscRsa_RSA_BASE;
        t = 0U;
    }
}

/*********************************
  Squares two field elements c := c * c
  The input c has the maximum length of RSA_SIZE_WORDS.
  By maintaining another copy of c in RAM,
  the multiplication algorithm can be used for this task.
 *********************************/
#if ( !defined( EscRsa_USE_MONTGOM_MUL ) ) || defined( EscRsa_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa_USE_SHORT_EXPONENT ) || defined( EscRsa_ENABLE_GENRSA )
static void
EscRsaFe_SquareClassically(
    const EscRsa_MultiplyDataT* pm )
{
    /* declarations */
#    if ( !defined( EscRsa_NO_MONTGOM_SHORT_EXP ) ) && ( !defined( EscRsa_USE_SLIDING_WINDOW ) )
    EscRsa_HWORD min;
    EscRsa_HWORD b_0, a_0;
#    endif
    EscRsa_HWORD i;
    EscRsa_HWORD j;
    EscRsa_FWORD uv;
    EscRsa_FWORD t = 0U;
    EscRsa_FWORD uv_t = 0U;

    EscRsa_FieldElementT inC;
    EscRsa_HWORD* cwords;

    cwords = pm->c->words;

    /* 1.) inC := c; c := 0 */
    Esc_ASSERT( cwords[ EscRsa_RSA_SIZE_WORDS ] == 0U );    /* single precision */
    EscRsaFe_FromLongFe( &inC, pm->c );

#    if defined( EscRsa_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa_USE_SLIDING_WINDOW )
    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        const EscRsa_HWORD max = ( i + 1U ) / 2U;
        const EscRsa_HWORD half = i / 2U;

        for ( j = 0U; j <= half; j++ ) {
            /* a[a_o + j] * b[b_o - j] */
            uv = (EscRsa_FWORD)inC.words[ j ] * (EscRsa_FWORD)inC.words[ i - j ];
            /* sum on uv_t */
            uv_t += uv;
            /* if overflow save carry to t */
            if ( uv_t < uv ) {
                t++;
            }
            if ( j < max ) {
                /* sum on uv_t */
                uv_t += uv;
                /* if overflow save carry to t */
                if ( uv_t < uv ) {
                    t++;
                }
            }
        }

        /* c[i] = v, v = u, u = t, t = 0 */
        /*lint -save -esym(960,17.4) cwords is an array*/
        cwords[ i ] = EscRsa_LO_HWORD( uv_t );
        /*lint -restore cwords is an array*/
        uv_t >>= EscRsa_RSA_BASE;
        uv_t |= t << EscRsa_RSA_BASE;
        t = 0U;
    }
    for (; i < ( EscRsa_RSA_SIZE_WORDS * 2U ); i++ ) {
        const EscRsa_HWORD max = ( i + 1U ) / 2U;
        const EscRsa_HWORD half = i / 2U;

        for ( j = ( ( i - EscRsa_RSA_SIZE_WORDS ) + 1U ); j <= half; j++ ) {
            /* a[a_o + j] * b[b_o - j] */
            uv = (EscRsa_FWORD)inC.words[ j ] * (EscRsa_FWORD)inC.words[ i - j ];
            /* sum on uv_t */
            uv_t += uv;
            /* if overflow save carry to t */
            if ( uv_t < uv ) {
                t++;
            }
            if ( j < max ) {
                /* sum on uv_t */
                uv_t += uv;
                /* if overflow save carry to t */
                if ( uv_t < uv ) {
                    t++;
                }
            }
        }

        /*lint -save -esym(960,17.4) cwords is an array*/
        cwords[ i ] = EscRsa_LO_HWORD( uv_t );
        /*lint -restore cwords is an array*/
        uv_t >>= EscRsa_RSA_BASE;
        uv_t |= t << EscRsa_RSA_BASE;
        t = 0U;
    }
#    else
    for ( i = 0U; i < ( 2U * EscRsa_RSA_SIZE_WORDS ); i++ ) {
        b_0 = EscRsa_MIN( ( EscRsa_RSA_SIZE_WORDS - 1U ), i );
        a_0 = i - b_0;
        min = EscRsa_MIN( ( EscRsa_RSA_SIZE_WORDS - a_0 ), ( b_0 + 1U ) );
        for ( j = 0U; j < min; j++ ) {
            /* a[a_o + j] * b[b_o - j] */
            uv = (EscRsa_FWORD)inC.words[ a_0 + j ] * (EscRsa_FWORD)inC.words[ b_0 - j ];
            /* sum on uv_t */
            uv_t += uv;
            /* if overflow save carry to t */
            if ( uv_t < uv ) {
                t++;
            }
        }

        /* c[i] = v, v = u, u = t, t = 0 */
        /*lint -save -esym(960,17.4) cwords is an array*/
        cwords[ i ] = EscRsa_LO_HWORD( uv_t );
        /*lint -restore cwords is an array*/
        uv_t >>= EscRsa_RSA_BASE;
        uv_t |= t << EscRsa_RSA_BASE;
        t = 0U;
    }
#    endif
}

/************************************************************************
 * c = c * c mod m                                                           *
 ************************************************************************/
static void
EscRsaFe_Square(
    const EscRsa_MultiplyDataT* pm )
{
    EscRsaFe_SquareClassically( pm );

    EscRsaFe_ModularReduction( pm );
}

#endif

/************************************************************************
 * c = c * x mod m                                                           *
 ************************************************************************/
static void
EscRsaFe_Multiply(
    const EscRsa_MultiplyDataT* pm )
{
    EscRsaFe_MultiplyClassically( pm );

    /* classical modular reduction */
    EscRsaFe_ModularReduction( pm );
}

/** returns whether or not the specified bit is set */
static BOOL
EscRsaFe_IsBitSet(
    const EscRsa_FieldElementT* e,
    UINT32 thebit )
{
    BOOL isSet;
    if ( ( e->words[ ( EscRsa_RSA_SIZE_WORDS - 1U ) - ( thebit / EscRsa_RSA_BASE ) ] & (EscRsa_HWORD)( (EscRsa_HWORD)1UL << ( ( EscRsa_RSA_BASE - 1U ) - ( thebit % EscRsa_RSA_BASE ) ) ) ) != 0U ) {
        isSet = TRUE;
    } else {
        isSet = FALSE;
    }

    return isSet;
}

/*********************************************************************
 * computes the power of a field element by the given exponent       *
 * x = x^e mod m                                                     *
 *                                                                   *
 * the exponent must not be 0                                        *
 *********************************************************************/
#ifdef EscRsa_USE_MONTGOM_MUL
#    ifdef EscRsa_USE_SLIDING_WINDOW
static void
EscRsaFe_BigPow(
    const EscRsa_MultiplyDataT* pm,
    const EscRsa_FieldElementT* e )
{
    EscRsa_FieldElementT x_, y_, one, x_2;
    EscRsa_MontGomElementT mg;
    UINT32 thebit = 0U;
    UINT32 i, n, windowsize, j;
    UINT16 exponent;
    /* array w/ size 2^k_m */
    EscRsa_FieldElementT fe_array[ (UINT16)( (UINT16)( (UINT16)2U << ( EscRsa_WINDOW_SIZE - 1U ) ) ) ];
    /* build word one */
    for ( i = 1U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        one.words[ i ] = 0U;
    }
    one.words[ 0 ] = 1U;

    /* build R = 2^mod_bit_size */
    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS_LONG; i++ ) {
        pm->c->words[ i ] = 0U;
    }
    pm->c->words[ EscRsa_RSA_SIZE_WORDS ] = 1U;

    EscRsaFe_ModularReduction( pm );
    /* x_ = R mod n */
    EscRsaFe_FromLongFe( &x_, pm->c );
    /* pm.c = c * x mod m  =  R * x mod m */
    EscRsaFe_Multiply( pm );
    /* y_ = x * R */
    EscRsaFe_FromLongFe( &y_, pm->c );

    /* init Montgomery variables */
    EscRsa_MontGom_Init( pm, &mg );

    /* fe_array[0] will not be used!! */
    EscRsaFe_Assign( &fe_array[ 0 ], &y_ );
    EscRsaFe_MontGom( &y_, &y_, &mg );
    EscRsaFe_Assign( &x_2, &fe_array[ 0 ] );
    /* precomputation - i < 2^k_m */
    for ( i = 1U; i < (UINT16)( (UINT16)( (UINT16)2U << ( EscRsa_WINDOW_SIZE - 1U ) ) ); i++ ) {
        EscRsaFe_MontGom( &x_2, &y_, &mg );
        EscRsaFe_Assign( &fe_array[ i ], &x_2 );
    }
    EscRsaFe_Assign( &x_, &fe_array[ 0 ] );
    while ( EscRsaFe_IsBitSet( e, thebit ) == FALSE ) {
        thebit++;
    }
    thebit++;

    while ( thebit < EscRsa_KEY_BITS ) {
        /* if Exponent = 0 double only */
        if ( EscRsaFe_IsBitSet( e, thebit ) == FALSE ) {
            EscRsaFe_MontGom( &x_, &x_, &mg );
            thebit++;
        } else {
            /* build window */
            exponent = 0U;
            n = EscRsa_WINDOW_SIZE - 1U;
            windowsize = EscRsa_WINDOW_SIZE;
            for ( j = thebit; ( j < ( thebit + EscRsa_WINDOW_SIZE ) ) && ( j < EscRsa_KEY_BITS ); j++ ) {
                /* build temporary exponent */
                if ( EscRsaFe_IsBitSet( e, j ) ) {
                    /*lint -save -e701 n is unsigned*/
                    exponent |= (UINT16)( (UINT16)1U << n );
                    /*lint -restore */
                }
                n--;
            }
            thebit += EscRsa_WINDOW_SIZE;
            /* make window odd and adjust i, reduce window size */
            while ( ( exponent % 2U ) == 0U ) {
                exponent = exponent >> 1U;
                thebit--;
                windowsize--;
            }

            /* double */
            for ( j = 0U; j < windowsize; j++ ) {
                EscRsaFe_MontGom( &x_, &x_, &mg );
            }
            /* add */
            EscRsaFe_MontGom( &x_, &fe_array[ (UINT16)( exponent / 2U ) ], &mg );
        }
    }
    EscRsaFe_MontGom( &x_, &one, &mg );
    EscRsaFe_Assign( pm->x, &x_ );
}

#    else /* EscRsa_USE_SLIDING_WINDOW */
static void
EscRsaFe_BigPow(
    const EscRsa_MultiplyDataT* pm,
    const EscRsa_FieldElementT* e )
{
    EscRsa_FieldElementT x_, y_, one;
    EscRsa_MontGomElementT mg;
    UINT32 thebit = 0U;
    UINT16 i;
    /* build word one */
    for ( i = 1U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        one.words[ i ] = 0U;
    }
    one.words[ 0 ] = 1U;

    /* build R = 2^mod_bit_size */
    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS_LONG; i++ ) {
        pm->c->words[ i ] = 0U;
    }
    pm->c->words[ EscRsa_RSA_SIZE_WORDS ] = 1U;

    EscRsaFe_ModularReduction( pm );
    /* x_ = R mod n */
    EscRsaFe_FromLongFe( &x_, pm->c );
    /* pm.c = c * x mod m  =  R * x mod m */
    EscRsaFe_Multiply( pm );
    /* y_ = x * R */
    EscRsaFe_FromLongFe( &y_, pm->c );

    /* init Montgomery variables */
    EscRsa_MontGom_Init( pm, &mg );

    /* find first bit in exponent != 0 */
    while ( EscRsaFe_IsBitSet( e, thebit ) == FALSE ) {
        thebit++;
    }

    /* square and multiply */
    while ( thebit < EscRsa_KEY_BITS ) {
        /* c := c * c mod m */
        EscRsaFe_MontGom( &x_, &x_, &mg );

        /* if e_i = 1 then c := c * x mod m */
        if ( EscRsaFe_IsBitSet( e, thebit ) ) {
            /* c := c * x mod m */
            EscRsaFe_MontGom( &x_, &y_, &mg );
        }
        thebit++;
    }
    EscRsaFe_MontGom( &x_, &one, &mg );
    /* x := c. We could actually copy the result directly into the
       output byte array, but stick to this for maintainability. */
    EscRsaFe_Assign( pm->x, &x_ );
}

#    endif

#else /* EscRsa_USE_MONTGOM_MUL */
#    ifdef EscRsa_USE_SLIDING_WINDOW
/* sliding window */
static void
EscRsaFe_BigPow(
    const EscRsa_MultiplyDataT* pm,
    const EscRsa_FieldElementT* e )
{
    UINT32 thebit = 0U;
    UINT32 i, n, windowsize, j;
    UINT16 exponent;
    /* array w/ size 2^k_m */
    EscRsa_FieldElementT fe_array[ (UINT16)( (UINT16)( (UINT16)2U << ( EscRsa_WINDOW_SIZE - 1U ) ) ) ];
    EscRsaFe_ToLongFe( pm->c, pm->x );
    EscRsaFe_FromLongFe( &fe_array[ 0 ], pm->c );
    EscRsaFe_Square( pm );
    EscRsaFe_FromLongFe( pm->x, pm->c );
    EscRsaFe_ToLongFe( pm->c, &fe_array[ 0 ] );

    /* precomputation - i < 2^k_m */
    for ( i = 1U; i < (UINT16)( (UINT16)( (UINT16)2U << ( EscRsa_WINDOW_SIZE - 1U ) ) ); i++ ) {
        EscRsaFe_Multiply( pm );
        EscRsaFe_FromLongFe( &fe_array[ i ], pm->c );
    }
    EscRsaFe_ToLongFe( pm->c, &fe_array[ 0 ] );

    while ( EscRsaFe_IsBitSet( e, thebit ) == FALSE ) {
        thebit++;
    }

    /* c == x => MSB has already been processed */
    thebit++;

    while ( thebit < EscRsa_KEY_BITS ) {
        /* if Exponent = 0 double only */
        if ( EscRsaFe_IsBitSet( e, thebit ) == FALSE ) {
            EscRsaFe_Square( pm );
            thebit++;
        } else {
            /* build window */
            exponent = 0U;
            n = EscRsa_WINDOW_SIZE - 1U;
            windowsize = EscRsa_WINDOW_SIZE;
            for ( j = thebit; ( j < ( thebit + EscRsa_WINDOW_SIZE ) ) && ( j < EscRsa_KEY_BITS ); j++ ) {
                /* build temporary exponent */
                if ( EscRsaFe_IsBitSet( e, j ) ) {
                    /*lint -save -e701 n is unsigned*/
                    exponent |= (UINT16)( (UINT16)1U << n );
                    /*lint -restore */
                }
                n--;
            }
            thebit += EscRsa_WINDOW_SIZE;
            /* make window odd and adjust i, reduce window size */
            while ( ( exponent & 1U ) == 0U ) {
                exponent >>= 1U;
                thebit--;
                windowsize--;
            }

            /* square */
            for ( j = 0U; j < windowsize; j++ ) {
                EscRsaFe_Square( pm );
            }
            /* multiply */
            EscRsaFe_Assign( pm->x, &fe_array[ (UINT16)( exponent / 2U ) ] );
            EscRsaFe_Multiply( pm );
        }
    }
    EscRsaFe_FromLongFe( pm->x, pm->c );
}

#    else /* EscRsa_USE_SLIDING_WINDOW */
/* square and multiply */
static void
EscRsaFe_BigPow(
    const EscRsa_MultiplyDataT* pm,
    const EscRsa_FieldElementT* e )
{
    /* Perform square and multiply. We left-shift the exponent
       and examine the most significant bit */
    UINT32 thebit = 0U;

    /* c := x */
    EscRsaFe_ToLongFe( pm->c, pm->x );

    /* find first bit in exponent != 0 */
    while ( EscRsaFe_IsBitSet( e, thebit ) == FALSE ) {
        thebit++;
    }

    /* c == x => MSB has already been processed */
    thebit++;

    /* square and multiply */
    while ( thebit < EscRsa_KEY_BITS ) {
        /* c := c * c mod m */
        EscRsaFe_Square( pm );

        /* if e_i = 1 then c := c * x mod m */
        if ( EscRsaFe_IsBitSet( e, thebit ) ) {
            /* c := c * x mod m */
            EscRsaFe_Multiply( pm );
        }
        thebit++;
    }

    /* x := c. We could actually copy the result directly into the
       output byte array, but stick to this for maintainability. */
    EscRsaFe_FromLongFe( pm->x, pm->c );
}

#    endif
#endif

/***********************************************
 * check if value of field element a is larger *
 * than the value of field element b           *
 ***********************************************/
static BOOL
EscRsaFe_GreaterThan(
    const EscRsa_FieldElementT* a,
    const EscRsa_FieldElementT* b )
{
    SINT16 i;
    BOOL isGreater = FALSE;

    for ( i = ( (SINT16)EscRsa_RSA_SIZE_WORDS - 1 ); i >= 0; i-- ) {
        if ( a->words[ i ] != b->words[ i ] ) {
            if ( a->words[ i ] > b->words[ i ] ) {
                isGreater = TRUE;
            } else {
                isGreater = FALSE;
            }
            break;
        }
    }

    return isGreater;
}

#ifdef EscRsa_USE_MONTGOM_MUL
static void
EscRsa_MontGom_Init(
    const EscRsa_MultiplyDataT* pm,
    EscRsa_MontGomElementT* mg )
{
    UINT8 i;
    EscRsa_HWORD y[ EscRsa_RSA_BASE + 1U ];
    /* assign modulus */
    EscRsaFe_Assign( &mg->mod, pm->m );
    mg->m = pm->m->words[ 0U ];
    /* calculate negative inverse of least significant word of the modulus mod 2^word_size
       see EUROCRYPT 90, Lecture Notes in ComputerScience, No. 473, pages 230-244 */
    y[ 1 ] = 1U;
    for ( i = 2U; i <= EscRsa_RSA_BASE; i++ ) {
        /*lint -save -e701 (i-1) is unsigned*/
        if ( i != EscRsa_RSA_BASE ) {
            y[ 0U ] = (EscRsa_HWORD)( ( mg->m * y[ i - 1U ] ) % ( (EscRsa_HWORD)( (EscRsa_HWORD)( 1U ) << i ) ) );
        } else {
            y[ 0U ] = (EscRsa_HWORD)( mg->m * y[ i - 1U ] );
        }
        if ( y[ 0U ] < (EscRsa_HWORD)( ( (EscRsa_HWORD)1U << ( i - 1U ) ) ) ) {
            y[ i ] = y[ i - 1U ];
        } else {
            y[ i ] = y[ i - 1U ] + ( (EscRsa_HWORD)( (EscRsa_HWORD)1U << ( i - 1U ) ) );
        }
        /*lint -restore */
    }
    /* calculate -y[RSA_BASE] mod 2^WORD_SIZE */
    mg->m = ( EscRsa_HWORD ) ~y[ EscRsa_RSA_BASE ];
    mg->m++;
}

/** Montgomery Multiplication w/ CIOS algorithm, x = MontGom(x, y)
 *  see Analyzing and Comparing Montgomery Multiplication Algorithms, Cetin Kaya Koc and Tolga Acar
 **/
static void
EscRsaFe_MontGom(
    EscRsa_FieldElementT* x,
    const EscRsa_FieldElementT* y,
    const EscRsa_MontGomElementT* mg )
{
    /* Initialization */
    EscRsa_HWORD i, j, n, uv_tmp;
    EscRsa_FWORD uv;
    EscRsa_HWORD c[ EscRsa_RSA_SIZE_WORDS + 2U ];
    EscRsa_HWORD t[ EscRsa_RSA_SIZE_WORDS + 2U ];

    /* zero c array */
    for ( i = 0U; i < ( EscRsa_RSA_SIZE_WORDS + 2U ); i++ ) {
        c[ i ] = 0U;
    }

    /* CIOS algorithm */
    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        uv = 0U;

        for ( j = 0U; j < EscRsa_RSA_SIZE_WORDS; j++ ) {
            uv_tmp = EscRsa_HI_HWORD( uv );
            uv = (EscRsa_FWORD)uv_tmp + ( (EscRsa_FWORD)x->words[ j ] * (EscRsa_FWORD)y->words[ i ] ) + (EscRsa_FWORD)c[ j ];
            c[ j ] = EscRsa_LO_HWORD( uv );
        }

        uv_tmp = EscRsa_HI_HWORD( uv );
        uv = (EscRsa_FWORD)c[ EscRsa_RSA_SIZE_WORDS ] + (EscRsa_FWORD)uv_tmp;
        c[ EscRsa_RSA_SIZE_WORDS ] = EscRsa_LO_HWORD( uv );
        c[ EscRsa_RSA_SIZE_WORDS + 1U ] = EscRsa_HI_HWORD( uv );
        n = (EscRsa_HWORD)( c[ 0U ] * mg->m );

        uv = (EscRsa_FWORD)c[ 0U ] + ( (EscRsa_FWORD)n * (EscRsa_FWORD)mg->mod.words[ 0U ] );
        for ( j = 1U; j < EscRsa_RSA_SIZE_WORDS; j++ ) {
            uv_tmp = EscRsa_HI_HWORD( uv );
            uv = (EscRsa_FWORD)uv_tmp + ( (EscRsa_FWORD)n * (EscRsa_FWORD)mg->mod.words[ j ] ) + (EscRsa_FWORD)c[ j ];
            c[ j - 1U ] = EscRsa_LO_HWORD( uv );
        }

        uv_tmp = EscRsa_HI_HWORD( uv );
        uv = (EscRsa_FWORD)c[ EscRsa_RSA_SIZE_WORDS ] + (EscRsa_FWORD)uv_tmp;
        c[ EscRsa_RSA_SIZE_WORDS - 1U ] = EscRsa_LO_HWORD( uv );
        c[ EscRsa_RSA_SIZE_WORDS ] = c[ EscRsa_RSA_SIZE_WORDS + 1U ] + EscRsa_HI_HWORD( uv );
    }

    /* third step of Montgomery, subtract modulus if necessary */
    uv = 0U;
    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        /* subtract mod and save carry in highword of uv (tricky one ;) ) */
        uv_tmp = (EscRsa_HWORD)( (EscRsa_HWORD)( ~EscRsa_HI_HWORD( uv ) ) + 1U );
        uv = ( (EscRsa_FWORD)c[ i ] - (EscRsa_FWORD)mg->mod.words[ i ] ) - (EscRsa_FWORD)uv_tmp;
        t[ i ] = EscRsa_LO_HWORD( uv );
    }
    /* last carry */
    uv_tmp = (EscRsa_HWORD)( (EscRsa_HWORD)( ~EscRsa_HI_HWORD( uv ) ) + 1U );
    uv = (EscRsa_FWORD)c[ EscRsa_RSA_SIZE_WORDS ] - (EscRsa_FWORD)uv_tmp;
    t[ EscRsa_RSA_SIZE_WORDS ] = EscRsa_LO_HWORD( uv );

    /* c > mod ? return t:return c */
    if ( EscRsa_HI_HWORD( uv ) == 0U ) {
        for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
            x->words[ i ] = t[ i ];
        }
    } else {
        for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
            x->words[ i ] = c[ i ];
        }
    }
}

#endif

#if defined( EscRsa_NO_MONTGOM_SHORT_EXP ) || defined( EscRsa_USE_SHORT_EXPONENT ) || defined( EscRsa_ENABLE_STACK_SAVING_INTERFACE )
/***************************************************************************
 * Modular exponentiation for very small exponents. c = m^e mod n           *
 * (expects big endian input)                                              *
 * maximum exponent = (2^32)-1                                             *
 ***************************************************************************/
#    ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
static BOOL
EscRsa_ShortPow(
    const UINT8 message[],
    const UINT8 modulus[],
    const UINT32 exponent,
    UINT8 result[] )
#    else
static BOOL
EscRsa_ShortPow(
    EscRsa_FieldElementT* message,
    const EscRsa_FieldElementT* modulus,
    const UINT32 exponent )
#    endif
{
    /* declarations */
    BOOL hasFailed = TRUE;
    SINT8 thebit = 31;
    EscRsa_FieldElementLongT multiplication_result;
    EscRsa_MultiplyDataT mult;
#    ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
    EscRsa_FieldElementT x, m;
#    endif

    mult.c = &multiplication_result;
#    ifdef EscRsa_ENABLE_STACK_SAVING_INTERFACE
    mult.m = modulus;
    mult.x = message;
#    else
    mult.m = &m;
    mult.x = &x;
#    endif

    if ( ( message != 0 ) &&
         ( modulus != 0 )
#    ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
         && ( result != 0 )
#    endif
         && ( exponent != 0U ) )
    {
#    ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
        EscRsaFe_FromBytesBE( &x, message, EscRsa_KEY_BYTES );

        /* We never calculate with the modulus directly, only the
            normalized version. */
        EscRsaFe_FromBytesBE( &m, modulus, EscRsa_KEY_BYTES );
#    endif

        if ( mult.m->words[ EscRsa_RSA_SIZE_WORDS - 1U ] != 0U ) {
            /* perform parameter and plausibility check   */
            /* make sure modulus > signature */
            if ( EscRsaFe_GreaterThan( mult.m, mult.x ) == TRUE ) {
                EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

                /* c := x */
                EscRsaFe_ToLongFe( mult.c, mult.x );

                /* find first bit in exponent != 0 */
                while ( ( exponent & (UINT32)( 1UL << thebit ) ) == 0U ) {
                    thebit--;
                }

                /* c == x => MSB has already been processed */
                thebit--;

                /* square and multiply */
                while ( thebit >= 0 ) {
                    /* c := c * c mod m */
                    EscRsaFe_Square( &mult );

                    /* if e_i = 1 then c := c * x mod m */
                    if ( ( exponent & (UINT32)( 1UL << thebit ) ) != 0U ) {
                        /* c := c * x mod m */
                        EscRsaFe_Multiply( &mult );
                    }
                    thebit--;
                }

                /* x := c. We could actually copy the result directly into the
                    output byte array, but stick to this for maintainability. */
                EscRsaFe_FromLongFe( mult.x, mult.c );

                EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

#    ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
                EscRsaFe_ToBytesBE( result, mult.x );
#    endif
                hasFailed = FALSE;
            }
        }
    }
    return hasFailed;
}

#endif

/***************************************************************************
 * Modular exponentiation c = m^e mod n (expects big endian input)          *
 ***************************************************************************/
#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
BOOL
EscRsa_ModExpLong(
    const UINT8 message[],
    const UINT8 modulus[],
    const UINT8 exponent[],
    UINT8 result[] )
#else
BOOL
EscRsa_ModExpLongSaveStack(
    EscRsa_FieldElementT* message,
    const EscRsa_FieldElementT* modulus,
    const EscRsa_FieldElementT* exponent_field )
#endif
{
    /* declarations */
    BOOL hasFailed = TRUE;
    EscRsa_FieldElementLongT multiplication_result;
    EscRsa_MultiplyDataT mult;
#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
    EscRsa_FieldElementT x, m;
    EscRsa_FieldElementT exponent_field;
#endif

    EscRsa_HWORD sum = 0U;
    UINT32 i;

    mult.c = &multiplication_result;
#ifdef EscRsa_ENABLE_STACK_SAVING_INTERFACE
    mult.m = modulus;
    mult.x = message;
#else
    mult.m = &m;
    mult.x = &x;
#endif

    if ( ( message != 0 ) &&
         ( modulus != 0 )
#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
         && ( result != 0 )
         && ( exponent != 0 ) )
#else
         && ( exponent_field != 0 ) )
#endif
    {
#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
        EscRsaFe_FromBytesBE( &exponent_field, exponent, EscRsa_KEY_BYTES );
#endif
        /* the exponent has to be at least 1 */
        for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
#ifdef EscRsa_ENABLE_STACK_SAVING_INTERFACE
            sum |= exponent_field->words[ i ];
#else
            sum |= exponent_field.words[ i ];
#endif
        }
        if ( sum != 0U ) {
#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
            EscRsaFe_FromBytesBE( &x, message, EscRsa_KEY_BYTES );

            /* We never calculate with the modulus directly, only the
               normalized version. */
            EscRsaFe_FromBytesBE( &m, modulus, EscRsa_KEY_BYTES );
#endif

            if ( mult.m->words[ EscRsa_RSA_SIZE_WORDS - 1U ] != 0U ) {
                /* perform parameter and plausibility check   */
                /* make sure modulus > signature */
                if ( EscRsaFe_GreaterThan( mult.m, mult.x ) == TRUE ) {
                    EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

#ifdef EscRsa_ENABLE_STACK_SAVING_INTERFACE
                    EscRsaFe_BigPow( &mult, exponent_field );
#else
                    EscRsaFe_BigPow( &mult, &exponent_field );
#endif
                    EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );

#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
                    EscRsaFe_ToBytesBE( result, mult.x );
#endif
                    hasFailed = FALSE;
                }
            }
        }
    }
    /* parameter check */
    return hasFailed;
}

/***************************************************************************
 * Modular exponentiation for small exponents. c = m^e mod n                *
 * (expects big endian input)                                              *
 ***************************************************************************/
#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
BOOL
EscRsa_ModExp(
    const UINT8 message[],
    const UINT8 modulus[],
    const UINT32 exponent,
    UINT8 result[] )
#else
BOOL
EscRsa_ModExpSaveStack(
    EscRsa_FieldElementT* message,
    const EscRsa_FieldElementT* modulus,
    const UINT32 exponent )
#endif
{
    BOOL hasFailed;

#if defined( EscRsa_ENABLE_STACK_SAVING_INTERFACE )
    hasFailed = EscRsa_ShortPow( message, modulus, exponent );
#elif defined( EscRsa_USE_SHORT_EXPONENT )
    hasFailed = EscRsa_ShortPow( message, modulus, exponent, result );
#else /* EscRsa_USE_SHORT_EXPONENT / EscRsa_ENABLE_STACK_SAVING_INTERFACE */
#    ifdef EscRsa_NO_MONTGOM_SHORT_EXP
    if ( exponent == 3U ) {
        hasFailed = EscRsa_ShortPow( message, modulus, exponent, result );
    } else {
#    endif /* EscRsa_NO_MONTGOM_SHORT_EXP */

        UINT8 longexp[ EscRsa_KEY_BYTES ];
        UINT32 i;

        for ( i = 0U; i < ( EscRsa_KEY_BYTES - 4U ); i++ ) {
            longexp[ i ] = 0U;
        }
        longexp[ EscRsa_KEY_BYTES - 1U ] = (UINT8)( ( exponent ) & 0xFFU );
        longexp[ EscRsa_KEY_BYTES - 2U ] = (UINT8)( ( exponent >> 8 ) & 0xFFU );
        longexp[ EscRsa_KEY_BYTES - 3U ] = (UINT8)( ( exponent >> 16 ) & 0xFFU );
        longexp[ EscRsa_KEY_BYTES - 4U ] = (UINT8)( ( exponent >> 24 ) & 0xFFU );

        hasFailed = EscRsa_ModExpLong( message, modulus, longexp, result );

#    ifdef EscRsa_NO_MONTGOM_SHORT_EXP
    }
#    endif /* EscRsa_NO_MONTGOM_SHORT_EXP */

#endif /* EscRsa_USE_SHORT_EXPONENT */
    return hasFailed;
}

#if defined ( EscRsa_ENABLE_GENRSA ) || defined ( EscRsa_ENABLE_CRT )
/*******************************************
 * compares two field elements `a' and `b' *
  +1: a>b
  -1: a<b
   0: a==b
 *******************************************/
static SINT8
EscRsaFe_AbsoluteCompare(
    const EscRsa_FieldElementT* a,
    const EscRsa_FieldElementT* b )
{
    SINT16 i;
    SINT8 compResult = 0;

    /* compare the corresponding words until a different word pair is found */
    for ( i = (SINT16)( (SINT16)EscRsa_RSA_SIZE_WORDS - 1 ); ( compResult == 0 ) && ( i >= 0 ); i-- ) {
        /* next lower pair */
        if ( a->words[ i ] > b->words[ i ] ) {
            /* a > b */
            compResult = 1;
        } else if ( a->words[ i ] < b->words[ i ] ) {
            /* a < b */
            compResult = -1;
        } else {
            /* do nothing */
        }
    }

    return compResult;
}

/************************
 * Calculate a := a - b *
 ************************/
static void
EscRsaFe_Subtract(
    EscRsa_FieldElementT* a,
    const EscRsa_FieldElementT* b )
{
    EscRsa_FWORD carry = 1U;
    EscRsa_FWORD diff;
    UINT32 i;

    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++) {
        diff = ( (EscRsa_FWORD) a->words[ i ] ) + ( (EscRsa_FWORD) ( b->words[ i ] ^ EscRsa_MAX_VAL ) ) + carry;
        carry = EscRsa_HI_FWORD( diff );
        a->words[ i ] = (EscRsa_HWORD) EscRsa_LO_FWORD( diff );
    }
}
#endif

#ifdef EscRsa_ENABLE_GENRSA
/*****************************************
 * shifts field element right by one bit *
 *****************************************/
static void
EscRsaFe_ShiftRight(
    EscRsa_FieldElementT* a )
{
    UINT16 i;

    /* shift first words */
    for ( i = 0U; i < ( EscRsa_RSA_SIZE_WORDS - 1U ); i++ ) {
        EscRsa_HWORD v;
        v = a->words[ i ] >> 1;
        v |= (EscRsa_HWORD)( a->words[ i + 1U ] << ( EscRsa_RSA_BASE - 1U ) );

        a->words[ i ] = v;
    }

    /* shift last word */
    a->words[ EscRsa_RSA_SIZE_WORDS - 1U ] >>= 1;   /* most significant word */
}

/*****************************************
 * shifts field element right by one bit *
 *****************************************/
static void
EscRsaFe_SignedShiftRight(
    EscRsa_FieldElementT* a,
    EscRsa_HWORD* aSign )
{
    UINT16 i;
    EscRsa_HWORD v;

    /* shift first words */
    for ( i = 0U; i < ( EscRsa_RSA_SIZE_WORDS - 1U ); i++ ) {
        v = a->words[ i ] >> 1;
        v |= (EscRsa_HWORD)( a->words[ i + 1U ] << ( EscRsa_RSA_BASE - 1U ) );
        a->words[ i ] = v;
    }

    /* shift last word */
    v = a->words[ EscRsa_RSA_SIZE_WORDS - 1U ] >> 1;
    v |= (EscRsa_HWORD) ( *aSign << ( EscRsa_RSA_BASE - 1U ) );
    a->words[ EscRsa_RSA_SIZE_WORDS - 1U ] = v;
    v = *aSign & ( (EscRsa_HWORD) ( (EscRsa_HWORD) 1U << ( EscRsa_RSA_BASE - 1U ) ) );
    *aSign >>= 1;
    *aSign |= v;
}

/**********************************
 * check if field element is zero *
 * note: sign irrelevant          *
 **********************************/
static BOOL
EscRsaFe_IsZero(
    const EscRsa_FieldElementT* a )
{
    UINT16 i;
    BOOL isZero = TRUE;

    /* for i from 0 to t do a[i] =? 0 */
    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        if ( a->words[ i ] != 0U ) {
            /* not zero */
            isZero = FALSE;
            break;
        }
    }

    return isZero;
}

/**********************************
 * checks if field element is one *
 **********************************/
static BOOL
EscRsaFe_IsOne(
    const EscRsa_FieldElementT* a )
{
    /* declarations */
    BOOL isOne = TRUE;

    /* a[0] =? 1 */
    if ( a->words[ 0 ] != 1U ) {
        isOne = FALSE;
    } else {
        UINT16 i;
        /* for i from 1 to t-1 do a[i] =? 0 */
        for ( i = 1U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
            if ( a->words[ i ] != 0U ) {
                /* not one */
                isOne = FALSE;
            }
        }
    }

    return isOne;
}

/************************
 * Calculate a := a + b *
 ************************/
static void
EscRsaFe_SignedAdd(
    EscRsa_FieldElementT* a,
    EscRsa_HWORD* aSign,
    const EscRsa_FieldElementT* b,
    const EscRsa_HWORD bSign )
{
    EscRsa_FWORD carry = 0U;
    EscRsa_FWORD sum;
    UINT32 i;

    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++) {
        sum = ( (EscRsa_FWORD) a->words[ i ] ) + ( (EscRsa_FWORD) b->words[ i ] ) + carry;
        carry = EscRsa_HI_FWORD( sum );
        a->words[ i ] = (EscRsa_HWORD) EscRsa_LO_FWORD( sum );
    }
    sum = ( (EscRsa_FWORD) *aSign ) + ( (EscRsa_FWORD) ( bSign ) ) + carry;
    *aSign = (EscRsa_HWORD) EscRsa_LO_FWORD( sum );
}

/************************
 * Calculate a := a - b *
 ************************/
static void
EscRsaFe_SignedSubtract(
    EscRsa_FieldElementT* a,
    EscRsa_HWORD* aSign,
    const EscRsa_FieldElementT* b,
    const EscRsa_HWORD bSign )
{
    EscRsa_FWORD carry = 1U;
    EscRsa_FWORD diff;
    UINT32 i;

    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++) {
        diff = ( (EscRsa_FWORD) a->words[ i ] ) + ( (EscRsa_FWORD) (  b->words[ i ] ^ EscRsa_MAX_VAL ) ) + carry;
        carry = EscRsa_HI_FWORD( diff );
        a->words[ i ] = (EscRsa_HWORD) EscRsa_LO_FWORD( diff );
    }
    diff = ( (EscRsa_FWORD) *aSign ) + ( (EscRsa_FWORD) ( bSign ^ EscRsa_MAX_VAL ) ) + carry;
    *aSign = (EscRsa_HWORD) EscRsa_LO_FWORD( diff );
}

/*******************************************
 * Add a 16/32 bit word to a field element *
 *******************************************/
static void
EscRsaFe_AddWord(
    EscRsa_FieldElementT* c,
    const EscRsa_HWORD a )
{
    EscRsa_HWORD* cwords;
    UINT32 i;

    cwords = c->words;

    /*lint -save -esym(960,17.4) cwords is an array*/
    cwords[ 0U ] += a;
    if ( cwords[ 0U ] < a ) {
        for ( i = 1U; i < EscRsa_RSA_SIZE_WORDS; i++) {
            cwords[ i ] += 1U;
            if ( cwords[ i ] >= 1U ) {
                break;
            }
        }
    }
    /*lint -restore cwords is an array*/
}

/******************************************************
 * Compute modulo of field element and a small number *
 ******************************************************/
static EscRsa_HWORD
EscRsaFe_ModWord(
    const EscRsa_FieldElementT* a,
    EscRsa_HWORD w )
{
    EscRsa_FWORD ret = 0U;
    SINT32 i;

    for ( i = (SINT32) EscRsa_RSA_SIZE_WORDS - 1; i >= 0; i--) {
        ret = ( ( ret << EscRsa_RSA_BASE ) | (EscRsa_FWORD) a->words[ i ] ) % (EscRsa_FWORD) w;
    }

    return (EscRsa_HWORD) ret;
}

/******************************************************
 * Test if a signed field element is positive (> 0)   *
 ******************************************************/
static BOOL
EscRsaFe_IsPositive(
    EscRsa_HWORD sign,
    const EscRsa_FieldElementT* fe)
{
    BOOL ret = FALSE;

    /* If MSB not set ... */
    if ( (sign & ( (EscRsa_HWORD) ( (EscRsa_HWORD) 1U << ( EscRsa_RSA_BASE - 1U ) ) )) == 0U) {
        /* .. and the entire value is non-zero ... */
        if ( (sign > 0) || (EscRsaFe_IsZero(fe) == FALSE) ) {
            /* ... then it's positive */
            ret = TRUE;
        }
    }

    return ret;
}

/*********************************************
 * Modular inverse using binary extended gcd *
 *********************************************/
static BOOL
EscRsaFe_ModInv(
    const EscRsa_FieldElementT* x,
    const EscRsa_FieldElementT* y,
    EscRsa_FieldElementT* result )
{
    EscRsa_FieldElementT u;
    EscRsa_FieldElementT v;
    EscRsa_FieldElementT a;
    EscRsa_FieldElementT b;
    EscRsa_FieldElementT c;
    EscRsa_FieldElementT d;
    UINT32 i;
    EscRsa_HWORD aSign;
    EscRsa_HWORD bSign;
    EscRsa_HWORD cSign;
    EscRsa_HWORD dSign;
    BOOL hasFailed = FALSE;

    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
        u.words[ i ] = x->words[ i ];
        v.words[ i ] = y->words[ i ];
        a.words[ i ] = 0U;
        b.words[ i ] = 0U;
        c.words[ i ] = 0U;
        d.words[ i ] = 0U;
    }
    a.words[ 0U ] = 1U;
    d.words[ 0U ] = 1U;
    aSign = 0U;
    bSign = 0U;
    cSign = 0U;
    dSign = 0U;

    do {
        while ( ( u.words[ 0U ] & 1U ) == 0U ) {
            EscRsaFe_ShiftRight( &u );
            if ( ( ( a.words[ 0U ] & 1U ) == 1U ) || ( ( b.words[ 0U ] & 1U ) == 1U ) ) {
                EscRsaFe_SignedAdd( &a, &aSign, y, 0U );
                EscRsaFe_SignedSubtract( &b, &bSign, x, 0U );
            }
            EscRsaFe_SignedShiftRight( &a, &aSign );
            EscRsaFe_SignedShiftRight( &b, &bSign );
        }
        while ( ( v.words[ 0U ] & 1U ) == 0U ) {
            EscRsaFe_ShiftRight( &v );
            if ( ( ( c.words[ 0U ] & 1U ) == 1U ) || ( ( d.words[ 0U ] & 1U ) == 1U ) ) {
                EscRsaFe_SignedAdd( &c, &cSign, y, 0U );
                EscRsaFe_SignedSubtract( &d, &dSign, x, 0U );
            }
            EscRsaFe_SignedShiftRight( &c, &cSign );
            EscRsaFe_SignedShiftRight( &d, &dSign );
        }
        if ( EscRsaFe_AbsoluteCompare( &u, &v ) > -1 ) {
            EscRsaFe_Subtract( &u, &v );
            EscRsaFe_SignedSubtract( &a, &aSign, &c, cSign );
            EscRsaFe_SignedSubtract( &b, &bSign, &d, dSign );
            /* "If B>0 then A <-- A+y and B <-- B-x." */
            if ( EscRsaFe_IsPositive(bSign, &b) != FALSE ) {
                EscRsaFe_SignedAdd( &a, &aSign, y, 0U );
                EscRsaFe_SignedSubtract( &b, &bSign, x, 0U );
            }
        } else {
            EscRsaFe_Subtract( &v, &u );
            EscRsaFe_SignedSubtract( &c, &cSign, &a, aSign );
            EscRsaFe_SignedSubtract( &d, &dSign, &b, bSign );
            /* "If D>0 then C <-- C+y and D <-- D-x." */
            if ( EscRsaFe_IsPositive(dSign, &d) != FALSE ) {
                EscRsaFe_SignedAdd( &c, &cSign, y, 0U );
                EscRsaFe_SignedSubtract( &d, &dSign, x, 0U );
            }
        }
    } while ( EscRsaFe_IsZero( &u ) == FALSE );

    if ( EscRsaFe_IsOne( &v ) == FALSE ) {
        hasFailed = TRUE;
    }

    if ( dSign ) {
        EscRsaFe_SignedAdd( &d, &dSign, x, 0U );
    }

    EscRsaFe_Assign( result, &d );

    return hasFailed;
}

/*******************************
 * Miller-Rabin primality test *
 *******************************/
static BOOL
EscRsa_MillerRabin(
    EscRandom_ContextT* randCtx,
    const EscRsa_FieldElementT* n,
    const UINT32 iter,
    BOOL* prime )
{
    EscRsa_FieldElementT n1;
    EscRsa_FieldElementT r;
    EscRsa_FieldElementT y;
    EscRsa_FieldElementLongT multiplication_result;
    EscRsa_MultiplyDataT mult;
    UINT8 a[ EscRsa_KEY_BYTES ];
    UINT32 s;
    UINT32 i;
    UINT32 j;
    BOOL hasFailed = FALSE;

    /* n1 = n - 1, requires that n is odd */
    EscRsaFe_Assign( &n1, n );
    n1.words[ 0 ] &= (EscRsa_HWORD) ( ~(EscRsa_HWORD)1U );

    /* r = n */
    EscRsaFe_Assign( &r, n );

    /* Decompose candidate n into r and s with n - 1 = 2^s * r */
    s = 0U;
    do {
        EscRsaFe_ShiftRight( &r );
        s++;
    } while ( ( r.words[ 0 ] & 1U ) == 0U );

    for ( i = 0U; i < ( EscRsa_KEY_BYTES / 2U ); i++ ) {
        a[ i ] = 0U;
    }

    *prime = TRUE;
    i = 0U;
    while ( ( *prime == TRUE ) && ( i < iter ) ) {
        do {
            /* Get random witness a with 2 <= a <= n - 2 */
            hasFailed |= EscRandom_GetRandom( randCtx, &a[ EscRsa_KEY_BYTES / 2U ], EscRsa_KEY_BYTES / 2U );
            a[ EscRsa_KEY_BYTES / 2U ] &= 0x7fU;
            EscRsaFe_FromBytesBE( &y, a, EscRsa_KEY_BYTES );
        } while ( ( EscRsaFe_IsZero( &y ) == TRUE )
              || ( EscRsaFe_IsOne( &y ) == TRUE )
              || ( EscRsaFe_AbsoluteCompare( &y, &n1 ) != -1 ) );

        /* y = a^r mod n */
        mult.c = &multiplication_result;
        mult.m = n;
        mult.x = &y;
        EscRsaFe_BigPow( &mult, &r );

        if ( ( EscRsaFe_IsOne( &y ) == FALSE ) && ( EscRsaFe_AbsoluteCompare( &y, &n1 ) != 0 ) ) {
            j = 1U;
            while ( ( j < s ) && ( EscRsaFe_AbsoluteCompare( &y, &n1 ) != 0 ) ) {
                /* y = y^2 mod n */
                mult.c = &multiplication_result;
                mult.m = n;
                mult.x = &y;
                EscRsaFe_Square( &mult );

                if ( EscRsaFe_IsOne( &y ) == TRUE ) {
                    *prime = FALSE;
                    break;
                }
                j++;
            }
            if ( ( *prime == TRUE ) && ( EscRsaFe_AbsoluteCompare( &y, &n1 ) != 0 ) ) {
                *prime = FALSE;
            }
        }
        i++;
    }

    return hasFailed;
}

/********************************
 * Generate random prime number *
 ********************************/
#ifdef CONFIG_KEYGEN_PROFILING
 #include <time.h>
 #include <stdio.h>
 inline long timevaldiff(struct timeval *starttime, struct timeval *finishtime)
{
  long msec;
  msec=(finishtime->tv_sec-starttime->tv_sec)*1000;
  msec+=(finishtime->tv_usec-starttime->tv_usec)/1000;
  return msec;
}
#endif /* CONFIG_KEYGEN_PROFILING */

static BOOL
EscRsa_GeneratePrime(
    EscRandom_ContextT* randCtx,
    EscRsa_FieldElementT* n )
{
    static const UINT8 primes[ EscRsa_NUM_PRIMES ] =
    {
          3U,   5U,   7U,  11U,  13U,  17U,  19U,  23U,  29U,  31U,  37U,  41U,  43U,  47U,  53U,  59U,
         61U,  67U,  71U,  73U,  79U,  83U,  89U,  97U, 101U, 103U, 107U, 109U, 113U, 127U, 131U, 137U,
        139U, 149U, 151U, 157U, 163U, 167U, 173U, 179U, 181U, 191U, 193U, 197U, 199U, 211U, 223U, 227U,
        229U, 233U, 239U, 241U, 251U
    };
    UINT8 candidate[ EscRsa_KEY_BYTES ];
    UINT8 mods[ EscRsa_NUM_PRIMES ];
    EscRsa_HWORD delta;
    UINT32 i;
    BOOL prime = FALSE;
    BOOL hasFailed;
#ifdef CONFIG_KEYGEN_PROFILING
    struct timeval funcS, funcE;
    UINT32 func = 0;

    gettimeofday(&funcS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING*/

    /* Get random number */
    for ( i = 0U; i < ( EscRsa_KEY_BYTES / 2U ); i++ ) {
        candidate[ i ] = 0U;
    }
    hasFailed = EscRandom_GetRandom( randCtx, &candidate[ EscRsa_KEY_BYTES / 2U ], EscRsa_KEY_BYTES / 2U );

    /* Set two most significant bits */
    candidate[ EscRsa_KEY_BYTES / 2U ] |= 0xc0U;

    /* Set least significant bit */
    candidate[ EscRsa_KEY_BYTES - 1U ] |= 1U;

    /* Convert to field element */
    EscRsaFe_FromBytesBE( n, candidate, EscRsa_KEY_BYTES );

#ifdef CONFIG_KEYGEN_PROFILING
    int abc = 0;
    int def = 0;
    int millRabs = 0;
    struct timeval intWhileS, intWhileE;
    struct timeval calcModS, calcModE;
    struct timeval addWordS, addWordE;
    struct timeval millS, millE;
    UINT32 intWhile = 0, calcMod = 0, addWord = 0, mill = 0;
#endif /* CONFIG_KEYGEN_PROFILING */
    while ( ( prime == FALSE ) && ( hasFailed == FALSE ) ) {
#ifdef CONFIG_KEYGEN_PROFILING
        abc++;
#endif /* CONFIG_KEYGEN_PROFILING */
        /* Calculate modulo */
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&calcModS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */
        for ( i = 0U; i < EscRsa_NUM_PRIMES; i++ ) {
            mods[ i ] = (UINT8) EscRsaFe_ModWord( n, (EscRsa_HWORD) primes[ i ] );
        }
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&calcModE, NULL);
        calcMod += timevaldiff(&calcModS, &calcModE);
        gettimeofday(&intWhileS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */
        delta = 0U;
        while ( ( prime == FALSE ) && ( delta < ( EscRsa_MAX_VAL - primes[ EscRsa_NUM_PRIMES - 1U ] ) ) ) {
#ifdef CONFIG_KEYGEN_PROFILING
        def++;
#endif /* CONFIG_KEYGEN_PROFILING */
            EscWatchdog_Call( EscWatchdog_STATIC_WATCHDOG_TRIGGER );
            prime = TRUE;
            for ( i = 0U; i < EscRsa_NUM_PRIMES; i++ ) {
                if ( ( ( (EscRsa_HWORD) mods[ i ] + delta ) % (EscRsa_HWORD) primes[ i ] ) <= 1U ) {
                    prime = FALSE;
                    delta += 2U;
                    break;
                }
            }
        }
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&intWhileE, NULL);
        intWhile += timevaldiff(&intWhileS, &intWhileE);
#endif /* CONFIG_KEYGEN_PROFILING */

        /* Add delta to candidate */
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&addWordS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */
        EscRsaFe_AddWord( n, delta );
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&addWordE, NULL);
        addWord += timevaldiff(&addWordS, &addWordE);
#endif /* CONFIG_KEYGEN_PROFILING */

        if ( prime == TRUE ) {
            /* Miller-Rabin primality test */
#ifdef CONFIG_KEYGEN_PROFILING
            millRabs++;
            gettimeofday(&millS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */
            hasFailed |= EscRsa_MillerRabin( randCtx, n, EscRsa_PRIMETEST_ITER, &prime );
#ifdef CONFIG_KEYGEN_PROFILING
            gettimeofday(&millE, NULL);
            mill += timevaldiff(&millS, &millE);
#endif /* CONFIG_KEYGEN_PROFILING */

            if ( prime == FALSE ) {
                EscRsaFe_AddWord( n, 2U );
            }
        }
    }
#ifdef CONFIG_KEYGEN_PROFILING
        gettimeofday(&funcE, NULL);
        func += timevaldiff(&funcS, &funcE);

    printf("outer while loop called %d times\n",abc);
    printf("inner while loop called %d times\n",def);
    printf("%s %ldms\n",__func__, func);
    printf("tot. calculate modulus %ldms\n", calcMod);
    printf("avg. calculate modulus %ldms\n", calcMod / abc);
    printf("tot. internal while loop %ldms\n", intWhile );
    printf("avg. internal while loop %ldms\n", intWhile / abc);
    printf("tot. addWord %ldms\n", addWord );
    printf("avg. addWord %ldms\n", addWord / abc);
    printf("tot. miller rabin %ldms\n", mill );
    printf("avg. miller rabin %ldms\n", mill / millRabs);
#endif /* CONFIG_KEYGEN_PROFILING */

    return hasFailed;
}

/*************************
 * Generate RSA key pair *
 *************************/
BOOL
EscRsa_GenerateKeyPair(
    EscRandom_ContextT* randCtx,
    EscRsa_KeyPairT* keyPair,
    UINT32 pubExp )
{
    EscRsa_FieldElementLongT multiplication_result;
    EscRsa_MultiplyDataT mult;
    EscRsa_FWORD dist;
    UINT32 i;
    BOOL hasFailed = FALSE;

#ifdef CONFIG_KEYGEN_PROFILING
    struct timeval funcS, funcE;
    UINT32 func = 0;

    gettimeofday(&funcS, NULL);
#endif /* CONFIG_KEYGEN_PROFILING */

    if ( ( randCtx != 0 ) && ( pubExp != 0U ) && ( keyPair != 0 ) ) {

        /* Set public exponent e */
        for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
            keyPair->e.words[ i ] = 0U;
        }
#    if EscRsa_WORD_SIZE == 4U
        keyPair->e.words[ 0U ] = pubExp;
#    else
        keyPair->e.words[ 0U ] = (EscRsa_HWORD)( pubExp >> EscRsa_RSA_BASE );
        keyPair->e.words[ 1U ] = (EscRsa_HWORD)( pubExp & EscRsa_MAX_VAL );
#    endif

        /* Generate random prime number for p */
#ifdef CONFIG_KEYGEN_PROFILING
        printf("%d start generate prime for p\n", time(NULL));
#endif /* CONFIG_KEYGEN_PROFILING */
        hasFailed = EscRsa_GeneratePrime( randCtx, &keyPair->p );
#ifdef CONFIG_KEYGEN_PROFILING
        printf("%d done generate prime for p\n\n", time(NULL));
#endif /* CONFIG_KEYGEN_PROFILING */

        /* Check if p - 1 is coprime to e */
        EscRsaFe_Assign( &keyPair->iqmp, &keyPair->p );
        keyPair->iqmp.words[ 0U ] &= (EscRsa_HWORD)( ~(EscRsa_HWORD)1U );
        hasFailed |= EscRsaFe_ModInv( &keyPair->e, &keyPair->iqmp, &keyPair->iqmp );

        /* Generate random prime number for q */
#ifdef CONFIG_KEYGEN_PROFILING
        printf("%d start generate prime for q\n", time(NULL));
        int genPrimeForQ = 0;
#endif /* CONFIG_KEYGEN_PROFILING */
        do {
#ifdef CONFIG_KEYGEN_PROFILING
            genPrimeForQ++;
#endif /* CONFIG_KEYGEN_PROFILING */
            hasFailed |= EscRsa_GeneratePrime( randCtx, &keyPair->q );

            /* Check if q - 1 is coprime to e */
            EscRsaFe_Assign( &keyPair->iqmp, &keyPair->q );
            keyPair->iqmp.words[ 0U ] &= (EscRsa_HWORD)( ~(EscRsa_HWORD)1U );
            hasFailed |= EscRsaFe_ModInv( &keyPair->e, &keyPair->iqmp, &keyPair->iqmp );

            /* Ensure p > q */
            if ( EscRsaFe_AbsoluteCompare( &keyPair->p, &keyPair->q ) <= 0 ) {
                /* Swap p and q (using xor swap to save memory) */
                for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
                    keyPair->p.words[ i ] ^= keyPair->q.words[ i ];
                    keyPair->q.words[ i ] ^= keyPair->p.words[ i ];
                    keyPair->p.words[ i ] ^= keyPair->q.words[ i ];
                }
            }

            /* Check distance between p and q */
            dist = ( (EscRsa_FWORD) keyPair->p.words[ ( EscRsa_RSA_SIZE_WORDS / 2U ) - 1U ] ) << ( EscRsa_RSA_BASE - 1U );
            dist /= ( (EscRsa_FWORD) keyPair->q.words[ ( EscRsa_RSA_SIZE_WORDS / 2U ) - 1U ] );
        } while ( dist <= EscRsa_PQ_LOWER_BOUND );
#ifdef CONFIG_KEYGEN_PROFILING
        printf("%d done generate prime for q, loop called %d times\n\n",
            time(NULL), genPrimeForQ);
#endif /* CONFIG_KEYGEN_PROFILING */

        /* Calculate n = p * q */
        mult.c = &multiplication_result;
        mult.x = &keyPair->p;
        EscRsaFe_ToLongFe( mult.c, &keyPair->q );
        EscRsaFe_MultiplyClassically( &mult );
        EscRsaFe_FromLongFe( &keyPair->n, mult.c );

        /* Calculate d = e^-1 mod phi(n) */
        /* phi(n) = (p - 1) * (q - 1) = p * q - p - q + 1 = n - p - q + 1 */
        EscRsaFe_Assign( &keyPair->iqmp, &keyPair->n );
        EscRsaFe_Subtract( &keyPair->iqmp, &keyPair->p );
        EscRsaFe_Subtract( &keyPair->iqmp, &keyPair->q );
        EscRsaFe_AddWord( &keyPair->iqmp, 1U );
        hasFailed |= EscRsaFe_ModInv( &keyPair->iqmp, &keyPair->e, &keyPair->d );

        /* Calculate d mod (p - 1) */
        EscRsaFe_Assign( &keyPair->dmp1, &keyPair->p );
        keyPair->dmp1.words[ 0U ] &= (EscRsa_HWORD)( ~(EscRsa_HWORD)1U );
        mult.c = &multiplication_result;
        mult.m = &keyPair->dmp1;
        EscRsaFe_ToLongFe( mult.c, &keyPair->d );
        EscRsaFe_ModularReduction( &mult );
        EscRsaFe_FromLongFe( &keyPair->dmp1, mult.c );

        /* Calculate d mod (q - 1) */
        EscRsaFe_Assign( &keyPair->dmq1, &keyPair->q );
        keyPair->dmq1.words[ 0U ] &= (EscRsa_HWORD)( ~(EscRsa_HWORD)1U );
        mult.c = &multiplication_result;
        mult.m = &keyPair->dmq1;
        EscRsaFe_ToLongFe( mult.c, &keyPair->d );
        EscRsaFe_ModularReduction( &mult );
        EscRsaFe_FromLongFe( &keyPair->dmq1, mult.c );

        /* Calculate q^-1 mod p */
        for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++ ) {
            keyPair->iqmp.words[ i ] = 0U;
        }
        hasFailed |= EscRsaFe_ModInv( &keyPair->p, &keyPair->q, &keyPair->iqmp );
    } else {
        hasFailed = TRUE;
    }

#ifdef CONFIG_KEYGEN_PROFILING
    gettimeofday(&funcE, NULL);
    func += timevaldiff(&funcS, &funcE);
    printf("%s %ldms\n",__func__, func);
#endif /* CONFIG_KEYGEN_PROFILING */

    return hasFailed;
}
#endif /* EscRsa_ENABLE_GENRSA */

#ifdef EscRsa_ENABLE_CRT
/************************
 * Calculate a := a + b *
 ************************/
static void
EscRsaFe_Add(
    EscRsa_FieldElementT* a,
    const EscRsa_FieldElementT* b )
{
    EscRsa_FWORD carry = 0U;
    EscRsa_FWORD sum;
    UINT32 i;

    for ( i = 0U; i < EscRsa_RSA_SIZE_WORDS; i++) {
        sum = ( (EscRsa_FWORD) a->words[ i ] ) + ( (EscRsa_FWORD) b->words[ i ] ) + carry;
        carry = EscRsa_HI_FWORD( sum );
        a->words[ i ] = (EscRsa_HWORD) EscRsa_LO_FWORD( sum );
    }
}

#    ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE
/***************************************************************************
 * Modular exponentiation for small exponents. c = m^e mod n ( with CRT )   *
 ***************************************************************************/
BOOL
EscRsa_ModExpCrt(
    UINT8 result[],
    const UINT8 message[],
    const EscRsa_KeyPairT* keyPair )
#    else /* EscRsa_ENABLE_STACK_SAVING_INTERFACE */
BOOL
EscRsa_ModExpCrtSaveStack(
    EscRsa_FieldElementT* result,
    const EscRsa_FieldElementT* message,
    const EscRsa_KeyPairT* keyPair )
#    endif /* EscRsa_ENABLE_STACK_SAVING_INTERFACE */
{
    EscRsa_FieldElementLongT multiplication_result;
    EscRsa_MultiplyDataT mult;
    EscRsa_FieldElementT mp;
    EscRsa_FieldElementT mq;
    EscRsa_FieldElementT temp;
    EscRsa_FieldElementT s;
    BOOL hasFailed = FALSE;

    if ( ( message != 0 ) && ( result != 0 ) && ( keyPair != 0 ) ) {

        /* mp = m^dmp1 mod p */
        mult.c = &multiplication_result;
        mult.x = &temp;
        mult.m = &keyPair->p;
#ifdef EscRsa_ENABLE_STACK_SAVING_INTERFACE
        EscRsaFe_Assign( mult.x, message );
#else
        EscRsaFe_FromBytesBE( mult.x, message, EscRsa_KEY_BYTES );
#endif
        EscRsaFe_BigPow( &mult, &keyPair->dmp1 );
        EscRsaFe_Assign( &mp, mult.x );

        /* mq = m^dmq1 mod q */
        mult.m = &keyPair->q;
#ifdef EscRsa_ENABLE_STACK_SAVING_INTERFACE
        EscRsaFe_Assign( mult.x, message );
#else
        EscRsaFe_FromBytesBE( mult.x, message, EscRsa_KEY_BYTES );
#endif
        EscRsaFe_BigPow( &mult, &keyPair->dmq1 );
        EscRsaFe_Assign( &mq, mult.x );

        /* s = (iqmp * (mp - mq) mod p) * q + mq */
        EscRsaFe_Assign( &s, &mp );
        if ( EscRsaFe_AbsoluteCompare( &mp, &mq ) == -1 ) {
            /* if mp < mq, add p to mp to ensure that (mp - mq) is positive */
            EscRsaFe_Add( &s, &keyPair->p );
        }
        EscRsaFe_Subtract( &s, &mq );

        mult.m = &keyPair->p;
        EscRsaFe_Assign( mult.x, &s );
        EscRsaFe_ToLongFe( mult.c, &keyPair->iqmp );
        EscRsaFe_Multiply( &mult );

        EscRsaFe_Assign( mult.x, &keyPair->q );
        EscRsaFe_MultiplyClassically( &mult );
        EscRsaFe_FromLongFe( &s, mult.c );

        EscRsaFe_Add( &s, &mq );

#ifdef EscRsa_ENABLE_STACK_SAVING_INTERFACE
        EscRsaFe_Assign( result, &s );
#else
        EscRsaFe_ToBytesBE( result, &s );
#endif
    } else {
        hasFailed = TRUE;
    }

    return hasFailed;
}
#endif /* EscRsa_ENABLE_CRT */

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
