/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        pkcs1_v15.c

   \brief       PKCS#1 v2.1 signature verification

   Implementation of RSASSA-PKCS1-v1_5_Verify

   Byteformat of all long numbers is Big-Endian.

   \see         ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1.pdf


   $Rev: 990 $
 */
/***************************************************************************/

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include "../inc/pkcs1_v15.h"

#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** Offset of PS within em */
#define EscPkcs1V15_PS_OFFSET 2U

/**
Describes a digest.
*/
typedef struct {
    /** The length of the digest in bytes. */
    UINT8 digestLen;
    /** The length of digestInfo in bytes. */
    UINT8 digestInfoLen;
    /** The DER encoded digestInfo. */
    const UINT8 * digestInfo;
} EscPkcs1V15_DigestDescriptionT;

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

static BOOL
EscPkcs1V15_GetDigestDescription(
    UINT8 digestType,
    const EscPkcs1V15_DigestDescriptionT ** descr );

static BOOL
EscPkcs1V15_MemIsDifferent(
    const UINT8 m1[],
    const UINT8 m2[],
    UINT8 memLen );

static BOOL
EscPkcs1V15_CheckDigest(
    const EscPkcs1V15_VerifyParamsT* params,
    const UINT8 em[],
    const EscPkcs1V15_DigestDescriptionT* digestDescr );

static BOOL
EscPkcs1V15_CheckEmsaHeader(
    const UINT8 em[],
    const EscPkcs1V15_DigestDescriptionT* digestDescr );

static BOOL
EscPkcs1V15_CheckEmsaEncoding(
    const EscPkcs1V15_VerifyParamsT* params,
    const UINT8 em[] );

/** Encodes a message according to EMSA-PKCS1-V1_5 */
static BOOL
EscPkcs1V15_EmsaV15Encode(
    const UINT8 hashDigest[],
    const EscPkcs1V15_SignParamsT* params,
    UINT8 em[] );

/***************************************************************************
 * 4. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

/* Returns the digest description for a certain digest type. */
static BOOL
EscPkcs1V15_GetDigestDescription(
    UINT8 digestType,
    const EscPkcs1V15_DigestDescriptionT ** descr )
{

/*
   Digest Infos from section 9.2, the note on
   http://www.rsa.com/rsalabs/node.asp?id=2125 and from IEEE-P1363a, 12.1.3.
   MD2: (0x)30 20 30 0c 06 08 2a 86 48 86 f7 0d 02 02 05 00 04 10 || H.
   MD5: (0x)30 20 30 0c 06 08 2a 86 48 86 f7 0d 02 05 05 00 04 10 || H.
   SHA-1: (0x)30 21 30 09 06 05 2b 0e 03 02 1a 05 00 04 14 || H.
   SHA-224: (0x)30 2d 30 0d 06 09 60 86 48 01 65 03 04 02 04 05 00 04 1c || H
   SHA-256: (0x)30 31 30 0d 06 09 60 86 48 01 65 03 04 02 01 05 00 04 20 || H.
   SHA-384: (0x)30 41 30 0d 06 09 60 86 48 01 65 03 04 02 02 05 00 04 30 || H.
   SHA-512: (0x)30 51 30 0d 06 09 60 86 48 01 65 03 04 02 03 05 00 04 40 || H.
   RIPEMD-160: (0x)30 21 30 09 06 05 2b 24 03 02 01 05 00 04 14 || H.
 */

#    ifndef EscPkcs1V15_DIGEST_MD2_DISABLE
/* The digest info. */
static const UINT8 EscPkcs1V15_DIGEST_INFO_MD2[] = {
    0x30U, 0x20U, 0x30U, 0x0cU, 0x06U, 0x08U, 0x2aU, 0x86U,
    0x48U, 0x86U, 0xf7U, 0x0dU, 0x02U, 0x02U, 0x05U, 0x00U,
    0x04U, 0x10U
};

/* The digest description. */
static const EscPkcs1V15_DigestDescriptionT EscPkcs1V15_DIGEST_DESCR_MD2 = {
    /* UINT8 digestLen */
    (UINT8)0x10U,

    /* UINT8 digestInfoLen */
    (UINT8)sizeof( EscPkcs1V15_DIGEST_INFO_MD2 ),

    /* const UINT8 *digestInfo */
    EscPkcs1V15_DIGEST_INFO_MD2,
};
#    endif

#    ifndef EscPkcs1V15_DIGEST_MD5_DISABLE
/* The digest info. */
static const UINT8 EscPkcs1V15_DIGEST_INFO_MD5[] = {
    0x30U, 0x20U, 0x30U, 0x0cU, 0x06U, 0x08U, 0x2aU, 0x86U,
    0x48U, 0x86U, 0xf7U, 0x0dU, 0x02U, 0x05U, 0x05U, 0x00U,
    0x04U, 0x10U
};

/* The digest description. */
static const EscPkcs1V15_DigestDescriptionT EscPkcs1V15_DIGEST_DESCR_MD5 = {
    /* UINT8 digestLen */
    (UINT8)0x10U,

    /* UINT8 digestInfoLen */
    (UINT8)sizeof( EscPkcs1V15_DIGEST_INFO_MD5 ),

    /* const UINT8 *digestInfo */
    EscPkcs1V15_DIGEST_INFO_MD5,
};
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA1_DISABLE
/* The digest info. */
static const UINT8 EscPkcs1V15_DIGEST_INFO_SHA1[] = {
    0x30U, 0x21U, 0x30U, 0x09U, 0x06U, 0x05U, 0x2bU, 0x0eU,
    0x03U, 0x02U, 0x1aU, 0x05U, 0x00U, 0x04U, 0x14U
};

/* The digest description. */
static const EscPkcs1V15_DigestDescriptionT EscPkcs1V15_DIGEST_DESCR_SHA1 = {
    /* UINT8 digestLen */
    (UINT8)0x14U,

    /* UINT8 digestInfoLen */
    (UINT8)sizeof( EscPkcs1V15_DIGEST_INFO_SHA1 ),

    /* const UINT8 *digestInfo */
    EscPkcs1V15_DIGEST_INFO_SHA1,
};
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA224_DISABLE
/* The digest info. */
static const UINT8 EscPkcs1V15_DIGEST_INFO_SHA224[] = {
    0x30U, 0x2dU, 0x30U, 0x0dU, 0x06U, 0x09U, 0x60U, 0x86U,
    0x48U, 0x01U, 0x65U, 0x03U, 0x04U, 0x02U, 0x04U, 0x05U,
    0x00U, 0x04U, 0x1cU
};

/* The digest description. */
static const EscPkcs1V15_DigestDescriptionT EscPkcs1V15_DIGEST_DESCR_SHA224 = {
    /* UINT8 digestLen */
    (UINT8)0x1cU,

    /* UINT8 digestInfoLen */
    (UINT8)sizeof( EscPkcs1V15_DIGEST_INFO_SHA224 ),

    /* const UINT8 *digestInfo */
    EscPkcs1V15_DIGEST_INFO_SHA224,
};
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA256_DISABLE
/* The digest info. */
static const UINT8 EscPkcs1V15_DIGEST_INFO_SHA256[] = {
    0x30U, 0x31U, 0x30U, 0x0dU, 0x06U, 0x09U, 0x60U, 0x86U,
    0x48U, 0x01U, 0x65U, 0x03U, 0x04U, 0x02U, 0x01U, 0x05U,
    0x00U, 0x04U, 0x20U
};

/* The digest description. */
static const EscPkcs1V15_DigestDescriptionT EscPkcs1V15_DIGEST_DESCR_SHA256 = {
    /* UINT8 digestLen */
    (UINT8)0x20U,

    /* UINT8 digestInfoLen */
    (UINT8)sizeof( EscPkcs1V15_DIGEST_INFO_SHA256 ),

    /* const UINT8 *digestInfo */
    EscPkcs1V15_DIGEST_INFO_SHA256,
};
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA384_DISABLE
/* The digest info. */
static const UINT8 EscPkcs1V15_DIGEST_INFO_SHA384[] = {
    0x30U, 0x41U, 0x30U, 0x0dU, 0x06U, 0x09U, 0x60U, 0x86U,
    0x48U, 0x01U, 0x65U, 0x03U, 0x04U, 0x02U, 0x02U, 0x05U,
    0x00U, 0x04U, 0x30U
};

/* The digest description. */
static const EscPkcs1V15_DigestDescriptionT EscPkcs1V15_DIGEST_DESCR_SHA384 = {
    /* UINT8 digestLen */
    (UINT8)0x30U,

    /* UINT8 digestInfoLen */
    (UINT8)sizeof( EscPkcs1V15_DIGEST_INFO_SHA384 ),

    /* const UINT8 *digestInfo */
    EscPkcs1V15_DIGEST_INFO_SHA384,
};
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA512_DISABLE
/* The digest info. */
static const UINT8 EscPkcs1V15_DIGEST_INFO_SHA512[] = {
    0x30U, 0x51U, 0x30U, 0x0dU, 0x06U, 0x09U, 0x60U, 0x86U,
    0x48U, 0x01U, 0x65U, 0x03U, 0x04U, 0x02U, 0x03U, 0x05U,
    0x00U, 0x04U, 0x40U
};

/* The digest description. */
static const EscPkcs1V15_DigestDescriptionT EscPkcs1V15_DIGEST_DESCR_SHA512 = {
    /* UINT8 digestLen */
    (UINT8)0x40U,

    /* UINT8 digestInfoLen */
    (UINT8)sizeof( EscPkcs1V15_DIGEST_INFO_SHA512 ),

    /* const UINT8 *digestInfo */
    EscPkcs1V15_DIGEST_INFO_SHA512,
};
#    endif

#    ifndef EscPkcs1V15_DIGEST_RIPEMD160_DISABLE
/* The digest info. */
static const UINT8 EscPkcs1V15_DIGEST_INFO_RIPEMD160[] = {
    0x30U, 0x21U, 0x30U, 0x09U, 0x06U, 0x05U, 0x2bU, 0x24U,
    0x03U, 0x02U, 0x01U, 0x05U, 0x00U, 0x04U, 0x14U
};

/* The digest description. */
static const EscPkcs1V15_DigestDescriptionT EscPkcs1V15_DIGEST_DESCR_RIPEMD160 = {
    /* UINT8 digestLen */
    (UINT8)0x14U,

    /* UINT8 digestInfoLen */
    (UINT8)sizeof( EscPkcs1V15_DIGEST_INFO_RIPEMD160 ),

    /* const UINT8 *digestInfo */
    EscPkcs1V15_DIGEST_INFO_RIPEMD160,
};
#    endif

    BOOL hasFailed = FALSE;

    switch ( digestType ) {
#    ifndef EscPkcs1V15_DIGEST_MD2_DISABLE
    case EscPkcs1V15_DIGEST_TYPE_MD2:
        *descr = &EscPkcs1V15_DIGEST_DESCR_MD2;
        break;
#    endif

#    ifndef EscPkcs1V15_DIGEST_MD5_DISABLE
    case EscPkcs1V15_DIGEST_TYPE_MD5:
        *descr = &EscPkcs1V15_DIGEST_DESCR_MD5;
        break;
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA1_DISABLE
    case EscPkcs1V15_DIGEST_TYPE_SHA1:
        *descr = &EscPkcs1V15_DIGEST_DESCR_SHA1;
        break;
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA224_DISABLE
    case EscPkcs1V15_DIGEST_TYPE_SHA224:
        *descr = &EscPkcs1V15_DIGEST_DESCR_SHA224;
        break;
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA256_DISABLE
    case EscPkcs1V15_DIGEST_TYPE_SHA256:
        *descr = &EscPkcs1V15_DIGEST_DESCR_SHA256;
        break;
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA384_DISABLE
    case EscPkcs1V15_DIGEST_TYPE_SHA384:
        *descr = &EscPkcs1V15_DIGEST_DESCR_SHA384;
        break;
#    endif

#    ifndef EscPkcs1V15_DIGEST_SHA512_DISABLE
    case EscPkcs1V15_DIGEST_TYPE_SHA512:
        *descr = &EscPkcs1V15_DIGEST_DESCR_SHA512;
        break;
#    endif

#    ifndef EscPkcs1V15_DIGEST_RIPEMD160_DISABLE
    case EscPkcs1V15_DIGEST_TYPE_RIPEMD160:
        *descr = &EscPkcs1V15_DIGEST_DESCR_RIPEMD160;
        break;
#    endif

    default:
        hasFailed = TRUE;
    }

    return hasFailed;
}

/* Returns FALSE if m1 and m2 are equal. */
static BOOL
EscPkcs1V15_MemIsDifferent(
    const UINT8 m1[],
    const UINT8 m2[],
    UINT8 memLen )
{
    UINT8 i;
    BOOL isDifferent = FALSE;

    for ( i = 0U; i < memLen; i++ ) {
        if ( m1[ i ] != m2[ i ] ) {
            isDifferent = TRUE;
            break;
        }
    }

    return isDifferent;
}

/*
   Checks the EMSA digest: digestInfo || digestValue.
   em has the length EscRsa_KEY_BYTES.
 */
static BOOL
EscPkcs1V15_CheckDigest(
    const EscPkcs1V15_VerifyParamsT* params,
    const UINT8 em[],
    const EscPkcs1V15_DigestDescriptionT* digestDescr )
{
    BOOL hasFailed;
    UINT16 digestValueStart;
    UINT16 digestInfoStart;

    digestValueStart = (UINT16)EscRsa_KEY_BYTES - (UINT16)digestDescr->digestLen;
    digestInfoStart = digestValueStart - (UINT16)digestDescr->digestInfoLen;

    /* Compare digest info. */
    hasFailed = EscPkcs1V15_MemIsDifferent( digestDescr->digestInfo, &em[ digestInfoStart ], digestDescr->digestInfoLen );

    if ( hasFailed == FALSE ) {
        /* Compare digest */
        hasFailed = EscPkcs1V15_MemIsDifferent( params->hashDigest, &em[ digestValueStart ], digestDescr->digestLen );
    }

    return hasFailed;
}

/*
   Checks the EMSA digest: digestInfo || digestValue.
   em has the length EscRsa_KEY_BYTES.
 */
static BOOL
EscPkcs1V15_CheckDigest4096(
    const EscPkcs1V15_VerifyParamsT* params,
    const UINT8 em[],
    const EscPkcs1V15_DigestDescriptionT* digestDescr )
{
    BOOL hasFailed;
    UINT16 digestValueStart;
    UINT16 digestInfoStart;

    digestValueStart = (UINT16)EscRsa4096_KEY_BYTES - (UINT16)digestDescr->digestLen;
    digestInfoStart = digestValueStart - (UINT16)digestDescr->digestInfoLen;

    /* Compare digest info. */
    hasFailed = EscPkcs1V15_MemIsDifferent( digestDescr->digestInfo, &em[ digestInfoStart ], digestDescr->digestInfoLen );

    if ( hasFailed == FALSE ) {
        /* Compare digest */
        hasFailed = EscPkcs1V15_MemIsDifferent( params->hashDigest, &em[ digestValueStart ], digestDescr->digestLen );
    }

    return hasFailed;
}


static BOOL
EscPkcs1V15_CheckDigest1024(
    const EscPkcs1V15_VerifyParamsT* params,
    const UINT8 em[],
    const EscPkcs1V15_DigestDescriptionT* digestDescr )
{
    BOOL hasFailed;
    UINT16 digestValueStart;
    UINT16 digestInfoStart;

    digestValueStart = (UINT16)EscRsa1024_KEY_BYTES - (UINT16)digestDescr->digestLen;
    digestInfoStart = digestValueStart - (UINT16)digestDescr->digestInfoLen;

    /* Compare digest info. */
    hasFailed = EscPkcs1V15_MemIsDifferent( digestDescr->digestInfo, &em[ digestInfoStart ], digestDescr->digestInfoLen );

    if ( hasFailed == FALSE ) {
        /* Compare digest */
        hasFailed = EscPkcs1V15_MemIsDifferent( params->hashDigest, &em[ digestValueStart ], digestDescr->digestLen );
    }

    return hasFailed;
}

/*
   Checks the EMSA header 00 || 01 | PS || 00.
   em has the length EscRsa_KEY_BYTES. */
static BOOL
EscPkcs1V15_CheckEmsaHeader(
    const UINT8 em[],
    const EscPkcs1V15_DigestDescriptionT* digestDescr )
{
    BOOL hasFailed = FALSE;
    UINT16 psLen;

    /* Calculate psLen. It has the length emLen - sizeof(T) - 3, with emLen = EscRsa_KEY_BYTES */
    psLen = ( ( EscRsa_KEY_BYTES - 3U ) - digestDescr->digestInfoLen ) - digestDescr->digestLen;
    /* PSLEN will be at least 8 octets according to PKCS#1 V2.1, 9.2 Step 4.
       This value can only be this low,
       if EscRsa_KEY_BYTES has an absurd low value, e.g. RSA-512 */
    Esc_ASSERT( psLen >= 8U );

    /* Check the leading 00 || 01 and the trailing 00 */
    if ( ( em[ 0 ] != 0x00U ) || ( em[ 1 ] != 0x01U ) || ( em[ psLen + EscPkcs1V15_PS_OFFSET ] != 0x00U ) ) {
        hasFailed = TRUE;
    } else {
        UINT16 i;
        /* Check the padding PS. */
        for ( i = EscPkcs1V15_PS_OFFSET; i < ( psLen + EscPkcs1V15_PS_OFFSET ); i++ ) {
            if ( em[ i ] != 0xffU ) {
                hasFailed = TRUE;
                break;
            }
        }
    }

    return hasFailed;
}

/*
   Checks the EMSA header 00 || 01 | PS || 00.
   em has the length EscRsa_KEY_BYTES. */
static BOOL
EscPkcs1V15_CheckEmsaHeader4096(
    const UINT8 em[],
    const EscPkcs1V15_DigestDescriptionT* digestDescr )
{
    BOOL hasFailed = FALSE;
    UINT16 psLen;

    /* Calculate psLen. It has the length emLen - sizeof(T) - 3, with emLen = EscRsa_KEY_BYTES */
    psLen = ( ( EscRsa4096_KEY_BYTES - 3U ) - digestDescr->digestInfoLen ) - digestDescr->digestLen;
    /* PSLEN will be at least 8 octets according to PKCS#1 V2.1, 9.2 Step 4.
       This value can only be this low,
       if EscRsa_KEY_BYTES has an absurd low value, e.g. RSA-512 */
    Esc_ASSERT( psLen >= 8U );

    /* Check the leading 00 || 01 and the trailing 00 */
    if ( ( em[ 0 ] != 0x00U ) || ( em[ 1 ] != 0x01U ) || ( em[ psLen + EscPkcs1V15_PS_OFFSET ] != 0x00U ) ) {
        hasFailed = TRUE;
    } else {
        UINT16 i;
        /* Check the padding PS. */
        for ( i = EscPkcs1V15_PS_OFFSET; i < ( psLen + EscPkcs1V15_PS_OFFSET ); i++ ) {
            if ( em[ i ] != 0xffU ) {
                hasFailed = TRUE;
                break;
            }
        }
    }

    return hasFailed;
}

static BOOL
EscPkcs1V15_CheckEmsaHeader1024(
    const UINT8 em[],
    const EscPkcs1V15_DigestDescriptionT* digestDescr )
{
    BOOL hasFailed = FALSE;
    UINT16 psLen;

    /* Calculate psLen. It has the length emLen - sizeof(T) - 3, with emLen = EscRsa_KEY_BYTES */
    psLen = ( ( EscRsa1024_KEY_BYTES - 3U ) - digestDescr->digestInfoLen ) - digestDescr->digestLen;
    /* PSLEN will be at least 8 octets according to PKCS#1 V2.1, 9.2 Step 4.
       This value can only be this low,
       if EscRsa_KEY_BYTES has an absurd low value, e.g. RSA-512 */
    Esc_ASSERT( psLen >= 8U );

    /* Check the leading 00 || 01 and the trailing 00 */
    if ( ( em[ 0 ] != 0x00U ) || ( em[ 1 ] != 0x01U ) || ( em[ psLen + EscPkcs1V15_PS_OFFSET ] != 0x00U ) ) {
        hasFailed = TRUE;
    } else {
        UINT16 i;
        /* Check the padding PS. */
        for ( i = EscPkcs1V15_PS_OFFSET; i < ( psLen + EscPkcs1V15_PS_OFFSET ); i++ ) {
            if ( em[ i ] != 0xffU ) {
                hasFailed = TRUE;
                break;
            }
        }
    }

    return hasFailed;
}

/*
   Checks the emsa encoding according to section 9.2.1 and additionally a leading.
   EM has the length EscRsa_KEY_BYTES-1.
 */
static BOOL
EscPkcs1V15_CheckEmsaEncoding(
    const EscPkcs1V15_VerifyParamsT* params,
    const UINT8 em[] )
{
    BOOL hasFailed = TRUE;
    const EscPkcs1V15_DigestDescriptionT* digestDescr = 0;

    /* EM = 00 || 01 || PS || 00 || digestInfo || digestValue */
    /* we devide in two portions, the header 00 || 01 || PS || 00 and the
       digest digestInfo || digestValue */
    if ( EscPkcs1V15_GetDigestDescription( params->digestType, &digestDescr ) == FALSE ) {
        if ( EscPkcs1V15_CheckEmsaHeader( em, digestDescr ) == FALSE ) {
            hasFailed = EscPkcs1V15_CheckDigest( params, em, digestDescr );
        }
    }

    return hasFailed;
}

/*
   Checks the emsa encoding according to section 9.2.1 and additionally a leading.
   EM has the length EscRsa_KEY_BYTES-1.
 */
static BOOL
EscPkcs1V15_CheckEmsaEncoding4096(
    const EscPkcs1V15_VerifyParamsT* params,
    const UINT8 em[] )
{
    BOOL hasFailed = TRUE;
    const EscPkcs1V15_DigestDescriptionT* digestDescr = 0;

    /* EM = 00 || 01 || PS || 00 || digestInfo || digestValue */
    /* we devide in two portions, the header 00 || 01 || PS || 00 and the
       digest digestInfo || digestValue */
    if ( EscPkcs1V15_GetDigestDescription( params->digestType, &digestDescr ) == FALSE ) {
        if ( EscPkcs1V15_CheckEmsaHeader4096( em, digestDescr ) == FALSE ) {
            hasFailed = EscPkcs1V15_CheckDigest4096( params, em, digestDescr );
        }
    }

    return hasFailed;
}

static BOOL
EscPkcs1V15_CheckEmsaEncoding1024(
    const EscPkcs1V15_VerifyParamsT* params,
    const UINT8 em[] )
{
    BOOL hasFailed = TRUE;
    const EscPkcs1V15_DigestDescriptionT* digestDescr = 0;

    /* EM = 00 || 01 || PS || 00 || digestInfo || digestValue */
    /* we devide in two portions, the header 00 || 01 || PS || 00 and the
       digest digestInfo || digestValue */
    if ( EscPkcs1V15_GetDigestDescription( params->digestType, &digestDescr ) == FALSE ) {
        if ( EscPkcs1V15_CheckEmsaHeader1024( em, digestDescr ) == FALSE ) {
            hasFailed = EscPkcs1V15_CheckDigest1024( params, em, digestDescr );
        }
    }

    return hasFailed;
}


/** Encodes a message according to EMSA-PKCS1-V1_5 */
static BOOL
    EscPkcs1V15_EmsaV15Encode(
    const UINT8 hashDigest[],  /** Message to encode */
    const EscPkcs1V15_SignParamsT* params, /** Sign parameters */
    UINT8 em[] )
{
    BOOL hasFailed;
    const EscPkcs1V15_DigestDescriptionT* digestDescr = 0;
    UINT16 psLen;
    UINT16 digestValueStart;
    UINT16 digestInfoStart;
    UINT16 i;

    if ( EscPkcs1V15_GetDigestDescription( params->digestType, &digestDescr ) == FALSE ) {

        /* First two bytes */
        em[ 0U ] = 0x00U;
        em[ 1U ] = 0x01U;

        /* PS */

        /* Calculate psLen. It has the length emLen - sizeof(T) - 3, with emLen = EscRsa_KEY_BYTES */
        psLen = ( ( EscRsa_KEY_BYTES - 3U ) - digestDescr->digestInfoLen ) - digestDescr->digestLen;
        /* PSLEN will be at least 8 octets according to PKCS#1 V2.1, 9.2 Step 4.
        This value can only be this low,
        if EscRsa_KEY_BYTES has an absurd low value, e.g. RSA-512 */
        Esc_ASSERT( psLen >= 8U );

        for ( i = EscPkcs1V15_PS_OFFSET; i < ( psLen + EscPkcs1V15_PS_OFFSET ); i++ ) {
            em[ i ] = 0xffU;
        }

        /* 0x00U */
        em[ psLen + EscPkcs1V15_PS_OFFSET ] = 0x00U;

        /* T */
        digestValueStart = (UINT16)EscRsa_KEY_BYTES - (UINT16)digestDescr->digestLen;
        digestInfoStart = digestValueStart - (UINT16)digestDescr->digestInfoLen;

        for ( i = 0U; i < (UINT16)digestDescr->digestInfoLen; i++ ) {
            /*lint -save -esym(960,17.4) em and digestInfo are both an array*/
            em[ digestInfoStart + i ] = digestDescr->digestInfo[ i ];
            /*lint -restore */
        }
        for ( i = 0U; i < (UINT16)digestDescr->digestLen; i++ ) {
            /*lint -save -esym(960,17.4) em and hashDigest are both an array*/
            em[ digestValueStart + i ] = hashDigest[ i ];
            /*lint -restore */
        }

        hasFailed = FALSE;

    } else {
        hasFailed = TRUE;
     }

    return hasFailed;
}

/** Encodes a message according to EMSA-PKCS1-V1_5 */
static BOOL
    EscPkcs1V15_EmsaV15Encode4096(
    const UINT8 hashDigest[],  /** Message to encode */
    const EscPkcs1V15_SignParamsT4096* params, /** Sign parameters */
    UINT8 em[] )
{
    BOOL hasFailed;
    const EscPkcs1V15_DigestDescriptionT* digestDescr = 0;
    UINT16 psLen;
    UINT16 digestValueStart;
    UINT16 digestInfoStart;
    UINT16 i;

    if ( EscPkcs1V15_GetDigestDescription( params->digestType, &digestDescr ) == FALSE ) {

        /* First two bytes */
        em[ 0U ] = 0x00U;
        em[ 1U ] = 0x01U;

        /* PS */

        /* Calculate psLen. It has the length emLen - sizeof(T) - 3, with emLen = EscRsa_KEY_BYTES */
        psLen = ( ( EscRsa4096_KEY_BYTES - 3U ) - digestDescr->digestInfoLen ) - digestDescr->digestLen;
        /* PSLEN will be at least 8 octets according to PKCS#1 V2.1, 9.2 Step 4.
        This value can only be this low,
        if EscRsa_KEY_BYTES has an absurd low value, e.g. RSA-512 */
        Esc_ASSERT( psLen >= 8U );

        for ( i = EscPkcs1V15_PS_OFFSET; i < ( psLen + EscPkcs1V15_PS_OFFSET ); i++ ) {
            em[ i ] = 0xffU;
        }

        /* 0x00U */
        em[ psLen + EscPkcs1V15_PS_OFFSET ] = 0x00U;

        /* T */
        digestValueStart = (UINT16)EscRsa4096_KEY_BYTES - (UINT16)digestDescr->digestLen;
        digestInfoStart = digestValueStart - (UINT16)digestDescr->digestInfoLen;

        for ( i = 0U; i < (UINT16)digestDescr->digestInfoLen; i++ ) {
            /*lint -save -esym(960,17.4) em and digestInfo are both an array*/
            em[ digestInfoStart + i ] = digestDescr->digestInfo[ i ];
            /*lint -restore */
        }
        for ( i = 0U; i < (UINT16)digestDescr->digestLen; i++ ) {
            /*lint -save -esym(960,17.4) em and hashDigest are both an array*/
            em[ digestValueStart + i ] = hashDigest[ i ];
            /*lint -restore */
        }

        hasFailed = FALSE;

    } else {
        hasFailed = TRUE;
     }

    return hasFailed;
}

/** Encodes a message according to EMSA-PKCS1-V1_5 */
static BOOL
    EscPkcs1V15_EmsaV15Encode1024(
    const UINT8 hashDigest[],  /** Message to encode */
    const EscPkcs1V15_SignParamsT1024* params, /** Sign parameters */
    UINT8 em[] )
{
    BOOL hasFailed;
    const EscPkcs1V15_DigestDescriptionT* digestDescr = 0;
    UINT16 psLen;
    UINT16 digestValueStart;
    UINT16 digestInfoStart;
    UINT16 i;

    if ( EscPkcs1V15_GetDigestDescription( params->digestType, &digestDescr ) == FALSE ) {

        /* First two bytes */
        em[ 0U ] = 0x00U;
        em[ 1U ] = 0x01U;

        /* PS */

        /* Calculate psLen. It has the length emLen - sizeof(T) - 3, with emLen = EscRsa_KEY_BYTES */
        psLen = ( ( EscRsa1024_KEY_BYTES - 3U ) - digestDescr->digestInfoLen ) - digestDescr->digestLen;
        /* PSLEN will be at least 8 octets according to PKCS#1 V2.1, 9.2 Step 4.
        This value can only be this low,
        if EscRsa_KEY_BYTES has an absurd low value, e.g. RSA-512 */
        Esc_ASSERT( psLen >= 8U );

        for ( i = EscPkcs1V15_PS_OFFSET; i < ( psLen + EscPkcs1V15_PS_OFFSET ); i++ ) {
            em[ i ] = 0xffU;
        }

        /* 0x00U */
        em[ psLen + EscPkcs1V15_PS_OFFSET ] = 0x00U;

        /* T */
        digestValueStart = (UINT16)EscRsa1024_KEY_BYTES - (UINT16)digestDescr->digestLen;
        digestInfoStart = digestValueStart - (UINT16)digestDescr->digestInfoLen;

        for ( i = 0U; i < (UINT16)digestDescr->digestInfoLen; i++ ) {
            /*lint -save -esym(960,17.4) em and digestInfo are both an array*/
            em[ digestInfoStart + i ] = digestDescr->digestInfo[ i ];
            /*lint -restore */
        }
        for ( i = 0U; i < (UINT16)digestDescr->digestLen; i++ ) {
            /*lint -save -esym(960,17.4) em and hashDigest are both an array*/
            em[ digestValueStart + i ] = hashDigest[ i ];
            /*lint -restore */
        }

        hasFailed = FALSE;

    } else {
        hasFailed = TRUE;
     }

    return hasFailed;
}

/*
   Signs a message according to
   RSASSA-PKCS1-v1_5_Sign.
 */
BOOL
EscPkcs1V15_Sign4096(
    const EscPkcs1V15_SignParamsT4096* params )
{
    BOOL hasFailed = TRUE;
#   ifdef EscRsa_ENABLE_CRT
    EscRsa4096_KeyPairT4096 keyPair;
#   endif

    if ( ( params != 0 ) &&
         ( params->hashDigest != 0 ) &&
         ( params->privKey != 0 ) &&
         ( params->signature != 0 ) )
    {
        /* 1. Encode message */
        hasFailed = EscPkcs1V15_EmsaV15Encode4096( params->hashDigest, params, params->signature );

        if ( hasFailed == FALSE ) {
            /* Do the RSA exponentiation */
#   ifdef EscRsa_ENABLE_CRT
            EscRsaFe_FromBytesBE4096( &keyPair.p, params->privKey->p, EscRsa4096_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE4096( &keyPair.q, params->privKey->q, EscRsa4096_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE4096( &keyPair.dmp1, params->privKey->dmp1, EscRsa4096_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE4096( &keyPair.dmq1, params->privKey->dmq1, EscRsa4096_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE4096( &keyPair.iqmp, params->privKey->iqmp, EscRsa4096_KEY_BYTES / 2U );
            hasFailed = EscRsa_ModExpCrt( params->signature, params->signature, &keyPair );
#   else
            hasFailed = EscRsa4096_ModExpLong( params->signature, params->privKey->modulus, params->privKey->privExp, params->signature );
#   endif
        }
    }

    return hasFailed;
}

/*
   Signs a message according to
   RSASSA-PKCS1-v1_5_Sign.
 */
BOOL
EscPkcs1V15_Sign(
    const EscPkcs1V15_SignParamsT* params )
{
    BOOL hasFailed = TRUE;
#   ifdef EscRsa_ENABLE_CRT
    EscRsa_KeyPairT keyPair;
#   endif

    if ( ( params != 0 ) &&
         ( params->hashDigest != 0 ) &&
         ( params->privKey != 0 ) &&
         ( params->signature != 0 ) )
    {
        /* 1. Encode message */
        hasFailed = EscPkcs1V15_EmsaV15Encode( params->hashDigest, params, params->signature );

        if ( hasFailed == FALSE ) {
            /* Do the RSA exponentiation */
#   ifdef EscRsa_ENABLE_CRT
            EscRsaFe_FromBytesBE( &keyPair.p, params->privKey->p, EscRsa_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE( &keyPair.q, params->privKey->q, EscRsa_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE( &keyPair.dmp1, params->privKey->dmp1, EscRsa_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE( &keyPair.dmq1, params->privKey->dmq1, EscRsa_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE( &keyPair.iqmp, params->privKey->iqmp, EscRsa_KEY_BYTES / 2U );
            hasFailed = EscRsa_ModExpCrt( params->signature, params->signature, &keyPair );
#   else
            hasFailed = EscRsa_ModExpLong( params->signature, params->privKey->modulus, params->privKey->privExp, params->signature );
#   endif
        }
    }

    return hasFailed;
}

/*
   Signs a message according to
   RSASSA-PKCS1-v1_5_Sign.
 */
BOOL
EscPkcs1V15_Sign1024(
    const EscPkcs1V15_SignParamsT1024* params )
{
    BOOL hasFailed = TRUE;
#   ifdef EscRsa_ENABLE_CRT
    EscRsa1024_KeyPairT1024 keyPair;
#   endif

    if ( ( params != 0 ) &&
         ( params->hashDigest != 0 ) &&
         ( params->privKey != 0 ) &&
         ( params->signature != 0 ) )
    {
        /* 1. Encode message */
        hasFailed = EscPkcs1V15_EmsaV15Encode1024( params->hashDigest, params, params->signature );

        if ( hasFailed == FALSE ) {
            /* Do the RSA exponentiation */
#   ifdef EscRsa_ENABLE_CRT
            EscRsaFe_FromBytesBE1024( &keyPair.p, params->privKey->p, EscRsa1024_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE1024( &keyPair.q, params->privKey->q, EscRsa1024_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE1024( &keyPair.dmp1, params->privKey->dmp1, EscRsa1024_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE1024( &keyPair.dmq1, params->privKey->dmq1, EscRsa1024_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE1024( &keyPair.iqmp, params->privKey->iqmp, EscRsa1024_KEY_BYTES / 2U );
            hasFailed = EscRsa_ModExpCrt( params->signature, params->signature, &keyPair );
#   else
            hasFailed = EscRsa1024_ModExpLong( params->signature, params->privKey->modulus, params->privKey->privExp, params->signature );
#   endif
        }
    }

    return hasFailed;
}

/*
   Verifies the signature of a message according to
   RSASSA-PKCS1-v1_5_Verify.
 */
BOOL
EscPkcs1V15_Verify(
    const EscPkcs1V15_VerifyParamsT* params )
{
    BOOL hasFailed = TRUE;
    UINT8 em[ EscRsa_KEY_BYTES ];

    if ( ( params != 0 ) &&
         ( params->signature != 0 ) &&
         ( params->modulus != 0 ) &&
         ( params->hashDigest != 0 ) )
    {
        /* Do RSA operation */
        if ( EscRsa_ModExp( params->signature, params->modulus, params->pubExp, em ) == FALSE ) {
            /* Check if the result is as expected. */
            hasFailed = EscPkcs1V15_CheckEmsaEncoding( params, em );
        }
    }

    return hasFailed;
}

/*
   Verifies the signature of a message according to
   RSASSA-PKCS1-v1_5_Verify.
 */
BOOL
EscPkcs1V15_Verify4096(
    const EscPkcs1V15_VerifyParamsT* params )
{
    BOOL hasFailed = TRUE;
    UINT8 em[ EscRsa4096_KEY_BYTES ];

    if ( ( params != 0 ) &&
         ( params->signature != 0 ) &&
         ( params->modulus != 0 ) &&
         ( params->hashDigest != 0 ) )
    {
        /* Do RSA operation */
        if ( EscRsa4096_ModExp( params->signature, params->modulus, params->pubExp, em ) == FALSE ) {
            /* Check if the result is as expected. */
            hasFailed = EscPkcs1V15_CheckEmsaEncoding4096( params, em );
        }
    }

    return hasFailed;
}

/*
   Verifies the signature of a message according to
   RSASSA-PKCS1-v1_5_Verify.
 */
BOOL
EscPkcs1V15_Verify1024(
    const EscPkcs1V15_VerifyParamsT* params )
{
    BOOL hasFailed = TRUE;
    UINT8 em[ EscRsa1024_KEY_BYTES ];

    if ( ( params != 0 ) &&
         ( params->signature != 0 ) &&
         ( params->modulus != 0 ) &&
         ( params->hashDigest != 0 ) )
    {
        /* Do RSA operation */
        if ( EscRsa1024_ModExp( params->signature, params->modulus, params->pubExp, em ) == FALSE ) {
            /* Check if the result is as expected. */
            hasFailed = EscPkcs1V15_CheckEmsaEncoding1024( params, em );
        }
    }

    return hasFailed;
}
#endif
/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
