/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        pkcs1_rsaes_v15.c

   \brief       PKCS#1 v1.5 RSAES encryption and decryption

   \see         ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1.pdf

   $Rev: 1094 $
 */
/***************************************************************************/

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include "../inc/pkcs1_rsaes_v15.h"

#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** The first byte of the fixed padding */
#define EscPkcs1RsaEsV15_PADDING_FIRST             0x00U
/** The second byte of the fixed padding */
#define EscPkcs1RsaEsV15_PADDING_SECOND            0x02U
/** The message saparator between padding and message */
#define EscPkcs1RsaEsV15_MESSAGE_SEPARATOR         0x00U

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

static BOOL
EscPkcs1RsaEsV15_EmeRsaEsEncode(
    const EscPkcs1RsaEsV15_EncryptDataT* encryptData );

static BOOL
EscPkcs1RsaEsV15_EmeRsaEsDecode(
    const UINT8 em[],
    UINT8 message[],
    UINT16* messageLen);

/***************************************************************************
 * 4. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

/** Function EME-RSAES-ENCODE. Encodes a message according to EME-PKCS#1 v1.5 */
static BOOL
EscPkcs1RsaEsV15_EmeRsaEsEncode(
    const EscPkcs1RsaEsV15_EncryptDataT* encryptData )
{
    UINT16  i;
    UINT16  encodedMessageLen = 2U;
    UINT16  paddingLen = ( EscRsa_KEY_BYTES - ( encryptData->messageLen + (UINT16)EscPkcs1RsaEsV15_FIXED_PADDING_SIZE ) );
    UINT8   padding;
    BOOL    hasFailed = FALSE;

    /*lint -save -esym(960,17.4) cipher is an array */
    encryptData->cipher[ 0U ] = EscPkcs1RsaEsV15_PADDING_FIRST;
    encryptData->cipher[ 1U ] = EscPkcs1RsaEsV15_PADDING_SECOND;
    /*lint -restore */

    /* Generate up to paddingLen random, but nonzero, padding bytes */
    for ( i = 0U; ( ( hasFailed == FALSE ) && ( i < paddingLen ) ); i++ ) {
        padding = 0U;

        do {
            hasFailed = EscRandom_GetRandom( encryptData->randCtx, &padding, 1U );
        } while ( ( padding == 0U ) && ( hasFailed == FALSE ) );

        if ( hasFailed == FALSE ) {
            /*lint -save -esym(960,17.4) cipher is an array */
            encryptData->cipher[ encodedMessageLen ] = padding;
            /*lint -restore */
            encodedMessageLen++;
        }
    }

    if ( hasFailed == FALSE ) {
        /* Append separating zero byte between padding and message. */
        /*lint -save -esym(960,17.4) cipher is an array */
        encryptData->cipher[ encodedMessageLen ] = EscPkcs1RsaEsV15_MESSAGE_SEPARATOR;
        /*lint -restore */
        encodedMessageLen++;

        /* Append original message. */
        for ( i = 0U; i < encryptData->messageLen; i++ ) {
            /*lint -save -esym(960,17.4) cipher is an array*/
            encryptData->cipher[ encodedMessageLen ] = encryptData->message[ i ];
            /*lint -restore */
            encodedMessageLen++;
        }
    }

    return hasFailed;
}

/** Function EME-RSAES-DECODE. Decodes an EME-PKCS#1 v1.5 encoded message. */
static BOOL
EscPkcs1RsaEsV15_EmeRsaEsDecode(
    const UINT8 em[],
    UINT8 message[],
    UINT16 *messageLen )
{
    UINT16 i = 0U;
    BOOL   hasFailed = FALSE;
    BOOL   dummyflag = TRUE;
    UINT16 byteCtr = 0U;
    UINT16 paddingLen = 0U;

    /**
    * No termination/cutoff in case of errors here!
    *
    * RFC 3447
    * Note.  Care shall be taken to ensure that an opponent cannot
    * distinguish the different error conditions in Step 3, whether by
    * error message or timing.  Otherwise an opponent may be able to obtain
    * useful information about the decryption of the ciphertext C, leading
    * to a strengthened version of Bleichenbacher's attack [6]; compare to
    * Manger's attack [36].
    * 
    * Due to the compiling process, the timing requirement can not be
    * guaranteed! Further measures can only be made in assembly language.
    */
    if ( em[ byteCtr ] != EscPkcs1RsaEsV15_PADDING_FIRST ) {
        hasFailed = TRUE;
    } else {
        dummyflag = FALSE;
    }
    byteCtr++;

    if ( em[ byteCtr ] != EscPkcs1RsaEsV15_PADDING_SECOND ) {
        hasFailed = TRUE;
    }
    byteCtr++;

    /* Skip padding and message separator. */
    while ( em[ byteCtr ] != EscPkcs1RsaEsV15_MESSAGE_SEPARATOR ) {
        byteCtr++;
        paddingLen++;

        if ( byteCtr >= EscRsa_KEY_BYTES ) {
            hasFailed = TRUE;
            break;
        }
    }

    /* Skip message separator */
    byteCtr++;

    /* Check for sufficient padding size. */
    if ( paddingLen < EscPkcs1RsaEsV15_MIN_RANDOM_PADDING_SIZE ) {
        hasFailed = TRUE;
    } else {
        dummyflag = FALSE;
    }

    *messageLen = (UINT16)EscRsa_KEY_BYTES - byteCtr;

    while ( byteCtr < EscRsa_KEY_BYTES ) {
        message[ i ] = em[ byteCtr ];
        i++;
        byteCtr++;
    }
    /* Here, hasFailed will still be FALSE if no error
     * condition was detected before. The dummyflag will also be FALSE */

    return ( hasFailed | dummyflag );
}

BOOL
EscPkcs1RsaEsV15_Encrypt(
    const EscPkcs1RsaEsV15_EncryptDataT* encryptData )
{
    BOOL hasFailed = TRUE;

    if ( ( encryptData != 0U ) &&
         ( encryptData->randCtx != 0U ) &&
         ( encryptData->message != 0U ) &&
         ( encryptData->pubKey != 0U ) &&
         ( encryptData->cipher != 0U ) )
    {
        if ( encryptData->messageLen <= EscPkcs1RsaEsV15_MAX_MESSAGE_SIZE ) {
            hasFailed = EscPkcs1RsaEsV15_EmeRsaEsEncode( encryptData );
        }

        if ( hasFailed == FALSE ) {
            hasFailed = EscRsa_ModExp(encryptData->cipher, encryptData->pubKey->modulus, encryptData->pubKey->pubExp, encryptData->cipher );
        }
    }

    return hasFailed;
}

BOOL
EscPkcs1RsaEsV15_Decrypt(
    const EscPkcs1RsaEsV15_DecryptDataT* decryptData )
{
    BOOL hasFailed = TRUE;
    UINT8 tmp[ EscRsa_KEY_BYTES ];
#   ifdef EscRsa_ENABLE_CRT
    EscRsa_KeyPairT keyPair;
#   endif

    /**
     * Remark: initially messageLen holds the maximum buffer size provided by
     * the caller!
     */
    if ( ( decryptData != 0U ) &&
         ( decryptData->message != 0U ) &&
         ( decryptData->messageLen != 0U ) &&
         ( decryptData->privKey != 0U ) &&
         ( decryptData->cipher != 0U ) )
    {
        if  ( *( decryptData->messageLen ) >= EscPkcs1RsaEsV15_MAX_MESSAGE_SIZE ) {
#   ifdef EscRsa_ENABLE_CRT
            EscRsaFe_FromBytesBE( &keyPair.p, decryptData->privKey->p, EscRsa_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE( &keyPair.q, decryptData->privKey->q, EscRsa_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE( &keyPair.dmp1, decryptData->privKey->dmp1, EscRsa_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE( &keyPair.dmq1, decryptData->privKey->dmq1, EscRsa_KEY_BYTES / 2U );
            EscRsaFe_FromBytesBE( &keyPair.iqmp, decryptData->privKey->iqmp, EscRsa_KEY_BYTES / 2U );
            hasFailed = EscRsa_ModExpCrt( tmp, decryptData->cipher, &keyPair );
#   else
            hasFailed = EscRsa_ModExpLong( decryptData->cipher, decryptData->privKey->modulus, decryptData->privKey->privExp, tmp );
#   endif
            if ( hasFailed == FALSE ) {
                hasFailed = EscPkcs1RsaEsV15_EmeRsaEsDecode( tmp, decryptData->message, decryptData->messageLen );
            }
        }
    }

    return hasFailed;
}

#endif
/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
