/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        pkcs1_oaep.c

   \brief       PKCS#1 v2.1 RSAES-OAEP encryption and decryption
   \see         ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1.pdf

   $Rev: 998 $
 */
/***************************************************************************/

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include "../inc/pkcs1_oaep.h"

#ifndef EscRsa_ENABLE_STACK_SAVING_INTERFACE

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

#    define EscPkcs1Oaep_MASK_DB_LEN ( ( EscRsa_KEY_BYTES - EscPkcs1Oaep_DIGEST_LEN ) - 1U )
#    define EscPkcs1Oaep_MASK_INPUT_LEN ( EscPkcs1Oaep_MASK_DB_LEN + 4U )
#    define EscPkcs1Oaep_MSG_LEN ( EscPkcs1Oaep_DIGEST_LEN + 4U )

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

/** Converts an integer into an octet string. (big endian) */
static void
EscPkcs1Oaep_I2OSP(
    UINT32 integer,
    UINT8 octet[] );

/** Wrapper function for the used hash function */
static BOOL
EscPkcs1Oaep_Hash(
    const UINT8 msg[],
    UINT32 msgLen,
    UINT8 digest[] );

/** Function EME-OAEP-ENCODE. Encodes a message according to EME-OAEP. */
static BOOL
EscPkcs1Oaep_EmeOaepEncode(
    const EscPkcs1Oaep_EncryptDataT* encryptData );

/** Function EME-OAEP-DECODE. Decodes an EME-OAEP encoded message. */
static BOOL
EscPkcs1Oaep_EmeOaepDecode(
    const UINT8 em[],
    const UINT8 label[],
    UINT32 labelLen,
    UINT8 message[],
    UINT32* messageLen );

/** Mask generation function according to PKCS#1 v2.1.*/
static BOOL
EscPkcs1Oaep_MGF1(
    const UINT8 mgfSeed[],
    UINT8 mask[],
    UINT32 maskLen );

/***************************************************************************
 * 4. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

/** Converts an integer into an octet string. (big endian) */
static void
EscPkcs1Oaep_I2OSP(
    UINT32 integer,
    UINT8 octet[] )
{
    octet[ 0 ] = (UINT8)( ( integer >> 24 ) & 0xffU );
    octet[ 1 ] = (UINT8)( ( integer >> 16 ) & 0xffU );
    octet[ 2 ] = (UINT8)( ( integer >> 8 ) & 0xffU );
    octet[ 3 ] = (UINT8)( ( integer ) & 0xffU );
}

/** Wrapper function for the used hash function. */
static BOOL
EscPkcs1Oaep_Hash(
    const UINT8 msg[],
    UINT32 msgLen,
    UINT8 digest[] )
{
    BOOL hasFailed;

#   if ( EscPkcs1Oaep_HASH_TYPE == EscPkcs1Oaep_USE_SHA1 )
/* SHA-1 */
    hasFailed = EscSha1_Calc( msg, msgLen, digest);
#   elif ( EscPkcs1Oaep_HASH_TYPE == EscPkcs1Oaep_USE_SHA256 )
/* SHA-256 */
    hasFailed = EscSha256_Calc( msg, msgLen, digest);
#   elif ( EscPkcs1Oaep_HASH_TYPE == EscPkcs1Oaep_USE_SHA512 )
/* SHA-512 */
    hasFailed = EscSha512_Calc( msg, msgLen, digest);
#   endif

    return hasFailed;
}

/**
 * Mask generation function according to PKCS#1 v2.1.
 * The size of input is fixed to EscPkcs1Oaep_DIGEST_LEN.
 **/
static BOOL
EscPkcs1Oaep_MGF1(
    const UINT8 mgfSeed[],
    /** seed from which the mask is generated */
    UINT8 mask[],
    UINT32 maskLen )
{
    /*lint --e{661} --e{662} It is confirmed that the array out of bounds warning is a false positive */
    BOOL hasFailed = FALSE;
    UINT8 msg[ EscPkcs1Oaep_MSG_LEN ];
    UINT8 digest[ EscPkcs1Oaep_DIGEST_LEN ];
    UINT32 i;
    UINT32 j;

    for ( i = 0U; i < EscPkcs1Oaep_DIGEST_LEN; i++ ) {
        msg[ i ] = mgfSeed[ i ];
    }

    for ( i = 0U; i < ( ( maskLen + ( EscPkcs1Oaep_DIGEST_LEN - 1U ) ) / EscPkcs1Oaep_DIGEST_LEN ); i++ ) {
        /* C = I2OSP( counter, 4 ) */
        /* T = T || Hash( mgfSeed || C ) */
        EscPkcs1Oaep_I2OSP( i, &msg[ EscPkcs1Oaep_DIGEST_LEN ] );

        hasFailed |= EscPkcs1Oaep_Hash( msg, EscPkcs1Oaep_MSG_LEN, digest );

        if ( hasFailed == FALSE ) {

            UINT32 copylen;

            copylen = ( ( ( ( i + 1U ) * EscPkcs1Oaep_DIGEST_LEN ) <= maskLen ) ? EscPkcs1Oaep_DIGEST_LEN : ( maskLen % EscPkcs1Oaep_DIGEST_LEN ) );

            for ( j = 0U; j < copylen; j++ ) {
                mask[ ( i * EscPkcs1Oaep_DIGEST_LEN ) + j ] = digest[ j ];
            }
        }
    }

    return hasFailed;
}

/** Function EME-OAEP-ENCODE. Encodes a message according to EME-OAEP.
    All quoted references refer to the section 7.1.1 from the PKCS#1 v2.1 standard. */
static BOOL
EscPkcs1Oaep_EmeOaepEncode(
    const EscPkcs1Oaep_EncryptDataT* encryptData )
{
    UINT8 seed[ EscPkcs1Oaep_DIGEST_LEN ];
    UINT8 digest[ EscPkcs1Oaep_DIGEST_LEN ];
    UINT32 i;
    BOOL hasFailed;
    UINT8 maskedDB[ EscPkcs1Oaep_MASK_INPUT_LEN ];

    if ( encryptData->messageLen > ( ( EscRsa_KEY_BYTES - ( 2U * EscPkcs1Oaep_DIGEST_LEN ) ) - 2U ) ) {
        hasFailed = TRUE;
    } else {
        /* 2.d) */
        hasFailed = EscRandom_GetRandom( encryptData->randCtx, seed, EscPkcs1Oaep_DIGEST_LEN );

        /* 2.e) dbMask = MGF1( seed, k - hLen - 1 ) */
        hasFailed |= EscPkcs1Oaep_MGF1( seed, maskedDB, ( EscRsa_KEY_BYTES - EscPkcs1Oaep_DIGEST_LEN ) - 1U );

        /* 2.a) lHash = Hash( L ) */
        hasFailed |= EscPkcs1Oaep_Hash( encryptData->label, encryptData->labelLen, digest );

        /* 2.b) PS consists of k - mLen - 2hLen - 2 zero octets */
        /* 2.c) / 2.f) maskedDB = ( lHash || PS || 0x01 || M ) XOR dbMask */
        for ( i = 0U; i < EscPkcs1Oaep_DIGEST_LEN; i++ ) {
            maskedDB[ i ] ^= digest[ i ];
        }

        maskedDB[ ( ( EscRsa_KEY_BYTES - encryptData->messageLen ) - EscPkcs1Oaep_DIGEST_LEN ) - 2U ] ^= 0x01U;

        for ( i = 0U; i < encryptData->messageLen; i++ ) {
            /*lint -save -esym(960,17.4) encryptData->message is an array*/
            maskedDB[ ( ( ( EscRsa_KEY_BYTES - EscPkcs1Oaep_DIGEST_LEN ) - 1U ) - encryptData->messageLen ) + i ] ^= encryptData->message[ i ];
            /*lint -restore encryptData->message is an array*/
        }

        /* 2.g) seedMask = MGF( maskedDB, hLen ) --> Hash( maskedDB || 00 00 00 00 ) */
        for ( i = 0U; i < 4U; i++ ) {
            maskedDB[ ( ( EscRsa_KEY_BYTES - EscPkcs1Oaep_DIGEST_LEN ) - 1U ) + i ] = 0x00U;
        }

        hasFailed |= EscPkcs1Oaep_Hash( maskedDB, EscPkcs1Oaep_MASK_INPUT_LEN, digest );

        /* 2.g) / 2.i) EM = 0x00 || ( seed XOR seedMask ) || maskedDB */
        /*lint -save -esym(960,17.4) encryptData->cipher is an array*/
        encryptData->cipher[ 0x00 ] = 0x00U;
        /*lint -restore encryptData->cipher is an array*/

        for ( i = 0U; i < EscPkcs1Oaep_DIGEST_LEN; i++ ) {
            /*lint -save -esym(960,17.4) encryptData->cipher is an array*/
            encryptData->cipher[ 1U + i ] = seed[ i ] ^ digest[ i ];
            /*lint -restore encryptData->cipher is an array*/
        }

        for ( i = 0U; i < EscPkcs1Oaep_MASK_DB_LEN; i++ ) {
            /*lint -save -esym(960,17.4) encryptData->cipher is an array*/
            encryptData->cipher[ 1U + EscPkcs1Oaep_DIGEST_LEN + i ] = maskedDB[ i ];
            /*lint -restore encryptData->cipher is an array*/
        }
    }
    
    return hasFailed;
}

/** Function EME-OAEP-DECODE. Decodes an EME-OAEP encoded message.
    All quoted references refer to the section 7.1.2 from the PKCS#1 v2.1 standard. */
static BOOL
EscPkcs1Oaep_EmeOaepDecode(
    const UINT8 em[],
    const UINT8 label[],
    UINT32 labelLen,
    UINT8 message[],
    UINT32* messageLen )
{
    UINT8 seed[ EscPkcs1Oaep_DIGEST_LEN ];
    UINT8 digest[ EscPkcs1Oaep_DIGEST_LEN ];
    UINT32 i;
    BOOL hasFailed;
    UINT8 maskedDB[ EscPkcs1Oaep_MASK_INPUT_LEN ];
    UINT8 DB[ EscPkcs1Oaep_MASK_DB_LEN ];

    /* 3.c) seedMask = MGF( maskedDB, hLen ) --> Hash( maskedDB || 00 00 00 00 ) */
    for ( i = 0U; i < EscPkcs1Oaep_MASK_DB_LEN; i++ ) {
        maskedDB[ i ] = em[ 1U + EscPkcs1Oaep_DIGEST_LEN + i ];
    }

    for ( i = 0U; i < 4U; i++ ) {
        maskedDB[ ( ( EscRsa_KEY_BYTES - EscPkcs1Oaep_DIGEST_LEN ) - 1U ) + i ] = 0x00U;
    }

    hasFailed = EscPkcs1Oaep_Hash( maskedDB, EscPkcs1Oaep_MASK_INPUT_LEN, digest );

    /* seed = maskedSeed XOR seedMask */
    for ( i = 0U; i < EscPkcs1Oaep_DIGEST_LEN; i++ ) {
        seed[ i ] = em[ 1U + i ] ^ digest[ i ];
    }

    /* 3.e) dbMask = MGF1( seed, k - hLen - 1 ) */
    hasFailed |= EscPkcs1Oaep_MGF1( seed, DB, ( EscRsa_KEY_BYTES - EscPkcs1Oaep_DIGEST_LEN ) - 1U );

    /* 3.f) DB = maskedDB XOR dbMask */
    for ( i = 0U; i < EscPkcs1Oaep_MASK_DB_LEN; i++ ) {
        DB[ i ] ^= maskedDB[ i ];
    }

    /* 3.a) lHash' = Hash( L ) */
    hasFailed |= EscPkcs1Oaep_Hash( label, labelLen, digest );

    for ( i = 0U; i < EscPkcs1Oaep_DIGEST_LEN; i++ ) {
        hasFailed |= ( digest[ i ] != DB[ i ] ) ? TRUE : FALSE;
    }

    if ( em[ 0x00 ] != 0x00U ) {
        hasFailed = TRUE;
    }

    /* find the end of padding PS */
    for ( i = EscPkcs1Oaep_DIGEST_LEN; ( i < ( EscPkcs1Oaep_MASK_DB_LEN - 1U ) ) && ( DB[ i ] == 0x00U ); i++ ) {}

    if ( DB[ i ] != 0x01U ) {
        hasFailed = TRUE;
    }

    *messageLen = ( EscPkcs1Oaep_MASK_DB_LEN - 1U ) - i;

    for ( i = 0U; i < *messageLen; i++ ) {
        message[ i ] = DB[ ( EscPkcs1Oaep_MASK_DB_LEN - *messageLen ) + i ];
    }

    return hasFailed;
}

BOOL
EscPkcs1Oaep_Encrypt(
    const EscPkcs1Oaep_EncryptDataT* encryptData )
{
    BOOL hasFailed = TRUE;

    if ( ( encryptData != 0 ) &&
         ( encryptData->randCtx != 0 ) &&
         ( encryptData->message != 0 ) &&
         ( encryptData->pubKey != 0 ) &&
         ( encryptData->label != 0 ) &&
         ( encryptData->cipher != 0 ) )
    {
        hasFailed = EscPkcs1Oaep_EmeOaepEncode( encryptData );

        if ( hasFailed == FALSE ) {
            hasFailed = EscRsa_ModExp( encryptData->cipher, encryptData->pubKey->modulus, encryptData->pubKey->pubExp, encryptData->cipher );
        }
    }

    return hasFailed;
}

BOOL
EscPkcs1Oaep_Decrypt(
    const EscPkcs1Oaep_DecryptDataT* decryptData )
{
    UINT8 tmp[ EscRsa_KEY_BYTES ];
#   ifdef EscRsa_ENABLE_CRT
    EscRsa_KeyPairT keyPair;
#   endif

    BOOL hasFailed = TRUE;

#   if ( EscRsa_KEY_BYTES < ( ( 2U * EscPkcs1Oaep_DIGEST_LEN ) + 2U ) )
#       error "PKCS OAEP 1.c) decryption error"
#   endif

    if ( ( decryptData != 0 ) &&
         ( decryptData->message != 0 ) &&
         ( decryptData->messageLen != 0U ) &&
         ( *( decryptData->messageLen ) >= ( ( EscRsa_KEY_BYTES - ( 2U * EscPkcs1Oaep_DIGEST_LEN ) ) - 2U ) ) &&
         ( decryptData->privKey != 0 ) &&
         ( decryptData->label != 0 ) &&
         ( decryptData->cipher != 0 ) )
    {
#   ifdef EscRsa_ENABLE_CRT
        EscRsaFe_FromBytesBE( &keyPair.p, decryptData->privKey->p, EscRsa_KEY_BYTES / 2U );
        EscRsaFe_FromBytesBE( &keyPair.q, decryptData->privKey->q, EscRsa_KEY_BYTES / 2U );
        EscRsaFe_FromBytesBE( &keyPair.dmp1, decryptData->privKey->dmp1, EscRsa_KEY_BYTES / 2U );
        EscRsaFe_FromBytesBE( &keyPair.dmq1, decryptData->privKey->dmq1, EscRsa_KEY_BYTES / 2U );
        EscRsaFe_FromBytesBE( &keyPair.iqmp, decryptData->privKey->iqmp, EscRsa_KEY_BYTES / 2U );
        hasFailed = EscRsa_ModExpCrt( tmp, decryptData->cipher, &keyPair );
#   else
        hasFailed = EscRsa_ModExpLong( decryptData->cipher, decryptData->privKey->modulus, decryptData->privKey->privExp, tmp );
#   endif

        if ( hasFailed == FALSE ) {
            hasFailed = EscPkcs1Oaep_EmeOaepDecode( tmp, decryptData->label, decryptData->labelLen, decryptData->message, decryptData->messageLen );
        }
    }

    return hasFailed;
}

#endif
/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
