/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        crl_utils.h

   \brief       Utilities used for CRL Handling

   ***

   $Rev: 817 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-28 14:43:45 +0200 (Mo, 28. Aug 2017) $
****************************************************************************/

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include "../inc/crl_utils.h"

#include "../../common/inc/esc_common.h"
#include "../../common/inc/esc_utils.h"

#include "../../cmp/inc/cmp.h"
#include "../../cmp/inc/asn1_utils.h"

#include "../../ecm/inc/ecm.h"
#include "../../ecm/inc/ecm_utils.h"
#include "../../ecm/inc/ecm_cm_main.h"
#include "../../common/inc/esc_debug.h"
#include "../../common/inc/esc_asn1_utils.h"
#include "../../common/inc/base64.h"

#include "../../cmp/inc/cmp_utils.h"

#include "../../cmp/asn1-2005-multi-file-os-asn1c/CertificateList.h"
#include "../../cmp/asn1-2005-multi-file-os-asn1c/CRLDistributionPoints.h"

#include "../../cycurlib/lib/inc/pkcs1_v15.h"

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif /* UNIT_TESTING */

#include "crl_utils.h"

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                         *
 ***************************************************************************/

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

BOOL loadCRL(
    char* path,
    const CR_LIST_t** crList,
    CERT_LIST_s* certList,
    CRL_HASHES_t* hashes )
{
    if (!path || !crList || !hashes )
        return TRUE;
    BOOL failed = FALSE;
    FILE *fp = NULL;

    UINT8* crlData = NULL;
    UINT32 crlDataSize = 0;
    UINT8* derData = NULL;
    UINT32 derDataSize = 0;
    long size = 0;
    size_t s = 0;

    /* open CRL File */
    ifNotFailed {
        if ((fp = fopen(path, "r")) == NULL) {

            dbgPrint( DBG_ERR, DBG_FLS "Error: Failed to open '%s' (%s)\n",
            DBG_FL, path);
            return failed = TRUE;
        }
    }

    /* get File Size */
    ifNotFailed {
        if (fseek(fp, 0, SEEK_END) != 0) {
            dbgPrint( DBG_ERR, DBG_FLS "Failed to read file size\n", DBG_FL);
            failed = TRUE;
        } else {
            size = ftell(fp);
            if ( size < 0 ) {
                failed = TRUE;
            } else {
                crlDataSize = size;
            }
            rewind(fp);
        }
        failed |= (crlDataSize > 0) ? FALSE : TRUE;
    }

    ifNotFailed {
        crlData = malloc (crlDataSize + 1);
        if ( ! crlData ) {
            failed = TRUE;
        }
    }

    ifNotFailed {
        memset ( crlData, 0x0, crlDataSize + 1);
        failed = (crlData == NULL) ? TRUE : FALSE;
    }

    /* read CRL - use SDC? */
    ifNotFailed {
        size_t ret = 0;
        ret = fread(crlData, 1, crlDataSize, fp);
        if (ret != crlDataSize) {
            dbgPrint( DBG_ERR, "reading from disk failed\n");
            failed = TRUE;
        }
    }

    /* Close File */
    if (fp != NULL)
        fclose(fp);

    /* Decode CRL */
    CertificateList_t *cl = NULL;

    ifNotFailed {
        failed = alloc((void *) &cl, sizeof(CertificateList_t));
    }

    if(!failed){

        /* Did we receive a PEM encoded CRL? */
        if ( checkForPEM( (char *)crlData) == TRUE ) {

            failed = decodePEM( crlData, crlDataSize, &derData, &derDataSize);

        } else {
            /* No, assume DER and just adjust pointer */
            derData = crlData;
            derDataSize = crlDataSize;
        }
    }


    ifNotFailed {
        failed = asn1Decode(derData, derDataSize, &asn_DEF_CertificateList, cl,
        FALSE);
    }
    // verify Signature of CRL
    STATUS_s stat;
    if (certList)
        failed = verifyCRLSignature(cl, certList, &stat );

    ifNotFailed {
        // Add digest of CRL
        if (hashes) {
            UINT8* digest = NULL;
            if (hashCRL(cl, &digest)) {
                dbgPrint(DBG_INFO, "Could not hash CR-List");
            } else{
                hashes->digest = digest;
                failed = (cmpGetObjFromRDNSequence(
                        &(cl->tbsCertList.issuer.choice.rdnSequence),
                        oid_commonName, TRUE, &hashes->issuerCN));
                hashes->next = NULL;

                /* place encoded CRL data as well */
                /*
                hashes->certList = crlData;
                hashes->certListLen = crlDataSize;
                */
                hashes->certList = derData;
                hashes->certListLen = derDataSize;

                hashes->cl = cl;

                switch(cl->tbsCertList.thisUpdate.present){
                case Time_PR_NOTHING:
                    hashes->thisUpdate = 0;
                    break;

                case Time_PR_utcTime:
                    hashes->thisUpdate = asn_UT2time(&cl->tbsCertList.thisUpdate.choice.utcTime, NULL, 0);
                    break;

                case Time_PR_generalTime:
                    hashes->thisUpdate = asn_GT2time(&cl->tbsCertList.thisUpdate.choice.generalTime, NULL, 0);
                    break;
                }

                /* extract CRL next update time */
                if ( cl->tbsCertList.nextUpdate ){
                    switch(cl->tbsCertList.nextUpdate->present){
                    case Time_PR_NOTHING:
                        hashes->nextUpdate = 0;
                        break;

                    case Time_PR_utcTime:
                        hashes->nextUpdate = asn_UT2time(&cl->tbsCertList.nextUpdate->choice.utcTime, NULL, 0);
                        break;

                    case Time_PR_generalTime:
                        hashes->nextUpdate = asn_GT2time(&cl->tbsCertList.nextUpdate->choice.generalTime, NULL, 0);
                        break;
                    }
                } else {
                    hashes->nextUpdate = 0;
                }

                if ( certList ) {
                    failed = determineCrlTypeByIssuerCn ( &hashes, certList, &stat);
                }
            }
            if(failed){
                free(digest);
                hashes->digest = NULL;
                free(hashes->issuerCN);
                hashes->issuerCN = NULL;
            }
        }
    }

    ifFailed {
         if ( cl ) {
             ASN_STRUCT_FREE( asn_DEF_CertificateList, cl );
         }
         if ( derData != crlData ) {
            escFreePtr((UINT8 **)&crlData);
         }
        escFreePtr((UINT8 **) &derData);
     } else {
         if ( derData != crlData ) {
            escFreePtr((UINT8 **)&crlData);
         }
     }

    CR_LIST_t **crListPtr = (CR_LIST_t **)crList;

    ifNotFailed {

        /* fast forward to end of crList */
        while ( *crListPtr ) {
            crListPtr = &(*crListPtr)->next;
        }

        // No revoked Certs
        if (!cl->tbsCertList.revokedCertificates) {
            *crListPtr = NULL;
        } else {
            unsigned int i = 0;
            //    CRL_LIST_t *crl_list = NULL;
            while (!failed && i < cl->tbsCertList.revokedCertificates->list.count) {

                *crListPtr = calloc(1, sizeof(CR_LIST_t));

                if ( NULL == *crListPtr ) {
                    failed = TRUE;
                }

                ifNotFailed {
                    failed = escInt2SizetAndCheckForOv(
                        cl->tbsCertList.revokedCertificates->list.array[i]->userCertificate.size, &s );
                }

                ifNotFailed {

                    (*crListPtr)->userID.buf = calloc (0x1, s);
                    if ( NULL == (*crListPtr)->userID.buf ) {
                        failed = TRUE;
                    }
                }
                ifNotFailed {
                    (*crListPtr)->userID.size = s;
                }

                ifNotFailed {
                    memcpy((*crListPtr)->userID.buf,
                        cl->tbsCertList.revokedCertificates->list.array[i]->userCertificate.buf, s );
                }

                ifNotFailed {
                    failed = (cmpGetObjFromRDNSequence( &(cl->tbsCertList.issuer.choice.rdnSequence), oid_commonName, TRUE, &(*crListPtr)->issuerCN));
                }

                crListPtr = &(*crListPtr)->next;
                i++;
            }
        }
    }

    return failed;
}


BOOL determineCrlTypeByIssuerCn (
    CRL_HASHES_t **crl,
    CERT_LIST_s *certList,
    STATUS_s *stat)
{
    BOOL failed = FALSE;
    CERT_LIST_s *issuerCert = NULL;
    CRL_HASHES_t **crlPtr = crl;

    dbgIn ( __func__ );

    while ( *crlPtr ) {

        (*crlPtr)->id = crl_id_none;

        /* get CRL issuer certificate */
        failed = cmpGetCrtFromListBySbjOrIssuerOid( subject, oid_commonName, (*crlPtr)->issuerCN, certList, &issuerCert, TRUE );

        ifFailed {
            dbgPrint (DBG_ERR, DBG_FLS "Could not find CLR issuer certificate\n", DBG_FL );
            ecmCmSetStatus(stat, EXT_IF_ERR_CERT_NOT_PRESENT, (*crlPtr)->issuerCN, NULL );
        } else {

            switch ( issuerCert->certId ) {

                /* check for root */
                case ( cert_id_root ):
                    (*crlPtr)->id = crl_id_root;
                    break;

                /* check for IC or EC */
                case ( cert_id_issuer ):
                case ( cert_id_endpoint ):

                    if ( issuerCert->devChain == TRUE ) {
                        /* issuer cert is part of device cert chain,
                         * means this is the EC CRL */
                        (*crlPtr)->id = crl_id_ec;
                    } else {
                        /* issuer cert is not part of device cert chain,
                         * since we only have 2 chains this must be the IC CRL */
                        (*crlPtr)->id = crl_id_ic;
                    }
                    break;

                default:
                    failed = TRUE;
                    break;
            }

            ifFailed {
                dbgPrint (DBG_ERR, DBG_FLS, "Could not determine CLR type\n", DBG_FL );
            }
        }

        crlPtr = &(*crlPtr)->next;
    }

    dbgOut ( __func__ );

    return ( failed );
}

int addCRLfromURI(
    char* uri,
    CR_LIST_t** list,
    CERT_LIST_s* certList,
    ull_time_t* crlCreationTime,
    ull_time_t* crlNextUpdTime,
    CRL_HASHES_t** crlHashes,
    STATUS_s *stat )
{
    if(!uri || !list)
        return -1;

    CR_LIST_t* crList = *list;
    if(crList != NULL){
        // if a non empty crList was provided --> fast forward to the end
        while(crList != NULL){
            crList = crList->next;
        }
    }

    BOOL failed = FALSE;
    UINT8* crlData = NULL;
    UINT32 crlDataSize = 0;
    size_t s = 0;
    dbgPrint ( DBG_INFO, "Download CRL %s\n", uri );
    failed = downloadFromURItoBuffer(uri, (char**) &crlData, &crlDataSize);
    if(failed){
        free(crlData);
        crlData = NULL;
        // Download failed
        return -2;
    }

    /* Decode CRL */
    CertificateList_t *cl = NULL;
    failed = alloc((void *) &cl, sizeof(CertificateList_t));

    ifNotFailed {
        failed = asn1Decode(crlData, crlDataSize, &asn_DEF_CertificateList, cl,
        FALSE);
    }

    /* extract CRL creation time */
    if(crlCreationTime){
        switch(cl->tbsCertList.thisUpdate.present){
        case Time_PR_NOTHING:
            *crlCreationTime = 0;
            break;

        case Time_PR_utcTime:
            *crlCreationTime = asn_UT2time(&cl->tbsCertList.thisUpdate.choice.utcTime, NULL, 0);
            break;

        case Time_PR_generalTime:
            *crlCreationTime = asn_GT2time(&cl->tbsCertList.thisUpdate.choice.generalTime, NULL, 0);
            break;
        }
    }

    /* extract CRL next update time */
    if(crlNextUpdTime){
        if ( cl->tbsCertList.nextUpdate ){
            switch(cl->tbsCertList.nextUpdate->present){
            case Time_PR_NOTHING:
                *crlNextUpdTime = 0;
                break;

            case Time_PR_utcTime:
                *crlNextUpdTime = asn_UT2time(&cl->tbsCertList.nextUpdate->choice.utcTime, NULL, 0);
                break;

            case Time_PR_generalTime:
                *crlNextUpdTime = asn_GT2time(&cl->tbsCertList.nextUpdate->choice.generalTime, NULL, 0);
                break;
            }
        } else {
            *crlNextUpdTime = 0;
        }
    }

    if(failed){
        asnFreeStruct((void**) &cl, asn_DEF_CertificateList);
        free(crlData);
        crlData = NULL;
        return -1;
    }

    // verify Signature of CRL
    // !!! if no cert list is given to verify with, we dont check
    if(certList){
        dbgPrint (DBG_INFO, "Verify CRL signature\n");
        failed = verifyCRLSignature(cl, certList, stat);
        if(failed){
            dbgPrint (DBG_ERR, DBG_FLS "CRL signature verification failed\n", DBG_FL );
            free(crlData);
            crlData = NULL;
            asnFreeStruct((void**) &cl, asn_DEF_CertificateList);
            return -1;
        }
    }

    unsigned int i = 0;
    CR_LIST_t *crl_old_entity = NULL;

    /* get CRL issuer */
    char *crlIssuerCN = NULL;

    failed = (cmpGetObjFromRDNSequence(
            &(cl->tbsCertList.issuer.choice.rdnSequence),
            oid_commonName, TRUE, &crlIssuerCN));

    /* get revoked certificates */
    if(cl->tbsCertList.revokedCertificates){
        while (!failed && i < cl->tbsCertList.revokedCertificates->list.count) {

            CR_LIST_t *crl_entity = calloc(1, sizeof(CR_LIST_t));
            if (!crl_entity)
                failed = TRUE;

            if (crl_old_entity != NULL) {
                crl_old_entity->next = crl_entity;
            }

            ifNotFailed {
                failed = escInt2SizetAndCheckForOv(
                    cl->tbsCertList.revokedCertificates->list.array[i]->userCertificate.size, &s );
            }
            ifNotFailed {
                crl_entity->userID.buf = calloc(1, s );
                if (!crl_entity->userID.buf) {
                    failed = TRUE;
                }
            }

            ifNotFailed {
                crl_entity->userID.size = s;

                memcpy(crl_entity->userID.buf,
                    cl->tbsCertList.revokedCertificates->list.array[i]->userCertificate.buf, s );

                failed = (cmpGetObjFromRDNSequence(
                        &(cl->tbsCertList.issuer.choice.rdnSequence),
                        oid_commonName, TRUE, &crl_entity->issuerCN));
            }

            if (i == 0)
                crList = crl_entity;

            crl_old_entity = crl_entity;
            i++;
        }
    }

    // check CRL Hashes - if CRL is not in List or Digest has changed add a copy of
    // the CertificateList
    if(!failed){
        if(crlHashes ) {

            BOOL failed = FALSE;
            UINT8* digest = NULL;
            CRL_HASHES_t** hashes = NULL;

            if ( *crlHashes == NULL ) {
                *crlHashes = calloc(1, sizeof(CRL_HASHES_t));

                if ( NULL == *crlHashes ) {
                    failed = TRUE;
                }
            }
            ifNotFailed {
                hashes = crlHashes;
                failed = hashCRL(cl, &digest);
            }

            // find CRL Hash for issuer
            if(!failed){
                while (*hashes && (*hashes)->issuerCN && strncmp(crlIssuerCN, (*hashes)->issuerCN, strlen ((*hashes)->issuerCN)) != 0) {
                    //nextEntry
                    hashes = &(*hashes)->next;
                }
                if(*hashes && (*hashes)->issuerCN){
                    if(memcmp(digest, (*hashes)->digest, EscSha1_DIGEST_LEN) != 0 ){
                        free((*hashes)->digest);
                        (*hashes)->digest = digest;
                        free((*hashes)->certList);
                        (*hashes)->certList = crlData;
                        (*hashes)->certListLen = crlDataSize;
                        (*hashes)->cl = cl;
                        (*hashes)->thisUpdate = *crlCreationTime;
                        (*hashes)->nextUpdate = *crlNextUpdTime;
                        (*hashes)->store = TRUE;
                        (*hashes)->outdated = FALSE;

                    } else {
                        // Same Digest -> do nothing
                        free(crlData);
                        crlData = NULL;
                        free(digest);
                        digest = NULL;
                    }
                } else {
                    if(!(*hashes)){
                    // Issuer not found --> create new entry
                        (*hashes) = calloc(1, sizeof(CRL_HASHES_t));
                        if ( NULL == *hashes ) {
                            failed = TRUE;
                        }
                    } else{
                        // issuerCN is NULL, this should only happen if the remainder is NULL to, but free anyway
                        free((*hashes)->digest);
                        free((*hashes)->certList);
                    }

                    ifNotFailed {

                        s = strlen ( crlIssuerCN );

                        failed = escCheckSizetForOv ( s );
                    }

                    ifNotFailed {

                        (*hashes)->issuerCN = calloc(1, s + 1);

                        if ( NULL != (*hashes)->issuerCN ) {
                            strncpy((*hashes)->issuerCN, crlIssuerCN, strlen (crlIssuerCN));
                            (*hashes)->digest = digest;
                            (*hashes)->certList = crlData;
                            (*hashes)->certListLen = crlDataSize;
                            (*hashes)->cl = cl;
                            (*hashes)->thisUpdate = *crlCreationTime;
                            (*hashes)->nextUpdate = *crlNextUpdTime;
                            (*hashes)->store = TRUE;
                            (*hashes)->next = NULL;

                            failed = determineCrlTypeByIssuerCn ( hashes, certList, stat );
                        } else {
                            failed = TRUE;
                        }
                    }
                }
            } else {
                free(crlData);
                crlData = NULL;
                free(digest);
                digest = NULL;
            }
        } else {
            free(crlData);
            crlData = NULL;
        }
    } else {
        free(crlData);
        crlData = NULL;
        freeCRL(crList);
        crList = NULL;
    }

    escFreePtr( (UINT8**) &crlIssuerCN);

    if (failed){
        freeCRL(crList);
        crList = NULL;
        return -1;
    } else{
        if(*list == NULL){
            *list = crList;
        } else{
            // append
            CR_LIST_t* tempList = *list;
            while(tempList->next){
                tempList = tempList->next;
            }
            tempList->next = crList;
        }
        return i;
    }
}

BOOL verifyCRLSignature(
    CertificateList_t* p_crList,
    CERT_LIST_s* certList,
    STATUS_s *stat )
{
    BOOL failed = FALSE;
    if (!certList || !p_crList)
        return failed = TRUE;

    /* Get Issuer CN */
    char* p_issuerCN = NULL;
    failed = (cmpGetObjFromRDNSequence(
            &(p_crList->tbsCertList.issuer.choice.rdnSequence), oid_commonName,
            TRUE, &p_issuerCN));

    if (failed || !p_issuerCN) {
        failed = TRUE;
    }

    /* Get Certificate of Issuer */
    CERT_LIST_s* p_issuerCertList = NULL;
    ifNotFailed {
        failed = cmpGetCrtFromListBySbjOrIssuerOid(oid_subject, oid_commonName,
                p_issuerCN, certList, &p_issuerCertList, TRUE);
    }
    if (!p_issuerCertList || failed) {

        dbgPrint ( DBG_INFO, "Certificate 'issuer CN=%s' not available to verify CRL signature\n", p_issuerCN );
        ecmCmSetStatus(stat, EXT_IF_ERR_CERT_NOT_PRESENT, p_issuerCN, NULL );
        failed = TRUE;
    }

    /* Decode Issuer Certificate */
    Certificate_t* p_issuerCert = NULL;
    ifNotFailed {
        failed = alloc((void *) &p_issuerCert, sizeof(Certificate_t));
    }
    ifNotFailed {
        failed = asn1Decode(p_issuerCertList->certPtr,
                p_issuerCertList->certSize, &asn_DEF_Certificate, p_issuerCert,
                FALSE);
    }

    UINT8 *encCRL = NULL;
    UINT32 encCRLSize = MAX_CERT_SIZE;
    ifNotFailed {

        /* Encode CRL TBSCertList part for verification */
        failed = asn1Encode(&p_crList->tbsCertList, &asn_DEF_TBSCertList,
                &encCRL, &encCRLSize, FALSE);
    }

    /* Get Hash Type */
    HASH_t hashType = 0;
    OID_t oid = oid_unsupported;
    ifNotFailed {
        /* determine the signature algorithm */
        failed = getObjIdentifier(&p_crList->signatureAlgorithm.algorithm,
                &oid);
    }
    ifNotFailed {
        switch (oid) {
        case (oid_shaWithRsaEnc):
            hashType = hash_sha1;
            break;
        case (oid_sha256WithRsaEnc):
            hashType = hash_sha256;
            break;
        case (oid_sha512WithRsaEnc):
            dbgPrint( DBG_ERR, DBG_FLS
            "RSA signatures using SHA 512 are not supported!\n",
            DBG_FL);
            break;
        default:
            dbgPrint( DBG_ERR,
            DBG_FLS "Unknown signature algorithm\n", DBG_FL);
            failed = TRUE;
            break;
        }
    }

    UINT32 keySize = 0;
    /* Get KeySize of Issuer */
    ifNotFailed {
        failed = EscCrypto_GetPubKeyModSize(
            p_issuerCert->tbsCertificate.subjectPublicKeyInfo.subjectPublicKey.buf,
            p_issuerCert->tbsCertificate.subjectPublicKeyInfo.subjectPublicKey.size,
            &keySize);
    }

    ifNotFailed {
        /* Get public key */
        switch (keySize) {
        case (128U): {
            EscRsa1024_KeyPairT keyPair;
            ifNotFailed {
                failed = EscCrypto_DecodePublicKey(&keyPair,
                    p_issuerCert->tbsCertificate.subjectPublicKeyInfo.subjectPublicKey.buf,
                    p_issuerCert->tbsCertificate.subjectPublicKeyInfo.subjectPublicKey.size,
                    1024);
            }

            ifNotFailed {
                failed = EscCrypto_VerifyMessage(&keyPair, encCRL, encCRLSize,
                        p_crList->signature.buf, hashType, 1024);
            }
            break;
        }

        case (256U): {
            EscRsa_KeyPairT keyPair;
            ifNotFailed {
                failed =
                        EscCrypto_DecodePublicKey(&keyPair,
                                p_issuerCert->tbsCertificate.\
subjectPublicKeyInfo.subjectPublicKey.buf,
                                p_issuerCert->tbsCertificate.\
subjectPublicKeyInfo.subjectPublicKey.size,
                                2048);
            }

            ifNotFailed {
                failed = EscCrypto_VerifyMessage(&keyPair, encCRL, encCRLSize,
                        p_crList->signature.buf, hashType, 2048);
            }
            break;
        }

        case (512U): {
            EscRsa4096_KeyPairT keyPair;
            ifNotFailed {
                failed =
                        EscCrypto_DecodePublicKey(&keyPair,
                                p_issuerCert->tbsCertificate.\
subjectPublicKeyInfo.subjectPublicKey.buf,
                                p_issuerCert->tbsCertificate.\
subjectPublicKeyInfo.subjectPublicKey.size,
                                4096);
            }

            ifNotFailed {
                failed = EscCrypto_VerifyMessage(&keyPair, encCRL, encCRLSize,
                        p_crList->signature.buf, hashType, 4096);
            }
            break;
        }

        default:
            failed = TRUE;
            break;
        }

        ifFailed {
            /* look for a certificate that has the same issuer as the CRL, this contains the CRL download URL */
            CERT_LIST_s *crt = NULL;
            cmpGetCrtFromListBySbjOrIssuerOid( issuer, oid_commonName, p_issuerCN, certList, &crt, TRUE);
            ecmCmSetStatus( stat, EXT_IF_ERR_CRL_INVALID, p_issuerCN, (( NULL != crt ) ? crt->crlUri : NULL));
        }
    }

    /* Free allocated Memory */
    escFreePtr((UINT8**)&p_issuerCN );
    asnFreeStruct((void**) &p_issuerCert, asn_DEF_Certificate);
    free(encCRL);

    return failed;
}

BOOL removeCrlHashEntryByIssuerCn (
    char *issuerCn,
    CRL_HASHES_t **crl,
    BOOL outdatedOnly)
{
    BOOL failed = FALSE;
    CRL_HASHES_t *nextPtr = NULL;
    CRL_HASHES_t **crlPtr = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !issuerCn || !crl || !(*crl) ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        crlPtr = crl;
        while ( (*crlPtr) ) {

            if ( !(*crlPtr)->issuerCN ) {
                dbgPrint( DBG_ERR, DBG_FLS "CRL issuer CN not available\n", DBG_FL );
                failed = TRUE;
                break;
            }

            if ( strncmp ((*crlPtr)->issuerCN, issuerCn, strlen(issuerCn)) == 0 ) {

                if ( ((*crlPtr)->outdated == TRUE ) || (FALSE == outdatedOnly)) {

                    escFreePtr((UINT8 **)&(*crlPtr)->issuerCN );
                    escFreePtr((UINT8**) &(*crlPtr)->digest );
                    escFreePtr((UINT8**) &(*crlPtr)->certList );

                    if ( (*crlPtr)->cl ) {
                        asnFreeStruct((void**) &(*crlPtr)->cl, asn_DEF_CertificateList);
                    }

                    nextPtr = (*crlPtr)->next;

                    escFreePtr((UINT8**) &(*crlPtr) );

                    (*crlPtr) = nextPtr;

                } else {

                    crlPtr = &(*crlPtr)->next;
                }

            } else {
                crlPtr = &(*crlPtr)->next;
            }
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}


BOOL removeCrListEntryByIssuerCn (
    char *issuerCn,
    CR_LIST_t **lst )
{
    CR_LIST_t *nextPtr = NULL;
    CR_LIST_t **crLstPtr = NULL;
    BOOL failed = FALSE;

    /* sanity checks */
    if ( !lst || !issuerCn ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        crLstPtr = lst;

        while ( (*crLstPtr) ) {

            if ( !(*crLstPtr)->issuerCN ) {
                dbgPrint( DBG_ERR, DBG_FLS "CRL issuer CN not available\n", DBG_FL );
                failed = TRUE;
                break;
            }

            if ( strncmp ((*crLstPtr)->issuerCN, issuerCn, strlen(issuerCn)) == 0 ) {

                escFreePtr((UINT8 **)&(*crLstPtr)->issuerCN );
                escFreePtr((UINT8**) &(*crLstPtr)->userID.buf );

                nextPtr = (*crLstPtr)->next;

                escFreePtr((UINT8**) &(*crLstPtr) );

                (*crLstPtr) = nextPtr;

            } else {
                crLstPtr = &(*crLstPtr)->next;
            }
        }
    }

    return ( failed );
}

void freeCRL(
    CR_LIST_t* p_crList)
{
    CR_LIST_t* ptr = p_crList;
    CR_LIST_t* p_crEntity;

    while ( ptr ) {
        if ( ptr->issuerCN ) {
            free(ptr->issuerCN);
            ptr->issuerCN = NULL;
        }
        if ( ptr->userID.buf ) {
            free(ptr->userID.buf);
            ptr->userID.buf = NULL;
        }

        p_crEntity = ptr;
        ptr = ptr->next;
        free(p_crEntity);
        p_crEntity = NULL;
    }
}

void freeCRLHashes(
    CRL_HASHES_t* crlHashes)
{
    CRL_HASHES_t* p_crEntity = NULL;

    while ( crlHashes ) {
        if ( crlHashes->issuerCN ) {
            free(crlHashes->issuerCN);
            crlHashes->issuerCN = NULL;
        }
        if ( crlHashes->digest ) {
            free(crlHashes->digest);
            crlHashes->digest = NULL;
        }
        if ( crlHashes->certList ) {
            free(crlHashes->certList);
            crlHashes->certList = NULL;
        }

        if ( crlHashes->cl ) {
            //asnFreeStruct((void**) &crlHashes->cl, asn_DEF_CertificateList);
            ASN_STRUCT_FREE( asn_DEF_CertificateList, crlHashes->cl );
        }

        if ( crlHashes->certList ) {
            free(crlHashes->certList);
            crlHashes->certList = NULL;
        }


        p_crEntity = crlHashes;
        crlHashes = crlHashes->next;
        free(p_crEntity);
        p_crEntity = NULL;
    }
}

BOOL verifyCertificateAgainstCRL(
    Certificate_t* p_certificate,
    CR_LIST_t* p_crList)
{
    BOOL failed = FALSE;

    if (!p_certificate || !p_crList)
        return failed = TRUE;

    /* get Issuer CN of Certificate */
    char* issuerCN = NULL;
    failed = (cmpGetObjFromRDNSequence(
            &(p_certificate->tbsCertificate.issuer.choice.rdnSequence),
            oid_commonName, TRUE, &issuerCN));

    while (p_crList && !failed) {
        /* compare Issuer CN*/
        failed =
                (strncmp(issuerCN, p_crList->issuerCN, strlen(issuerCN)) == 0) ?
                        TRUE : FALSE;

        ifFailed {
            /* compare length of INTEGER_t */
            failed =
                    (p_certificate->tbsCertificate.serialNumber.size
                            == p_crList->userID.size) ? TRUE : FALSE;

            /* compare Buffer if sizes match */
            unsigned int i = 0;
            while (failed && i < p_crList->userID.size) {
                failed =
                        (p_certificate->tbsCertificate.serialNumber.buf[i]
                                == p_crList->userID.buf[i]) ? TRUE : FALSE;
                i++;
            }
        }
        p_crList = p_crList->next;
    }
    free(issuerCN);
    return failed;
}

BOOL verifyCertListAgainstCRL(
    CERT_LIST_s* p_certList,
    CR_LIST_t* crList,
    STATUS_s *stat )
{
    BOOL failed = FALSE;

    // no entries in cr list -> so all Certs are valid
    if(!crList){
        return FALSE;
    }
    // no Certs given to verify -> thats unexpected
    if(!p_certList){
        return TRUE;
    }
    while(p_certList){
        Certificate_t* p_certificate = p_certList->crt;
        if (!p_certificate){
            failed = alloc((void**)&p_certificate, sizeof(Certificate_t));
            if(!failed){
                failed = asn1Decode( p_certList->certPtr, p_certList->certSize,
                    &asn_DEF_Certificate, p_certificate, FALSE);
            }
            if(failed){
                ASN_STRUCT_FREE(asn_DEF_Certificate, p_certificate);
                return TRUE;
            }
        }

        /* get Issuer CN of Certificate */
        char* issuerCN = NULL;
        failed = (cmpGetObjFromRDNSequence(
            &(p_certificate->tbsCertificate.issuer.choice.rdnSequence),
            oid_commonName, TRUE, &issuerCN));
        CR_LIST_t* p_crList = crList;
        while (p_crList && !failed) {
            /* compare Issuer CN*/
            failed =
                    (strncmp(issuerCN, p_crList->issuerCN, strlen(issuerCN)) == 0) ?
                            TRUE : FALSE;

            ifFailed {
                /* compare length of INTEGER_t */
                failed =
                        (p_certificate->tbsCertificate.serialNumber.size
                                == p_crList->userID.size) ? TRUE : FALSE;

                /* compare Buffer if sizes match */
                unsigned int i = 0;
                while (failed && i < p_crList->userID.size) {
                    failed =
                            (p_certificate->tbsCertificate.serialNumber.buf[i]
                                    == p_crList->userID.buf[i]) ? TRUE : FALSE;
                    i++;
                }
            }
            p_crList = p_crList->next;
        }
        if(failed){
            ecmCmSetStatus( stat, EXT_IF_ERR_CERT_REVOKED, p_certList->subjectCn, NULL );
            free(issuerCN);
            break;
        }
        free(issuerCN);
        issuerCN = NULL;
        if(!p_certList->crt){
            ASN_STRUCT_FREE(asn_DEF_Certificate, p_certificate);
        }
        p_certList = p_certList->next;
    }
    return failed;
}

BOOL downloadCRL(
    char* uri)
{
    BOOL failed = FALSE;

    if (!uri || 3 > strlen(uri))
        return failed = TRUE;

    char* filename = "downloadedCRL.crl";

    failed = downloadFromURItoDisk(uri, filename);

    return failed;
}

char* getCRLURIfromCertificate(
    Certificate_t* crt)
{
    char* uri = NULL;

    if (!crt || !crt->tbsCertificate.extensions ||
        !crt->tbsCertificate.extensions->list.array)
    {
        return NULL;
    }
    /* get CRL Distribution Points Extension --> 2.5.29.31 */
    const unsigned int oidSize = 4;
    unsigned int authInfoOid[4] = { 2, 5, 29, 31 };
    Extension_t** extArray = (crt->tbsCertificate.extensions->list.array);
    unsigned int i = 0;
    BOOL match = FALSE;
    size_t s = 0;

    while (!match && i < crt->tbsCertificate.extensions->list.count) {
        OBJECT_IDENTIFIER_t *oid = &extArray[i]->extnID;
        unsigned int fixed_arcs[oidSize];
        unsigned int *arcs = fixed_arcs;
        int arc_type_size = sizeof(fixed_arcs[0]);  // sizeof(long)
        int arc_slots = sizeof(fixed_arcs) / sizeof(fixed_arcs[0]); // 4
        int count;  // Real number of arcs.

        count = OBJECT_IDENTIFIER_get_arcs(oid, arcs, arc_type_size, arc_slots);
        if (count != arc_slots) {
            /* if count of OID Args do not match try the next OID */
            ++i;
            continue;
        }

        /* Check if OIDs match */
        match = TRUE;
        unsigned int j = 0;
        while (match && j < oidSize) {
            if (authInfoOid[j] != arcs[j])
                match = FALSE;
            ++j;
        }
        ++i;
    }

    if (!match) {
        dbgPrint(DBG_INFO, "Could not find \"CRL Distribution Points\" OID\n");
        return NULL;
    }

    // get CRL Distribution Points Info as extension value
    UINT8* p_aiVal = extArray[i - 1]->extnValue.buf;
    if (!p_aiVal)
        return NULL;
    UINT32 bufSize = extArray[i - 1]->extnValue.size;

    // Try to decode the Buffer as cRLDistributionPoints
    CRLDistributionPoints_t* temp = NULL;
    BOOL failed = alloc((void **) &temp, sizeof(CRLDistributionPoints_t));
    ifNotFailed {
        if (!asn1Decode(p_aiVal, bufSize, &asn_DEF_CRLDistributionPoints, temp, FALSE)) {
            if (temp->list.count > 0) {
                /* Find Full Name */
                i = 0;
                while (i < temp->list.count &&
                    temp->list.array[i]->distributionPoint->present != DistributionPointName_PR_fullName)
                {
                    ++i;
                }
                if (i < temp->list.count) {
                    GeneralNames_t* generalNames = &temp->list.array[i]->distributionPoint->choice.fullName;
                    if (generalNames->list.count > 0) {
                        i = 0;
                        /* Find Unifrom Resource Identifier */
                        while (i < generalNames->list.count &&
                            generalNames->list.array[i]->present != GeneralName_PR_uniformResourceIdentifier)
                        {
                            ++i;
                        }
                        if (i < generalNames->list.count) {

                            failed = escInt2SizetAndCheckForOv(
                                generalNames->list.array[i]->choice.uniformResourceIdentifier.size, &s);

                            ifNotFailed {

                                uri = malloc(s + 1);
                                if ( NULL != uri ) {
                                    memcpy(uri, generalNames->list.array[i]->choice.uniformResourceIdentifier.buf, s);

                                    // Append trailing 0
                                    uri[s] = 0;
                                }
                            }
                            ASN_STRUCT_FREE(asn_DEF_CRLDistributionPoints, temp);
                            return uri;
                        }
                    }
                }
            }
        }
    }
    ASN_STRUCT_FREE(asn_DEF_CRLDistributionPoints, temp);
    return NULL;
}

BOOL hashCRL(
    CertificateList_t* crl,
    UINT8** digest)
{
    if(!crl || !digest)
        return FALSE;

    BOOL failed = FALSE;

    UINT8* bufferBackup = NULL;
    UINT32 bufferSize = 0;
    Time_t* nextUpdateTimeBackup = NULL;

    // Remove Time and nextUpdataTime from Data before hashing
    if(crl->tbsCertList.thisUpdate.present != Time_PR_NOTHING){
        // Time is a Union with both Times implemented as Octet String, so we do
        // not need to distinguish here
        bufferBackup = crl->tbsCertList.thisUpdate.choice.generalTime.buf;
        bufferSize = crl->tbsCertList.thisUpdate.choice.generalTime.size;
        crl->tbsCertList.thisUpdate.choice.generalTime.buf = NULL;
        crl->tbsCertList.thisUpdate.choice.generalTime.size = 0;
    }
    if(crl->tbsCertList.nextUpdate){
        nextUpdateTimeBackup = crl->tbsCertList.nextUpdate;
        crl->tbsCertList.nextUpdate = NULL;
    }

    UINT8* data = NULL;
    UINT32 datalength = 0;

    // Encode remaining Data
    failed = asn1Encode(&crl->tbsCertList, &asn_DEF_TBSCertList, &data, &datalength, FALSE);

    ifNotFailed{
        *digest = malloc(EscSha1_DIGEST_LEN);
        if(!*digest)
            failed = TRUE;
        ifNotFailed{
            // Hash the data
            failed = EscSha1_Calc(data, datalength, *digest);
            if(failed){
                free(*digest);
                *digest = NULL;
            }
        }
    }

    // Reinsert backup'd Data
    crl->tbsCertList.thisUpdate.choice.generalTime.buf = bufferBackup;
    crl->tbsCertList.thisUpdate.choice.generalTime.size = bufferSize;
    crl->tbsCertList.nextUpdate = nextUpdateTimeBackup;

    free(data);
    data = NULL;
    datalength = 0;

    return failed;
}

BOOL getCrlFromListByLocalId (
    CRL_HASHES_t *lst,
    CRL_ID_e id,
    CRL_HASHES_t **crlPtr )
{
    BOOL failed = FALSE;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !lst || !crlPtr ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        *crlPtr = NULL;
        failed = TRUE;

        while ( lst ) {
            if ( lst->id == id ) {
                *crlPtr = lst;
                failed = FALSE;
                break;
            }

            lst = lst->next;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/
