/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        esc_utils.c

   \brief       Common utilities module

   $Rev: 814 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-28 10:42:47 +0200 (Mo, 28. Aug 2017) $
****************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

#include "../inc/esc_utils.h"
#include "../../ecm/inc/ecm.h"

/* linux headers */
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <dirent.h>

/* OpenSSL */
#include <openssl/crypto.h>

/* common headers */
#include "../inc/esc_common.h"
//#include <ecm.h>
#include "../../cmp/inc/asn1_utils.h"
#include "../inc/esc_utils.h"
#ifdef CONFIG_SIMPLE_DEBUG
# include "../inc/esc_debug_simple.h"
#else
# include "../inc/esc_debug.h"
#endif /* CONFIG_SIMPLE_DEBUG */



#include "../../crypto/inc/crypto.h"
#include "../../cmp/inc/cmp.h"
#include "../../ecm/inc/ecm_utils.h"


#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

void remTrailingBlanks ( char **str );

BOOL escCheckSizetForOv ( size_t s ) {
    BOOL failed = FALSE;

    if ( SIZE_MAX == s ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Value is SIZE_MAX\n", DBG_FL );
        failed = TRUE;
    }

    return ( failed );
}

BOOL escInt2SizetAndCheckForOv ( int i, size_t *s ) {
    BOOL failed = FALSE;

    /* check for negative and 0 */
    if ( i <= 0 ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Cannot convert int to size_t\n", DBG_FL );
        failed = TRUE;
    }

    /* convert to size_t */
    ifNotFailed {
        *s = (size_t) i;
    }

    /* check for size_t max to avoid overflow if +1 */
    ifNotFailed {
        if ( SIZE_MAX == *s ) {
            dbgPrint ( DBG_ERR, DBG_FLS "Value is SIZE_MAX\n", DBG_FL );
            failed = TRUE;
        }
    }

    return ( failed );
}

char *escId2StringLookup (
    const STR_LOOKUP_t *tab,
    const UINT32 tabSize,
    UINT32 id )
{
    int i;

    /* sanity checks */
    if ( !tab ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        return ( "" );
    }

    /* check parameter */
    if ( id > tabSize ) {
        return ( "" );
    }

    for ( i = 0; i < tabSize; i++ ) {
        if ( id == (&tab[i])->val ) {
            return ( (char *)(&tab[i])->str );
        }
    }

    return ( "" );
}

BOOL escString2IdLookup (
    const STR_LOOKUP_t *tab,
    const UINT32 tabSize,
    const char* str,
    UINT32 *id )
{
    int i;
    size_t len = 0;
    char *tmpStr = NULL;
    BOOL success = FALSE;
    BOOL failed = FALSE;

    /* sanity checks */
    if ( !tab || !str || !id ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {
        len = strlen ( str );
        failed = escCheckSizetForOv( len );
    }

    ifNotFailed {
        failed = escMalloc ((void **)&tmpStr, len + 1);
    }

    ifNotFailed {
        memcpy ( tmpStr, str, len);

        remTrailingCrLf( &tmpStr );
        remTrailingBlanks( &tmpStr );

        for ( i = 0; i < tabSize; i++ ) {

            if ( strncmp ( tmpStr, (&tab[i])->str, strlen ((&tab[i])->str )) == 0 ) {
                *id = (&tab[i])->val;
                success = TRUE;
                break;
            }
        }
    }

    escFreePtr( (UINT8 **) &tmpStr );

    return ( (success == TRUE && failed == FALSE) ? FALSE : TRUE );
}

#ifdef UNIT_TESTING
#undef free
#endif
void escFreeDirent (
    void **dir,
    UINT32 len )
{
    int i;
    struct dirent **item;
    struct dirent **src = (struct dirent **) dir;

    /* free dirent list */
    if (( src != NULL ) && ( len )) {
        item = src;
        for ( i = 0; i < len; i++, item++ ) {
            free ( *item );
        }
        free ( src );
    }
}

#ifdef UNIT_TESTING
#define free(ptr) _test_free(ptr, __FILE__, __LINE__)
#endif

void escFreePtr ( UINT8 **ptr ){

    if ( ptr ) {
        if ( *ptr ) {
            free ( *ptr );
            *ptr = NULL;
        }
    }
}

BOOL escMalloc (
    void **ptr,
    int size )
{
    BOOL failed = FALSE;

    if ( ptr == NULL || size <= 0 ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        return ( TRUE );
    }

    *ptr = malloc( size + 1 );
    if (!(*ptr)) {
        dbgPrint ( DBG_ERR, DBG_FLS "malloc() failed\n", DBG_FL );
        failed = TRUE;
    } else {
        memset(*ptr, 0x0, size + 1);
    }

    return ( failed );
}

/** Function that removes trailing blanks */
void remTrailingBlanks ( char **str )
{
    char *ptr;

    /* sanity checks */
    if ( !str ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        return;
    }

    ptr = *str + strlen ( *str ) - 1;
    while ( (*ptr == '\t') || (*ptr == ' ') ) {
       *ptr-- = '\0';
    }
}

/** Function that removes leading blanks */
void remLeadingBlanks ( char **str ) {

}

/** Function that removes characters */
void escRemoveChars ( UINT32 flags, char **str )
{
    /* remove crlf first */
    if ( flags & ESC_REM_TRAIL_CRLF ) {
        remTrailingCrLf( str );
    }

    if ( flags & ESC_REM_TRAIL_BLANKS ) {
        remTrailingBlanks( str );
    }

    if ( flags & ESC_REM_LEAD_BLANKS ) {
        remLeadingBlanks ( str );
    }
}

BOOL copyParameterValue (
    const char *str,
    char **buf )
{
    dbgIn( __func__ );

    if ( str == NULL || buf == NULL ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        return ( TRUE );
    }

    /* Free buffer if already allocated.
       This is necessary if e.g. the modification script is used that
       re-writes some parameters */
    escFreePtr( (UINT8 **) buf );

    if ( TRUE == escCheckSizetForOv( strlen ( str ))) {
        return ( TRUE );
    }

    *buf = calloc (1, strlen ( str ) + 1);

    if ( NULL == *buf ) {
        return ( TRUE );
    }

    /* remove trailing CR and or LF from str */
    escRemoveChars( ESC_REM_TRAIL_BLANKS | ESC_REM_TRAIL_CRLF, (char **) &str );
    memcpy ( *buf, str, (strlen ( str )));

    dbgOut( __func__ );

    return ( FALSE );
}

BOOL str2Bool (
    const char *str,
    BOOL *boolVal )
{
    BOOL failed = FALSE;
    char *tmpStr = NULL;

    dbgIn( __func__ );

    if ( str == NULL || boolVal == NULL ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        return ( TRUE );
    }

    if ( TRUE == escCheckSizetForOv( strlen ( str ))) {
        return ( TRUE );
    }

    tmpStr = calloc(1, strlen(str)+1);
    if ( !tmpStr ) {
        return ( TRUE );
    }
    strncpy (tmpStr, str, strlen ( str ));

    /* remove trailing CR and or LF from str */
    escRemoveChars( ESC_REM_TRAIL_BLANKS | ESC_REM_TRAIL_CRLF, (char **) &tmpStr );

    if ( strncmp ( tmpStr, "true", strlen ( "true") ) == 0 ) {
        *boolVal = TRUE;
    } else if ( strncmp ( tmpStr, "false", strlen ( "false" ) ) == 0 ) {
        *boolVal = FALSE;
    } else {
        failed = TRUE;
    }

    escFreePtr( (UINT8 **)&tmpStr );

    dbgOut( __func__ );

    return ( failed );
}

void remTrailingCrLf ( char **str )
{

    /* sanity checks */
    if ( !str ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        return;
    }

    /* set pointer to last character of str */
    char *ptr = *str + strlen ( *str ) - 1;
    while ( ( *ptr == '\n') || ( *ptr == '\r') ) {
        *ptr-- = '\0';
    }
}


#ifdef UNIT_TESTING
#undef malloc
#endif
BOOL escCsvList2StringList (
    char *src,
    STRING_LIST_s **dst )
{
    BOOL failed = FALSE;
    char *tok = NULL;
    STRING_LIST_s **sp = NULL;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !src || !dst ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        tok = src;
        remTrailingCrLf ( &tok );

        failed = alloc ((void **)dst, sizeof ( STRING_LIST_s ));
    }

    ifNotFailed {

        sp = dst;

        tok = strtok ( tok, "," );

        while ( tok ) {

            if ( !(*sp) ) {
                failed = alloc ((void **)sp, sizeof (STRING_LIST_s));
                ifFailed {
                    break;
                }
            }

            (*sp)->s = strdup ( tok );

            tok = strtok ( NULL, ",");
            sp = &(*sp)->next;
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}
#ifdef UNIT_TESTING
#define malloc(num) _test_malloc(num, __FILE__, __LINE__)
#endif /* UNIT_TESTING */

void escFreeStringList (
    STRING_LIST_s **lst )
{
    STRING_LIST_s *next, *ptr;

    dbgIn ( __func__ );

    if ( lst ) {
        if ( *lst ) {

            ptr = *lst;
            while ( ptr ) {
                next = ptr->next;
                escFreePtr( (UINT8 **)&ptr->s );
                escFreePtr( (UINT8 **)&ptr );
                ptr = next;
            }
        }
    }

    dbgOut ( __func__ );

    return;
}

/* processes the configuration file */
BOOL readConfFile (
    CONF_FILES_t *confFile,
    void *par )
{
    BOOL failed = FALSE;
    FILE *fp = NULL;
    char currLine[MAX_CONF_FILE_LINE_LEN];
    char *linePtr = "n/a";
    char nameBuf[64];
    int numRead;
    int i;

    ECM_PARAMS_t *params = (ECM_PARAMS_t *) par;

    dbgIn(__func__);

    if ( confFile == NULL || params == NULL ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        for ( i = 0; i < CONF_FILE_LIST_SIZE; i++ ) {
            if ( confFile[i].name == NULL ) {
                continue;
            }

            fp = fopen ( confFile[i].name, "r" );

            if ( fp ) {
                confFile[i].used = TRUE;
                break;
            }
        }

        if ( !fp ) {
            dbgPrint ( DBG_ERR,
                DBG_FLS "No config file available, QUIT\n", DBG_FL );
            return ( TRUE );
        }
    }

    ifNotFailed {

        /* read until EOF is detected */
        while ( (!feof ( fp )) && ( failed == FALSE ) ) {

            linePtr = fgets ( currLine, MAX_CONF_FILE_LINE_LEN, fp );

            if ( linePtr ) {

                if ( *linePtr == '\n' ||
                     *linePtr == '\r' ||
                     *linePtr == '#' )
                {
                    /* skip empty lines or comments */
                    continue;

                } else {

                    /* extract the parameter name and make sure at least
                       one blank as separator between name and value
                       is available */
                    numRead = sscanf ( linePtr, "%s", nameBuf);

                    if ( numRead != 1 ) {
                        dbgPrint ( DBG_ERR, DBG_FLS
                            "Failed to read configuration file line\n", DBG_FL );
                        break;
                    }

                    /* skip all leading blanks or tabs */
                    while ( *linePtr == ' ' || *linePtr == '\t') {

                        linePtr++;
                    }

                    linePtr = linePtr + strlen ( nameBuf );

                    /* skip all trailing blanks or tabs */
                    while ( *linePtr == ' ' || *linePtr == '\t') {

                        linePtr++;
                    }

                    /* if empty line */
                    if ( *linePtr == '\r' || *linePtr == '\n' ) {

                        //retVal = EcmRetFailed;
                        continue;

                    } else {

                        failed = handleConfigFileLine( nameBuf, linePtr, params );
                    }
                }
            }
        }
    }

    /* log error on failure */
    ifFailed {

        dbgPrint ( DBG_ERR, DBG_FLS
            "Config file %s entry invalid: %s %s\n",
            DBG_FL, confFile, nameBuf, linePtr );
    }

    /* check if there are all mandatory entries processed */
    ifNotFailed {
        failed = checkMandatoryOptions( params );
    }

    /* set default parameters */
    ifNotFailed {
        failed = setDefaultOptions( params );
    }

    if ( fp ) {
        fclose ( fp );
    }

    dbgOut( __func__ );

    return ( failed );
}

void escDaemonize (
    void )
{
    int i;
    pid_t pid = 45;

    /* terminate parent process */
    if ((pid = fork()) != 0) {
        exit(EXIT_FAILURE);
    }

    /* become session leader */
    if ( setsid() < 0) {
        exit(EXIT_FAILURE);
    }

    /* ignor SIGHUP */
    signal(SIGHUP, SIG_IGN);

    /* terminate child process */
    if ((pid = fork()) != 0) {
        exit(EXIT_FAILURE);
    }

    /* change working director */
    if ( chdir ("/") < 0 ) {
        exit(EXIT_FAILURE);
    }

    /* set bitmask */
    umask(0);

    /* close all file descriptors of our parent */
    for ( i = sysconf(_SC_OPEN_MAX); i > 0; i-- ) {
        close(i);
    }
}

BOOL escGetFileSizeByFp (
    FILE *fp,
    UINT32 *size )
{
    BOOL failed = FALSE;

    dbgIn ( __func__ );

    /* sanity checks */
    if ( !fp || !size ) {
        dbgPrint ( DBG_ERR, DBG_FLS "Invalid parameter!\n", DBG_FL );
        failed = TRUE;
    }

    ifNotFailed {

        if ( fseek ( fp, 0 , SEEK_END ) != 0 ) {
            dbgPrint ( DBG_ERR, DBG_FLS "Failed to read file size\n", DBG_FL );
            failed = TRUE;
        } else {
            *size = ftell ( fp );
            rewind ( fp );
        }
    }

    dbgOut ( __func__ );

    return ( failed );
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/

