/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        esc_debug.c

   \brief       Common debug module

   $Rev: 812 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-25 15:34:36 +0200 (Fr, 25. Aug 2017) $
****************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

/* linux headers */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <pthread.h>

/* local header */
#include "../inc/esc_common.h"
#include "../inc/esc_debug.h"

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif


/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** Maximum line length of a debug message */
#define MAX_LINE_LEN 128

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/** Debug Mutex */
pthread_mutex_t dbgMtx;

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

/** Maxium size of file pointer array */
#define MAX_FPS_ARRAY_SIZE 2
/** Array with file pointers we log to*/
static FILE *fps[MAX_FPS_ARRAY_SIZE];
/** Array with va_list arguments, one for each file pointer to log to
    (Required for 64bit compatibility */
static va_list args[MAX_FPS_ARRAY_SIZE];
/** global debug mask */
static unsigned int dbgMask;
/** Flag to signal initialization finished */
static BOOL initDone = FALSE;

/** Lookup table mapping log level to debug mask */
#if defined CONFIG_ECM
static UINT32 logMapping[8][2] = {
    { 0, DBG_ERR },
    { 1, DBG_ERR | DBG_CERT },
    { 2, DBG_ERR | DBG_CERT | DBG_STATE },
    { 3, DBG_ERR | DBG_CERT | DBG_STATE | DBG_DATA },
    { 4, DBG_ERR | DBG_CERT | DBG_STATE | DBG_DATA | DBG_INFO },
    { 5, DBG_ERR | DBG_CERT | DBG_STATE | DBG_DATA | DBG_INFO | DBG_API },
    { 6, DBG_ERR | DBG_CERT | DBG_STATE | DBG_DATA | DBG_INFO | DBG_API |
         DBG_ASN1 },
    { 7, DBG_ERR | DBG_CERT | DBG_STATE | DBG_DATA | DBG_INFO | DBG_API |
         DBG_ASN1 | DBG_TRACE }
};
#elif defined CONFIG_LRA
static UINT32 logMapping[6][2] = {
    { 0, DBG_ERR },
    { 1, DBG_ERR | DBG_CERT },
    { 2, DBG_ERR | DBG_CERT | DBG_DATA },
    { 3, DBG_ERR | DBG_CERT | DBG_DATA | DBG_INFO },
    { 4, DBG_ERR | DBG_CERT | DBG_DATA | DBG_INFO | DBG_ASN1 },
    { 5, DBG_ERR | DBG_CERT | DBG_DATA | DBG_INFO | DBG_ASN1 | DBG_TRACE }
};
#elif defined CONFIG_LRA_TEST_CLIENT
static UINT32 logMapping[7][2] = {
    { 0, DBG_ERR },
    { 1, DBG_ERR | DBG_TEST },
    { 2, DBG_ERR | DBG_TEST | DBG_TESTD },
    { 3, DBG_ERR | DBG_TEST | DBG_TESTD | DBG_STATE },
    { 4, DBG_ERR | DBG_TEST | DBG_TESTD | DBG_STATE | DBG_DATA },
    { 5, DBG_ERR | DBG_TEST | DBG_TESTD | DBG_STATE | DBG_DATA | DBG_INFO },
    { 6, DBG_ERR | DBG_TEST | DBG_TESTD | DBG_STATE | DBG_DATA | DBG_INFO | DBG_TRACE },
};

#endif /* CONFIG_ECM */

/** Lookup table mapping debug mask value to string representation */
static const STR_LOOKUP_t maskStrTab[] = {
    { DBG_ENTRY, "ENTRY" },
    { DBG_EXIT,  "EXIT " },
    { DBG_DATA,  "DATA " },
    { DBG_INFO,  "INFO " },
    { DBG_ASN1,  "ASN1 " },
    { DBG_ERR,   "ERROR" },
    { DBG_CERT,  "CERT " },
    { DBG_STATE, "STATE" },
    { DBG_API,   "API  " },
    { DBG_TEST,  "TEST " },
    { DBG_TRACE, "TRACE" },
};

static char *mask2Str ( 
    UINT32 lvl )
{
    int i, max = sizeof ( maskStrTab ) / sizeof ( STR_LOOKUP_t );

    for ( i = 0; i < max; i++ ) {
        if ( lvl == maskStrTab[i].val ) {
            return ( (char *)maskStrTab[i].str );
        }
    }

    return ( "" );
}

BOOL escInitLogging (
    UINT32 logLevel,
    BOOL quiet,
    char *fileName
    )
{
    BOOL failed = FALSE;
    int i = 0;

    memset ( fps, 0x0, sizeof ( fps ));

    /* map log level to local debug mask */
    dbgMask = logMapping[logLevel][1];

    /* open log file */
    if ( !(fps[i++] = fopen ( fileName, "a")) ) {
        fprintf ( stderr, "ERROR: Failed to open / create log file, exit\n");
        return ( TRUE );
    }

    /* set the additional FILE pointers we want to log to */
    if ( quiet == FALSE ) {

        /* don't be quiet, so log to stdout as well */
        fps[i++] = stdout;
    }

    if ( pthread_mutex_init ( &dbgMtx, NULL ) ) {
        fprintf ( stderr, "ERROR: Failed to initialize debug mutex, exit\n" );
        exit ( EXIT_FAILURE );
    }

    if ( failed == FALSE ) {
        initDone = TRUE;
    }

    return ( failed );
}

BOOL escCloseLogging (
    )
{
    int i;

    if ( initDone ) {
        for ( i = 0; i < MAX_FPS_ARRAY_SIZE; i++ ) {
            if ( fps[i] != NULL && fps[i] != stdout ) {
                fclose ( fps[i] );
            }
            fps[i] = 0;
        }

        pthread_mutex_destroy ( &dbgMtx );
    }

    return ( FALSE );
}

# ifndef DBG_IN_DEFINED
inline void 
dbgIn (
    const char *funcName )
{
    dbgPrint ( DBG_TRACE, "Enter %s()\n", funcName );
}
/** Define to signal dbgIn() is already defined */
# define DBG_IN_DEFINED
# endif

# ifndef DBG_OUT_DEFINED
inline void 
dbgOut ( 
    const char *funcName )
{
    dbgPrint ( DBG_TRACE, "Leave %s()\n", funcName );
}
/** Define to signal dbgOut() is already defined */
# define DBG_OUT_DEFINED
# endif

# ifndef DBG_PRINT_DEFINED
inline void 
dbgPrint (
    const UINT32 _lvl_,
    const char* _str_,
    ... )
{
#ifdef UNIT_TESTING
    return;
#endif

    int i;
    time_t t;
    struct tm *tm;
    char str[MAX_LINE_LEN] = { 0 };
    size_t cnt = 0;
#if defined CONFIG_ECM
    char *app = "ECM";
#elif defined CONFIG_LRA
    char *app = "LRA";
#elif defined CONFIG_LRA_TEST_CLIENT
    char *app = "LRATC";
#endif /* CONFIG_ECM */

    pthread_mutex_lock( &dbgMtx );

    va_start ( args[0], _str_);
    va_copy ( args[1], args[0]);

    /* Print errors to stderr if logging is not initialized yet */
    if ( initDone == FALSE ){
        memset ( fps, 0x0, sizeof ( fps ));
        fps[0] = stderr;
        dbgMask = DBG_ERR;
    }

    if( dbgMask & _lvl_ ) {

        t = time ( NULL );
        tm = localtime ( &t );
        cnt += strftime ( str, sizeof ( str ), "%F %T", tm );
        /*
        cnt += snprintf ( &str[cnt], sizeof ( str ) - cnt, " %s (%d) %s: ",
            app, getpid(), mask2Str ( _lvl_ ));
            */
        cnt += snprintf ( &str[cnt], sizeof ( str ) - cnt, " %s (%lu) %s: ",
            app, pthread_self(), mask2Str ( _lvl_ ));

        for ( i = 0; i < 2; i++  ) {
            if(fps[i]){
                fprintf ( fps[i], "%s", str );
                vfprintf ( fps[i], _str_, args[i] );
                fflush ( fps[i] );
            }
        }
    }

    for ( i = 0; i < MAX_FPS_ARRAY_SIZE; i++ ) {
        va_end ( args[i] );
    }

    pthread_mutex_unlock( &dbgMtx );
}

/** Define to signal dbgPrint() is already defined */
# define DBG_PRINT_DEFINED
# endif

# ifndef DBG_PRINT_BYTE_ARRAY
inline void
dbgPrintByteArray (
    const UINT32 _lvl_,
    UINT32 _size_,
    char* _u8ptr_ )
{
    char *_dataPtr_ = _u8ptr_;
    int i;

    int bufSize = _size_ * 2 + 3;
    size_t cnt = 0;
    char *buf = malloc ( bufSize );

    if ( buf ) {

        cnt += snprintf ( &buf[cnt], bufSize - cnt, "\n" );

        for ( i = 1; i <= _size_; i++, _dataPtr_++) {

            cnt += snprintf (
                &buf[cnt], bufSize - cnt, "%02X", (UINT8) *_dataPtr_ );
        }

        cnt += snprintf ( &buf[cnt], bufSize - cnt, "\n" );
        buf[cnt] = 0;

        dbgPrint ( _lvl_, buf );

        free ( buf );
    }
}
# endif /* DBG_PRINT_BYTE_ARRAY */

# ifndef DBG_PRINT_BLOCK_DEFINED
inline void 
dbgPrintBlock (
    const UINT32 _lvl_,
    UINT32 _size_, 
    char* _u8ptr_ )
{
    char *_dataPtr_ = _u8ptr_;
    int i;

    size_t cnt = 0;
    int bufSize = _size_ * 3 + _size_ / PRINT_BLOCK_NL_AFTER_BYTE + 3;
    char *buf = malloc ( bufSize );

    if ( buf ) {

        cnt = snprintf ( &buf[cnt], bufSize - cnt, "\n" );

        for ( i = 1; i <= _size_; i++, _dataPtr_++) {

            cnt += snprintf (
                &buf[cnt], bufSize - cnt, "%02X ", (UINT8) *_dataPtr_ );


            if (((i%PRINT_BLOCK_NL_AFTER_BYTE)==0) && (i>0)) {
                cnt += snprintf ( &buf[cnt], bufSize - cnt, "\n" );
            };
        }

        cnt += snprintf ( &buf[cnt], bufSize - cnt, "\n" );

        dbgPrint ( _lvl_, buf );

        free ( buf );
    }
}
/** Define to signal dbgPrintBlock() is already defined */
# define DBG_PRINT_BLOCK_DEFINED
# endif

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/

