/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        base64.c

   \brief       Base 64 encoder / decoder

   $Rev: 812 $

   This code is based on an implementation found on stackoverflow.com [1].
   The code is adapted to ESCRYPTs data types, coding style and
   modified in this way that it compiles without warnings.

   See OpenSource attribution document for details.

   [1] http://stackoverflow.com/questions/342409/how-do-i-base64-encode-decode-in-c
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-08-25 15:34:36 +0200 (Fr, 25. Aug 2017) $
****************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *
****************************************************************************/

/* linux headers */
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

/* common headers */
#include <esc_common.h>
#include <esc_debug.h>

/* local header */
#include <base64.h>

#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

static char encoding_table[] = {
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
    'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
    'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
    'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
    'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
    'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
    'w', 'x', 'y', 'z', '0', '1', '2', '3',
    '4', '5', '6', '7', '8', '9', '+', '/'
};

static char *decoding_table = NULL;
static int mod_table[] = {0, 2, 1};

char *escBase64Encode(
    const UINT8 *data,
    const UINT32 input_length,
    UINT32 *output_length)
{
    size_t len = 4 * ((input_length + 2) / 3);

    char *encoded_data = malloc(len);
    if (encoded_data == NULL) {
        return NULL;
    }

    int i = 0;
    int j = 0;

    while ( i < input_length ) {

        UINT32 octet_a = i < input_length ? (unsigned char)data[i++] : 0;
        UINT32 octet_b = i < input_length ? (unsigned char)data[i++] : 0;
        UINT32 octet_c = i < input_length ? (unsigned char)data[i++] : 0;

        // create 24bit val from 3x8bit
        UINT32 triple = ((octet_a << 0x10) + (octet_b << 0x08) + octet_c) & 0xFFFFFF;

        encoded_data[j++] = encoding_table[( (triple >> (3 * 6) ) & 0x3F )];
        encoded_data[j++] = encoding_table[( (triple >> (2 * 6) ) & 0x3F )];
        encoded_data[j++] = encoding_table[( (triple >> (1 * 6) ) & 0x3F )];
        encoded_data[j++] = encoding_table[( (triple >> (0 * 6) ) & 0x3F )];
    }

    // Add Padding
    for (i = 0; i < mod_table[input_length % 3]; i++) {
        encoded_data[len - 1 - i] = '=';
    }

    *output_length = len;
    return ( encoded_data );
}

BOOL decodePEM(UINT8* pem, UINT32 pemSize,
		UINT8** decodedData, UINT32* decodedDataSize){

	if(!pem || pemSize < 1 || !decodedData || !decodedDataSize ){
	    /*
		*decodedData = NULL;
		*decodedDataSize = 0;
		*/
		return TRUE;
	}

	UINT32 dataSize = pemSize;

	//find End of Header
	UINT8* pemStart = NULL;
	BOOL winLineBreak = FALSE;
	pemStart = (UINT8*) strstr((char*)pem, "-\n");
	if(!pemStart){
		pemStart = (UINT8*) strstr((char*)pem, "-\r\n");
		if(pemStart) winLineBreak = TRUE;
	}

	// exclude Header, if found
	if(pemStart){
		// Increase pemStart to first base64 (after Newline)
		pemStart += 2;
		if(winLineBreak) pemStart++;
		dataSize = pemSize - (pemStart-pem);
	}
	else
		pemStart = pem;

	// find length of Trailer
	UINT8* pemEnd = NULL;
	if(!winLineBreak)
		pemEnd = (UINT8*) strstr((char*)pem, "\n-");
	else
		pemEnd = (UINT8*) strstr((char*)pem, "\r\n-");

	// exclude Trailer, if found
	if(pemEnd){
		//decrease pemEnd to last base64 position
//		pemEnd += 1;
		dataSize -= (pemSize - (pemEnd+1-pem) );
	}

	*decodedData = escBase64Decode(pemStart, dataSize, decodedDataSize);
	if(*decodedData && *decodedDataSize > 0)
		return FALSE;
	else
		return TRUE;
}

UINT8 *escBase64Decode(const UINT8* data, const UINT32 input_length,
		UINT32 *output_length) {

	unsigned long t, x, y, z;
	UINT8 c;
	int g;

	if (!data || input_length == 0 || !output_length)
		return NULL;

	if (decoding_table == NULL) {
		build_decoding_table();
	}

	// allocate Buffer - the output_length will be < input_length
	// --> use input_length as upper bound
	UINT8* decoded_data = malloc(input_length);
	if (!decoded_data) {
		return NULL;
	}
	memset(decoded_data, 0x00, input_length);
	*output_length = input_length;

	g = 3;
	for (x = y = z = t = 0; x < input_length; x++) {
		c = decoding_table[data[x] & 0xFF];
		if (c == 255)
			continue;
		/* the final = symbols are read and used to trim the remaining bytes */
		if (c == 254) {
			c = 0;
			/* prevent g < 0 which would potentially allow an overflow later */
			if (--g < 0) {
				free(decoded_data);
				return decoded_data = NULL;
			}
		} else if (g != 3) {
			/* we only allow = to be at the end */
			free(decoded_data);
			return decoded_data = NULL;
		}

		t = (t << 6) | c;

		if (++y == 4) {
			if (z + g > *output_length) {
				free(decoded_data);
				return decoded_data = NULL;
			}
			decoded_data[z++] = (unsigned char) ((t >> 16) & 255);
			if (g > 1)
				decoded_data[z++] = (unsigned char) ((t >> 8) & 255);
			if (g > 2)
				decoded_data[z++] = (unsigned char) (t & 255);
			y = t = 0;
		}
	}
	if (y != 0) {
		free(decoded_data);
		return decoded_data = NULL;
	}
	*output_length = z;

	// reallocate Buffer to actual size
	decoded_data = realloc(decoded_data, *output_length);

	return (decoded_data);
}

void build_decoding_table ()
{
    decoding_table = malloc(256);
    memset(decoding_table, 0xFF, 256);
    int i = 0;
    for (i = 0; i < 64; i++) {
        decoding_table[(unsigned char) encoding_table[i]] = i;
    }
    // mark "=" (0x3D) as special character
    decoding_table[0x3D] = 254;
}

void base64_cleanup() {
    if ( decoding_table ) {
        free(decoding_table);
        decoding_table = NULL;
    }
}

/***************************************************************************
 * 6. END                                                                  *
 ***************************************************************************/

