/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        esc_debug.h

   \brief       Common debug module

   ***

   $Rev: 937 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2019-02-20 16:27:46 +0100 (Mi, 20. Feb 2019) $
****************************************************************************/

#ifndef __ESC_DEBUG_H__
#define __ESC_DEBUG_H__

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/
#include "esc_common.h"

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/** Flag to print function entry - deprecated, DBG_TRACE is used */
#define DBG_ENTRY ((UINT32)(1 << 0))
/** Flag to print function exit - deprecated, DBG_TRACE is used */
#define DBG_EXIT  ((UINT32)(1 << 1))
/** Flag to print data */
#define DBG_DATA  ((UINT32)(1 << 2))
/** Flag to print information */
#define DBG_INFO  ((UINT32)(1 << 3))
/** Flag to print ASN1 XML representation */
#define DBG_ASN1  ((UINT32)(1 << 4))
/** Flag to print error messages */
#define DBG_ERR   ((UINT32)(1 << 5))
/** Flag to print certificate related information */
#define DBG_CERT  ((UINT32)(1 << 6))
/** Flag to print state changes */
#define DBG_STATE ((UINT32)(1 << 7))
/** Flag to print API calls */
#define DBG_API   ((UINT32)(1 << 8))
/** Flag to print TEST calls */
#define DBG_TEST  ((UINT32)(1 << 9))
/** Flag to print trace information */
#define DBG_TRACE ((UINT32)(1<< 10))
/** Flag to print test data */
#define DBG_TESTD ((UINT32)(1<< 11))

/** Defines after how much printed bytes of a data block a new line is used */
#define PRINT_BLOCK_NL_AFTER_BYTE 16

/** Define string with format specifier to print __FILE__:__LINE__ */ 
#define DBG_FLS "%s:%d "
/** Define arguments for __FILE__:__LINE__ */
#define DBG_FL __FILE__, __LINE__

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. FUNCTION PROTOTYPES                                                  *
 ***************************************************************************/

/**
This function initializes the logging facilities.

\param[in] logLevel 
    Log level that should be used.
\param[in] quiet
    Flag to signal if the application should be quiet (TRUE).
    In this case only file logging is enabled,
    logging to stdout is additionally enabled otherwise (FALSE).
\param[in] fileName 
    Name of the log file.

\return #BOOL
\retval #FALSE on success
\retval #TRUE  on failure

*/
BOOL 
escInitLogging (
    UINT32 logLevel,
    BOOL quiet,
    char *fileName );

/**
This function closes the logging facilities.
*/
BOOL 
escCloseLogging (
    void );

/**
This function logs the function name on entry.

\param[in] funcName 
    Name of the function that should be printed.
*/
// Anjani inline void dbgIn (
extern void dbgIn (
    const char *funcName );

/**
This function logs the function name on exit.

\param[in] funcName 
    Name of the function that should be printed.
*/
// Anjani inline void dbgOut (
extern void dbgOut (
    const char *funcName );

/**
This function logs the given string.

\param[in] _lvl_
    Reference debug level.
    String is logged only if this level is set in the global debug mask.
\param[in] _str_
    String to log.
*/
// Anjani inline void dbgPrint (
extern void dbgPrint (
    const UINT32 _lvl_,
    const char* _str_, ... ); 

/**
This function logs the given data block in hex representation.

\param[in] _lvl_
    Reference debug level.
    Data is logged only if this level is set in the global debug mask.
\param[in] _size_
    Size of the data block.
\param[in] _u8ptr_
    Pointer to data.
*/
extern void
dbgPrintBlock (
    const UINT32 _lvl_,
    UINT32 _size_, 
    char* _u8ptr_ );

/**
This function logs the given data as contiguous byte array

\param[in] _lvl_
    Reference debug level.
    Data is logged only if this level is set in the global debug mask.
\param[in] _size_
    Size of the data block.
\param[in] _u8ptr_
    Pointer to data.
*/
extern  void
dbgPrintByteArray (
    const UINT32 _lvl_,
    UINT32 _size_,
    char* _u8ptr_ );

/***************************************************************************
 * 6. MACRO FUNCTIONS                                                      *
 ***************************************************************************/

/***************************************************************************
 * 7. END                                                                  *
 ***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* __ESC_DEBUG_H__ */
