/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        cmp_utils.h

   \brief       Utilities module of CMP library

   ***

   $Rev: 780 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-07-28 15:40:59 +0200 (Fr, 28. Jul 2017) $
****************************************************************************/

#ifndef __CMP_UTILS_H__
#define __CMP_UTILS_H__

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

#include "cmp.h"
#include "../../common/inc/esc_common.h"
#include "asn1_utils.h"
#include "../asn1-2005-multi-file-os-asn1c/PKIStatusInfo.h"
#include "../asn1-2005-multi-file-os-asn1c/CertResponse.h"
#include "../asn1-2005-multi-file-os-asn1c/PKIBody.h"


/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

/** Enumeration of certificate OID sources */
typedef enum {
    issuer = 0,
    subject,
} CRT_OID_SRC_e;

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. FUNCTION PROTOTYPES                                                  *
 ***************************************************************************/

/**
This function is used to check the device certificate

\param[in] cmpData 
    Pointer to CMP data
\param[in] checkForRenew
    Flag to signal if renew threshold should be considered or not
\param[in] checkForCertValidity
    Flag to check the validity of the certificate or not

\param[out] cmpData 
    Updated CMP data

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
cmpCheckCert (
    CMP_DATA_s *cmpData,
    BOOL checkForRenew,
    BOOL checkForCertValidity);

/**
This function is used to verify the signatures of all certificates available
in the certificate list.

\param[in] certList
    Pointer to Certificate List
\param[in] stat
    Pointer to status structure to record errors
\param[in] stopOnError
    Flag to signal if verification of the chain will be aborted after the
    the first verification failed
\param[in] quiet
    Flag to suppress (TRUE) debug output or not (FALSE)

\param[out] certList
    Pointer to Certificate List

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL 
cmpVerifyCertSignatures (
    CERT_LIST_s *certList,
    STATUS_s *stat,
    BOOL stopOnError,
    BOOL quiet );

/**
This function is used to verify the Issuer of all certificates available
in the certificate list against a predefined wildcard string.

\param[in] cmpData
    Pointer to CMP_DATA_s data

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
cmpVerifyIssuer (
        CMP_DATA_s *cmpData );

/**
This function is used to match a String with optional wildcards against
another String.

\param[in] wildcardIssuer
    String with permitted Issuer with optional wildcards

\param[in] issuerCN
    Actual IssuerCN

\return #BOOL
\retval #TRUE on match
\retval #FALSE on not matching

*/
BOOL match(char *wildcardIssuer, char *issuerCN);


/**
This function is used to retrieve the issuer and subject common name (CN)
from a encoded certificate.

\param[in] cert
    Pointer to encoded certificate
\param[in] certSize
    Size of the encoded certificate

\param[out] issuerCn
    Pointer where issuer CN will be stored
\param[out] subjectCn
    Pointer where subject CN will be stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL cmpGetIssuerAndSubjectCnFromEncCert (
    UINT8 *cert,
    UINT32 certSize,
    char **issuerCn,
    char **subjectCn );

/**
This function is used to free CMP specific data from the global structure

\param[in] cmp
    Pointer to CMP data to free

\param[out] cmp
    Updated CMP data structure

*/
BOOL freeCmpData (
    CMP_DATA_s *cmp );

// Anjani 21-03-2017 #ifdef CONFIG_ECM

/**
This function is used to free the certificate list

\param[in] lst
    Pointer to certificate list

\param[out] lst
    Pointer to updates certificate list

\return none

*/
BOOL freeCertList (
    CERT_LIST_s **lst );

/**
This function is used to check the certificate list if it contains a
complete certificate chain

\param[in] lst
    Pointer to certificate list
\param[in] stat
    Pointer to status structure to record errors
\param[in] phony
    Flag to signal if error message will be logged if chain is incomplete,
    yes if TRUE, no otherwise.

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL checkCertListForCompleteChain (
    CERT_LIST_s *lst,
    STATUS_s *stat,
    BOOL phony );
	
	// Anjani 21-03-2017#endif /* CONFIG_ECM */


/**
This function is used to retrieve the device certificate from the
certificate list.

\param[in] crtLst
    Complete certificate list

\param[out] devCrt
    Pointer to device certificate

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
cmpGetDeviceCertFromCertList (
    CERT_LIST_s *crtLst,
    CERT_LIST_s **devCrt );

/**
This function is used to extract an object, identified by tartetOid from a given
Relative Distinguished Name (RDN) used in certificates as
issuer and subject identifier.

\param[in] rdnSequence
    Structur containing the CN to extract
\param[in] targetOid
    OID of object to extract
\param[in] handleError
    Flag to signal if the absence of a requested OID is handled as an error.

\param[out] str
    Pointer where object string will be stored.
    Buffer is allocated by this function and must be freed by the caller!

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
cmpGetObjFromRDNSequence (
    RDNSequence_t *rdnSequence,
    OID_t targetOid,
    BOOL handleError,
    char **str );

/**
This function is used to extract certificate parameters for internal use.

\param[in] crtEntry
    Pointer to certificate structure which contains the certificate data and
    the location to store the extracted parameters

\param[out] crtEntry
    Updated certificate structure.

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
cmpExtractCrtParameters (
    CERT_LIST_s *crtEntry);

/**
This function is used to retrieve the pointer to the certificate structure
that contains the referenced subject OID, e.g. oid_title == "Device".

\param[in] src
    Defines where for the given OID should be searched (issuer or subject).
\param[in] oid
    Object identifier that is requested
\param[in] ref
    Reference string to compare with
\param[in] crtLst
    Pointer to certificate list
\param[in] quiet
    Flag to signal if error message should be printed (false) or not (true)
    if requested certificate was not found.

\param[out] crt
    Pointer to requested certificate if found

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
cmpGetCrtFromListBySbjOrIssuerOid (
    CRT_OID_SRC_e src,
    OID_t oid,
    char *ref,
    CERT_LIST_s *crtLst,
    CERT_LIST_s **crt,
    BOOL quiet );

/**
This function is used to identify the certificate types by different mechanisms:
- Device certificate by comparing the title and / or by comparing the
public key
- Root certificate by comparing if issuer and subject CN matches
- Issuer certificates by searching for certificates that have the current one
as issuer
- End point certificates are the remaining

\note
The device certificate is a special end point certificate.

\param[in] cmp
Pointer to CMP data

\param[out] cmp
Pointer to updated CMP data

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
cmpDetermineCrtTypesAndHierarchy (
    CMP_DATA_s *cmp );

/**
This function is similar to cmpDetermineCrtTypesAndHierarchy but
has different parameters.

\param[in] lst
Pointer to certificate list
\param[in] devPubKey
Pointer to device public key
\param[in] devPubKeySize
Size of device public key

\param[out] lst
Pointer to updated certificate list

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL cmpDetermineCrtTypesAndHierarchy2 (
    CERT_LIST_s *lst,
    UINT8* devPubKey,
    UINT32 devPubKeySize );

/**
This function is used to search the certificate list for
a certificate with a given local ID.
The pointer to the certificate is returned.

\param[in] lst
Pointer to certificate list
\param[in] certId
Certificate ID to search

\param[out] crtPtr
Pointer to requested certificate within certificate list,
NULL if not within list

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL checkCertListForCertByLocalId (
    CERT_LIST_s *lst,
    CERT_ID_e certId,
    CERT_LIST_s **crtPtr );

/**
This function is used to check the certificate list if it contains a
complete device certificate chain

\param[in] lst
    Pointer to certificate list

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL checkCertListForCompleteDeviceCertChain (
    CERT_LIST_s *lst );

/**
This function is used to extract Certificate Parameters from a TBSCertificate
stated in the CMP_DATA_s structure.

\param[in] tbsCrt
    TBSCertificate from which to extract the Parameters

\param[out] crtEntry
    Updated CERT_LIST_s entry

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL cmpExtractCrtParamsFromTBSCertificate(
        TBSCertificate_t *tbsCrt,
        CERT_LIST_s *crtEntry );

/**
This function is used to check whether a certificate,
identified by given subject name string,
is included in a given certificate list

\param[in] lst
    Certificate list to check
\param[in] subjName
    Subject name string of the certificate

\return CERT_LIST_s *
\retval Pointer to certificate if it was found, NULL otherwise

*/
CERT_LIST_s
*checkCertListForCertBySubjectCN (
    CERT_LIST_s *lst,
    char *subjName );

/**
This function is used to copy a given CERT_LIST_s item
to a given CERT_LIST_s structure.

\param[in] src
    Item to copy
\param[in] dst
    Destination where item will be copied to

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
copyCertListItem (
    CERT_LIST_s *src,
    CERT_LIST_s **dst );

/**
This function is used to check the local certificate list for a certificate
with the given issuer common name.

\param[in] lst
    Pointer to local certificate list
\param[in] issuerName
    Issuer common name to search for

\returns Pointer to certificate (#CERT_LIST_s) if found or NULL

*/
CERT_LIST_s
*checkCertListForCertByIssuerCN (
    CERT_LIST_s *lst,
    char *issuerName );

/**
This function is used to extract ASN1 type Time_t.

\param[in] asnTime
    ASN1 encoded time to extract

\return ull_time_t

*/
ull_time_t
cmpExtractAsn1Time (
    Time_t asnTime );
/***************************************************************************
 * 6. MACRO FUNCTIONS                                                      *
 ***************************************************************************/

/***************************************************************************
 * 7. END                                                                  *
 ***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* __CMP_UTILS_H__ */
