/***************************************************************************
 *                                                                         *
 * Copyright                                                               *
 *     escrypt GmbH, Bochum, Germany                                       *
 *     Lise-Meitner-Allee 4                                                *
 *     D-44801 Bochum, Germany                                             *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************/
/*!
   \file        asn1_utils.h

   \brief       ASN1 utilities module of CMP library

   ***

   $Rev: 634 $
 */
/***************************************************************************
$Author: mlange $
$Date: 2017-02-15 14:59:54 +0100 (Mi, 15. Feb 2017) $
****************************************************************************/

#ifndef __ASN1_UTILS_H__
#define __ASN1_UTILS_H__

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/
/* common headers */
#include "../../common/inc/esc_common.h"

/* asn1 header */

#include "../asn1-2005-multi-file-os-asn1c/OBJECT_IDENTIFIER.h"
#include "../asn1-2005-multi-file-os-asn1c/INTEGER.h"
#include "../asn1-2005-multi-file-os-asn1c/ANY.h"
#include "../asn1-2005-multi-file-os-asn1c/RDNSequence.h"

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/

/** Object identifier (OID) enumeration */
typedef enum {
    /** OID country */
    oid_country = 0,
    /** OID common name */
    oid_commonName,
    /** OID serial number */
    oid_serialNumber,
    /** OID state or province */
    oid_stateOrProvince,
    /** OID locality name */
    oid_localityName,
    /** OID name */
    oid_name,
    /** OID distinguished name */
    oid_distinguishedName,
    /** OID organization */
    oid_organization,
    /** OID organization unit */
    oid_organizationUnit,
    /** OID description */
    oid_description,
    /** OID unique identifier */
    oid_uniqueIdentifier,
    /** OID UUID pair */
    oid_uuidpair,
    /** OID user password */
    oid_userPw,
    /** OID registration token */
    oid_regToken,
    /** OID RSA encryption */
    oid_rsaEnc,
    /** OID SHA with RSA encryption */
    oid_shaWithRsaEnc,
    /** OID SHA 256 with RSA encryption */
    oid_sha256WithRsaEnc,
    /** OID SHA 512 with RSA encryption */
    oid_sha512WithRsaEnc,
    /** OID title */
    oid_title,
    /** OID printable string */
    oid_printableString,
    /** extension subject alternative name */
    oid_ext_subjectAlternativeName,
    /** Unsupported OID */
    oid_unsupported
} OID_t;

/** OID lookup table structure */
typedef struct {
    /** Internally used OID enumeration */
    OID_t type;
    /** OID number string */
    char noStr[32];
    /** OID name string */
    char nameStr[32];
} OIDS_t;

/***************************************************************************
 * 4. CONSTANTS                                                            *
 ***************************************************************************/

/***************************************************************************
 * 5. FUNCTION PROTOTYPES                                                  *
 ***************************************************************************/

/**
This function is used to retrieve the name string of an OID

\param[in] oid
    OID to request the name string of

\param[out] str
    Pointer where name of requested OID is stored.
    Buffer is allocated by this function and must be freed by the caller!

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
asn1GetOidNameStr (
    OID_t oid,
    char **str );

/**
This function is used to set an OID 

\param[in] id 
    OID value to store

\param[out] dst 
    Pointer where ASN1 representation of OID is stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
setObjIdentifier (
    OBJECT_IDENTIFIER_t *dst,
    OID_t id );

/**
This function is used to add an OID and an string to an 
attributeTypeAndValue list

\param[in] id 
    OID we want to add
\param[in] str 
    String we want to add

\param[out] list
    Pointer where the list is stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
addToAttrList (
    void *list,
    OID_t id,
    char *str );

/**
This function is used to add an OID and an string to an 
relative distinguished name list

\param[in] id 
    OID we want to add
\param[in] str 
    String we want to add

\param[out] rdnSeq
    Pointer where the list is stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
addToRdnList (
    void *rdnSeq,
    OID_t id,
    char *str );

/**
This function is used to allocate memory for a given ASN1 type

\param[in] ptr
    Pointer to pointer where memory should be allocated
\param[in] size
    Amount of memory that should be allocated

\param[out] ptr 
    Pointer to pointer to allocated memory

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
alloc ( 
    void **ptr,
    int size );

/**
This function is used to print an ASN1 structure as XML representation

\param[in] asnType 
    Pointer to ASN1 type we want to print
\param[in] asnPtr
    Pointer to ASN1 structure

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL asn1DebugPrint ( 
    asn_TYPE_descriptor_t *asnType,
    void *asnPtr );

/**
This function returns the enumerated type of a given ASN1 encoded OID.

\param[in] oid
    Pointer to ASN1 OID

\param[out] result
    Pointer where enumerated type will be stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
getObjIdentifier (
    OBJECT_IDENTIFIER_t *oid,
    OID_t *result );

/**
This function is used to convert a given ASN.1 type to the
more general type ANY.

\param[in] type
    Pointer to ASN1 type to convert
\param[in] td
    Pointer to type descriptor of ASN1 type to convert

\param[out] any
    Pointer where ANY type will be stored

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
asn1Type2Any (
    void *type,
    asn_TYPE_descriptor_t *td,
    ANY_t *any );

/**
This function is used to convert the ASN.1 general type ANY to a ASN.1 type.

\param[in] any
    Pointer to ASN1 ANY type to convert
\param[in] td
    Pointer to type descriptor of resulting ASN1 type

\param[out] type
    Pointer where resulting ASN.1 type will be stored.
    Buffer is allocated by this function and must be freed by the caller!

\return #BOOL
\retval #FALSE on success
\retval #TRUE on failure

*/
BOOL
asn1Any2Type (
    ANY_t *any,
    asn_TYPE_descriptor_t *td,
    void **type);

/***************************************************************************
 * 6. MACRO FUNCTIONS                                                      *
 ***************************************************************************/

/***************************************************************************
 * 7. END                                                                  *
 ***************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* __ASN1_UTILS_H__ */
