/******************************************************************
 *FILE: UserEncryptDecrypt_IConverter.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Jan, 2018
 */

#ifndef UPGRADE_USERENCRYPTDECRYPT_ICONVERT_H_
#define UPGRADE_USERENCRYPTDECRYPT_ICONVERT_H_

#include <dlfcn.h>
#include <boost/current_function.hpp>
#include <functional>
#include <vector>
#include "UserEncryptDecrypt_Version.h"
#include "backends/sdc/UserEncryptDecrypt_SDCBackend.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"

class IConvert {
 public:
  /**
   * @brief Constructor for the class.
   */
  IConvert() {}

  /**
   * @brief Destructor for the class.
   */
  virtual ~IConvert() {}

  /**
   * @brief This function converts the provided data to the previously provided
   * destination version.
   *
   * @param data is the data that was generated by our lib that may require
   * upgrade.
   * @param sourceVersion is the current version of the system.
   * @param destinationVersion is the version the data needs to be upgraded to.
   *
   * @return returns the upgraded data.
   */
  virtual std::vector<unsigned char> convertData(
      std::vector<unsigned char> data, int sourceVersion,
      int destinationVersion) = 0;

  /**
   * @brief This function checks if the provided data requires conversion.
   *
   * @param sourceVersion is the current version of the system.
   * @param destinationVersion is the version the data needs to be upgraded to.
   *
   * @return returns true if conversion is needed and false otherwise.
   * @throw ErrorMessage ErrType::UPGRADE_DestinationVersionDoesNotExist
   * @throw ErrorMessage ErrType::UPGRADE_SourceVersionDoesNotExist
   */
  bool isConversionNeeded(int sourceVersion, int destinationVersion) {
    if (_dataConversionNeeded.find(sourceVersion) !=
        _dataConversionNeeded.end()) {
      if (_dataConversionNeeded[sourceVersion].find(destinationVersion) !=
          _dataConversionNeeded[sourceVersion].end()) {
        return _dataConversionNeeded[sourceVersion][destinationVersion];
      } else {
        THROW_UED_EXCEPTION(ErrType::UPGRADE_DestinationVersionDoesNotExist);
      }
    } else {
      THROW_UED_EXCEPTION(ErrType::UPGRADE_SourceVersionDoesNotExist);
    }
  }

  /**
   * @brief this function serves as the function to init the conversion maps
   * with the conversion information (function pointers and booleans)
   */
  virtual void init() = 0;

  /**
   * @brief This function serves as a function to call when the provided data
   * does not require conversion
   *  from the source version to the destination.
   *
   * @param data is the data that was generated by our lib that may require
   * upgrade.
   *
   * @return returns the received data.
   */
  static std::vector<unsigned char> noop(std::vector<unsigned char> data) {
    return data;
  }

 protected:
  void convertionIsNeeded(int src, int dst, bool b) {
    _dataConversionNeeded[src][dst] = b;
  }

 private:
  std::map<int, std::map<int, bool>> _dataConversionNeeded;
};

#endif /* UPGRADE_USERENCRYPTDECRYPT_ICONVERT_H_ */
