/******************************************************************
 *FILE: UserEncryptDecrypt_ConversionProcess.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Jan, 2018
 */
#include <core/UserEncryptDecrypt_ConsistentFile.h>
#include <core/authentication/UserEncryptDecrypt_Authentication.h>
#include <core/uedpacket/UserEncryptDecrypt_UEDPacketManager.h>
#include <string>
#include <thread>

#include "UserEncryptDecrypt_Version.h"
#include "core/UserEncryptDecrypt_Configurations.h"
#include "core/logger/UserEncryptDecrypt_Logger.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"
#include "upgrade/UserEncryptDecrypt_ConversionProcess.h"

#include "UserEncryptDecrypt_AuthenticationConvert.h"
#include "domains/UserEncryptDecrypt_UEDPacketDomain.h"
#include "upgrade/UserEncryptDecrypt_DataConvert.h"
#include "upgrade/UserEncryptDecrypt_Notifier.h"
#include "upgrade/UserEncryptDecrypt_TokenConvert.h"

std::shared_ptr<ConversionProcess> ConversionProcess::_instance = nullptr;
std::mutex ConversionProcess::_mutex;

ConversionProcess::ConversionProcess() {
  _notifier           = std::make_shared<Notifier>();
  _destinationVersion = UED_CURRENT_VERSION;
  _updating           = false;
  _mapTypeDataConverter.emplace("CYPHERTEXT", new DataConvert());
  _mapTypeDataConverter.emplace("TOKEN", new TokenConvert());
  _logger = Logger1::getLogger();
  _ctx = _logger->registerContext("AUTH", "UserEncryptDecrypt_Authentication");
}

ConversionProcess::~ConversionProcess() {
  delete _mapTypeDataConverter["CYPHERTEXT"];
  delete _mapTypeDataConverter["TOKEN"];
  _logger->unregisterContext(_ctx);
}

std::shared_ptr<ConversionProcess>& ConversionProcess::getInstance() {
  if (!_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (!_instance) {
      _instance.reset(new ConversionProcess());
    }
  }
  return _instance;
}

void ConversionProcess::startData() {
  if (_updating == false) {
    _updating = true;

    auto authenticationConvert = AuthenticationConvert::getConverter();
    authenticationConvert->init();
    std::string file = Configurations::getInstance()->get<std::string>(
        Configurations::UED_AUTH_FILE);
    auto _consistentFile = ConsistentFile::getFile(file);
    _consistentFile->openFile();

    auto line = _consistentFile->readLine();
    if (line.empty()) {
      Authentication::getInstance();
      return;
    }

    unsigned int version = std::stoul(line);
    if (authenticationConvert->isConversionNeeded(version,
                                                  _destinationVersion)) {
      std::vector<unsigned char> em = std::vector<unsigned char>();
      authenticationConvert->convertData(em, version, _destinationVersion);
    }
  }
}

void ConversionProcess::stopData() {
  AuthenticationConvert::getConverter()->end();
  _updating = false;
}

bool ConversionProcess::isConversionNeeded(std::vector<unsigned char> data) {
  if (_updating == false) {
    _logger->log(_ctx, LogLevel::ERROR, "startData was not called");
    _updating = false;
    THROW_UED_EXCEPTION(ErrType::UPGRADE_ProcessNotStarted);
  }
  UEDPacket packet;
  packet.decode(std::string(data.begin(), data.end()));

  isPacketUsable(packet);

  boost::optional<std::string> type =
      packet.getHeaderField<std::string>("dataType");
  _mapTypeDataConverter[type.get()]->init();
  return _mapTypeDataConverter[type.get()]->isConversionNeeded(
      packet.getHeaderField<int>("libVersion").get(), _destinationVersion);
}

std::vector<unsigned char> ConversionProcess::convertData(
    std::vector<unsigned char> data) {
  if (_updating == false) {
    _logger->log(_ctx, LogLevel::ERROR, "startData was not called");
    _updating = false;
    THROW_UED_EXCEPTION(ErrType::UPGRADE_ProcessNotStarted);
  }
  std::lock_guard<std::mutex> lock(_mutex);
  UEDPacket packet;
  packet.decode(std::string(data.begin(), data.end()));

  isPacketUsable(packet);

  boost::optional<std::string> type =
      packet.getHeaderField<std::string>("dataType");

  _mapTypeDataConverter[type.get()]->init();
  std::vector<unsigned char> d = _mapTypeDataConverter[type.get()]->convertData(
      UEDPacketManager::getDataFromUEDPacket(data).get(),
      packet.getHeaderField<int>("libVersion").get(), _destinationVersion);

  UEDPacketManager::createUEDPacket(d);
  return UEDPacketManager::createUEDPacket(d);
}

void ConversionProcess::notifyUpgradeProcess(uint8_t msB, uint8_t middleByte,
                                             uint8_t lsB) {
  unsigned int destinationVersion = UED_VERSION(msB, middleByte, lsB);

  if (destinationVersion == 0) {
    _logger->log(_ctx, LogLevel::ERROR, "Version creation overflowed");
    _updating = false;
    THROW_UED_EXCEPTION(ErrType::UPGRADE_VersionProvidedOverflowed);
  }

  if (destinationVersion >= _destinationVersion) {
    _destinationVersion = destinationVersion;
    _notifier->notifyAllObservers();
  }
}

void ConversionProcess::addObserver(Observer& observer) {
  _notifier->addObserver(observer);
}

void ConversionProcess::removeObserver(Observer& observer) {
  _notifier->removeObserver(observer);
}

std::vector<unsigned char> ConversionProcess::startRuntimeUpgradeProcess(
    std::vector<unsigned char> data) {
  int destinationVersion = UED_HEADER_CURRENT_VERSION;
  if (destinationVersion == 0) {
    _logger->log(_ctx, LogLevel::ERROR, "Version creation overflowed");
    _updating = false;
    THROW_UED_EXCEPTION(ErrType::UPGRADE_VersionProvidedOverflowed);
  }

  if (destinationVersion > _destinationVersion) {
    _destinationVersion = destinationVersion;
    std::thread notify(&Notifier::notifyAllObservers, _notifier);
    notify.detach();
  }

  return SDCBackend::getInstance()->Decrypt(0, convertData(data),
                                            SDCBackend::AES, SDCBackend::CBC);
}

void ConversionProcess::isPacketUsable(UEDPacket packet) {
  boost::optional<std::string> headerVersionValue =
      packet.getHeaderField<std::string>("headerVersion");
  if (headerVersionValue != boost::none && headerVersionValue.get().empty()) {
    _updating = false;
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }
  if (_mapTypeDataConverter.find(headerVersionValue.get()) ==
      _mapTypeDataConverter.end()) {
    _updating = false;
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }
}
