/******************************************************************
 *FILE: UserEncryptDecrypt_ErrorMessage.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: ErrorMessage
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 *
 * Date      | Author             | Modification
 * 14/08/2018| AKM7COB			  | changed class name logger to logger1. class name logger is already used in ASF framework
 */

#ifndef SRC_ERROR_USERENCRYPTDECRYPT_ERRORMESSAGE_H_
#define SRC_ERROR_USERENCRYPTDECRYPT_ERRORMESSAGE_H_

#include <boost/current_function.hpp>
#include <boost/function.hpp>
#include <exception>
#include <map>
#include <string>
#include "core/logger/UserEncryptDecrypt_Logger.h"

/**
 * This enumerator data type contains all of the possible error codes that
 * UserEncryptDecrypt product can throw to a process that includes it.
 */

enum class ErrType {
  /**
   *0-999 System Errors
   */
  SYSTEM_noMemory       = 0,
  VERSION_notCompatible = 1,
  /**
  *1000-1999 Authentication Errors
  */
  AUTH_CreateUser                             = 1000,
  AUTH_CreateUserSaltExists                   = 1001,
  AUTH_CreateUserAlreadyExists                = 1002,
  AUTH_AuthenticateUser                       = 1003,
  AUTH_AuthenticateUserMaxAuthenticatedUsers  = 1004,
  AUTH_AuthenticateUserWrongCredentials       = 1005,
  AUTH_AuthenticateUserDoesNotExist           = 1006,
  AUTH_UnauthenticateUser                     = 1007,
  AUTH_UnauthenticateUserDoesNotExist         = 1008,
  AUTH_DeleteUser                             = 1009,
  AUTH_DeleteUserDoesNotExist                 = 1010,
  AUTH_DeleteUserWrongCredentials             = 1011,
  AUTH_ModifyUserPassphrase                   = 1012,
  AUTH_ModifyUserPassphraseSaltExists         = 1013,
  AUTH_ModifyUserPassphraseWrongOldPassphrase = 1014,
  AUTH_ModifyUserPassphraseUserDoesNotExist   = 1015,
  AUTH_AuthenticationRequired                 = 1016,
  AUTH_UnauthorizedOperation                  = 1017,
  AUTH_UserIDNotValid                         = 1018,
  AUTH_PassphraseNotValid                     = 1019,
  AUTH_SaltNotValid                           = 1020,
  AUTH_MaximumAuthenticatedUsersReached       = 1021,
  AUTHDB_fileDataCorrupted                    = 1100,
  UM_UserDoesNotExist                         = 1200,
  /**
  * 2000-2999 Core Functions Errors
  */
  CORE_Encrypt                      = 2000,
  CORE_EncryptUserIdNotFound        = 2001,
  CORE_EncryptGuestUserCantDecrypt  = 2002,
  CORE_EncryptUnableToEncrypt       = 2003,
  CORE_Decrypt                      = 2004,
  CORE_DecryptUserIdNotFound        = 2005,
  CORE_DecryptGuestUserCantDecrypt  = 2006,
  CORE_DecryptUnableToDecrypt       = 2007,
  CORE_Digest                       = 2008,
  CORE_DigestUserIdNotFound         = 2009,
  CORE_DigestGuestUserCantDecrypt   = 2010,
  CORE_EncryptionModuleNotAvailable = 2011,
  /**
  *3000-3999 Packet Errors
  */
  PCKT_InvalidPayload        = 3000,
  PCKT_InvalidFormat         = 3001,
  PCKT_InvalidTokenFormat    = 3002,
  PCKT_LibVersionNotFound    = 3003,
  PCKT_HeaderVersionNotFound = 3004,
  PCKT_SignatureNotFound     = 3005,
  /**
  *4000-4999 Upgrade Errors
  */
  UPGRADE_NotInScope                     = 4000,
  UPGRADE_ProcessNotStarted              = 4001,
  UPGRADE_VersionProvidedOverflowed      = 4002,
  UPGRADE_DestinationVersionDoesNotExist = 4003,
  UPGRADE_SourceVersionDoesNotExist      = 4004,
  /**
  *5000-5999 Consistent Errors
  */
  CONSISTENT_filenameNotValid      = 5000,
  CONSISTENT_fileNotOpen           = 5001,
  CONSISTENT_fileoperationNotValid = 5002,
  /**
  *6000-6999 Configurations Errors
  */
  CONFIGURATIONS_fileNotOpen              = 6000,
  CONFIGURATIONS_failedToWriteDefaultFile = 6001,
  CONFIGURATIONS_failedReadingConfFile    = 6002,
  CONFIGURATIONS_fileBadlyFormatted       = 6003,
  CONFIGURATIONS_valueNotDefined          = 6004,
  CONFIGURATIONS_ConfNotDefined           = 6005,
  /**
  *7000-7999 Domain Errors
  */
  DOMAIN_notValid            = 7000,
  DOMAIN_InvalidBase64Format = 7001,
  DOMAIN_plainDataEmpty      = 7002,
  /**
   *8000-8999 Cryptographic System Errors
   */
  BACKEND_invalidTypeOfBackend     = 8000,
  BACKEND_BackendsNotLoaded        = 8001,
  OPENSSL_keylenCannotFit          = 8100,
  OPENSSL_keyDerivationFailed      = 8101,
  SDC_invalidParameter             = 8200,
  SDC_EncryptionModuleNotAvailable = 8201,
  SDC_KeyError                     = 8202,
  SDC_unknown                      = 8203,
  SDC_KeyExists                    = 8204
};

#define THROW_UED_EXCEPTION(arg)                              \
  throw ErrorMessage(arg, __FILE__, std::to_string(__LINE__), \
                     BOOST_CURRENT_FUNCTION)
#define THROW_UED_EXCEPTION_WITH_TRIES(arg, tries)                     \
  throw ErrorMessageWithTries(arg, __FILE__, std::to_string(__LINE__), \
                              BOOST_CURRENT_FUNCTION, tries)
/**
 * ErrorMessage is a representation of the userEncryptDecrypt run-time errors.
 *
 * Run-time errors occur when something unexpected happen.\n
 * ErrorMessage inherits std::exception that makes it Throwable.\n
 * Who call methods that throw ErrorMessage need to catch exceptions thrown by
 * ErrorMessage.\n
 * Error codes are at ::ErrType
 * */
class ErrorMessage : public std::exception {
 private:
  ErrType _errType;
  std::string _file;
  std::string _line;
  std::string _errCallerInfo;
  std::string _errMessage;
  static std::map<ErrType, std::string> _errorTypeMessagesMap;

  std::shared_ptr<Logger1> _logger;
  std::shared_ptr<LogContext> _ctx;

 protected:
  /**
   * @brief Is a complete representation of ErrorMessage exception in string
   * format.
   */
  std::string _fullMsg;
  /**
   * @brief Builds a default ErrorMessage string format exception.
   *
   * @returns Default ErrorMessage string format
   */
  std::string buildDefaultMessage();

 public:
  /**
   * @brief This constructor receives an error code (@param errType) and the
   * name of method (@param errCallerInfo) that instantiate it.
   *
   * @param file The current file where exception was thrown
   * @param line The current line where exception was thrown
   *
   * INFO: errCallerInfo must contains class and method name.
   * HINT: errCallerInfo can be obtained by including
   * <boost/current_function.hpp> in class that will use ErrorMessage and use
   * the MACRO BOOST_CURRENT_FUNCTION.
   */
  ErrorMessage(ErrType errType, std::string file, std::string line,
               std::string errCallerInfo);
  ErrorMessage(ErrType errType, std::string errCallerInfo);

  /**
   * @brief Destructor for the class
   */
  virtual ~ErrorMessage();

  /**
   * @brief This function returns the error code of exception thrown. Error
   * codes can be found at ::ErrType
   * @return Returns ::ErrType exception thrown by ErrorMessage.
   */
  ErrType getErrorType();

  /**
   * @brief This function returns the message of exception thrown and is related
   * to ::ErrType.
   * @return returns the message exception thrown by ErrorMessage.
   */
  std::string getErrorMessage();

  /**
   * @brief This function returns the class and method name that throws
   * ErrorMessage.
   * @return returns the errorCallerInfo.
   */
  std::string getErrorCallerInfo();

  /**
   * @brief This function returns the message of exception thrown and is related
   * to ::ErrType, including errorCallerInfo.
   * @return returns the message exception thrown by ErrorMessage.
   */
  std::string getFullErrorMessage();
  /**
   * @brief This method enables ErrorMessage to have the same behavior as
   * std::exception when a throw is not handled.
   */
  const char* what() const throw() { return _fullMsg.c_str(); }
};

class ErrorMessageWithTries : public ErrorMessage {
 private:
  int _errTries;

 public:
  /**
   * @brief This constructor receives an error code (@param errType) and the
   * name of method (@param errCallerInfo) that instantiate it.
   *
   * @param file The current file where exception was thrown
   * @param line The current line where exception was thrown
   * @param tries The number of error tries to show in exception.
   *
   * INFO: errCallerInfo must contains class and method name.
   * HINT: errCallerInfo can be obtained by including
   * <boost/current_function.hpp> in class that will use ErrorMessage and use
   * the MACRO BOOST_CURRENT_FUNCTION.
   */
  ErrorMessageWithTries(ErrType errType, std::string file, std::string line,
                        std::string errCallerInfo, int tries);
  /**
   * @brief Destructor for the class
   */
  ~ErrorMessageWithTries();

  /**
   * @brief This function return the number of tries related to the ::ErrType.
   * @return returns the number of tries related to the ::ErrType
   */
  int getErrorTries();
};
#endif /* SRC_ERROR_USERENCRYPTDECRYPT_ERRORMESSAGE_H_ */
