/******************************************************************
 *FILE: UserEncryptDecrypt_ErrorMessage.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: ErrorMessage
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#include "UserEncryptDecrypt_ErrorMessage.h"
#include <iostream>

std::map<ErrType, std::string> ErrorMessage::_errorTypeMessagesMap = {

    {ErrType::VERSION_notCompatible,
     "Data not compatible with software version"},
    {ErrType::SYSTEM_noMemory,
     "Memory allocation error - possibly the system has no available memory"},
    {ErrType::DOMAIN_InvalidBase64Format,
     "Domain not valid: Wrong base64 string format"},
    {ErrType::DOMAIN_notValid, "Domain not valid"},
    {ErrType::DOMAIN_plainDataEmpty, "Plain Data is empty"},
    {ErrType::AUTH_CreateUser,
     "An error occurred while requesting a new user profile creation"},
    {ErrType::AUTH_CreateUserSaltExists,
     "An error occurred while creating new user profile: Provided salt already "
     "exists"},
    {ErrType::AUTH_CreateUserAlreadyExists,
     "An error occurred while creating new user profile: User already exists"},
    {ErrType::AUTH_AuthenticateUser,
     "An error occurred trying to authenticate user"},
    {ErrType::AUTH_AuthenticateUserMaxAuthenticatedUsers,
     "An error occurred trying to authenticate user: Maximum number of "
     "authenticated users reached"},
    {ErrType::AUTH_AuthenticateUserWrongCredentials,
     "An error occurred trying to authenticate user: Wrong credentials"},
    {ErrType::AUTH_AuthenticateUserDoesNotExist,
     "An error occurred trying to authenticate user: User does not exist"},
    {ErrType::AUTH_UnauthenticateUser,
     "An error occurred while trying to unauthenticate user"},
    {ErrType::AUTH_UnauthenticateUserDoesNotExist,
     "An error occurred trying to unauthenticate user: User does not exist"},
    {ErrType::AUTH_DeleteUser,
     "An error occurred while requesting user profile deletion"},
    {ErrType::AUTH_DeleteUserDoesNotExist,
     "An error occurred while deleting user profile: User does not exist"},
    {ErrType::AUTH_DeleteUserWrongCredentials,
     "An error occurred while deleting user profile: Wrong credentials"},
    {ErrType::AUTH_ModifyUserPassphrase,
     "An error occurred while trying to modify user passphrase"},
    {ErrType::AUTH_ModifyUserPassphraseSaltExists,
     "An error occurred while modifying user passphrase: Provided salt already "
     "exists"},
    {ErrType::AUTH_ModifyUserPassphraseWrongOldPassphrase,
     "An error occurred while modifying user passphrase: Wrong old password"},
    {ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist,
     "An error occurred while modifying user passphrase: User does not exist"},
    {ErrType::AUTH_AuthenticationRequired, "Authentication required"},
    {ErrType::AUTH_UnauthorizedOperation,
     "Permission denied. Unauthorized operation"},
    {ErrType::AUTH_UserIDNotValid, "The provided userID is invalid"},
    {ErrType::AUTH_PassphraseNotValid, "The provided passphrase is invalid"},
    {ErrType::AUTH_MaximumAuthenticatedUsersReached,
     "Maximum number of authenticated users reached"},
    {ErrType::AUTH_SaltNotValid, "The provided salt is invalid"},
    {ErrType::AUTHDB_fileDataCorrupted,
     "An error occurred trying to read from the database file"},
    {ErrType::CORE_Encrypt, "An error occurred while trying to encrypt data"},
    {ErrType::CORE_EncryptUserIdNotFound,
     "An error occurred while trying to encrypt data: Invalid userID or not "
     "found"},
    {ErrType::CORE_EncryptGuestUserCantDecrypt,
     "An error occurred while trying to encrypt data: Guest user can't encrypt "
     "data"},
    {ErrType::CORE_EncryptUnableToEncrypt,
     "An error occurred while trying to encrypt data: Data not encryptable"},
    {ErrType::CORE_Decrypt, "An error occurred while trying to decrypt data"},
    {ErrType::CORE_DecryptUserIdNotFound,
     "An error occurred while trying to decrypt data: Invalid userID or not "
     "found"},
    {ErrType::CORE_DecryptGuestUserCantDecrypt,
     "An error occurred while trying to decrypt data: Guest user can't decrypt "
     "data"},
    {ErrType::CORE_DecryptUnableToDecrypt,
     "An error occurred while trying to decrypt data: Data not decryptable"},
    {ErrType::CORE_Digest, "An error occurred while trying to digest data"},
    {ErrType::CORE_DigestUserIdNotFound,
     "An error occurred while trying to digest data: Invalid userID or not "
     "found"},
    {ErrType::CORE_DigestGuestUserCantDecrypt,
     "An error occurred while trying to digest data: Guest user can't digest "
     "data"},
    {ErrType::CORE_EncryptionModuleNotAvailable,
     "Encryption hardware interface not available"},
    {ErrType::UM_UserDoesNotExist,
     "An error occurred trying to get userId: User does not exist"},
    {ErrType::PCKT_InvalidPayload,
     "An error occurred trying to get packet payload: Invalid payload"},
    {ErrType::PCKT_InvalidFormat, "Provided packet is in a wrong format"},
    {ErrType::PCKT_InvalidTokenFormat, "Provided token is in a wrong format"},
    {ErrType::PCKT_LibVersionNotFound,
     "Provided packet does not contain an lib version: wrong format"},
    {ErrType::PCKT_HeaderVersionNotFound,
     "Provided packet does not contain an header version: wrong format"},
    {ErrType::PCKT_SignatureNotFound,
     "Provided packet does not contain an valid signature: wrong format"},
    {ErrType::UPGRADE_ProcessNotStarted,
     "An error occurred trying to upgrade data: startData was not called"},
    {ErrType::UPGRADE_NotInScope,
     "Upgrade between the required version is not possible"},
    {ErrType::UPGRADE_VersionProvidedOverflowed,
     "An error occurred trying to create the new version: Version creation "
     "overflowed"},
    {ErrType::UPGRADE_DestinationVersionDoesNotExist,
     "An error occurred trying to upgrade to the destination version"},
    {ErrType::UPGRADE_SourceVersionDoesNotExist,
     "An error occurred trying to upgrade from the source version"},
    {ErrType::CONSISTENT_filenameNotValid,
     "File name is not a valid path or is empty"},
    {ErrType::CONSISTENT_fileNotOpen, "File was not open"},
    {ErrType::CONSISTENT_fileoperationNotValid, "File operation is not valid"},
    {ErrType::BACKEND_invalidTypeOfBackend, "Invalid input type of backend"},
    {ErrType::BACKEND_BackendsNotLoaded, "Load of OpenSSL and SDC failed"},
    {ErrType::CONFIGURATIONS_ConfNotDefined, "Configuration not defined"},
    {ErrType::OPENSSL_keylenCannotFit,
     "The system can not handle the key length required"},
    {ErrType::SDC_KeyExists, "key already exists in the key store"},
    {ErrType::SDC_EncryptionModuleNotAvailable,
     "It was unable to connect/use SDC deamon"},
    {ErrType::SDC_invalidParameter, "Passed parameter is not valid"},
    {ErrType::SDC_KeyError, "Problem using the key"},
    {ErrType::SDC_unknown, "Unknow SDC error"},
    {ErrType::OPENSSL_keyDerivationFailed, "Key Derivation failed"},
    {ErrType::CONFIGURATIONS_fileNotOpen, "Configurations file is not open"},
    {ErrType::CONFIGURATIONS_failedToWriteDefaultFile,
     "Failed to create default Configurations file"},
    {ErrType::CONFIGURATIONS_failedReadingConfFile,
     "Failed to read from the Configurations file"},
    {ErrType::CONFIGURATIONS_fileBadlyFormatted,
     "Configurations file is badly formatted"},
    {ErrType::CONFIGURATIONS_valueNotDefined,
     "Configurations value is not defined"},

};

ErrorMessage::ErrorMessage(ErrType errType, std::string errCallerInfo)
    : std::exception() {
  _errType       = errType;
  _errCallerInfo = errCallerInfo;
  _line          = "Line not specified";
  _file          = "File not specified";

  _fullMsg = buildDefaultMessage();

  _logger = Logger1::getLogger();
  _ctx = _logger->registerContext("ERRMSG", "UserEncryptDecrypt_ErrorMessage");
}

ErrorMessage::ErrorMessage(ErrType errType, std::string file, std::string line,
                           std::string errCallerInfo)
    : std::exception() {
  _errType       = errType;
  _errCallerInfo = errCallerInfo;
  _line          = line;
  _file          = file;

  _fullMsg = buildDefaultMessage();

  _logger = Logger1::getLogger();
  _ctx = _logger->registerContext("ERRMSG", "UserEncryptDecrypt_ErrorMessage");
}

ErrorMessage::~ErrorMessage() {
  _logger->log(_ctx, LogLevel::ERROR, _fullMsg);
  _logger->unregisterContext(_ctx);
}

std::string ErrorMessage::buildDefaultMessage() {
  if (_errorTypeMessagesMap.find(_errType) != _errorTypeMessagesMap.end()) {
    _errMessage = _errorTypeMessagesMap[_errType];
  } else {
    _errMessage = "Unknown error message type";
  }

  return ("\n\tFile: " + _file + "\n\tLine: " + _line + "\n\tFunction: " +
          _errCallerInfo + "\n\tErrorMessage: " + _errMessage +
          "\n\tErrorCode: " + std::to_string(static_cast<int>(_errType)));
}

ErrType ErrorMessage::getErrorType() { return _errType; }

std::string ErrorMessage::getErrorMessage() { return _errMessage; }

std::string ErrorMessage::getFullErrorMessage() { return _fullMsg; }

std::string ErrorMessage::getErrorCallerInfo() { return _errCallerInfo; }

// ErrorMessageWithTries
ErrorMessageWithTries::ErrorMessageWithTries(ErrType errType, std::string file,
                                             std::string line,
                                             std::string errCallerInfo,
                                             int tries)
    : ErrorMessage(errType, file, line, errCallerInfo) {
  _errTries              = tries;
  std::string tmpFullMsg = buildDefaultMessage();
  tmpFullMsg = tmpFullMsg + "\n\tTries: " + std::to_string(_errTries);
  _fullMsg   = tmpFullMsg;
}

ErrorMessageWithTries::~ErrorMessageWithTries() {}

int ErrorMessageWithTries::getErrorTries() { return _errTries; }
