/******************************************************************
 *FILE: UserEncryptDecrypt_UserManager.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/ /*
/**
* @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
* @date Dec, 2017
*/

#ifndef SRC_CORE_USER_USERENCRYPTDECRYPT_USERMANAGER_H_
#define SRC_CORE_USER_USERENCRYPTDECRYPT_USERMANAGER_H_

#include <boost/optional.hpp>
#include <list>
#include <map>
#include <memory>
#include <mutex>
#include "core/user/UserEncryptDecrypt_User.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"

/**
 * This class provides the management of all the users.
 *
 * The Authentication class uses this class in order to access user related
 * data.
 *
 * A typical class permits callers to create as many instances of the class as
 * they want, but this class use the Singleton Pattern.
 * A singleton object provides a global point of access to the resources of its
 * class and returns the same instance no matter how many times an application
 * requests it.
 * The function UserManager::getInstance() provides the access to the
 * resources of this class.
 */
class UserManager {
 public:
  /*
   * @brief Destructor for the class.
   */
  virtual ~UserManager();

/**
 *   @brief Provides an instance of the UserManager class object.
 *
 *   This function provides the access to class object, but only one instance
 * is returned as mentioned above in the definition of singleton pattern.
 *
 *   @return returns a shared pointer of the instance of the UserManager
 */
#ifdef TESTING
  static std::shared_ptr<UserManager>& getInstance(bool createNew = false);
#else
  static std::shared_ptr<UserManager>& getInstance();
#endif

  /*
   * @brief This method checks if the provided user is the guest user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   * @return returns true if the provided ID matches with the guest ID.
   */
  bool isGuestUser(unsigned int userID);

  /*
   * @brief This method checks if the provided user exists.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   * @return returns true if the ID matches with a created user.
   */
  bool hasUser(unsigned int userID);

  /*
   * @brief Uses the data of the provided user and generates a string to be
   * inserted in the database.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   * @return returns a string with the MUK(formatted data to insert in the
   * database).
   */
  std::string createMUK(unsigned int userID);

  bool isMasterUser(unsigned int userID);

  /**
   * @brief Creates a user and associates the provided data with him.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param hashPassSalt is the hash of the concatenated passphrase and salt.
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   * @param encDataKey is the wrapped data key.
   * @param hashDataKey is the hash of the user data key.
   * @param hashWrapperKey is the hash of the user wrapper key.
   * @param salt is the random number used for the generation of the user hash.
   *
   * @return returns true if the credentials are created and false if the user
   * already exists.
   */
  bool createUser(unsigned int userID, std::vector<unsigned char> hashPassSalt,
                  unsigned int wrapperKeyID,
                  std::vector<unsigned char> encDataKey,
                  std::vector<unsigned char> hashDataKey,
                  std::vector<unsigned char> hashWrapperKey,
                  std::vector<unsigned char> salt);

  /**
   * @brief Creates a Master user and associates the provided data with him.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param hashPassSalt is the hash of the concatenated passphrase and salt.
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   * @param encDataKey is the wrapped data key.
   * @param hashDataKey is the hash of the user data key.
   * @param hashWrapperKey is the hash of the user wrapper key.
   * @param salt is the random number used for the generation of the user hash.
   *
   * @return returns true if the credentials are created and false if the user
   * already exists.
   */

  bool createMaster(unsigned int userID,
                    std::vector<unsigned char> hashPassSalt,
                    unsigned int wrapperKeyID,
                    std::vector<unsigned char> encDataKey,
                    std::vector<unsigned char> hashDataKey,
                    std::vector<unsigned char> hashWrapperKey,
                    std::vector<unsigned char> salt);

  /**
   * @brief Creates a user that does not have a passphrase and associates the
   * provided data with him.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   * @param encDataKey is the wrapped data key.
   * @param hashDataKey is the hash of the user data key.
   * @param hashWrapperKey is the hash of the user wrapper key.
   *
   * @return returns true if the credentials are created and false if the user
   * already exists.
   */
  bool createUser(unsigned int userID, unsigned int wrapperKeyID,
                  std::vector<unsigned char> encDataKey,
                  std::vector<unsigned char> hashDataKey,
                  std::vector<unsigned char> hashWrapperKey);

  /**
   * @brief Creates a Master user that does not have a passphrase and associates
   * the provided data with him.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   * @param encDataKey is the wrapped data key.
   * @param hashDataKey is the hash of the user data key.
   * @param hashWrapperKey is the hash of the user wrapper key.
   *
   * @return returns true if the credentials are created and false if the user
   * already exists.
   */
  bool createMaster(unsigned int userID, unsigned int wrapperKeyID,
                    std::vector<unsigned char> encDataKey,
                    std::vector<unsigned char> hashDataKey,
                    std::vector<unsigned char> hashWrapperKey);

  /*
   * @brief Deletes the user with the provided user ID.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   * @return returns true if the user is deleted and false if the user does not
   * exit.
   */
  bool deleteUser(unsigned int userID);

  /*
   * @brief Compares the provided hash with the hash associated with the user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param hash is the digested data of the concatenated passphrase and salt.
   *
   * @return returns true if the values match and false if they do not.
   */
  bool compareHashPassSalt(unsigned int userID,
                           std::vector<unsigned char> hashPassSalt);

  /*
   * @brief Checks if the given user has a passphrase.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   * @return returns true if the user has a passphrase and false if he does not.
   */
  bool userHasPassphrase(unsigned int userID);

  /*
   * @brief Associates the given hash to the user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param hashPassSalt is the digested data of the concatenated passphrase and
   * salt.
   */
  void insertHashPassSalt(unsigned int userID,
                          std::vector<unsigned char> hashPassSalt);

  /*
   * @brief Deletes the association between the given user and his hash.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   */
  void deleteHashPassSalt(unsigned int userID);

  /*
   * @brief Associates the given data key ID to the user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param dataKeyID is the ID of the wrapped data key.
   */
  void insertDataKeyID(unsigned int userID, unsigned int dataKeyID);

  /**
   * @brief Deletes the association between the given user and his data key ID.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   */
  void deleteDataKeyID(unsigned int userID);

  /**
   * @brief Associates the given wrapper key ID to the user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param wrapperKeyID is the ID of the key used to wrap the user data key.
   */
  void insertWrapperKeyID(unsigned int userID, unsigned int wrapperKeyID);

  /**
   * @brief Associates the given encrypted data key to the user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param encDataKey is the user's wrapped data key.
   */
  void insertEncDataKey(unsigned int userID,
                        std::vector<unsigned char> encDataKey);

  /**
   * @brief Associates the given hash of a wrapper key to the user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param hashWrapperKey is the hash of the user wrapper key.
   */
  void insertHashWrapperKey(unsigned int userID,
                            std::vector<unsigned char> hashWrapperKey);

  /**
   * @brief Associates the given salt to the user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param salt is the random number used for the generation of the user hash.
   */
  void insertSalt(unsigned int userID, std::vector<unsigned char> salt);

  /**
   * @brief Deletes the association between the given user and his salt.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   */
  void deleteSalt(unsigned int userID);

  /**
   * @brief Provides the wrapper key ID of the given user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   * @return returns the ID of the key used to wrap the user's data key.
   */
  unsigned int getWrapperKeyID(unsigned int userID);

  /**
   * @brief Provides the data key ID of the given user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   * @return returns the ID of the user's wrapped data key.
   */
  unsigned int getDataKeyID(unsigned int userID);

  /**
   * @brief Provides the wrapper key ID of the given user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   * @return returns the encrypted wrapped data key of the user.
   */
  std::vector<unsigned char> getEncDataKey(unsigned int userID);

  /**
   * @brief Provides the salt from a given user ID
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @return the salt for a given user
   */
  std::vector<unsigned char> getSalt(unsigned int userID);

  /**
   * @brief Provides the users that are exist in the system.
   *
   * @return returns a list with the existing users.
   */
  std::list<unsigned int> getCreatedUsers();

  /**
   * @brief Checks if the provided passhphrase already exists in the system.
   *
   * @param hash is the digested data of the concatenated passphrase and salt.
   *
   * @return returns true if the provided hash matches one already in the system
   * and false if it does not.
   */
  bool keyCompare(std::vector<unsigned char> hash);

  /**
   * tell if the user is able to encrypt/decrypt data
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @return true if enabled false otherwise
   */
  bool canEncrypt(unsigned int userID);

  /**
   * Tells if master can be created in the system or not
   *
   * @return true if the master is enabled and false otherwise
   */
  bool hasActiveMaster();

  /**
   * @brief Provides the encryption wrapper key of the given user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   *
   * @return returns the encrypted wrapped data key.
   */
  std::vector<unsigned char> getEncWrapperKey(unsigned int userID);
  /**
   * @brief Set encryption wrapper key of the given user.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   * @param data is the encrypted wrapped data key.
   */
  void insertEncWrapperKey(unsigned int userID,
                           std::vector<unsigned char> data);

 private:
  /*
   * @brief Constructor for the class.
   */
  UserManager();

  static std::mutex _mutex;
  static std::shared_ptr<UserManager> _instance;
  std::shared_ptr<User> getUserByID(unsigned int userID) throw(ErrorMessage);
  std::map<unsigned int, std::shared_ptr<User>> _mapUserIDUser;
  static bool _masterCreated;
  static bool _masterExist;
};

#endif /* SRC_CORE_USER_USERENCRYPTDECRYPT_USERMANAGER_H_ */
