/******************************************************************
 *FILE: UserEncryptDecrypt_UserManager.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/ /*
/**
* @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
* @date Dec, 2017
*/

#include "core/user/UserEncryptDecrypt_UserManager.h"
#include <boost/current_function.hpp>
#include <boost/optional/optional.hpp>
#include <stdexcept>
#include "core/UserEncryptDecrypt_Configurations.h"
#include "domains/UserEncryptDecrypt_SaltDomain.h"
#include "domains/UserEncryptDecrypt_UserIDDomain.h"

std::mutex UserManager::_mutex;

std::shared_ptr<UserManager> UserManager::_instance = nullptr;
bool UserManager::_masterExist;
bool UserManager::_masterCreated;

UserManager::UserManager() {
  _mapUserIDUser.emplace(Configurations::getInstance()->get<unsigned int>(
                             Configurations::Conf::GUEST_AUTH_USER_ID),
                         nullptr);
  _masterExist = Configurations::getInstance()->get<bool>(
      Configurations::Conf::ACTIVE_MASTER_USER);
  _masterCreated = false;
}

UserManager::~UserManager() {}

#ifdef TESTING
std::shared_ptr<UserManager>& UserManager::getInstance(bool createNew) {
  if (createNew || !_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (createNew || !_instance) {
      _instance.reset(new UserManager());
    }
  }
  return _instance;
}
#else
std::shared_ptr<UserManager>& UserManager::getInstance() {
  if (!_instance) {
    std::lock_guard<std::mutex> lock(_mutex);
    if (!_instance) {
      try {
        _instance.reset(new UserManager());
      } catch (std::bad_alloc&) {
        THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
      }
    }
  }
  return _instance;
}
#endif

bool UserManager::isMasterUser(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  if (dynamic_cast<MasterUser*>(getUserByID(userID).get()) != nullptr) {
    return true;
  }

  return false;
}

bool UserManager::isGuestUser(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  return userID ==
         Configurations::getInstance()->get<unsigned int>(
             Configurations::Conf::GUEST_AUTH_USER_ID);
}

bool UserManager::createUser(unsigned int userID,
                             std::vector<unsigned char> hashPassSalt,
                             unsigned int wrapperKeyID,
                             std::vector<unsigned char> encDataKey,
                             std::vector<unsigned char> hashDataKey,
                             std::vector<unsigned char> hashWrapperKey,
                             std::vector<unsigned char> salt) {
  isValidInDomain<UserID>(userID);
  isValidInDomain<Salt>(salt);
  if (!hasUser(userID)) {
    std::shared_ptr<User> u =
        std::make_shared<User>(userID, hashPassSalt, wrapperKeyID, encDataKey,
                               hashDataKey, hashWrapperKey, salt);
    _mapUserIDUser.emplace(userID, u);
    return true;
  }
  return false;
}

bool UserManager::createMaster(unsigned int userID,
                               std::vector<unsigned char> hashPassSalt,
                               unsigned int wrapperKeyID,
                               std::vector<unsigned char> encDataKey,
                               std::vector<unsigned char> hashDataKey,
                               std::vector<unsigned char> hashWrapperKey,
                               std::vector<unsigned char> salt) {
  isValidInDomain<UserID>(userID);
  isValidInDomain<Salt>(salt);
  if (!hasUser(userID) && !hasActiveMaster()) {
    std::shared_ptr<User> u = std::make_shared<MasterUser>(
        userID, hashPassSalt, wrapperKeyID, encDataKey, hashDataKey,
        hashWrapperKey, salt);
    _mapUserIDUser.emplace(userID, u);
    _masterCreated = true;
    return true;
  }
  return false;
}

bool UserManager::createUser(unsigned int userID, unsigned int wrapperKeyID,
                             std::vector<unsigned char> encDataKey,
                             std::vector<unsigned char> hashDataKey,
                             std::vector<unsigned char> hashWrapperKey) {
  isValidInDomain<UserID>(userID);
  if (!hasUser(userID)) {
    std::shared_ptr<User> u = std::make_shared<User>(
        userID, wrapperKeyID, encDataKey, hashDataKey, hashWrapperKey);
    _mapUserIDUser.emplace(userID, u);
    return true;
  }
  return false;
}

bool UserManager::createMaster(unsigned int userID, unsigned int wrapperKeyID,
                               std::vector<unsigned char> encDataKey,
                               std::vector<unsigned char> hashDataKey,
                               std::vector<unsigned char> hashWrapperKey) {
  isValidInDomain<UserID>(userID);
  if (!hasUser(userID) && !hasActiveMaster()) {
    std::shared_ptr<User> u = std::make_shared<MasterUser>(
        userID, wrapperKeyID, encDataKey, hashDataKey, hashWrapperKey);
    _mapUserIDUser.emplace(userID, u);
    _masterCreated = true;
    return true;
  }
  return false;
}

bool UserManager::deleteUser(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  if (hasUser(userID)) {
    if (isMasterUser(userID) && _masterCreated == true) {
      _masterCreated = false;
    }
    auto iterator = _mapUserIDUser.find(userID);
    _mapUserIDUser.erase(iterator);
    return true;
  }
  return false;
}

bool UserManager::hasUser(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  if (!isGuestUser(userID)) {
    return _mapUserIDUser.find(userID) != _mapUserIDUser.end();
  }
  return false;
}

bool UserManager::userHasPassphrase(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  if (hasUser(userID)) {
    return getUserByID(userID)->getHashPassSalt() != boost::none;
  }
  return false;
}

std::shared_ptr<User> UserManager::getUserByID(unsigned int userID) throw(
    ErrorMessage) {
  isValidInDomain<UserID>(userID);
  if (hasUser(userID)) {
    return _mapUserIDUser[userID];
  }

  THROW_UED_EXCEPTION(ErrType::UM_UserDoesNotExist);
}

std::string UserManager::createMUK(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  std::shared_ptr<User> u = getUserByID(userID);
  return u->createMUK();
}

bool UserManager::compareHashPassSalt(unsigned int userID,
                                      std::vector<unsigned char> hashPassSalt) {
  isValidInDomain<UserID>(userID);
  if (hasUser(userID)) {
    return getUserByID(userID)->getHashPassSalt() == hashPassSalt;
  }
  return false;
}

void UserManager::insertHashPassSalt(unsigned int userID,
                                     std::vector<unsigned char> hashPassSalt) {
  isValidInDomain<UserID>(userID);
  getUserByID(userID)->setHashPassSalt(hashPassSalt);
}

void UserManager::deleteHashPassSalt(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  getUserByID(userID)->setHashPassSalt(boost::none);
}

void UserManager::insertDataKeyID(unsigned int userID, unsigned int dataKeyID) {
  isValidInDomain<UserID>(userID);
  getUserByID(userID)->setDataKeyID(dataKeyID);
}

void UserManager::deleteDataKeyID(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  getUserByID(userID)->setDataKeyID(boost::none);
}

void UserManager::insertWrapperKeyID(unsigned int userID,
                                     unsigned int wrapperKeyID) {
  isValidInDomain<UserID>(userID);
  getUserByID(userID)->setWrapperKeyID(wrapperKeyID);
}

void UserManager::insertEncDataKey(unsigned int userID,
                                   std::vector<unsigned char> encDataKey) {
  isValidInDomain<UserID>(userID);
  getUserByID(userID)->setOldEncDataKey(getUserByID(userID)->getEncDataKey());
  getUserByID(userID)->setEncDataKey(encDataKey);
}

void UserManager::insertHashWrapperKey(
    unsigned int userID, std::vector<unsigned char> hashWrapperKey) {
  isValidInDomain<UserID>(userID);
  getUserByID(userID)->setHashWrapperKey(hashWrapperKey);
}

void UserManager::insertSalt(unsigned int userID,
                             std::vector<unsigned char> salt) {
  isValidInDomain<UserID>(userID);
  isValidInDomain<Salt>(salt);
  getUserByID(userID)->setSalt(salt);
}

void UserManager::deleteSalt(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  getUserByID(userID)->setSalt(boost::none);
}

unsigned int UserManager::getWrapperKeyID(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  return getUserByID(userID)->getWrapperKeyID();
}

unsigned int UserManager::getDataKeyID(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  return getUserByID(userID)->getDataKeyID().get();
}

std::vector<unsigned char> UserManager::getEncDataKey(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  return getUserByID(userID)->getEncDataKey();
}

std::vector<unsigned char> UserManager::getSalt(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  if (getUserByID(userID)->getSalt().is_initialized())
    return getUserByID(userID)->getSalt().get();
  else
    return std::vector<unsigned char>();
}

bool UserManager::keyCompare(std::vector<unsigned char> hash) {
  for (auto& iterator : _mapUserIDUser) {
    if (!isGuestUser(iterator.first) &&
        (iterator.second->getHashWrapperKey() == hash ||
         iterator.second->getHashDataKey() == hash)) {
      return true;
    }
  }
  return false;
}

std::list<unsigned int> UserManager::getCreatedUsers() {
  std::list<unsigned int> users;
  for (auto& iterator : _mapUserIDUser) {
    if (!isGuestUser(iterator.first)) {
      users.push_front(iterator.first);
    }
  }

  return users;
}

bool UserManager::canEncrypt(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  User* u = getUserByID(userID).get();
  if (hasUser(userID)) {
    if (isMasterUser(userID)) {
      MasterUser* m = dynamic_cast<MasterUser*>(u);
      return m->canEncrypt();
    }
    return u->canEncrypt();
  }
  return false;
}

bool UserManager::hasActiveMaster() { return (_masterExist && _masterCreated); }

std::vector<unsigned char> UserManager::getEncWrapperKey(unsigned int userID) {
  isValidInDomain<UserID>(userID);
  return getUserByID(userID)->getEncWrapperKey();
}

void UserManager::insertEncWrapperKey(unsigned int userID,
                                      std::vector<unsigned char> data) {
  isValidInDomain<UserID>(userID);
  getUserByID(userID)->set_encWrapperKey(data);
}
