/******************************************************************
 *FILE: UserEncryptDecrypt_UEDPacketManager.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UEDPacketManager
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#include "UserEncryptDecrypt_UEDPacketManager.h"
#include <stdexcept>

std::string UEDPacketManager::createUEDPacketToken(unsigned int userID) {
  Token token             = Token(userID);
  std::string tokenString = token.encode();
  UEDPacket uedPacket     = UEDPacket(UEDPacketDataType::TOKEN, tokenString);
  return uedPacket.encode();
}

boost::optional<unsigned int> UEDPacketManager::getUserIDFromUEDPacketToken(
    std::string& packetString) {
  boost::optional<Token> token = getTokenObject(packetString);

  if (token != boost::none) {
    return token.get().getUserID();
  }

  return boost::none;
}

bool UEDPacketManager::validateUEDPacketToken(std::string& packetString) {
  boost::optional<Token> token = getTokenObject(packetString);

  if (token != boost::none) {
    return true;
  }

  return false;
}

std::vector<unsigned char> UEDPacketManager::createUEDPacket(
    std::vector<unsigned char>& data) {
  std::string dataString;
  std::string encodedPacket;

  try {
    dataString.append(data.begin(), data.end());
  } catch (std::bad_alloc&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  } catch (std::length_error&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  UEDPacket uedPacket = UEDPacket(UEDPacketDataType::CYPHERTEXT, dataString);
  encodedPacket       = uedPacket.encode();

  return std::vector<unsigned char>(encodedPacket.begin(), encodedPacket.end());
}

boost::optional<std::vector<unsigned char>>
UEDPacketManager::getDataFromUEDPacket(std::vector<unsigned char>& packet) {
  boost::optional<std::vector<unsigned char>> payload =
      getDataFromUEDPacketPrivate(packet);

  if (payload != boost::none) {
    return payload.get();
  }

  return boost::none;
}

bool UEDPacketManager::validateUEDPacket(std::vector<unsigned char>& packet) {
  std::string packetString;

  try {
    packetString.append(packet.begin(), packet.end());
  } catch (std::bad_alloc&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  } catch (std::length_error&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  UEDPacket uedPacket = UEDPacket();
  uedPacket.decode(packetString);
  return uedPacket.isValid();
}

boost::optional<std::vector<unsigned char>>
UEDPacketManager::getDataFromUEDPacketPrivate(
    std::vector<unsigned char>& packet) {
  std::string packetString;
  try {
    packetString.append(packet.begin(), packet.end());
  } catch (std::bad_alloc&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  } catch (std::length_error&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  UEDPacket uedPacket = UEDPacket();
  uedPacket.decode(packetString);

  if (uedPacket.isValid()) {
    boost::optional<std::string> payload = uedPacket.getPayload<std::string>();
    if (payload != boost::none) {
      return std::vector<unsigned char>(payload.get().begin(),
                                        payload.get().end());
    }
  }
  return boost::none;
}

boost::optional<Token> UEDPacketManager::getTokenObject(
    std::string& packetString) {
  std::vector<unsigned char> packetBytes;
  try {
    packetBytes.insert(packetBytes.end(), packetString.begin(),
                       packetString.end());
  } catch (std::bad_alloc&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  } catch (std::length_error&) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  boost::optional<std::vector<unsigned char>> tokenBytes =
      getDataFromUEDPacketPrivate(packetBytes);

  if (tokenBytes != boost::none) {
    std::string tokenString;
    try {
      tokenString.append(tokenBytes.get().begin(), tokenBytes.get().end());
    } catch (std::bad_alloc&) {
      THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
    } catch (std::length_error&) {
      THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
    }
    Token token = Token();
    token.decode(tokenString);
    if (token.isValid()) {
      return token;
    }
  }
  return boost::none;
}
