/******************************************************************
 *FILE: UserEncryptDecrypt_Token.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author José Oliveira  (josearmando.oliveira@altran.com)
 * @date Dec, 2017
 */

#ifndef USERENCRYPTDECRYPT_USERENCRYPTDECRYPT_TOKEN_H
#define USERENCRYPTDECRYPT_USERENCRYPTDECRYPT_TOKEN_H

#include <boost/optional.hpp>
#include "core/packet/UserEncryptDecrypt_Packet.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"

/**
 * This class inherits Packet class that. The Token class is JSON Web Token
 * (JWT) compliant.
 * JSON Web Token (JWT) is a compact, safe means of representing claims to
 * be transferred between two parties. The claims in a JWT are encoded as a JSON
 * object that is used as the payload of a JSON Web Signature (JWS) structure
 * enabling the
 * claims to be digitally signed or integrity protected and encrypted.
 *
 * Tokens have the following structure (see [https://jwt.io/] to have a visual
 * representation of a JWT token):
 * - TokenHeader with the signature Algorithm and Token type metadata.
 * - TokenPayload with the actual data being encoded into the Token. This field
 * will be encrypted.
 * - TokenSignature where the hashed and signed content of the first two parts
 * ('TokenHeader'+encrypted('TokenPayload') are.
 *
 * This class can parse, validate, encode and sign such tokens.
 *
 * The format of tokens will be in the form of:
 * -Header: contains typ and alg that corresponds to data inside the payload
 * -Payload: contains the userID
 * -Signature: contains an hash to verify if the Header and Payload are the
 * same as the ones presents on the Signature
 *
 * With the encrypted TokenPayload we are able to hide the relation with userID
 * and Token. And with the TokenSignature being hashed and signed we are able
 * to validate the issuer and verify integrity of the Token.
 * This Token can then be encoded to a base64 string with an encrypted
 * TokenPayload making it safe to transmit through open channels.
 */
class Token : private Packet {
 private:
  static std::string _userIDStr;
  static std::string _typStr;
  static std::string _algStr;
  boost::optional<unsigned int> _userID;

  void buildToken();

 public:
  /**
   * This constructor only can be used when a decode action is needed.
   * If an encode request is called and the header and payload are empty, it
   * will generate an invalid Token
   */
  Token();
  /**
   * This constructor enables to build a token with userID
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   */
  Token(unsigned int userID);

  /**
     * @brief Destructor for the class.
     */
  ~Token();

  /**
   * @brief This method retrieves a field content inside the token header.
   *
   * @param key Name of header field
   *
   * @return Returns the content of an header field
   */
  template <typename T>
  boost::optional<T> getHeaderField(std::string key) {
    return _header.getField<T>(key);
  }
  /**
   *@brief Gets the value of the userID associated with the token in the
   * TokenPayload.
   *
   *@return returns the value of payload userID.
   */
  boost::optional<unsigned int> getUserID();

  /**
   * @brief Encodes a Token in a JWT compliant format.
   *
   *   @return returns a base64 string with format TokenHeader::encode()+'.'+
   * TokenPayload::encode() +'.'+ TokenSignature::encode()
   */
  std::string encode();

  /**
   * @brief Decodes and verify the authenticity of a Token from a given base64
   * string.
   *
   * @param str base64 string from designated Token
   *
   *   @return returns True if the TokenSignature is valid, returns false
   * otherwise.
   */
  bool decode(std::string str);

  /**
   * @brief Verify if the Token is valid
   *   @return return True if the Token is valid, returns false otherwise.
   */
  bool isValid();
};

#endif  // USERENCRYPTDECRYPT_USERENCRYPTDECRYPT_TOKEN_H
