/******************************************************************
 *FILE: UserEncryptDecrypt_PacketSignature.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: PacketSignature
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#ifdef ROBS
#include <mock/RobsMock.h>
#else
#include <boost/property_tree/exceptions.hpp>
#endif

#include <algorithm>
#include <string>
#include "UserEncryptDecrypt_PacketSignature.h"
#include "UserEncryptDecrypt_Utils.h"
#include "backends/UserEncryptDecrypt_BackendManager.h"
#include "core/UserEncryptDecrypt_Configurations.h"
#include "domains/UserEncryptDecrypt_Base64Domain.h"

PacketSignature::PacketSignature() {
  _sdcBackend = BackendManager::getInstance()->get<SDCBackend>();
  initSignatureKey();
}

PacketSignature::PacketSignature(PacketHeader ph, PacketPayload pp) {
  _payload    = pp;
  _header     = ph;
  _sdcBackend = BackendManager::getInstance()->get<SDCBackend>();
  initSignatureKey();
}

PacketSignature::~PacketSignature() {}

void PacketSignature::initSignatureKey() {
  unsigned int kid = Configurations::getInstance()->get<unsigned int>(
      Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID);
  std::vector<unsigned char> key = _sdcBackend->GetRandom(32);
  _sdcBackend->Store(key, kid);
}

void PacketSignature::generateSignature() throw(ErrorMessage) {
  std::vector<unsigned char> hash1 = digestHeaderData(_header);
  std::vector<unsigned char> hash2 = digestPayloadData(_payload);

  unsigned int kid = Configurations::getInstance()->get<unsigned int>(
      Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID);

  std::vector<unsigned char> hmac1 = _sdcBackend->Encrypt(
      kid, std::vector<unsigned char>(hash1.begin(), hash1.end()),
      SDCBackend::Algorithm::AES, SDCBackend::AlgorithmMode::CBC);
  std::vector<unsigned char> hmac2 = _sdcBackend->Encrypt(
      kid, std::vector<unsigned char>(hash2.begin(), hash2.end()),
      SDCBackend::Algorithm::AES, SDCBackend::AlgorithmMode::CBC);

  std::string sigHeader  = base64_encode(hmac1.data(), hmac1.size());
  std::string sigPayload = base64_encode(hmac2.data(), hmac2.size());

  isValidInDomain<Base64>(sigHeader);
  isValidInDomain<Base64>(sigPayload);
  try {
    put("sigHeader", sigHeader);
    put("sigPayload", sigPayload);
  } catch (boost::property_tree::ptree_error& e) {
    THROW_UED_EXCEPTION(ErrType::DOMAIN_notValid);
  }
}

std::string PacketSignature::encode() throw(ErrorMessage) {
  generateSignature();
  std::ostringstream ss;
  bptree::ptree* pt = this;

  try {
    bptree::write_json(ss, *pt);
  } catch (boost::property_tree::json_parser_error&) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }
  std::string str = ss.str();

  std::vector<unsigned char> tst(str.begin(), str.end());
  std::string encodedSignature = base64_encode(&tst[0], tst.size());
  isValidInDomain<Base64>(encodedSignature);

  return encodedSignature;
}

void PacketSignature::decode(std::string str) throw(ErrorMessage) {
  isValidInDomain<Base64>(str);

  std::vector<unsigned char> dec = base64_decode(str);
  std::stringstream ss(std::string(dec.begin(), dec.end()));
  bptree::ptree pt;
  try {
    bptree::read_json(ss, pt);
    swap(pt);
  } catch (boost::property_tree::json_parser_error&) {
    THROW_UED_EXCEPTION(ErrType::PCKT_InvalidFormat);
  }
}

bool PacketSignature::verifySignature(PacketHeader ph,
                                      PacketPayload pp) throw(ErrorMessage) {
  std::vector<unsigned char> hash1 = digestHeaderData(ph);
  std::vector<unsigned char> hash2 = digestPayloadData(pp);

  unsigned int kid = Configurations::getInstance()->get<unsigned int>(
      Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID);

  std::string sigHeader  = get<std::string>("sigHeader");
  std::string sigPayload = get<std::string>("sigPayload");
  isValidInDomain<Base64>(sigHeader);
  isValidInDomain<Base64>(sigPayload);
  std::vector<unsigned char> lhmac1 = base64_decode(sigHeader);
  std::vector<unsigned char> lhmac2 = base64_decode(sigPayload);

  std::vector<unsigned char> lhash1 =
      _sdcBackend->Decrypt(kid, lhmac1, SDCBackend::AES, SDCBackend::CBC);
  std::vector<unsigned char> lhash2 =
      _sdcBackend->Decrypt(kid, lhmac2, SDCBackend::AES, SDCBackend::CBC);

  return (lhash1 == hash1) && (lhash2 == hash2);
}

std::vector<unsigned char> PacketSignature::digestHeaderData(PacketHeader ph) {
  std::string enc = ph.encode();
  std::vector<unsigned char> dataArr(enc.begin(), enc.end() + 1);
  std::vector<unsigned char> hash = _sdcBackend->Digest(dataArr);

  return hash;
}

std::vector<unsigned char> PacketSignature::digestPayloadData(
    PacketPayload pp) {
  std::string enc = pp.encodeB64();
  std::vector<unsigned char> dataArr(enc.begin(), enc.end() + 1);
  std::vector<unsigned char> hash = _sdcBackend->Digest(dataArr);

  return hash;
}
