/******************************************************************
 *FILE: UserEncryptDecrypt_PacketPayload.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: PacketPayload
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#ifndef USERENCRYPTDECRYPT_PACKETPAYLOAD_H_
#define USERENCRYPTDECRYPT_PACKETPAYLOAD_H_

#include <boost/optional.hpp>
#include <map>
#include <string>
#include "error/UserEncryptDecrypt_ErrorMessage.h"

#ifdef ROBS
#include <mock/RobsMock.h>
#else
#include <boost/property_tree/json_parser.hpp>
#include <boost/property_tree/ptree.hpp>
#endif

namespace bptree = boost::property_tree;

/**
 * This class extends boost::property_tree::ptree
 * [http://www.boost.org/doc/libs/1_65_1/doc/html/property_tree.html]
 * and generates a PacketPayload used by Packet.\n
 * PacketPayload has the core information to be stored into Packet. This class
 * will be encoded and can be encrypted.
 */
class PacketPayload : private bptree::ptree {
 public:
  /**
   * @brief Default constructor for the class.
   */
  PacketPayload();
  /**
   * @brief Destructor for the class
   */
  ~PacketPayload();

  /**
   * @brief Encrypt and encodes a PacketPayload in a base64 string
   *
   *   @return returns a base64 string with the encoded information.
   */
  std::string encode();

  /**
   * @brief Encodes a PacketPayload in a base64 string.
   *
   *   @return returns a base64 string with the encoded information.
   */
  std::string encodeB64() throw(ErrorMessage);

  /**
   * @brief Decodes a PacketPayload from a given base64 string and decrypt using
   * an algorithm
   *
   * @param str base64 string from designated PacketPayload
   *
   */
  void decode(std::string str) throw(ErrorMessage);

  /**
   * @brief Decodes a PacketPayload from a given base64 string.
   *
   * @param str base64 string from designated PacketPayload
   *
   */
  void decodeB64(std::string) throw(ErrorMessage);

  /**
   * @brief Get a field from PacketPayload
   *
   * @param key is the name of field inside PacketPayload
   *
   * @returns returns the value, the type is dynamic and assigned in runtime
   */
  template <typename T>
  boost::optional<T> getField(std::string key = "payload") {
    if (find(key) != not_found()) {
      return get<T>(key);
    }

    return boost::none;
  }

  /**
   * @brief add the value related with key inside the ptree inherited from boost
   * lib
   *
   * @param key is the identifier of value
   * @param value is the value that we want to store in the payload
   */
  void addField(std::string key, std::string value);
  void addField(std::string key, int value);
  void addField(std::string key, double value);
  void addField(std::string key, unsigned int value);
  /**
   * @brief add the value related with key inside the ptree inherited from boost
   * lib
   *
   * @param fields is a collection of values that we want to store in the
   * payload
   */
  void addField(std::map<std::string, std::string> fields);
  /**
   * @brief add the value related with key inside the ptree inherited from boost
   * lib
   *
   * @param key is the identifier of value
   * @param fields is a ptree and will be a child of ptree
   */
  void addField(std::string key, bptree::ptree fields) throw(ErrorMessage);
  /**
   * @brief remove the value and key inside the ptree inherited from boost lib
   *
   * @param key is the identifier to remove from ptree. If it exists True is
   * returned, otherwise False is returned
   */
  bool removeField(std::string key);

#ifdef TESTING
  void print() {
    std::ostringstream ss;
    bptree::ptree* pt = this;
    bptree::write_json(ss, *pt);
    std::cout << "Payload: " << ss.str() << std::endl;
  }
#endif
 private:
  template <typename T>
  void addFieldPrivate(std::string key, T value) throw(ErrorMessage);
};
#endif  // USERENCRYPTDECRYPT_PACKETPAYLOAD_H_
