/******************************************************************
 *FILE: UserEncryptDecrypt_Packet.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: Packet
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Artur Bento  (artur.bento@altran.com)
 * @date Jan, 2018
 */

#ifndef USERENCRYPTDECRYPT_PACKET_H_
#define USERENCRYPTDECRYPT_PACKET_H_

#include <string>
#include "UserEncryptDecrypt_PacketHeader.h"
#include "UserEncryptDecrypt_PacketPayload.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"

/**
 * This class is JSON Web Token (JWT) compliant.
 * JSON Web Token (JWT) is a compact, safe means of representing claims to
 * be transferred between two parties. The claims in a JWT are encoded as a JSON
 * object that is used as the payload of a JSON Web Signature (JWS) structure
 * enabling the
 * claims to be digitally signed or integrity protected and encrypted.
 *
 * Tokens have the following structure (see [https://jwt.io/] to have a visual
 * representation of a JWT token):\n
 * - TokenHeader with the signature Algorithm and Token type metadata.
 * - TokenPayload with the actual data being encoded into the Token. This field
 * will be encrypted.
 * - TokenSignature where the hashed and signed content of the first two parts
 * ('TokenHeader'+encrypted('TokenPayload') are.
 *
 * This class can parse, validate, encode and sign such tokens.
 *
 * The format of Packet will be in the form of: PacketHeader + PacketPayload +
 * PacketSignature
 * With the TokenSignature being hashed and signed we are able to validate the
 * issuer and verify integrity of the Packet.
 * This Packet can then be encoded to a base64 string with an encrypted
 * PacketPayload making it safe to transmit through open channels.
 */
class Packet {
 private:
  bool _isValid;
  bool _encryptPayload;
#ifdef TESTING
 public:
#else
 protected:
#endif
  PacketHeader _header;
  PacketPayload _payload;

 public:
  /**
   * @brief Constructor for the class.
   *
   * @param encryptPayload In case of the caller want to encrypt the payload
   * data.
   */
  Packet(bool encryptPayload);
  /**
   * @brief Destructor for the class
   */
  virtual ~Packet();

  /**
   * @brief Encodes a Packet in a JWT compliant format.
   *
   *   @return returns a base64 string with format:
   *   PacketHeader::encode()+'.'+ PacketPayload::encode() +'.'+
   * TokenSignature::encode() or
   *   PacketHeader::encode()+'.'+ PacketPayload::encodeB64() +'.'+
   * TokenSignature::encode()
   *   depending on the _encryptPayload member assigned in the constructor
   */
  std::string encode() throw(ErrorMessage);

  /**
   * @brief Decodes and verify the authenticity of a Packet from a given base64
   * string.
   *
   * @param str base64 string from designated Packet
   *
   *   @return returns True if the PacketSignature is valid, returns false
   * otherwise.
   */
  bool decode(std::string str) throw(ErrorMessage);

  /**
   * @brief Verify if the Packet is valid
   *   @return return True if the Packet is valid, returns false otherwise.
   */
  bool isValid();
};
#endif  // USERENCRYPTDECRYPT_PACKET_H_
