/******************************************************************
 *FILE: UserEncryptDecrypt_Logger.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: Logger interface for UserEncryptDecrypt using DLT
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Ulisses Costa (marioulisses.costa@altran.com)
 * @date Dec, 2017
 *
 * Date      | Author             | Modification
 * 14/08/2018| AKM7COB			  | changed class name logger to logger1. class name logger is already used in ASF framework
 */

#ifndef SRC_CORE_USERENCRYPTDECRYPT_LOGGER_H_
#define SRC_CORE_USERENCRYPTDECRYPT_LOGGER_H_

#include <cstring>
#include <iostream>
#include <list>
#include <memory>
#include <mutex>
#include <sstream>
#include <vector>
#include "core/logger/UserEncryptDecrypt_LogContext.h"

/**
 * Definitions of Logger log levels, same as DLT (dlt_types.h)
 */
enum class LogLevel {
  DEFAULT = -1,   /**< Default log level */
  OFF     = 0x00, /**< Log level off */
  FATAL   = 0x01, /**< fatal system error */
  ERROR   = 0x02, /**< error with impact to correct functionality */
  WARN    = 0x03, /**< warning, correct behavior could not be ensured */
  INFO    = 0x04, /**< informational */
  DEBUG1   = 0x05, /**< debug  */
  VERBOSE = 0x06  /**< highest grade of information */
};

/**
 * Definitions of Logger output.
 * DLT: Send the message to DLT Viewer.
 * STDOUT: Send the message to stdout console.
 */
enum class LoggerOutput { DLT, STDOUT };

/**
 * This class allow to register important system messages in DLT Viewer
 * application.
 * It provides methods to handle the logging system, such as register/unregister
 * application, context and log the messages.
 *
 * A typical class permits callers to create as many instances of the class as
 * they want, but this class use the Singleton Pattern.
 * A singleton object provides a global point of access to the resources of its
 * class and returns the same instance no matter how many times an application
 * requests it.
 * The function Logger1::getInstance() provides the access to the
 * resources of this class.
 *
 * To log a message it is necessary to register our application and a LogContext
 * in the DLT Viewer application.
 * This can be done following the next example:
 * Logger1::getLogger()->registerApplication("UEDS","User Encrypt Decrypt
 * Application");
 * std::shared_ptr<Logger1> logger = Logger1::getLogger();
 * std::shared_ptr<LogContext> logContext = logger->registerContext("LOG",
 * "UserEncryptDecrypt_LogContext");
 * Finally to log the message we use the context and a LogLevel to identify the
 * message in the DLT Viewer application.
 * logger->log(logContext, LogLevel::INFO, "This is an example to Log a message
 * in DLT Viewer, using the context 'LOG' with a description of
 * 'UserEncryptDecrypt_LogContext'");
 */
class Logger1 {
 public:
  /**
   * @brief Destructor for the class
   */
  virtual ~Logger1();
  /**
   *   @brief Provides an instance of the Logger1 class object.
   *
   *   This function provides the access to class object, but only one instance
   * is returned as mentioned above in the definition of singleton pattern.
   *
   *   @return returns a shared pointer of the instance of the Logger1
   */
  static std::shared_ptr<Logger1>& getLogger();

  void setLoggerOutput(LoggerOutput out) { _loggerOut = out; }
  /**
   * @brief This function is used to register our application in the DLT Viewer
   * application.
   *
   * @param appID an application identifier
   * @param appDescription an application description
   */
  void registerApplication(std::string appID, std::string appDescription);
  /**
   * @brief This function is used to unregister our application in the DLT
   * Viewer application.
   * Log messages can't be logged using the application identifier that was
   * unregistered.
   */
  void unregisterApplication();
  /**
   * @brief This function is used to register an application context in the DLT
   * Viewer application.
   *
   * @param contextID an application context identifier
   * @param contextDescription an application context description
   *
   * @return Returns a LogContext shared pointer object allowing to register log
   * messages with the intended context.
   */
  std::shared_ptr<LogContext> registerContext(
      const std::string& contextID, const std::string& contextDescription);
  /**
   * @brief This function is used to unregister a context.
   *
   * @param contextID an application context identifier string.
   */
  void unregisterContext(const std::string& contextID);
  /**
   * @brief This function is used to unregister a context.
   *
   * @param ctx an application context shared pointer object.
   */
  void unregisterContext(std::shared_ptr<LogContext> ctx);

  /**
   * @brief This function is used to log messages. It required at least four
   * arguments.
   *
   * @param context The context of message that want to be logged
   * @param logLevel The priority of this message
   * @param value The message to be logged
   * @param args This enable to send more parameters to DLT Viewer
   */

  template <typename T, typename... Args>
  void log(std::shared_ptr<LogContext> context, LogLevel logLevel,
           const T& value, const Args&... args) {
    std::stringstream stream;

    stream << value;
    _string.insert(_string.size(), stream.str());

    log(context, logLevel, args...);
  }
  /**
   * @brief This function is used to log messages.
   *
   * @param context The context of message that want to be logged
   * @param logLevel The priority of this message
   * @param value The message to be logged
   */
  template <typename T>
  void log(std::shared_ptr<LogContext> context, LogLevel logLevel,
           const T& value) {
    std::stringstream stream;
    LogLevel logLevelFixed;

    if (logLevel == LogLevel::DEBUG1 || logLevel == LogLevel::VERBOSE) {
      logLevelFixed = LogLevel::INFO;
    } else {
      logLevelFixed = logLevel;
    }

    stream << value;
    _string.insert(_string.size(), stream.str());

    if (_loggerOut == LoggerOutput::DLT) {
      DltContext ctx        = context->getDltContext();
      DltLogLevelType level = static_cast<DltLogLevelType>(logLevelFixed);

      DLT_LOG(ctx, level, DLT_CSTRING(_string.c_str()));
    } else {
      std::cout << "[" << context->getContextId() << "] " << _string
                << std::endl;
    }
    _string.clear();
  }

 private:
  Logger1();

  std::string _string;
  static std::mutex _mutex;
  static std::mutex _logActionLock;
  static std::shared_ptr<Logger1> _instance;
  std::list<std::shared_ptr<LogContext>> _contexts;
  LoggerOutput _loggerOut;
  std::string _appID;
};

#endif /* SRC_CORE_USERENCRYPTDECRYPT_LOGGER_H_ */
