/******************************************************************
 *FILE: UserEncryptDecrypt_SaltDatabase.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2018 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author José Oliveira  (josearmando.oliveira@altran.com)
 * @date Jan, 2018
 */

#ifndef USERENCRYPTDECRYPT_SALTDATABASE_H
#define USERENCRYPTDECRYPT_SALTDATABASE_H

#include <memory>
#include <mutex>
#include <vector>
#include "core/UserEncryptDecrypt_ConsistentFile.h"

/**
 * This class provides methods that support the Authentication class in regards
 * to the salt database file management.
 *
 * A typical class permits callers to create as many instances of the class as
 * they want, but this class use the Singleton Pattern.
 * A singleton object provides a global point of access to the resources of its
 * class and returns the same instance no matter how many times an application
 * requests it.
 * The function AuthenticationDatabase::getInstance() provides the access to the
 * resources of this class.
 */
class SaltDatabase {
 public:
  /**
   * @brief Destructor for the class
   */
  virtual ~SaltDatabase();

/**
 *   @brief Provides an instance of the SaltDatabase class object.
 *
 *   This function provides the access to class object, but only one instance
 * is returned as mentioned above in the definition of singleton pattern.
 *
 *   @return returns a shared pointer of the instance of the SaltDatabase
 */
#ifdef TESTING
  static std::shared_ptr<SaltDatabase> getInstance(bool reload = true);
#else
  static std::shared_ptr<SaltDatabase> getInstance();
#endif

  /**
   * @brief Checks if the given salt exists in the database
   *
   * @param salt is the salt used for the generation of the user hash.
   *
   * @return returns true if the salt already exists and false otherwise.
   */
  bool exists(std::vector<unsigned char> salt);

  /**
   * @brief Checks if the given salt exists in the database and if it does not
   * it inserts the provided salt in the database
   *
   * @param salt is the salt used for the generation of the user hash.
   *
   * @return returns true if the salt did not exist and the operation was
   * performed and false otherwise.
   */
  bool insertIfNotExists(std::vector<unsigned char> salt);

 private:
  static std::shared_ptr<SaltDatabase> _instance;
  static std::mutex _mutex;
  std::shared_ptr<ConsistentFile> _file;
  SaltDatabase();
  bool insert(std::vector<unsigned char> salt);
};

#endif  // USERENCRYPTDECRYPT_USERENCRYPTDECRYPT_SALTDATABASE_H
