/******************************************************************
 *FILE: UserEncryptDecrypt_AuthenticationDatabase.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Dec, 2017
 *
 * Date      | Author             | Modification
 * 14/08/2018| AKM7COB			  | changed class name logger to logger1. class name logger is already used in ASF framework
 */

#ifndef SRC_BACKENDS_AUTHENTICATION_USERENCRYPTDECRYPT_AUTHENTICATIONDATABASE_H_
#define SRC_BACKENDS_AUTHENTICATION_USERENCRYPTDECRYPT_AUTHENTICATIONDATABASE_H_

#include <memory>
#include <mutex>
#include <string>
#include "core/UserEncryptDecrypt_ConsistentFile.h"
#include "error/UserEncryptDecrypt_ErrorMessage.h"

class Logger1;
class LogContext;
class UserManager;

/**
 * This class provides methods that support the Authentication class in regards
 * to the database file management.
 *
 * A typical class permits callers to create as many instances of the class as
 * they want, but this class use the Singleton Pattern.
 * A singleton object provides a global point of access to the resources of its
 * class and returns the same instance no matter how many times an application
 * requests it.
 * The function AuthenticationDatabase::getInstance() provides the access to the
 * resources of this class.
 */

class AuthenticationDatabase {
 public:
  /**
   * @brief Destructor for the class
   */
  virtual ~AuthenticationDatabase();

/**
 *   @brief Provides an instance of the AuthenticationDatabase class object.
 *
 *   This function provides the access to class object, but only one instance
 * is returned as mentioned above in the definition of singleton pattern.
 *
 *   @return returns a shared pointer of the instance of the
 * AuthenticationDatabase
 */
#ifdef TESTING
  static std::shared_ptr<AuthenticationDatabase>& getInstance(bool createNew);
  static std::shared_ptr<AuthenticationDatabase>& getInstance();
#else
  static std::shared_ptr<AuthenticationDatabase>& getInstance();

 private:
  static std::shared_ptr<AuthenticationDatabase>& getInstance(bool createNew);

 public:
#endif

  /**
   * @brief Loads the stored values from the database file.
   */
  void loadFromFile() throw(ErrorMessage);

  /**
   * @brief Saves the MUK(formatted data to insert in the database) of the user
   * with the provided user ID into the database.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   */
  void saveMUK(unsigned int userID) throw(ErrorMessage);

  /**
   * @brief Deletes the MUK of the user with the provided user ID from the
   * database.
   *
   * @param userID is the unique identifier of the user inside the A-IVI.
   */
  void deleteFromFile(unsigned int userID) throw(ErrorMessage);

  /**
   * @brief Returns the current value of _wrongTries.
   */
  int getWrongTries();

  /**
   * @brief Increments the value of the _wrongTries variable.
   *
   * @return returns the current number of wrong tries.
   */
  int incrementWrongTries();

  /**
   * @brief Resets the value of wrongTries to 0.
   */
  void resetWrongTries();

 private:
  AuthenticationDatabase();

  void createFile();
  int checkKey(std::string line);

  void checkFile();

  std::string createMUK(unsigned int userID) throw(ErrorMessage);

  void saveWrongTries(int wrongTries) throw(ErrorMessage);

  size_t getNextDelimiter(std::string line) throw(ErrorMessage);

  static int _wrongTries;
  static std::shared_ptr<AuthenticationDatabase> _instance;
  static std::mutex _mutex;
  std::shared_ptr<ConsistentFile> _consistentFile;
  std::shared_ptr<UserManager> _userManager;
  char _delimiter;
  std::shared_ptr<Logger1> _logger;
  std::shared_ptr<LogContext> _ctx;
};

#endif /* SRC_BACKENDS_AUTHENTICATION_USERENCRYPTDECRYPT_AUTHENTICATIONDATABASE_H_ \
          */
