/******************************************************************
 *FILE: UserEncryptDecrypt_Authentication.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Nov, 2017
 */

#include "core/authentication/UserEncryptDecrypt_Authentication.h"
#include <UserEncryptDecrypt_Utils.h>
#include <fstream>
#include "UserEncryptDecrypt_SaltDatabase.h"
#include "backends/UserEncryptDecrypt_BackendManager.h"
#include "backends/openssl/UserEncryptDecrypt_OpensslBackend.h"
#include "core/uedpacket/UserEncryptDecrypt_UEDPacketManager.h"
#include "core/user/UserEncryptDecrypt_UserManager.h"
#include "domains/UserEncryptDecrypt_PassphraseDomain.h"
#include "domains/UserEncryptDecrypt_SaltDomain.h"
#include "domains/UserEncryptDecrypt_UserIDDomain.h"

const std::array<unsigned int, 7> Authentication::_incrementalDelay{
	{0, 5, 60, 3600, 3600 * 3, 3600 * 6, 3600 * 12}};

int Authentication::_maxAuthUser;
std::mutex Authentication::_mutex;
std::shared_ptr<Authentication> Authentication::_instance = nullptr;
const unsigned int Authentication::SALT_SIZE              = 32;

#define KEYID_RANGE_START 14000
#define KEYID_RANGE_END 14049

Authentication::Authentication()
: _guestAuthUserID(Configurations::getInstance()->get<unsigned int>(
		Configurations::Conf::GUEST_AUTH_USER_ID)) {

	_logger = Logger1::getLogger();
	_logger->setLoggerOutput(LoggerOutput::DLT);
	_ctx = _logger->registerContext("AUTH", "UserEncryptDecrypt_Authentication");


	_logger->log(_ctx, LogLevel::INFO, "UserEncryptDecrypt_Authentication constructor");
	initKey();
	_maxAuthUser = Configurations::getInstance()->get<int>(
			Configurations::Conf::MAX_AUTH_USER);
	_userManager = UserManager::getInstance();

	try {
		_currentAuthUserID.insert(_currentAuthUserID.end(), _guestAuthUserID);
	} catch (std::bad_alloc &) {
		THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
	}

	_authenticationDatabase = AuthenticationDatabase::getInstance();
	_authenticationDatabase->loadFromFile();

	_sdcBackend     = BackendManager::getInstance()->get<SDCBackend>();
	_opensslBackend = BackendManager::getInstance()->get<OpensslBackend>();

}

Authentication::~Authentication() { _logger->unregisterContext(_ctx); }

std::shared_ptr<Authentication> &Authentication::getInstance(bool createNew) {
	if (createNew || !_instance) {
		std::lock_guard<std::mutex> lock(_mutex);
		if (createNew || !_instance) {
			_instance.reset(new Authentication());
		}
	}
	return _instance;
}

#ifdef TESTING

std::vector<unsigned char> Authentication::getWrappingKey() { return UserKey; }

#endif

std::vector<unsigned char> Authentication::callWithUserIDKey(
		unsigned int userID,
		std::function<std::vector<unsigned char>(unsigned int)> fun) {
	auto keyID = _userManager->getDataKeyID(userID);

	return fun(keyID);
}

bool Authentication::createUserCredentials(unsigned int userID,
		std::vector<unsigned char> pass,
		std::vector<unsigned char> salt,
		bool master) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<Passphrase>(pass);
	isValidInDomain<Salt>(salt);

	if (!SaltDatabase::getInstance()->insertIfNotExists(salt)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_CreateUserSaltExists);
	}

	if ((!_userManager->hasUser(userID) && master &&
			!_userManager->hasActiveMaster()) ||
			(!_userManager->hasUser(userID) && !master)) {
		bool ret = CreateUserCredentialsLogic(userID, pass, salt, master);
		_sdcBackend->Overwrite(pass);
		return ret;
	}
	THROW_UED_EXCEPTION(ErrType::AUTH_CreateUserAlreadyExists);
}

bool Authentication::createUserCredentials(unsigned int userID,
		std::vector<unsigned char> pass,
		bool master) {
	return createUserCredentials(userID, pass, generateSalt(), master);
}

bool Authentication::createUserCredentials(unsigned int userID,
		bool master) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);

	if ((!_userManager->hasUser(userID) && master &&
			!_userManager->hasActiveMaster()) ||
			(!_userManager->hasUser(userID) && !master)) {
		return CreateUserCredentialsLogic(userID, boost::none, boost::none, master);
	}
	THROW_UED_EXCEPTION(ErrType::AUTH_CreateUserAlreadyExists);
}

bool Authentication::createUserCredentials(unsigned int userID, unsigned int wKeyID,
		bool master) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);

	if ((!_userManager->hasUser(userID) && master &&
			!_userManager->hasActiveMaster()) ||
			(!_userManager->hasUser(userID) && !master)) {
		if( (wKeyID >= KEYID_RANGE_START ) || (wKeyID <= KEYID_RANGE_END))
			return CreateUserCredentialsLogic_FixedKey(userID, master, wKeyID);
		else
			return false;
	}
	THROW_UED_EXCEPTION(ErrType::AUTH_CreateUserAlreadyExists);
}

bool Authentication::CreateUserCredentialsLogic(
		unsigned int userID, boost::optional<std::vector<unsigned char>> pass,
		boost::optional<std::vector<unsigned char>> salt, bool master) {
	unsigned int wrapperKeyID;
	std::vector<unsigned char> hashPassSalt;
	std::vector<unsigned char> encDataKey;
	std::vector<unsigned char> dataKey;
	std::vector<unsigned char> wrapperKey;
	std::vector<unsigned char> hashDataKey;
	std::vector<unsigned char> hashWrapperKey;
	boost::optional<std::tuple<unsigned int, std::vector<unsigned char>,
	std::vector<unsigned char>>>
	wrapperKeyIDAndHash;
	std::vector<unsigned char> encWrapperKey;

	if (pass != boost::none && salt != boost::none) {
		_logger->log(_ctx, LogLevel::INFO, "key derivation ");
		wrapperKeyIDAndHash = keyDerivation(pass.get(), salt.get());
		hashPassSalt        = concatAndHash(pass.get(), salt.get());
		_sdcBackend->Overwrite(pass.get());
		if (wrapperKeyIDAndHash == boost::none) {
			THROW_UED_EXCEPTION(ErrType::AUTH_CreateUserSaltExists);
		}
		wrapperKeyID   = std::get<0>(wrapperKeyIDAndHash.get());
		hashWrapperKey = std::get<1>(wrapperKeyIDAndHash.get());
		encWrapperKey  = std::get<2>(wrapperKeyIDAndHash.get());

	} else {
		_logger->log(_ctx, LogLevel::INFO, "random key generation");
		hashWrapperKey = generateKey(wrapperKey);
		wrapperKeyID   = _sdcBackend->Store(wrapperKey);
		encWrapperKey  = _sdcBackend->Encrypt(
				Configurations::getInstance()->get<unsigned int>(
						Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID),
						wrapperKey, SDCBackend::AES, SDCBackend::CBC);
		_sdcBackend->Overwrite(wrapperKey);
	}

	hashDataKey = generateKey(dataKey);
	encDataKey  = _sdcBackend->Encrypt(wrapperKeyID, dataKey, SDCBackend::AES,
			SDCBackend::CBC);
	_sdcBackend->Overwrite(dataKey);

	if (!master && hashPassSalt.empty()) {
		_userManager->createUser(userID, wrapperKeyID, encDataKey, hashDataKey,
				hashWrapperKey);
		_logger->log(_ctx, LogLevel::INFO, "User ", userID, " created");
	}
	if (!master && !hashPassSalt.empty()) {
		_userManager->createUser(userID, hashPassSalt, wrapperKeyID, encDataKey,
				hashDataKey, hashWrapperKey, salt.get());
		_logger->log(_ctx, LogLevel::INFO, "User ", userID, " created");
	}
	if (master && hashPassSalt.empty()) {
		_userManager->createMaster(userID, wrapperKeyID, encDataKey, hashDataKey,
				hashWrapperKey);
		_logger->log(_ctx, LogLevel::INFO, "Master ", userID, " created");
	}
	if (master && !hashPassSalt.empty()) {
		_userManager->createMaster(userID, hashPassSalt, wrapperKeyID, encDataKey,
				hashDataKey, hashWrapperKey, salt.get());
		_logger->log(_ctx, LogLevel::INFO, "Master ", userID, " created");
	}
	_userManager->insertEncWrapperKey(userID, encWrapperKey);

	_authenticationDatabase->saveMUK(userID);

	return true;
}

bool Authentication::CreateUserCredentialsLogic_FixedKey(
		unsigned int userID, bool master, unsigned int uWKeyID) {
	unsigned int wrapperKeyID = uWKeyID;
	std::vector<unsigned char> encDataKey;
	std::vector<unsigned char> dataKey;
	std::vector<unsigned char> wrapperKey;
	std::vector<unsigned char> hashDataKey;
	std::vector<unsigned char> hashWrapperKey;
	std::vector<unsigned char> encWrapperKey;

	if( (uWKeyID >= KEYID_RANGE_START) || (uWKeyID <= KEYID_RANGE_END)) {
		hashWrapperKey = generateKey(wrapperKey);
		if(!_sdcBackend->Store(wrapperKey, uWKeyID, true))
		{
			encWrapperKey  = _sdcBackend->Encrypt(
					Configurations::getInstance()->get<unsigned int>(
							Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID),
							wrapperKey, SDCBackend::AES, SDCBackend::CBC);
			_logger->log(_ctx, LogLevel::INFO, "Wkey generated and stored");
			_sdcBackend->Overwrite(wrapperKey);
		}
		else
		{
			_logger->log(_ctx, LogLevel::INFO, "SDC wkey store error");
			return false;
		}
	}else
	{
		_logger->log(_ctx, LogLevel::INFO, "Invalid keyid specified");
		return false;
	}

	hashDataKey = generateKey(dataKey);
	encDataKey  = _sdcBackend->Encrypt(wrapperKeyID, dataKey, SDCBackend::AES,
			SDCBackend::CBC);
	_sdcBackend->Overwrite(dataKey);

	if (!master) {
		_userManager->createUser(userID, wrapperKeyID, encDataKey, hashDataKey,
				hashWrapperKey);
		_logger->log(_ctx, LogLevel::INFO, "User ", userID, " created");
	}
	if (master ) {
		_userManager->createMaster(userID, wrapperKeyID, encDataKey, hashDataKey,
				hashWrapperKey);
		_logger->log(_ctx, LogLevel::INFO, "Master ", userID, " created");
	}
	_userManager->insertEncWrapperKey(userID, encWrapperKey);

	_authenticationDatabase->saveMUK(userID);

	return true;
}


bool Authentication::isUserExist(unsigned int userID) {
	isValidInDomain<UserID>(userID);

	return _userManager->hasUser(userID);
}

boost::optional<std::string> Authentication::authenticateUser(
		unsigned int userID, std::vector<unsigned char> pass,
		std::vector<unsigned char> salt) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<Passphrase>(pass);
	isValidInDomain<Salt>(salt);

	std::vector<unsigned char> hashPassSalt = concatAndHash(pass, salt);
	_sdcBackend->Overwrite(pass);

	if (_userManager->hasUser(userID) &&
			_userManager->compareHashPassSalt(userID, hashPassSalt)) {
		_logger->log(_ctx, LogLevel::INFO, "User", userID,
				"HASH Exists in UserManager");

		return authenticateUserLogic(userID);
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticateUserDoesNotExist);
	}

	if (!_userManager->compareHashPassSalt(userID, hashPassSalt)) {
		THROW_UED_EXCEPTION_WITH_TRIES(
				ErrType::AUTH_AuthenticateUserWrongCredentials,
				_authenticationDatabase->getWrongTries());
	}
	return boost::none;
}

boost::optional<std::string> Authentication::authenticateUser(
		unsigned int userID, std::vector<unsigned char> pass) {
	if (_userManager->hasUser(userID)) {
		return authenticateUser(userID, pass, _userManager->getSalt(userID));
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticateUserDoesNotExist);
	}
	return boost::none;
}

boost::optional<std::string> Authentication::authenticateUser(
		unsigned int userID) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);

	if (_userManager->hasUser(userID) &&
			!_userManager->userHasPassphrase(userID)) {
		return authenticateUserLogic(userID);
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticateUserDoesNotExist);
	}
	return boost::none;
}

boost::optional<std::string> Authentication::authenticateUserLogic(
		unsigned int userID) {
	std::vector<unsigned char> DataKey;
	unsigned int DataKeyID;

	if (isUserAuthenticated(userID)) {
		return UEDPacketManager::createUEDPacketToken(userID);
	}

	if (_currentAuthUserID.size() >= _maxAuthUser &&
			getCurrentAuthenticatedUsers().front() != _guestAuthUserID) {
		THROW_UED_EXCEPTION(ErrType::AUTH_MaximumAuthenticatedUsersReached);
	}

	insertCurrentAuthenticatedUser(userID);

	_authenticationDatabase->resetWrongTries();
	DataKey = _sdcBackend->Decrypt(_userManager->getWrapperKeyID(userID),
			_userManager->getEncDataKey(userID),
			SDCBackend::AES, SDCBackend::CBC);
	DataKeyID = _sdcBackend->Store(DataKey, false);
	_sdcBackend->Overwrite(DataKey);

	_userManager->insertDataKeyID(userID, DataKeyID);

	return UEDPacketManager::createUEDPacketToken(userID);
}

bool Authentication::isUserAuthenticated(unsigned int userID) {
	isValidInDomain<UserID>(userID);

	return std::find(_currentAuthUserID.begin(), _currentAuthUserID.end(),
			userID) != _currentAuthUserID.end();
}

bool Authentication::unauthenticateUser(unsigned int userID) throw(
		ErrorMessage) {
	isValidInDomain<UserID>(userID);

	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_UnauthenticateUserDoesNotExist);
	}

	if (isUserAuthenticated(userID)) {
		_currentAuthUserID.remove(userID);
		_sdcBackend->DeleteKeyID(_userManager->getDataKeyID(userID));
		_userManager->deleteDataKeyID(userID);
		if (_currentAuthUserID.empty()) {
			try {
				_currentAuthUserID.push_front(_guestAuthUserID);
			} catch (std::bad_alloc &) {
				THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
			}
		}
		return true;
	}
	return false;
}

bool Authentication::deleteUserCredentials(
		unsigned int userID, std::vector<unsigned char> pass,
		std::vector<unsigned char> salt) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<Passphrase>(pass);
	isValidInDomain<Salt>(salt);

	_logger->log(_ctx, LogLevel::INFO, "deleteUserCredentials full auth");

	std::vector<unsigned char> hashPassSalt = concatAndHash(pass, salt);

	_logger->log(_ctx, LogLevel::INFO,
			_userManager->compareHashPassSalt(userID, hashPassSalt)
			? "DELETE OK"
					: "DELETE WRONG");

	if (isUserAuthenticated(userID) &&
			_userManager->compareHashPassSalt(userID, hashPassSalt) &&
			(_currentAuthUserID.front() == userID ||
					(_userManager->isMasterUser(userID) &&
							_userManager->hasActiveMaster()))) {
		return deleteUserLogic(userID);
	}

	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_DeleteUserDoesNotExist);
	}

	if (!_userManager->compareHashPassSalt(userID, hashPassSalt)) {
		THROW_UED_EXCEPTION_WITH_TRIES(ErrType::AUTH_DeleteUserWrongCredentials,
				_authenticationDatabase->getWrongTries());
	}
	return false;
}

bool Authentication::deleteUserCredentials(unsigned int userID,
		std::vector<unsigned char> pass) {
	_logger->log(_ctx, LogLevel::INFO, "deleteUserCredentials full auth");
	if (_userManager->hasUser(userID)) {
		return deleteUserCredentials(userID, pass, _userManager->getSalt(userID));
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_DeleteUserDoesNotExist);
	}
	return false;
}

bool Authentication::deleteUserCredentials(unsigned int userID) throw(
		ErrorMessage) {
	isValidInDomain<UserID>(userID);

	if (Configurations::getInstance()->get<bool>(
			Configurations::RE_AUTHENTICATION) &&
			_userManager->userHasPassphrase(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticationRequired);
	}

	if (isUserAuthenticated(userID) && (_currentAuthUserID.front() == userID ||
			(_userManager->isMasterUser(userID) &&
					_userManager->hasActiveMaster()))) {
		return deleteUserLogic(userID);
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_DeleteUserDoesNotExist);
	}
	return false;
}

bool Authentication::deleteUserCredentials(
		unsigned int userID, std::vector<unsigned char> pass,
		std::vector<unsigned char> salt,
		unsigned int userIDToDelete) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<UserID>(userIDToDelete);
	isValidInDomain<Passphrase>(pass);
	isValidInDomain<Salt>(salt);

	_logger->log(_ctx, LogLevel::INFO, "deleteUserCredentials full auth");

	std::vector<unsigned char> hashPassSalt = concatAndHash(pass, salt);

	_logger->log(_ctx, LogLevel::INFO,
			_userManager->compareHashPassSalt(userID, hashPassSalt)
			? "DELETE OK"
					: "DELETE WRONG");

	if ((_currentAuthUserID.front() == userID &&
			_userManager->compareHashPassSalt(userID, hashPassSalt) &&
			!_userManager->hasActiveMaster()) ||
			(_userManager->isMasterUser(userID) &&
					_userManager->compareHashPassSalt(userID, hashPassSalt) &&
					_userManager->hasActiveMaster()) ||
					(_currentAuthUserID.front() == userID && userID == userIDToDelete)) {
		return deleteUserLogic(userIDToDelete);
	}

	wrongTry();
	if (!_userManager->hasUser(userID) ||
			!_userManager->hasUser(userIDToDelete)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_DeleteUserDoesNotExist);
	}

	if (!_userManager->compareHashPassSalt(userID, hashPassSalt)) {
		THROW_UED_EXCEPTION_WITH_TRIES(ErrType::AUTH_DeleteUserWrongCredentials,
				_authenticationDatabase->getWrongTries());
	}
	return false;
}

bool Authentication::deleteUserCredentials(unsigned int userID,
		std::vector<unsigned char> pass,
		unsigned int userIDToDelete) {
	_logger->log(_ctx, LogLevel::INFO, "deleteUserCredentials full auth");
	if (_userManager->hasUser(userID)) {
		return deleteUserCredentials(userID, pass, _userManager->getSalt(userID),
				userIDToDelete);
	}
	wrongTry();
	if (!_userManager->hasUser(userID) ||
			!_userManager->hasUser(userIDToDelete)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_DeleteUserDoesNotExist);
	}
	return false;
}

bool Authentication::deleteUserCredentials(
		unsigned int userID, unsigned int userIDToDelete) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<UserID>(userIDToDelete);

	if (Configurations::getInstance()->get<bool>(
			Configurations::RE_AUTHENTICATION) &&
			_userManager->userHasPassphrase(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticationRequired);
	}
	if ((_currentAuthUserID.front() == userID &&
			!_userManager->hasActiveMaster()) ||
			(_userManager->isMasterUser(userID) && _userManager->hasActiveMaster()) ||
			(_currentAuthUserID.front() == userID && userID == userIDToDelete)) {
		return deleteUserLogic(userIDToDelete);
	}

	wrongTry();
	if (!_userManager->hasUser(userID) ||
			!_userManager->hasUser(userIDToDelete)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_DeleteUserDoesNotExist);
	}
	return false;
}

bool Authentication::deleteAllUsers(
		unsigned int userID, std::vector<unsigned char> pass,
		std::vector<unsigned char> salt) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<Passphrase>(pass);
	isValidInDomain<Salt>(salt);

	std::vector<unsigned char> hashPassSalt = concatAndHash(pass, salt);

	if ((_currentAuthUserID.front() == userID &&
			_userManager->compareHashPassSalt(userID, hashPassSalt) &&
			!_userManager->hasActiveMaster()) ||
			(_userManager->isMasterUser(userID) &&
					_userManager->compareHashPassSalt(userID, hashPassSalt) &&
					_userManager->hasActiveMaster())) {
		bool sdcRet       = true;
		auto createdUsers = _userManager->getCreatedUsers();

		for (auto iterator = createdUsers.begin(); iterator != createdUsers.end();
				iterator++) {
			sdcRet &= deleteUserLogic(*iterator);
		}
		return sdcRet;
	}

	wrongTry();
	throwDeleteAllUsersException(userID);
	return false;
}

bool Authentication::deleteAllUsers(
		unsigned int userID, std::vector<unsigned char> pass) throw(ErrorMessage) {
	if (_userManager->hasUser(userID)) {
		return deleteAllUsers(userID, pass, _userManager->getSalt(userID));
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_DeleteUserDoesNotExist);
	}
	return false;
}

bool Authentication::deleteAllUsers(unsigned int userID) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);

	if (Configurations::getInstance()->get<bool>(
			Configurations::RE_AUTHENTICATION) &&
			_userManager->userHasPassphrase(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticationRequired);
	}

	if ((_currentAuthUserID.front() == userID &&
			!_userManager->hasActiveMaster()) ||
			(_userManager->isMasterUser(userID) && _userManager->hasActiveMaster())) {
		bool sdcRet       = true;
		auto createdUsers = _userManager->getCreatedUsers();

		for (auto iterator = createdUsers.begin(); iterator != createdUsers.end();
				iterator++) {
			sdcRet &= deleteUserLogic(*iterator);
		}
		return sdcRet;
	}
	wrongTry();
	throwDeleteAllUsersException(userID);
	return false;
}

bool Authentication::deleteUserLogic(unsigned int userID) {
	bool sdcRet;

	_authenticationDatabase->deleteFromFile(userID);
	sdcRet = _sdcBackend->DeleteKeyID(_userManager->getWrapperKeyID(userID));

	if (isUserAuthenticated(userID)) {
		sdcRet &= unauthenticateUser(userID);
	}

	_userManager->deleteUser(userID);
	_authenticationDatabase->resetWrongTries();
	return sdcRet;
}

bool Authentication::changePassphrase(
		unsigned int userID, std::vector<unsigned char> oldPassphrase,
		std::vector<unsigned char> oldSalt,
		std::vector<unsigned char> newPassphrase,
		std::vector<unsigned char> newSalt) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<Passphrase>(oldPassphrase);
	isValidInDomain<Salt>(oldSalt);
	isValidInDomain<Passphrase>(newPassphrase);
	isValidInDomain<Salt>(newSalt);

	std::vector<unsigned char> hashPassSalt =
			concatAndHash(oldPassphrase, _userManager->getSalt(userID));
	if (_userManager->compareHashPassSalt(userID, hashPassSalt)) {
		insertKeysAndPass(userID, newPassphrase, newSalt);
		return true;
	}
	wrongTry();

	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist);
	}
	if (!_userManager->compareHashPassSalt(userID, hashPassSalt)) {
		THROW_UED_EXCEPTION_WITH_TRIES(
				ErrType::AUTH_ModifyUserPassphraseWrongOldPassphrase,
				_authenticationDatabase->getWrongTries());
	}
	return false;
}

bool Authentication::changePassphrase(
		unsigned int userID, std::vector<unsigned char> oldPassphrase,
		std::vector<unsigned char> newPassphrase) throw(ErrorMessage) {
	if (_userManager->hasUser(userID)) {
		return changePassphrase(userID, oldPassphrase,
				_userManager->getSalt(userID), newPassphrase,
				generateSalt());
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist);
	}
	return false;
}

bool Authentication::setPassphrase(
		unsigned int userID, std::vector<unsigned char> newPassphrase,
		std::vector<unsigned char> newSalt) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<Passphrase>(newPassphrase);
	isValidInDomain<Salt>(newSalt);

	if (isUserAuthenticated(userID) && !_userManager->userHasPassphrase(userID)) {
		insertKeysAndPass(userID, newPassphrase, newSalt);
		return true;
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist);
	}
	return false;
}

bool Authentication::setPassphrase(
		unsigned int userID,
		std::vector<unsigned char> newPassphrase) throw(ErrorMessage) {
	if (_userManager->hasUser(userID)) {
		return setPassphrase(userID, newPassphrase, generateSalt());
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist);
	}
	return false;
}

bool Authentication::removePassphrase(
		unsigned int userID, std::vector<unsigned char> oldPassphrase,
		std::vector<unsigned char> oldSalt) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<Passphrase>(oldPassphrase);
	isValidInDomain<Salt>(oldSalt);

	std::vector<unsigned char> hashPassSalt =
			concatAndHash(oldPassphrase, oldSalt);

	if (isUserAuthenticated(userID) &&
			_userManager->compareHashPassSalt(userID, hashPassSalt)) {
		deleteKeysAndPass(userID);
		return true;
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist);
	}
	if (!_userManager->compareHashPassSalt(userID, hashPassSalt)) {
		THROW_UED_EXCEPTION_WITH_TRIES(
				ErrType::AUTH_ModifyUserPassphraseWrongOldPassphrase,
				_authenticationDatabase->getWrongTries());
	}
	return false;
}

bool Authentication::removePassphrase(
		unsigned int userID,
		std::vector<unsigned char> oldPassphrase) throw(ErrorMessage) {
	if (_userManager->hasUser(userID)) {
		return removePassphrase(userID, oldPassphrase,
				_userManager->getSalt(userID));
	}
	wrongTry();
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist);
	}
	return false;
}

bool Authentication::removeAnotherUserPassphrase(
		unsigned int userID, std::vector<unsigned char> pass,
		std::vector<unsigned char> salt,
		unsigned int userIDToDelete) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<UserID>(userIDToDelete);
	isValidInDomain<Passphrase>(pass);
	isValidInDomain<Salt>(salt);

	std::vector<unsigned char> hashPassSalt = concatAndHash(pass, salt);

	if (isUserAuthenticated(userID) &&
			_userManager->compareHashPassSalt(userID, hashPassSalt) &&
			_userManager->isMasterUser(userID) &&
			_userManager->hasUser(userIDToDelete)) {
		deleteKeysAndPass(userIDToDelete);
		return true;
	}
	wrongTry();
	if (!_userManager->hasUser(userID) ||
			!_userManager->hasUser(userIDToDelete)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist);
	}
	if (!_userManager->compareHashPassSalt(userID, hashPassSalt)) {
		THROW_UED_EXCEPTION_WITH_TRIES(
				ErrType::AUTH_ModifyUserPassphraseWrongOldPassphrase,
				_authenticationDatabase->getWrongTries());
	}
	return false;
}

bool Authentication::removeAnotherUserPassphrase(
		unsigned int userID, std::vector<unsigned char> pass,
		unsigned int userIDToDelete) throw(ErrorMessage) {
	if (_userManager->hasUser(userID)) {
		return removeAnotherUserPassphrase(
				userID, pass, _userManager->getSalt(userID), userIDToDelete);
	}
	wrongTry();
	if (!_userManager->hasUser(userID) ||
			!_userManager->hasUser(userIDToDelete)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist);
	}
	return false;
}

bool Authentication::removeAnotherUserPassphrase(
		unsigned int userID, unsigned int userIDToDelete) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);
	isValidInDomain<UserID>(userIDToDelete);

	if (Configurations::getInstance()->get<bool>(
			Configurations::RE_AUTHENTICATION) &&
			_userManager->userHasPassphrase(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticationRequired);
	}

	if (isUserAuthenticated(userID) && _userManager->isMasterUser(userID) &&
			_userManager->hasUser(userIDToDelete)) {
		deleteKeysAndPass(userIDToDelete);
		return true;
	}
	wrongTry();
	if (!_userManager->hasUser(userID) ||
			!_userManager->hasUser(userIDToDelete)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseUserDoesNotExist);
	}
	return false;
}

void Authentication::deleteKeysAndPass(unsigned int userID) {
	unsigned int wrapperKeyID = _userManager->getWrapperKeyID(userID);
	std::vector<unsigned char> wrapperKey;
	std::vector<unsigned char> dataKey;
	std::vector<unsigned char> encDataKey = _userManager->getEncDataKey(userID);
	std::vector<unsigned char> hashWrapperKey;

	_authenticationDatabase->deleteFromFile(userID);

	_authenticationDatabase->resetWrongTries();
	dataKey = _sdcBackend->Decrypt(wrapperKeyID, encDataKey, SDCBackend::AES,
			SDCBackend::CBC);

	_sdcBackend->DeleteKeyID(wrapperKeyID);

	hashWrapperKey = generateKey(wrapperKey);
	wrapperKeyID   = _sdcBackend->Store(wrapperKey);
	_sdcBackend->Overwrite(wrapperKey);

	encDataKey = _sdcBackend->Encrypt(wrapperKeyID, dataKey, SDCBackend::AES,
			SDCBackend::CBC);
	_sdcBackend->Overwrite(dataKey);

	_userManager->deleteHashPassSalt(userID);
	_userManager->insertWrapperKeyID(userID, wrapperKeyID);
	_userManager->insertEncDataKey(userID, encDataKey);
	_userManager->insertHashWrapperKey(userID, hashWrapperKey);
	_userManager->deleteSalt(userID);

	_authenticationDatabase->saveMUK(userID);
}

void Authentication::insertKeysAndPass(
		unsigned int userID, std::vector<unsigned char> newPassphrase,
		std::vector<unsigned char> newSalt) throw(ErrorMessage) {
	boost::optional<std::tuple<unsigned int, std::vector<unsigned char>,
	std::vector<unsigned char>>>
	wrapperKeyIDAndHash;
	unsigned int wrapperKeyID = _userManager->getWrapperKeyID(userID);
	std::vector<unsigned char> dataKey;
	std::vector<unsigned char> encDataKey = _userManager->getEncDataKey(userID);
	std::vector<unsigned char> hashWrapperKey;
	std::vector<unsigned char> hashPassSalt;

	if (!SaltDatabase::getInstance()->insertIfNotExists(newSalt)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseSaltExists);
	}

	hashPassSalt = concatAndHash(newPassphrase, newSalt);

	wrapperKeyIDAndHash = keyDerivation(newPassphrase, newSalt);
	_sdcBackend->Overwrite(newPassphrase);
	if (wrapperKeyIDAndHash == boost::none) {
		THROW_UED_EXCEPTION(ErrType::AUTH_ModifyUserPassphraseSaltExists);
	}

	_authenticationDatabase->deleteFromFile(userID);

	_authenticationDatabase->resetWrongTries();
	dataKey = _sdcBackend->Decrypt(wrapperKeyID, encDataKey, SDCBackend::AES,
			SDCBackend::CBC);
	_sdcBackend->DeleteKeyID(wrapperKeyID);

	wrapperKeyID   = std::get<0>(wrapperKeyIDAndHash.get());
	hashWrapperKey = std::get<1>(wrapperKeyIDAndHash.get());
	std::vector<unsigned char> encWrapperKey =
			std::get<2>(wrapperKeyIDAndHash.get());

	encDataKey = _sdcBackend->Encrypt(wrapperKeyID, dataKey, SDCBackend::AES,
			SDCBackend::CBC);
	_sdcBackend->Overwrite(dataKey);

	_userManager->insertHashPassSalt(userID, hashPassSalt);
	_userManager->insertWrapperKeyID(userID, wrapperKeyID);
	_userManager->insertEncDataKey(userID, encDataKey);
	_userManager->insertHashWrapperKey(userID, hashWrapperKey);
	_userManager->insertSalt(userID, newSalt);
	_userManager->insertEncWrapperKey(userID, encWrapperKey);

	_authenticationDatabase->saveMUK(userID);
}

std::list<unsigned int> Authentication::getCreatedUsers()
{
	return _userManager->getCreatedUsers();
}

std::list<unsigned int> Authentication::getCurrentAuthenticatedUsers() {
	return _currentAuthUserID;
}

void Authentication::insertCurrentAuthenticatedUser(unsigned int userID) {
	if (_currentAuthUserID.back() == _guestAuthUserID) {
		_currentAuthUserID.pop_back();
	}
	try {
		_currentAuthUserID.push_front(userID);
	} catch (std::bad_alloc &) {
		THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
	}
}

boost::optional<std::tuple<unsigned int, std::vector<unsigned char>,
std::vector<unsigned char>>>
Authentication::keyDerivation(std::vector<unsigned char> &pass,
		std::vector<unsigned char> &salt) {
	unsigned int kid;
	unsigned int size = Configurations::getInstance()->get<unsigned int>(
			Configurations::Conf::DERIVED_KEY_SIZE);
	int iter =
			Configurations::getInstance()->get<int>(Configurations::Conf::ITERATIONS);
	std::vector<unsigned char> key;
	std::vector<unsigned char> hash;

	key  = _opensslBackend->DeriveKey(pass, salt, iter, size);
	hash = _sdcBackend->Digest(key);

	if (_userManager->keyCompare(hash)) {
		return boost::none;
	}

	kid = _sdcBackend->Store(key);
	std::vector<unsigned char> encKey =
			_sdcBackend->Encrypt(Configurations::getInstance()->get<unsigned int>(
					Configurations::USER_ENCRYPT_DECRYPT_SIG_KEYID),
					key, SDCBackend::AES, SDCBackend::CBC);

#ifdef TESTING
UserKey = key;
#endif
_sdcBackend->Overwrite(key);

std::tuple<unsigned int, std::vector<unsigned char>,
std::vector<unsigned char>>
tuple(kid, hash, encKey);

return tuple;
}

std::vector<unsigned char> Authentication::generateKey(
		std::vector<unsigned char> &random) {
	std::vector<unsigned char> hash;

	do {
		random =
				_sdcBackend->GetRandom(Configurations::getInstance()->get<unsigned int>(
						Configurations::Conf::DERIVED_KEY_SIZE));
		hash = _sdcBackend->Digest(random);
	} while (_userManager->keyCompare(hash));
#ifdef TESTING
UserKey = random;
#endif
return hash;
}

std::vector<unsigned char> Authentication::generateSalt() {
	std::vector<unsigned char> salt;

	do {
		salt = _sdcBackend->GetRandom(SALT_SIZE);
	} while (SaltDatabase::getInstance()->exists(salt));

	return salt;
}

#ifdef TESTING

std::vector<unsigned char> Authentication::getKey(unsigned int userID) {
	return _sdcBackend->Decrypt(_userManager->getWrapperKeyID(userID),
			_userManager->getEncDataKey(userID),
			SDCBackend::AES, SDCBackend::CBC);
}

unsigned int Authentication::getKID(unsigned int userID) {
	return _userManager->getWrapperKeyID(userID);
}

#endif

std::vector<unsigned char> Authentication::concatAndHash(
		std::vector<unsigned char> &pass, std::vector<unsigned char> salt) {
	std::vector<unsigned char> hashPassSalt;
	try {
		std::vector<unsigned char> concatenated(pass.begin(), pass.end());
		concatenated.insert(concatenated.end(), salt.begin(), salt.end());
		hashPassSalt = _sdcBackend->Digest(concatenated);
		_sdcBackend->Overwrite(concatenated);
	} catch (std::bad_alloc &) {
		THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
	}

	return hashPassSalt;
}

void Authentication::delUserDiagnosis(unsigned int userID) throw(ErrorMessage) {
	isValidInDomain<UserID>(userID);

	_logger->log(_ctx, LogLevel::INFO, "delUserDiagnosis");
	if (isUserAuthenticated(userID)) {
		_sdcBackend->DeleteKeyID(_userManager->getDataKeyID(userID));
		unauthenticateUser(userID);
	}
	_userManager->deleteUser(userID);
}

void Authentication::wrongTry() {
	_authenticationDatabase->incrementWrongTries();
#ifndef ROBS
	sleep(_incrementalDelay[_authenticationDatabase->getWrongTries()]);
#endif
}

void Authentication::throwDeleteAllUsersException(unsigned int userID) throw(
		ErrorMessage) {
	if (!_userManager->hasUser(userID)) {
		THROW_UED_EXCEPTION(ErrType::AUTH_DeleteUserDoesNotExist);
	}
	if (_currentAuthUserID.front() != userID) {
		THROW_UED_EXCEPTION(ErrType::AUTH_UnauthorizedOperation);
	}
}
