/******************************************************************
 *FILE: UserEncryptDecrypt_CoreFunctions.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: CoreFunctions
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Nov, 2017
 */

#include "UserEncryptDecrypt_CoreFunctions.h"
#include <UserEncryptDecrypt_Version.h>
#include <core/user/UserEncryptDecrypt_UserManager.h>
#include <domains/UserEncryptDecrypt_PacketStringDomain.h>
#include <domains/UserEncryptDecrypt_PlainDataDomain.h>
#include <domains/UserEncryptDecrypt_UEDPacketDomain.h>
#include <upgrade/UserEncryptDecrypt_ConversionProcess.h>
#include <upgrade/UserEncryptDecrypt_ConversionProcess.h>
#include <boost/current_function.hpp>
#include <fstream>
#include <fstream>
#include <iostream>
#include <mutex>
#include <sstream>
#include "UserEncryptDecrypt_Utils.h"
#include "backends/UserEncryptDecrypt_BackendManager.h"
#include "core/authentication/UserEncryptDecrypt_Authentication.h"
#include "core/uedpacket/UserEncryptDecrypt_UEDPacketManager.h"

/*
 * In order to avoid receiving the keyID and passing it to the
 * '_sdcBackend->Encrypt' method,
 * we call the  'callWithUserIDKey' from Authentication with the implementation
 * that will rely
 * on the keyID.
 *
 * */

std::vector<unsigned char> CoreFunctions::AESEncrypt(
    std::string token, std::vector<unsigned char> plainData) {
  isValidInDomain<PlainData>(plainData);
  isValidInDomain<UEDPacketDomain>(token);

  std::shared_ptr<SDCBackend> _sdcBackend;
  try {
    _sdcBackend = BackendManager::getInstance()->get<SDCBackend>();
  } catch (...) {
    THROW_UED_EXCEPTION(ErrType::CORE_EncryptionModuleNotAvailable);
  }
  std::shared_ptr<Authentication> _authentication =
      Authentication::getInstance();
  unsigned int _guestUserID = Configurations::getInstance()->get<unsigned int>(
      Configurations::Conf::GUEST_AUTH_USER_ID);

  std::vector<unsigned char> encData;
  boost::optional<unsigned int> userIDTmp =
      UEDPacketManager::getUserIDFromUEDPacketToken(token);

  if (userIDTmp != boost::none) {
    unsigned int userID = userIDTmp.get();

    if (userID != _guestUserID) {
      if (UserManager::getInstance()->canEncrypt(userID)) {
        if (_authentication->isUserAuthenticated(userID)) {
          encData = _authentication->callWithUserIDKey(
              userID, [&](unsigned int keyID) {
                return _sdcBackend->Encrypt(keyID, plainData,
                                            SDCBackend::Algorithm::AES,
                                            SDCBackend::AlgorithmMode::CBC);
              });

        } else {
          THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticationRequired);
        }
      } else {
        THROW_UED_EXCEPTION(ErrType::AUTH_UnauthorizedOperation);
      }
    } else {
      THROW_UED_EXCEPTION(ErrType::CORE_EncryptGuestUserCantDecrypt);
    }
  } else {
    THROW_UED_EXCEPTION(ErrType::CORE_EncryptUserIdNotFound);
  }

  if (encData.size() == 0) {
    THROW_UED_EXCEPTION(ErrType::CORE_EncryptUnableToEncrypt);
  }

  return UEDPacketManager::createUEDPacket(encData);
}

std::vector<unsigned char> CoreFunctions::AESDecrypt(
    std::string token, std::vector<unsigned char> encPacketData) {
  std::shared_ptr<SDCBackend> _sdcBackend;

  isValidInDomain<UEDPacketDomain>(
      std::string(encPacketData.begin(), encPacketData.end()));
  isValidInDomain<UEDPacketDomain>(token);

  try {
    _sdcBackend = BackendManager::getInstance()->get<SDCBackend>();
  } catch (...) {
    THROW_UED_EXCEPTION(ErrType::CORE_EncryptionModuleNotAvailable);
  }
  std::shared_ptr<Authentication> _authentication =
      Authentication::getInstance();
  unsigned int _guestUserID = Configurations::getInstance()->get<unsigned int>(
      Configurations::Conf::GUEST_AUTH_USER_ID);

  std::vector<unsigned char> decData;
  boost::optional<unsigned int> userIDTmp =
      UEDPacketManager::getUserIDFromUEDPacketToken(token);

  if (userIDTmp != boost::none) {
    unsigned int userID = userIDTmp.get();

    if (userID != _guestUserID) {
      if (UserManager::getInstance()->canEncrypt(userID)) {
        if (_authentication->isUserAuthenticated(userID)) {
          UEDPacket packet;
          packet.decode(
              std::string(encPacketData.begin(), encPacketData.end()));

          boost::optional<unsigned int> libVersionValue =
              packet.getHeaderField<unsigned int>("libVersion");

          if (libVersionValue != boost::none &&
              libVersionValue.get() != UED_CURRENT_VERSION) {
            boost::optional<std::vector<unsigned char>> payload =
                UEDPacketManager::getDataFromUEDPacket(encPacketData);
            if (payload != boost::none) {
              std::vector<unsigned char> resp =
                  ConversionProcess::getInstance()->startRuntimeUpgradeProcess(
                      payload.get());
              ConversionProcess::getInstance()->stopData();
              return resp;
            }
          }

          boost::optional<std::vector<unsigned char>> payload =
              UEDPacketManager::getDataFromUEDPacket(encPacketData);

          if (payload != boost::none) {
            decData = _authentication->callWithUserIDKey(
                userID, [&](unsigned int keyID) {

                  return _sdcBackend->Decrypt(keyID, payload.get(),
                                              SDCBackend::Algorithm::AES,
                                              SDCBackend::AlgorithmMode::CBC);
                });
          } else {
            THROW_UED_EXCEPTION(ErrType::PCKT_InvalidPayload);
          }
        } else {
          THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticationRequired);
        }
      } else {
        THROW_UED_EXCEPTION(ErrType::AUTH_UnauthorizedOperation);
      }
    } else {
      THROW_UED_EXCEPTION(ErrType::CORE_DecryptGuestUserCantDecrypt);
    }
  } else {
    THROW_UED_EXCEPTION(ErrType::CORE_DecryptUserIdNotFound);
  }

  if (decData.size() == 0) {
    THROW_UED_EXCEPTION(ErrType::CORE_DecryptUnableToDecrypt);
  }

  return decData;
}

std::vector<unsigned char> CoreFunctions::Digest(
    std::string token, std::vector<uint8_t> &plainData) {
  isValidInDomain<PlainData>(plainData);
  isValidInDomain<UEDPacketDomain>(token);

  std::shared_ptr<SDCBackend> _sdcBackend;
  try {
    _sdcBackend = BackendManager::getInstance()->get<SDCBackend>();
  } catch (...) {
    THROW_UED_EXCEPTION(ErrType::CORE_EncryptionModuleNotAvailable);
  }
  std::shared_ptr<Authentication> _authentication =
      Authentication::getInstance();
  unsigned int _guestUserID = Configurations::getInstance()->get<unsigned int>(
      Configurations::Conf::GUEST_AUTH_USER_ID);

  std::vector<unsigned char> hashedData;
  boost::optional<unsigned int> userIDTmp =
      UEDPacketManager::getUserIDFromUEDPacketToken(token);

  if (userIDTmp != boost::none) {
    unsigned int userID = userIDTmp.get();

    if (userID != _guestUserID) {
      if (_authentication->isUserAuthenticated(userID)) {
        hashedData = _sdcBackend->Digest(plainData);
      } else {
        THROW_UED_EXCEPTION(ErrType::AUTH_AuthenticationRequired);
      }
    } else {
      THROW_UED_EXCEPTION(ErrType::CORE_DigestGuestUserCantDecrypt);
    }
  } else {
    THROW_UED_EXCEPTION(ErrType::CORE_DigestUserIdNotFound);
  }

  if (hashedData.size() == 0) {
    THROW_UED_EXCEPTION(ErrType::CORE_Digest);
  }

  return UEDPacketManager::createUEDPacket(hashedData);
}
