/******************************************************************
 *FILE: UserEncryptDecrypt_Configurations.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: Configurations
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Ulisses Costa (marioulisses.costa@altran.com)
 * @date Dec, 2017
 *
 * Date      | Author             | Modification
 * 14/08/2018| AKM7COB			  | changed class name logger to logger1. class name logger is already used in ASF framework
 * 24/07/2019| AKM7COB			  | File path modified and new member function implemented for folder creation. Changed file path to persistent folder. As per requirement from telematics key needs to be retained over software update.
 */

#ifndef SRC_CORE_USERENCRYPTDECRYPT_CONFIGURATIONS_H_
#define SRC_CORE_USERENCRYPTDECRYPT_CONFIGURATIONS_H_

#include <error/UserEncryptDecrypt_ErrorMessage.h>
#include <boost/blank.hpp>
#include <boost/optional.hpp>
#include <boost/variant.hpp>
#include <cstring>
#include <iostream>
#include <memory>
#include <mutex>
#include "core/logger/UserEncryptDecrypt_Logger.h"

/**
 * Configurations is a representation of the userEncryptDecrypt configuration
 * file.
 *
 * Here are stored several configurations for different parts of the component
 * code.
 * These configurations have a default value if not set via the configuration
 * file.
 * Here is a list of the configuration items supported at the moment:
 * - UED_AUTH_FILE: The filepath of the Authentication database (a file in text
 * format);
 * - AES_BLOCK_SIZE: The block size of AES256 cypher;
 * - DELIMITER: The delimiter used in the Authentication database serialization;
 * - DERIVED_KEY_SIZE: The size of the result of key derivation function;
 * - ITERATIONS: The number of iterations for the key derivation function;
 * - MAX_AUTH_USER: The maximum number of authenticated users at the same time;
 * - GUEST_AUTH_USER_ID: The guest userID;
 * - USER_ENCRYPT_DECRYPT_SIG_KEYID: The master key identifier.
 * - NR_DEFAULT_PROFILES: The number of default profiles to be created
 * - ENC_DEC_MASTER: Flag that represents if an master user has permission to
 * perform encrypt/decrypt operations
 * - ACTIVE_MASTER_USER: Flag that enable/disable the existence of master user
 * - RE_AUTHENTICATION: Flag to enable/disable re-authentication on critical
 * operations(eg. deleting an user profile)
 * - SALT_FILE: The filepath of the generated salt database (a file in text
 * format)
 * - KEY_SIZE: The default size of all keys
 *
 * A typical class permits callers to create as many instances of the class as
 * they want, but this class use the Singleton Pattern.
 * A singleton object provides a global point of access to the resources of its
 * class and returns the same instance no matter how many times an application
 * requests it.
 * The function Configurations::getInstance() provides the access to the
 * resources of this class.
 * */

class Configurations {
 public:
  // Don't remove or change the position of '_LAST' and '_FIRST' elements.
  enum Conf {
    _FIRST,
    UED_AUTH_FILE,
    AES_BLOCK_SIZE,
    DELIMITER,
    DERIVED_KEY_SIZE,
    ITERATIONS,
    MAX_AUTH_USER,
    GUEST_AUTH_USER_ID,
    USER_ENCRYPT_DECRYPT_SIG_KEYID,
    NR_DEFAULT_PROFILES,
    ENC_DEC_MASTER,
    ACTIVE_MASTER_USER,
    RE_AUTHENTICATION,
    SALT_FILE,
    KEY_SIZE,
    _LAST
  };

  /**
   * @brief Destructor for the class.
   */
  virtual ~Configurations();

  /**
   *   @brief Provides an instance of the Configurations class object.
   *
   *   This function provides the access to class object, but only one instance
   * is returned as mentioned above in the definition of singleton pattern.
   *
   *   @return returns a shared pointer of the instance of the Configurations
   */
  static std::shared_ptr<Configurations> getInstance(bool createNew = false);

  /**
   * @brief Get the value for a particular Conf value. The value is either
   * loaded from
   * file or if not exist in the configuration file is a default value.
   *
   * @return The value of the Conf value
   *
   * */
  template <typename T>
  T get(Conf c) {
    if (c <= _FIRST || c >= _LAST) {
      THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_valueNotDefined);
    } else {
      return boost::get<T>(_configVector[c].data);
    }
  }

  /**
   * @brief Modify the value of a Conf with 'val'
   *
   * @param conf if the Conf item to set the value 'val'
   * @param val is the actual value 'conf' Conf will have
   *
   * */
  template <typename T>
  void changeConf(Conf conf, T val) {
    _configVector[conf].data = val;
  }

 private:
  typedef boost::variant<boost::blank, std::string, unsigned int, int, char,
                         bool>
      ConfigurationType;
  enum Type { _STRING, _UNSIGNED_INT, _INT, _CHAR, _BOOL, _NONE };

  class Config {
   public:
    Config() { type = _NONE; }
    Config(std::string _name, Type _type, ConfigurationType _data)
        : name(_name), type(_type), data(_data) {}

    std::string name;
    Type type;
    ConfigurationType data;
  };

  Configurations();
  /**
   * @brief Create an configuration file with default values
   */
  void createFile();
  /**
   * @brief Load the configurations that exists on the file
   */
  void loadFile();
  /**
   * @brief Createdirectory if not exists
   */
  bool createDirIfNotExists(const std::string path);

  std::shared_ptr<Logger1> _logger;
  std::shared_ptr<LogContext> _ctx;

  static std::mutex _mutex;
  static std::shared_ptr<Configurations> _instance;
  const std::string _path =  "/var/opt/bosch/persistent/cryptoapp";
  const std::string _Configurations = "/var/opt/bosch/persistent/cryptoapp/Configuration.txt";
  std::vector<Config> _configVector;
};

#endif /* SRC_CORE_USERENCRYPTDECRYPT_CONFIGURATIONS_H_ */
