/******************************************************************
 *FILE: UserEncryptDecrypt_Configuration.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: Configuration
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Ulisses Costa (marioulisses.costa@altran.com)
 * @date Dec, 2017
 */

#include "core/UserEncryptDecrypt_Configurations.h"
#include <fstream>
#include <map>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#define RETURN_SUCESS 0

std::mutex Configurations::_mutex;
std::shared_ptr<Configurations> Configurations::_instance = nullptr;

Configurations::Configurations() {
	_logger = Logger1::getLogger();
	_logger->setLoggerOutput(LoggerOutput::DLT);
	_ctx = _logger->registerContext("CONF", "UserEncryptDecrypt_Configurations");

	try {
		_configVector = std::vector<Config>(Conf::_LAST);
	} catch (std::bad_alloc &e) {
		THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
	}
	_configVector[UED_AUTH_FILE] =
			Config("UED_AUTH_FILE", _STRING, (std::string) "file.txt");
	_configVector[SALT_FILE] =
			Config("SALT_FILE", _STRING, (std::string) "saltDatabase.txt");
	_configVector[AES_BLOCK_SIZE] =
			Config("AES_BLOCK_SIZE", _UNSIGNED_INT, (unsigned int)16);
	_configVector[DELIMITER] = Config("DELIMITER", _CHAR, (char)31);
	_configVector[DERIVED_KEY_SIZE] =
			Config("DERIVED_KEY_SIZE", _UNSIGNED_INT, (unsigned int)128);
	_configVector[ITERATIONS] = Config("ITERATIONS", _INT, (int)1000);
	_configVector[USER_ENCRYPT_DECRYPT_SIG_KEYID] = Config(
			"USER_ENCRYPT_DECRYPT_SIG_KEYID", _UNSIGNED_INT, (unsigned int)5000);
	_configVector[MAX_AUTH_USER] = Config("MAX_AUTH_USER", _INT, (int)5);
	_configVector[NR_DEFAULT_PROFILES] =
			Config("NR_DEFAULT_PROFILES", _UNSIGNED_INT, (unsigned int)5);
	_configVector[GUEST_AUTH_USER_ID] =
			Config("GUEST_AUTH_USER_ID", _UNSIGNED_INT, (unsigned int)0);
	_configVector[ENC_DEC_MASTER] = Config("ENC_DEC_MASTER", _BOOL, (bool)true);
	_configVector[RE_AUTHENTICATION] =
			Config("RE_AUTHENTICATION", _BOOL, (bool)true);
	_configVector[ACTIVE_MASTER_USER] =
			Config("ACTIVE_MASTER_USER", _BOOL, (bool)false);
	_configVector[KEY_SIZE] = Config("KEY_SIZE", _UNSIGNED_INT, (unsigned int)32);
	_logger->log(_ctx, LogLevel::INFO, ("configurations constructor"));
}

Configurations::~Configurations() { _logger->unregisterContext(_ctx); }

std::shared_ptr<Configurations> Configurations::getInstance(bool createNew) {
	if (createNew || !_instance) {
		std::lock_guard<std::mutex> lock(_mutex);
		if (createNew || !_instance) {
			try {
				_instance.reset(new Configurations());
			} catch (ErrorMessage &e) {
				THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
			}
			_instance->loadFile();
		}
	}
	return _instance;
}

void throwDefaultFileFailed(std::ofstream &writer) {
	if (writer.bad() || writer.fail() || writer.exceptions()) {
		THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_failedToWriteDefaultFile);
	}
}

bool Configurations::createDirIfNotExists(const std::string path){
	struct stat st;
	int status = 0;

	try{
		if (stat(path.data(), &st) != RETURN_SUCESS)
		{
			/* Directory does not exist. EEXIST for race condition */
			if ((mkdir(path.data(), 0775) != RETURN_SUCESS) && (errno != EEXIST))
			{
				status = -1;
				_logger->log(_ctx, LogLevel::INFO, ("Error: Error in directory creation.."));
			}
			else
			{
				_logger->log(_ctx, LogLevel::INFO, ("Directory created successfully.."));
			}
		}
		else if (!S_ISDIR(st.st_mode))
		{
			errno = ENOTDIR;
			status = -1;
			_logger->log(_ctx, LogLevel::INFO, ("Error: Given name is not directory.."));
		}
		else if ((errno == EACCES)|(errno == EPERM))
		{// Directory exists
			_logger->log(_ctx, LogLevel::INFO, ("Error: Access permission error or operation not permitted.."), errno);
			status = -1;
		}
		else if (errno == RETURN_SUCESS){
			_logger->log(_ctx, LogLevel::INFO, ("Directory exists"));
		}
		else
		{
			_logger->log(_ctx, LogLevel::INFO, ("Error: other directory creation error.."));
			status = -1;
		}
	}
	catch(ErrorMessage &e){
		THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_failedToWriteDefaultFile);
	}


	if (!status)
		return true;
	else
		return false;
}

void Configurations::createFile() {
	_logger->log(_ctx, LogLevel::INFO, ("create file entered"));
	std::ofstream writer;

	createDirIfNotExists(_path);

	try{
		writer.open(_Configurations.data(), std::ios::app);

		if (!writer.is_open() || writer.bad() || writer.fail() ||
				writer.exceptions()) {
			_logger->log(_ctx, LogLevel::INFO, ("file open failed"));
			THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileNotOpen);
		}
	} catch	(ErrorMessage &e){
		THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileNotOpen);
	}


	_logger->log(_ctx, LogLevel::INFO, ("file open success, before write"));
	writer << _configVector[UED_AUTH_FILE].name << "="
			<< _configVector[UED_AUTH_FILE].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[AES_BLOCK_SIZE].name << "="
			<< _configVector[AES_BLOCK_SIZE].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[DELIMITER].name << "="
			<< _configVector[DELIMITER].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[DERIVED_KEY_SIZE].name << "="
			<< _configVector[DERIVED_KEY_SIZE].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[ITERATIONS].name << "="
			<< _configVector[ITERATIONS].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[USER_ENCRYPT_DECRYPT_SIG_KEYID].name << "="
			<< _configVector[USER_ENCRYPT_DECRYPT_SIG_KEYID].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[MAX_AUTH_USER].name << "="
			<< _configVector[MAX_AUTH_USER].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[GUEST_AUTH_USER_ID].name << "="
			<< _configVector[GUEST_AUTH_USER_ID].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[ENC_DEC_MASTER].name << "="
			<< _configVector[ENC_DEC_MASTER].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[NR_DEFAULT_PROFILES].name << "="
			<< _configVector[NR_DEFAULT_PROFILES].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[ACTIVE_MASTER_USER].name << "="
			<< _configVector[ACTIVE_MASTER_USER].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[RE_AUTHENTICATION].name << "="
			<< _configVector[RE_AUTHENTICATION].data << "\n";
	throwDefaultFileFailed(writer);
	writer << _configVector[SALT_FILE].name << "="
			<< _configVector[SALT_FILE].data << "\n";
	throwDefaultFileFailed(writer);

	writer.close();

	throwDefaultFileFailed(writer);
}

void Configurations::loadFile() {
	std::string line;
	std::string key;
	std::string value;
	std::ifstream reader(_Configurations);
	std::list<std::string> unusedFlagsList;

	_logger->log(_ctx, LogLevel::INFO, (" loadfile entered"));
	if (!reader) {
		createFile();
		reader.open(_Configurations.c_str(), std::ifstream::in);
		if (reader.bad() || reader.fail() || reader.exceptions()) {
			_logger->log(_ctx, LogLevel::INFO, ("Config reader open failed"));
			THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileNotOpen);
		}
	}

	while (getline(reader, line)) {
		if (reader.bad() || reader.fail() || reader.exceptions()) {
			THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_failedReadingConfFile);
		}
		size_t pos = line.find("=");

		if (pos >= line.size() && !line.empty()) {
			std::cerr << line << std::endl;
			THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileBadlyFormatted);
		} else if (line.empty()) {
			break;
		}

		try {
			key = line.substr(0, line.find("="));
		} catch (std::bad_alloc &) {
			THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
		}
		try {
			unusedFlagsList.push_back(key);
		} catch (std::bad_alloc &) {
			THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
		}
		for (int i = Conf::_FIRST; i < Conf::_LAST; i++) {
			bool comp = false;
			try {
				comp = key.compare(_configVector[i].name);
			} catch (std::out_of_range &) {
				THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileBadlyFormatted);
			}
			if (!comp) {
				unusedFlagsList.remove(key);
				pos = line.find("=") + 1;
				if (pos >= line.size()) {
					THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileBadlyFormatted);
				}
				value = line.substr(pos);
				switch (_configVector[i].type) {
				case _STRING:
					try {
						_configVector[i].data = std::string(value);
					} catch (std::bad_alloc &) {
						THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
					}
					break;
				case _UNSIGNED_INT:
					try {
						_configVector[i].data = (unsigned int)stoul(value, nullptr, 10);
					} catch (std::invalid_argument &) {
						THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileBadlyFormatted);
					} catch (std::out_of_range &) {
						THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileBadlyFormatted);
					}

					break;
				case _INT:
					try {
						_configVector[i].data = (int)stoi(value, nullptr, 10);
					} catch (std::invalid_argument &) {
						THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileBadlyFormatted);
					} catch (std::out_of_range &) {
						THROW_UED_EXCEPTION(ErrType::CONFIGURATIONS_fileBadlyFormatted);
					}
					break;

				case _CHAR:
					_configVector[i].data = (char)value[0];
					break;
				case _BOOL:
					_configVector[i].data = value[0] != '0';
					break;
				case _NONE:
					break;
				default:
					break;
				}
			}
		}
	}
	reader.close();

	if (!unusedFlagsList.empty()) {
		std::string unusedFlags = "";
		for (auto flag : unusedFlagsList) {
			try {
				unusedFlags = unusedFlags + flag + ",";
			} catch (...) {
				THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
			}
		}
		unusedFlags.substr(0, unusedFlags.size() - 1);
		_logger->log(_ctx, LogLevel::INFO, ("Unused flags: " + unusedFlags + "\n"));
	}
}
