/******************************************************************
 *FILE: UserEncryptDecrypt_SDCBackend.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Ulisses Costa (marioulisses.costa@altran.com)
 * @date Oct, 2017
 */

#ifndef SRC_BACKENDS_SDC_SDCBACKEND_H_
#define SRC_BACKENDS_SDC_SDCBACKEND_H_

#include "core/logger/UserEncryptDecrypt_Logger.h"

#include <map>
#include <memory>
#include <mutex>
#include <vector>

#ifdef ROBS
#include <mock/RobsMock.h>
#else
#include <sdc_op_common.h>
#endif

/**
 * This class provides the communication between the Core package of
 *UserEncryptDecrypt Lib and the SDC library
 * The Authentication and CoreFunctions classes uses this class to Digest,
 *Encrypt and Decrypt data.
 *
 * A typical class permits callers to create as many instances of the class as
 *they want, but this class use the Singleton Pattern.
 * A singleton object provides a global point of access to the resources of its
 *class and returns the same instance no matter how many times an application
 *requests it.
 * The function SDCBackend::getInstance() provides the access to the resources
 *of this class.
 **/
class SDCBackend {
 public:
#ifdef TESTING
  long keys_count;
#endif
  /**
   * Encryption algorithm types
   */
  enum Algorithm { RSA, AES };
  /**
   * Encryption algorithm mode
   * The purpose of cipher modes is to mask the patterns which exists in
   * encrypted data.\n
   * CBC: Cipher Block Chaining\n
   * ECB: Electronic Codebook\n
   * CTR: Counter\n
   * CFB: Cipher Feedback\n
   * OFB: Output Feedback\n
   */
  enum AlgorithmMode { CBC, ECB, CTR, CFB, OFB };

  /**
   * @brief Destructor for the class
   */
  virtual ~SDCBackend();

  /**
   *   @brief Provides an instance of the SDCBackend class object.
   *
   *   This function provides the access to class object, but only one instance
   * is returned as mentioned above in the definition of singleton pattern.
   *
   *   @return returns a shared pointer of the instance of the SDCBackend
   */
  static std::shared_ptr<SDCBackend> &getInstance();

  /**
   *   @brief Deletes a key from the SDC key storage
   *
   *   @param  keyID is the unique identifier of the key to be deleted
   *
   *   @return returns true if the key is deleted and false if the the deletion
   * fails.
   */
  bool DeleteKeyID(unsigned int keyID);

  /*
   *   @brief Returns a random number with the given size
   *
   *   @param size is the length of characters of the number to generate
   *
   *   @return returns the randomly generated number
   */
  std::vector<unsigned char> GetRandom(int size);

  /*
   *   @brief Writes 0 over the given unsigned char vector. This is used to make
   * sure that @param pass is emptied
   *
   *   @param pass is the secret data to be overwritten
   */
  void Overwrite(std::vector<unsigned char> &pass);

/*
 *   @brief Stores the given key into the SDC with default permissions
 *
 *   @param key is the data to be stored in the SDC container
 *   @param persist is a flag that tells if the key is supposed to be persistent
 * or volatile on the SDC
 *
 *   @return returns the unique identifier associated with the key
 */
#ifdef TESTING

  unsigned int Store(std::vector<unsigned char> key, bool persist = false);

#else

  unsigned int Store(std::vector<unsigned char> key, bool persist = true);

#endif

/*
 *   @brief Stores the given key into the SDC with default permissions
 *
 *   @param key is the data to be stored in the SDC container
 *   @param keyID is the unique identifier in which data that will be stored in
 * the SDC container can be obtained from
 *   @param persist is a flag that tells if the key is supposed to be persistent
 * or volatile on the SDC
 *
 *   @return returns the unique identifier associated with the key
 */
#ifdef TESTING

  unsigned int Store(std::vector<unsigned char> key, unsigned int keyID,
                     bool persist = false);

#else

  unsigned int Store(std::vector<unsigned char> key, unsigned int keyID,
                     bool persist = true);

#endif

  /*
   *   @brief Encrypts the given data with the key stored in the given key ID
   *
   *   @param kid is the unique identifier of the key to be used in the
   * encryption
   *   @param data is the plain text to be encrypted
   *
   *   @param alg is the algorithm to be used in the encryption
   *
   *   @param blk is the block mode cipher to be used in the encryption
   *
   *   @return returns the encrypted data
   */
  std::vector<unsigned char> Encrypt(unsigned int kid,
                                     std::vector<unsigned char> data,
                                     Algorithm alg, AlgorithmMode blk);

  /*
   *   @brief Decrypts the given data with the key stored in the given key ID
   *
   *   @param kid is the unique identifier of the key to be used in the
   * encryption
   *   @param data is the encrypted text to be decrypted
   *
   *   @param alg is the algorithm to be used in the encryption
   *
   *   @param blk is the block mode cipher to be used in the encryption
   *
   *   @return returns the plain data
   */
  std::vector<unsigned char> Decrypt(unsigned int kid,
                                     std::vector<unsigned char> data,
                                     Algorithm alg, AlgorithmMode blk);

  /*
   *  @brief Performs the digest of given plain data using the SHA-256 protocol
   *
   * @param data to be digested
   *
   * @return The generated hash of data
   */
  std::vector<unsigned char> Digest(std::vector<unsigned char> data);

 private:
  /**
   * @brief Constructor for the class
   */
  SDCBackend();

  /**
   * @brief Set the encrypt/ decrypt algorithm and mode to cryptography
   * operations
   */
  std::pair<sdc_encrypt_decrypt_type_t *, sdc_session_t *> setEncryptDecrypt(
      unsigned int kid, Algorithm alg, AlgorithmMode blk);

  std::shared_ptr<Logger1> _logger;
  std::shared_ptr<LogContext> _ctx;
  sdc_session *session;

  static std::mutex _mutex;
  static std::shared_ptr<SDCBackend> _instance;

  size_t sizeCipherdataWithIV(size_t);

  unsigned int BLOCKSIZE;
  std::map<Algorithm, sdc_encrypt_decrypt_alg_t> _algo;
  std::map<AlgorithmMode, sdc_encrypt_decrypt_blk_t> _algoMode;
};

#endif /* SRC_BACKENDS_SDC_SDCBACKEND_H_ */
