/******************************************************************
 *FILE: UserEncryptDecrypt_SDCBackend.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Ulisses Costa (marioulisses.costa@altran.com)
 * @date Oct, 2017
 */

#include "UserEncryptDecrypt_SDCBackend.h"
#include <error/UserEncryptDecrypt_ErrorMessage.h>
#include "core/UserEncryptDecrypt_Configurations.h"

#ifdef ROBS
#include <mock/RobsMock.h>
#else
#include <sdc_keystore_keys.h>
#include <sdc_op_adv.h>
#include <sdc_op_conv.h>
#include <sdc_perm.h>
#include <sdc_random.h>
#include <sdc_session.h>
#endif
#include <string.h>
#include <cstring>
#include <iostream>

std::mutex SDCBackend::_mutex;
std::shared_ptr<SDCBackend> SDCBackend::_instance = nullptr;

void handleSDCErros(sdc_error_t &err,
                    std::string func = BOOST_CURRENT_FUNCTION) {
  if (err == SDC_OK) {
    return;
  }
  if (err == SDC_KID_EXISTS) {
    throw ErrorMessage(
        ErrType::SDC_KeyExists,
        std::string(func) + " sdc error code: " + std::to_string(err));
  }

  if (err == SDC_NO_MEM) {
    throw ErrorMessage(
        ErrType::SYSTEM_noMemory,
        std::string(func) + " sdc error code: " + std::to_string(err));
  }

  if (SDC_INVALID_PARAMETER >= err <= SDC_KEYLEN_INVALID) {
    throw ErrorMessage(
        ErrType::SDC_invalidParameter,
        std::string(func) + " sdc error code: " + std::to_string(err));
  } else if (SDC_DAEMON_COMMUNICATION_ERROR >= err <=
                 SDC_KEY_CONTAINER_IMPORT_FAILED ||
             err == SDC_CONFIG_ACCESS_FAILED || err == SDC_NOT_SUPPORTED ||
             err == SDC_AUTHENTICATION_FAILED || err == SDC_OP_NOT_SUPPORTED ||
             err == SDC_ACCESS_DENIED) {
    throw ErrorMessage(
        ErrType::SDC_EncryptionModuleNotAvailable,
        std::string(func) + " sdc error code: " + std::to_string(err));
  } else if (SDC_KID_NOT_AVAILABLE >= err <= SDC_KEY_TAMPERED) {
    throw ErrorMessage(
        ErrType::SDC_KeyError,
        std::string(func) + " sdc error code: " + std::to_string(err));
  } else if (SDC_KEY_INFO_INVALID >= err <= SDC_KEY_ENC_INVALID) {
    throw ErrorMessage(
        ErrType::SDC_KeyError,
        std::string(func) + " sdc error code: " + std::to_string(err));
  } else if (SDC_ARCHITECTURE_SPECIFIC_ERROR_01 >= err <=
                 SDC_ARCHITECTURE_SPECIFIC_ERROR_30 ||
             err == SDC_UNKNOWN_ERROR) {
    throw ErrorMessage(
        ErrType::SDC_unknown,
        std::string(func) + " sdc error code: " + std::to_string(err));
  }
  throw ErrorMessage(
      ErrType::SDC_EncryptionModuleNotAvailable,
      std::string(func) + " sdc error code: " + std::to_string(err));
}

SDCBackend::SDCBackend() {
  try {
    _algo.emplace(Algorithm::AES, SDC_ENCDEC_ALG_AES);
    _algo.emplace(Algorithm::RSA, SDC_ENCDEC_ALG_RSA);

    _algoMode.emplace(AlgorithmMode::CBC, SDC_ENCDEC_BLK_CBC);
    _algoMode.emplace(AlgorithmMode::ECB, SDC_ENCDEC_BLK_ECB);
    _algoMode.emplace(AlgorithmMode::CTR, SDC_ENCDEC_BLK_CTR);
    _algoMode.emplace(AlgorithmMode::CFB, SDC_ENCDEC_BLK_CFB);
    _algoMode.emplace(AlgorithmMode::OFB, SDC_ENCDEC_BLK_OFB);
  } catch (std::bad_alloc) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  BLOCKSIZE = Configurations::getInstance()->get<unsigned int>(
      Configurations::Conf::AES_BLOCK_SIZE);

  _logger = Logger1::getLogger();
  _ctx    = _logger->registerContext("SDCB", "SDC backend");
}

SDCBackend::~SDCBackend() { _logger->unregisterContext(_ctx); }

std::shared_ptr<SDCBackend> &SDCBackend::getInstance() {
  if (!_instance) {
    std::lock_guard<std::mutex> lock(_mutex);

    if (!_instance) {
      _instance.reset(new SDCBackend());
    }
  }
  return _instance;
}

bool SDCBackend::DeleteKeyID(unsigned int keyID) {
  sdc_error_t err;

  err = sdc_remove_storage_key(keyID);
  try {
    handleSDCErros(err);
  } catch (...) {
    return false;
  }
  return true;
}

void SDCBackend::Overwrite(std::vector<unsigned char> &pass) {
  sdc_error_t err;

  err = sdc_overwrite_secret(pass.data(), pass.size());
  try {
    handleSDCErros(err);
  } catch (...) {
  }
}

unsigned int SDCBackend::Store(std::vector<unsigned char> key, bool persist) {
  sdc_permissions_t *permissions;
  unsigned int kid;
  sdc_error_t err;
  sdc_key_storage_options_t storageOption = SDC_PERSISTENT_STORAGE_KEY;

  if (persist == false) {
    std::cout << "Store(std::vector<unsigned char> key, bool persist) "
                 "persistance is off";
    storageOption = SDC_VOLATILE_STORAGE_KEY;
  }

  /* key permission definition*/
  err = sdc_permissions_alloc(&permissions);

  handleSDCErros(err);

  err = sdc_set_default_permissions_current_gid(permissions);

  handleSDCErros(err);
  int key_size = Configurations::getInstance()->get<unsigned int>(
      Configurations::KEY_SIZE);
  err = sdc_insert_plain_storage_key(&kid, SDC_CREATE_KEY_AUTOMATIC_ID,
                                     key.data(), key_size, storageOption,
                                     permissions);

  handleSDCErros(err);

  err = sdc_permissions_free(permissions);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  return kid;
}

unsigned int SDCBackend::Store(std::vector<unsigned char> key,
                               unsigned int keyID, bool persist) {
  sdc_permissions_t *permissions;
  unsigned int kid = keyID;
  sdc_error_t err;
  sdc_key_storage_options_t storageOption = SDC_PERSISTENT_STORAGE_KEY;

  if (persist == false) {
    storageOption = SDC_VOLATILE_STORAGE_KEY;
  }

  /* key permission definition*/
  err = sdc_permissions_alloc(&permissions);

  handleSDCErros(err);

  err = sdc_set_default_permissions_current_gid(permissions);

  handleSDCErros(err);

  int key_size = Configurations::getInstance()->get<unsigned int>(
      Configurations::KEY_SIZE);
  sdc_error_t err1 =
      sdc_insert_plain_storage_key(&kid, SDC_CREATE_KEY_FIXED_ID, key.data(),
                                   key_size, storageOption, permissions);

  try {
    handleSDCErros(err1);
  } catch (ErrorMessage &e) {
    if (e.getErrorType() != ErrType::SDC_KeyExists) {
      __throw_exception_again;
    }
  }

  err = sdc_permissions_free(permissions);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  return err1;
}

std::vector<unsigned char> SDCBackend::GetRandom(int size) {
  unsigned char *out = (unsigned char *)malloc(size);
  if (out == nullptr) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  sdc_session *session;
  sdc_error_t err;
  std::vector<unsigned char> random;

  err = sdc_open_session(&session);

  handleSDCErros(err);

  err = sdc_random_gen(session, size, out);

  handleSDCErros(err);

  err = sdc_close_session(session);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  random = std::vector<unsigned char>(out, out + size);

  free(out);

  return random;
}

std::vector<unsigned char> SDCBackend::Encrypt(unsigned int kid,
                                               std::vector<unsigned char> data,
                                               Algorithm alg,
                                               AlgorithmMode blk) {
  sdc_session_t *session;
  sdc_encrypt_decrypt_type_t *type;
  std::pair<sdc_encrypt_decrypt_type_t *, sdc_session_t *> typeSessionPair;
  sdc_error_t err;
  unsigned int outSize;
  unsigned char *out = nullptr;
  unsigned char *iv  = (unsigned char *)malloc(BLOCKSIZE);
  if (iv == nullptr) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }

  std::vector<unsigned char> salt;
  std::vector<unsigned char> encrypted;

  memset(iv, '0', BLOCKSIZE);

  typeSessionPair = setEncryptDecrypt(kid, alg, blk);

  type    = typeSessionPair.first;
  session = typeSessionPair.second;

  salt = GetRandom(BLOCKSIZE);

  try {
    salt.insert(salt.end(), data.begin(), data.end());
  } catch (std::bad_alloc) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }
  err = sdc_encrypt(session, type, salt.data(), salt.size(), &iv, (size_t*) &BLOCKSIZE,
                    &out, (size_t*) &outSize);

  handleSDCErros(err);

  err = sdc_close_session(session);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  encrypted = std::vector<unsigned char>(out, out + outSize);
  err       = sdc_cipher_buffer_free(out, outSize);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  err = sdc_encrypt_decrypt_type_free(type);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  free(iv);
  return encrypted;
}

std::vector<unsigned char> SDCBackend::Decrypt(unsigned int kid,
                                               std::vector<unsigned char> data,
                                               Algorithm alg,
                                               AlgorithmMode blk) {
  sdc_session_t *session;
  sdc_encrypt_decrypt_type_t *type;
  std::pair<sdc_encrypt_decrypt_type_t *, sdc_session_t *> typeSessionPair;
  unsigned int outSize;
  unsigned char *out = nullptr;
  unsigned char *iv  = (unsigned char *)malloc(BLOCKSIZE);
  if (iv == nullptr) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }
  sdc_error_t err;
  std::vector<unsigned char> decrypted;

  typeSessionPair = setEncryptDecrypt(kid, alg, blk);

  type    = typeSessionPair.first;
  session = typeSessionPair.second;

  memset(iv, '0', BLOCKSIZE);

  err = sdc_decrypt(session, type, data.data(), data.size(), iv, BLOCKSIZE,
                    &out, (size_t*) &outSize);
  if (err != SDC_OK) {
    handleSDCErros(err);

  } else {
    decrypted = std::vector<unsigned char>(out, out + outSize);

    if (BLOCKSIZE < decrypted.size()) {
      decrypted.erase(decrypted.begin(), decrypted.begin() + BLOCKSIZE);
    }
  }

  err = sdc_close_session(session);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  sdc_cipher_buffer_free(out, outSize);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  err = sdc_encrypt_decrypt_type_free(type);
  try {
    handleSDCErros(err);
  } catch (...) {
  }
  free(iv);
  return decrypted;
}

std::vector<unsigned char> SDCBackend::Digest(std::vector<unsigned char> data) {
  unsigned char *out = nullptr;

  unsigned int outSize = SIZE_MAX;
  std::vector<unsigned char> hash;
  sdc_session_t *session;
  sdc_dgst_type_t *type;
  sdc_error_t err;

  err = sdc_dgst_type_alloc(&type);

  handleSDCErros(err);

  err = sdc_dgst_type_set_hash(type, SDC_DGST_HASH_SHA256);

  handleSDCErros(err);

  err = sdc_open_session(&session);

  handleSDCErros(err);

  // perform the digest of the concatenated result

  err = sdc_dgst(session, type, data.data(), data.size(), &out, (size_t*) &outSize);

  handleSDCErros(err);

  err = sdc_dgst_type_free(type);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  err = sdc_close_session(session);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  hash = std::vector<unsigned char>(out, out + outSize);

  sdc_cipher_buffer_free(out, outSize);
  try {
    handleSDCErros(err);
  } catch (...) {
  }

  return hash;
}

size_t SDCBackend::sizeCipherdataWithIV(size_t s) {
  size_t cipherdata_len;

  cipherdata_len = s / BLOCKSIZE;
  if (s % BLOCKSIZE != 0) {
    cipherdata_len++;
  }
  cipherdata_len++;
  cipherdata_len *= 16;
  return cipherdata_len;
}

std::pair<sdc_encrypt_decrypt_type_t *, sdc_session_t *>
SDCBackend::setEncryptDecrypt(unsigned int kid, Algorithm alg,
                              AlgorithmMode blk) {
  sdc_session_t *session;
  sdc_encrypt_decrypt_type_t *type;

  sdc_error_t err = sdc_encrypt_decrypt_type_alloc(&type);

  handleSDCErros(err);

  err = sdc_encrypt_decrypt_type_set_algorithm(type, _algo.at(alg));

  handleSDCErros(err);

  err = sdc_encrypt_decrypt_type_set_block_mode(type, _algoMode.at(blk));

  handleSDCErros(err);

  err = sdc_open_session(&session);

  handleSDCErros(err);

  err = sdc_session_load_storage_key(session, kid);

  handleSDCErros(err);

  return std::make_pair(type, session);
}
