/******************************************************************
 *FILE: UserEncryptDecrypt_OpensslBackend.h
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Nov, 2017
 */

#ifndef SRC_BACKENDS_OPENSSL_OPENSSLBACKEND_H_
#define SRC_BACKENDS_OPENSSL_OPENSSLBACKEND_H_

#include <memory>
#include <mutex>
#include <vector>

/**
 * This class provides the communication between the Core package of
 * UserEncryptDecrypt Lib and the OpenSSL library
 * The Authentication class uses this class to derive keys
 *
 * A typical class permits callers to create as many instances of the class as
 * they want, but this class use the Singleton Pattern.
 * A singleton object provides a global point of access to the resources of its
 * class and returns the same instance no matter how many times an application
 * requests it.
 * The function OpensslBackend::getInstance() provides the access to the
 * resources of this class.
 */
class OpensslBackend {
 private:
  /**
   * @brief Constructor for the class
   */
  OpensslBackend();

  static std::mutex _mutex;
  static std::shared_ptr<OpensslBackend> _instance;

 public:
  /**
   * @brief Destructor for the class
   */
  virtual ~OpensslBackend();

  /**
   *   @brief Provides an instance of the OpensslBackend class object.
   *
   *   This function provides the access to class object, but only one instance
   * is returned as mentioned above in the definition of singleton pattern.
   *
   *   @return returns a shared pointer of the instance of the OpensslBackend
   */
  static std::shared_ptr<OpensslBackend> &getInstance();

  /*@brief Derives a key using the OpenSSL method PKCS5_PBKDF2_HMAC method
   * @param pass is the PIN or the password the user issues via the A-IVI HMI.
   * @param salt is the salt used for the key derivation cryptographic function.
   * @param iter is the number of iterations to use in the key derivation.
   * @param keylen is the size of the key to be generated.
   *
   * @return The generated key
   *
   **/
  std::vector<unsigned char> DeriveKey(std::vector<unsigned char> &pass,
                                       std::vector<unsigned char> &salt,
                                       int iter, int keylen);
};

#endif /* SRC_BACKENDS_OPENSSL_OPENSSLBACKEND_H_ */
