/******************************************************************
 *FILE: UserEncryptDecrypt_OpensslBackend.cpp
 *SW-COMPONENT: UserEncryptDecrypt
 *DESCRIPTION: UserEncryptDecrypt
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 ******************************************************************/
/**
 * @author Guilherme Ferreira  (guilhermedaniel.ferreira@altran.com)
 * @date Nov, 2017
 */

#include "UserEncryptDecrypt_OpensslBackend.h"
#include <domains/UserEncryptDecrypt_PassphraseDomain.h>
#include <domains/UserEncryptDecrypt_SaltDomain.h>
#include <error/UserEncryptDecrypt_ErrorMessage.h>

#ifdef ROBS
#include <mock/RobsMock.h>
#else
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#endif

std::mutex OpensslBackend::_mutex;
std::shared_ptr<OpensslBackend> OpensslBackend::_instance = nullptr;

OpensslBackend::OpensslBackend() {
  // TODO Auto-generated constructor stub
}

OpensslBackend::~OpensslBackend() {
  // TODO Auto-generated destructor stub
}

std::shared_ptr<OpensslBackend> &OpensslBackend::getInstance() {
  if (!_instance) {
    try {
      _instance.reset(new OpensslBackend());
    } catch (std::bad_alloc &) {
      THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
    }
  }
  return _instance;
}

std::vector<unsigned char> OpensslBackend::DeriveKey(
    std::vector<unsigned char> &pass, std::vector<unsigned char> &salt,
    int iter, int keylen) {
  isValidInDomain<Passphrase>(pass);
  isValidInDomain<Salt>(salt);

  unsigned char *out;
  try {
    out = new unsigned char[keylen];
  } catch (std::bad_alloc &) {
    THROW_UED_EXCEPTION(ErrType::SYSTEM_noMemory);
  }
  std::vector<unsigned char> key;

  if (key.max_size() < keylen) {
    THROW_UED_EXCEPTION(ErrType::OPENSSL_keylenCannotFit);
  }

  key.reserve(keylen);

#ifndef ROBS
  int ret = PKCS5_PBKDF2_HMAC((const char *)pass.data(), pass.size(),
                              (const unsigned char *)salt.data(), salt.size(),
                              iter, EVP_sha256(), keylen, out);
#else
  int ret = PKCS5_PBKDF2_HMAC((const char *)pass.data(), pass.size(),
                              (const unsigned char *)salt.data(), salt.size(),
                              iter, EVP_sha256(0), keylen, out);
#endif

  if (ret != 1) {
    THROW_UED_EXCEPTION(ErrType::OPENSSL_keyDerivationFailed);
  }

  key = std::vector<unsigned char>(out, out + keylen);

  delete[] out;

  return key;
}
