/**************************************************************************//**
 * \file       Sds_TextDB.cpp
 *
 * Implements a fixtext data base with multi-language support.
 * Data structures are designed so that the module has a small ROM
 * footprint.
 * All text data is included from the generated Sds_TextDB.dat.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds_TextDB.h"

#include <string.h>

/**
 *  Generated array with offsets into the ROM strings.
 */
static const tU32 g_aIdLookup[] =
{
#define SDS_TEXT(id, isMultiLang, offset)      (((tU32)((isMultiLang) << 31)) | (offset)),
#include "Sds_TextDB.dat"
#undef SDS_TEXT
};


/**
 *  Generated array with language names.
 */
static const char* g_aLanguageNames[] =
{
#define SDS_TEXT_LANGUAGE(id)                  #id,
#include "Sds_TextDB.dat"
#undef SDS_TEXT_LANGUAGE
};


/**
 *  Generated array with multi-language lookup data.
 */
static const tU32 g_aLanguageLookup[] =
{
#define SDS_TEXT_LANG_OFFSETS
#include "Sds_TextDB.dat"
#undef SDS_TEXT_LANG_OFFSETS
   0x00000000
};


/**
 *  Generated byte array with all concatenated fixtexts.
 */
static const tChar g_acFixtext[] =
#define SDS_TEXT_STRINGS
#include "Sds_TextDB.dat"
#undef SDS_TEXT_STRINGS
   ;


static Sds_TextLanguageId g_currentLanguage = SDS_TEXT_LANGUAGE_ID_LIMIT;


/***********************************************************************//**
*
***************************************************************************/
static bool endsWith(const std::string& str, const std::string& isoCode)
{
   return !(str.compare(str.size() - isoCode.size(), isoCode.size(), isoCode));
}


/***********************************************************************//**
*
***************************************************************************/
Sds_TextLanguageId Sds_TextDB_enFindLanguageId(const std::string& langCode, const std::string& countryCode)
{
   std::string isoLang = langCode + "_" + countryCode;
   for (size_t i = 0; i < (sizeof(g_aLanguageNames) / sizeof(g_aLanguageNames[0])); i++)
   {
      if (endsWith(g_aLanguageNames[i], isoLang))
      {
         return static_cast<Sds_TextLanguageId>(i);
      }
   }
   return SDS_TEXT_LANGUAGE_ID_LIMIT;
}


/**************************************************************************//**
* Tells the text data base to subsequently retrieve
* strings for "g_currentLanguage"
******************************************************************************/
void Sds_TextDB_vSetLanguage(const std::string& isoLangCode, const std::string& isoCountryCode)
{
   Sds_TextLanguageId enLanguageId = Sds_TextDB_enFindLanguageId(isoLangCode, isoCountryCode);

   if (enLanguageId < SDS_TEXT_LANGUAGE_ID_LIMIT)
   {
      g_currentLanguage = enLanguageId;
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();     // invalid language id
      g_currentLanguage = static_cast<Sds_TextLanguageId>(0);      //lint !e527 Unreachable code - jnd2hi: behind assertion
   }
}


/**************************************************************************//**
* Returns the name of the language id.
******************************************************************************/
const char* Sds_TextDB_pGetLanguageName(Sds_TextLanguageId enLanguage)
{
   if (enLanguage < SDS_TEXT_LANGUAGE_ID_LIMIT)
   {
      return g_aLanguageNames[enLanguage];
   }
   return "undefined language";
}


/**************************************************************************//**
* Returns a language-dependent text offset into g_acFixtext[].
* "u32MaskIndex" points to a multi-language data set. The first entry is a
* language mask followed by as many text offsets as the language mask has bits.
* If the bit for the current language is not set in the mask, the text offset
* for the first language - the default language - will be returned.
******************************************************************************/
static tU32 Sds_TextDB_u32GetMultiLanguageTextOffset(tU32 u32MaskIndex, tU32 u32Language)
{
   const tU32 u32Mask = g_aLanguageLookup[u32MaskIndex];
   tU32 u32LanguageFlag = 1 << u32Language;
   tU32 u32Index = 1;
   if ((u32Mask & u32LanguageFlag) != 0)
   {
      /* determine offset position in lookup for current language */
      u32LanguageFlag >>= 1;
      while (u32LanguageFlag != 0)
      {
         if ((u32LanguageFlag & u32Mask) != 0)
         {
            u32Index++;
         }
         u32LanguageFlag >>= 1;
      }
   }
   return g_aLanguageLookup[u32MaskIndex + u32Index];
}


/**************************************************************************//**
* Returns an offset into g_acFixtext[] which points to the beginning of the
* string for "u32TextId".
******************************************************************************/
static tU32 Sds_TextDB_u32GetTextOffset(tU32 u32TextId, tU32 u32Language)
{
   const tU32 u32Descr = g_aIdLookup[u32TextId];
   const tU32 u32Offset = u32Descr & 0x7FFFFFFF;
   const bool blIsMultiLang = (u32Descr & 0x80000000) != 0;
   if (blIsMultiLang == TRUE)
   {
      return Sds_TextDB_u32GetMultiLanguageTextOffset(u32Offset, u32Language);
   }
   else
   {
      return u32Offset;
   }
}


/**************************************************************************//**
* Retrieves the fixtext for "u32TextId" from the fixtext arrays.
* The pointer to the string is returned in "ppubString" and its length in
* "pu32Length". Note that the string is not zero-terminated.
******************************************************************************/
static const tChar* Sds_TextDB_pubGetFixString(tU32 u32TextId, tU32 u32Language)
{
   if (u32TextId < (tU32)(SDS_TEXT_ID_LIMIT))
   {
      return g_acFixtext + Sds_TextDB_u32GetTextOffset(u32TextId, u32Language);
   }
   else if (u32TextId == (tU32)(SDS_TEXT_ID_LIMIT))
   {
      return "";
   }
   NORMAL_M_ASSERT_ALWAYS();
   return "";  //lint !e527 Unreachable code - jnd2hi: behind assertion
}


/**************************************************************************//**
* Returns a pointer to the constant string identified by "enTextId". The string
* is returned in the currently set language.
******************************************************************************/
const tChar* Sds_TextDB_vGetText(Sds_TextId enTextId)
{
   const tChar* text = Sds_TextDB_pubGetFixString((tU32)(enTextId), g_currentLanguage);
   if (strlen(text) == 0)
   {
      // for missing translations return US English
      return Sds_TextDB_pubGetFixString((tU32)(enTextId), 0);
   }
   return text;
}


/**************************************************************************//**
*
******************************************************************************/
Sds_TextId Sds_TextDB_enFindTextId(const std::string& oString)
{
   for (tU32 u32Index = 0; u32Index < (tU32)(SDS_TEXT_ID_LIMIT); u32Index++)
   {
      if ((oString == ((Sds_TextDB_vGetText(((Sds_TextId)(u32Index)))))))
      {
         return ((Sds_TextId)u32Index);
      }
   }
   return SDS_TEXT_ID_LIMIT;
}


/***********************************************************************//**
*
***************************************************************************/
unsigned char Sds_TextDB_bIsMatchingTextId(Sds_TextId enView, const std::string& oString)
{
   if (enView == Sds_TextDB_enFindTextId(oString))
   {
      return TRUE;
   }
   return FALSE;
}
