/**************************************************************************//**
 * \file       FileUtils.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "FileUtils.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>


static std::string stripTrailing(std::string s, char c)
{
   while ((s.size() > 1) && (s[s.size() - 1] == c))
   {
      s.erase(s.size() - 1);
   }
   return s;
}


std::string dirname(std::string path)
{
   path = stripTrailing(path, '/');
   size_t pos = path.rfind('/');
   if (pos == std::string::npos)
   {
      return ".";    // no slash - current directory
   }
   if (pos == 0)
   {
      return "/";    // root directory
   }

   return stripTrailing(path.substr(0, pos), '/');
}


bool dirExists(const std::string& path)
{
   struct stat st;
   if (stat(path.c_str(), &st) == 0)
   {
      if (S_ISDIR(st.st_mode))
      {
         return true;
      }
   }
   return false;
}


bool fileExists(const std::string& path)
{
   struct stat st;
   if (stat(path.c_str(), &st) == 0)
   {
      if (S_ISREG(st.st_mode))
      {
         return true;
      }
   }
   return false;
}


bool makeDir(const std::string& path)
{
   if (dirExists(path))
   {
      return true;
   }
   return (mkdir(path.c_str(), 0777) == 0);  // TODO jnd2hi: is 0777 the correct permission mask?
}


bool makePath(const std::string& path)
{
   if (dirExists(path))
   {
      return true;
   }
   if (makePath(dirname(path)))
   {
      return makeDir(path);
   }
   return false;
}


bool deleteFile(const std::string& path)
{
   if (fileExists(path))
   {
      return (unlink(path.c_str()) == 0);
   }
   return true;
}
