/**************************************************************************//**
 * \file       clSDS_View.cpp
 *
 * clSDS_View class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_View.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "application/clSDS_ListItems.h"
#include "application/clSDS_StringVarList.h"
#include "application/clSDS_SDSStatus.h"
#include "application/clSDS_PhoneNumberFormatter.h"
#include "application/StringUtils.h"
#include "view_db/Sds_TextDB.h"
#include "view_db/Sds_ViewDB.h"
#include "SdsAdapter_Trace.h"
#include <stdio.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_View.cpp.trc.h"
#endif


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_View::~clSDS_View()
{
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_View::clSDS_View() : _commandListCount(0), _maxItemsPerPage(1)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vCreateView(
   sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal,
   const std::string& viewId,
   std::vector<clSDS_ListItems>& oListItems,
   const std::map<std::string, std::string>& oScreenVariableData,
   std::string enTemplateId,
   const std::map<std::string, std::string>& oHeaderValueMap,
   uint32 cursorIndex,
   uint32 pageNumber,
   unsigned int maxItemsperPage)
{
   vInitView(viewPopupRequestSignal);
   if (!viewId.empty())
   {
      _maxItemsPerPage = maxItemsperPage;
      viewPopupRequestSignal.setLayout(Sds_ViewDB_getLayout(viewId));

      vWriteHeadline(viewPopupRequestSignal, oScreenVariableData, enTemplateId, oHeaderValueMap, viewId);

      vWriteHelpline(viewPopupRequestSignal, oScreenVariableData, enTemplateId, oHeaderValueMap, viewId);

      vWriteInfo(viewPopupRequestSignal, oScreenVariableData, enTemplateId, oHeaderValueMap, viewId);

      vWriteCommands(viewPopupRequestSignal, oScreenVariableData, viewId, pageNumber);

      ETG_TRACE_USR1(("vCreateView"));
      if (Sds_ViewDB_getLayout(viewId) == "R")
      {
         vWriteSubCommands(viewPopupRequestSignal, oScreenVariableData, viewId, cursorIndex);
      }
      else if (Sds_ViewDB_getLayout(viewId) == "R2")
      {
         vWriteSubCommands(viewPopupRequestSignal, oScreenVariableData, viewId);
      }
      else
      {
         vWriteExampleCommands(viewPopupRequestSignal, oScreenVariableData, viewId);
      }
      vWriteListData(viewPopupRequestSignal, oListItems);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vCreateViewHeader(
   sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal,
   std::string templateId,
   const std::map<std::string, std::string>& oHeaderValueMap,
   const std::string& viewId) const
{
   //vWriteHeadline(viewPopupRequestSignal, templateId, oHeaderValueMap, viewId);
}


static std::string formatUint(size_t number, const char* format)
{
   char text[100];
   snprintf(text, sizeof text, format, static_cast<uint32_t>(number));
   text[sizeof text - 1] = 0;
   return text;
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vWriteListData(
   sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal,
   std::vector<clSDS_ListItems>& oListItems) const
{
   ::std::vector< sds_gui_fi::PopUpService::TextField >& textFields(viewPopupRequestSignal.getTextFieldsMutable());

   for (tU32 u32Index = 0;
         (u32Index < oListItems.size()) && (u32Index < _maxItemsPerPage);
         u32Index++)
   {
      const clSDS_ListItems& listItem = oListItems[u32Index];

      if (!(oListItems[u32Index].bIsListScreenWithoutIndex))
      {
         sds_gui_fi::PopUpService::TextField textFieldCommand;
         fillTextFieldData(textFieldCommand,
                           std::string("cmd") + StringUtils::toString(u32Index),
                           formatUint(u32Index  + 1, Sds_TextDB_vGetText(SDS_TEXT_LIST_INDEX_FORMAT)),
                           colorToTextAttribute(listItem.oCommand.enTextColor));
         textFields.push_back(textFieldCommand);
      }
      else
      {
         if (listItem.oCommand.szString != "")
         {
            sds_gui_fi::PopUpService::TextField textFieldCommand;
            fillTextFieldData(textFieldCommand,
                              std::string("cmd") + StringUtils::toString(u32Index),
                              listItem.oCommand.szString,
                              colorToTextAttribute(listItem.oCommand.enTextColor));
            textFields.push_back(textFieldCommand);
         }
      }

      if (listItem.oDescription.szString != "")
      {
         sds_gui_fi::PopUpService::TextField textFieldDescription;
         fillTextFieldData(textFieldDescription,
                           std::string("descr")  + StringUtils::toString(u32Index),
                           listItem.oDescription.szString,
                           colorToTextAttribute(listItem.oDescription.enTextColor));
         textFields.push_back(textFieldDescription);
      }

      if (listItem.oDistance.szString != "")
      {
         sds_gui_fi::PopUpService::TextField textFieldDistance;
         fillTextFieldData(textFieldDistance,
                           std::string("dist") + StringUtils::toString(u32Index),
                           listItem.oDistance.szString,
                           colorToTextAttribute(listItem.oDistance.enTextColor));
         textFields.push_back(textFieldDistance);
      }
      if (listItem.oDirectionSymbol.szString != "")
      {
         sds_gui_fi::PopUpService::TextField textFieldDirectionSymbol;
         fillTextFieldData(textFieldDirectionSymbol,
                           std::string("dir") + StringUtils::toString(u32Index),
                           listItem.oDirectionSymbol.szString,
                           colorToTextAttribute(listItem.oDirectionSymbol.enTextColor));
         textFields.push_back(textFieldDirectionSymbol);
      }
      if (listItem.oPrice.szString != "")
      {
         sds_gui_fi::PopUpService::TextField textFieldPrice;
         fillTextFieldData(textFieldPrice,
                           std::string("price") + StringUtils::toString(u32Index),
                           listItem.oPrice.szString,
                           colorToTextAttribute(listItem.oPrice.enTextColor));
         textFields.push_back(textFieldPrice);
      }
      if (listItem.oLastPriceUpdate.szString != "")
      {
         sds_gui_fi::PopUpService::TextField textFieldLastPriceUpdate;
         fillTextFieldData(textFieldLastPriceUpdate,
                           std::string("priceU") + StringUtils::toString(u32Index),
                           listItem.oLastPriceUpdate.szString,
                           colorToTextAttribute(listItem.oLastPriceUpdate.enTextColor));
         textFields.push_back(textFieldLastPriceUpdate);
      }
      if (listItem.oMessage.szString != "")
      {
         sds_gui_fi::PopUpService::TextField textFieldMessage;
         fillTextFieldData(textFieldMessage,
                           std::string("msg") + StringUtils::toString(u32Index),
                           listItem.oMessage.szString,
                           colorToTextAttribute(listItem.oMessage.enTextColor));
         textFields.push_back(textFieldMessage);
      }
      if (listItem.oTime.szString != "")
      {
         sds_gui_fi::PopUpService::TextField textFieldDateTime;
         fillTextFieldData(textFieldDateTime,
                           std::string("time") + StringUtils::toString(u32Index),
                           listItem.oTime.szString,
                           colorToTextAttribute(listItem.oTime.enTextColor));
         textFields.push_back(textFieldDateTime);
      }
      if (listItem.oIcon.szString != "")
      {
         sds_gui_fi::PopUpService::TextField textFieldIcon;
         fillTextFieldData(textFieldIcon,
                           std::string("icon") + StringUtils::toString(u32Index),
                           listItem.oIcon.szString,
                           colorToTextAttribute(listItem.oIcon.enTextColor));
         textFields.push_back(textFieldIcon);
      }
      if (listItem.oIcon2.szString != "")
      {
         sds_gui_fi::PopUpService::TextField textFieldIcon;
         fillTextFieldData(textFieldIcon,
                           std::string("calltype") + StringUtils::toString(u32Index),
                           listItem.oIcon2.szString,
                           colorToTextAttribute(listItem.oIcon2.enTextColor));
         textFields.push_back(textFieldIcon);
      }
   }
}


tVoid clSDS_View::fillTextFieldData(
   sds_gui_fi::PopUpService::TextField& textField,
   std::string tagName,
   std::string value,
   sds_gui_fi::PopUpService::TextAttribute colorOfText) const
{
   textField.setTagName(tagName);
   textField.setString(value);
   textField.setAttrib(colorOfText);
}


/***********************************************************************//**
 *
 ***************************************************************************/
sds_gui_fi::PopUpService::TextAttribute clSDS_View::colorToTextAttribute(clSDS_ListItems::tenColorofText colorOfText) const
{
   sds_gui_fi::PopUpService::TextAttribute textAttrib;

   switch (colorOfText)
   {
      case clSDS_ListItems::NORMAL:
         textAttrib = sds_gui_fi::PopUpService::TextAttribute__NORMAL;
         break;

      case clSDS_ListItems::COMMAND:
         textAttrib = sds_gui_fi::PopUpService::TextAttribute__COMMAND;
         break;

      case clSDS_ListItems::GREYED_OUT:
         textAttrib = sds_gui_fi::PopUpService::TextAttribute__GREYED_OUT;
         break;

      case clSDS_ListItems::PRICE_AGE_CURRENT:
         textAttrib = sds_gui_fi::PopUpService::TextAttribute__AGE_TODAY;
         break;

      case clSDS_ListItems::PRICE_AGE_1_DAY:
         textAttrib = sds_gui_fi::PopUpService::TextAttribute__AGE_ONE_DAY;
         break;

      case clSDS_ListItems::PRICE_AGE_2_DAYS:
         textAttrib = sds_gui_fi::PopUpService::TextAttribute__AGE_TWO_DAYS;
         break;

      case clSDS_ListItems::PRICE_AGE_3_DAYS_PLUS:
         textAttrib = sds_gui_fi::PopUpService::TextAttribute__AGE_THREE_PLUS_DAYS;
         break;

      case clSDS_ListItems::clSDS_ListItems::SELECTED:
         textAttrib = sds_gui_fi::PopUpService::TextAttribute__SELECTED;
         break;

      default:
         textAttrib = sds_gui_fi::PopUpService::TextAttribute__NORMAL;
         break;
   }

   return textAttrib;
}


/***********************************************************************//**
 *
 ***************************************************************************/

std::string clSDS_View::mapInfoFromHeadline(
   sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal,
   std::string oHeaderString,
   std::map<std::string, std::string> const& oHeaderValueMap,
   const std::string& viewId)const
{
   if (Sds_ViewDB_oGetInfo(viewId).size())
   {
      if (Sds_ViewDB_oGetInfo(viewId) == "&(Headline)")
      {
         viewPopupRequestSignal.getTextFieldsMutable().push_back(sds_gui_fi::PopUpService::TextField("info", oResolveVariableContent(oHeaderString, oHeaderValueMap).c_str(), sds_gui_fi::PopUpService::TextAttribute__NORMAL));
         oHeaderString = "";
      }
      else
      {
         viewPopupRequestSignal.getTextFieldsMutable().push_back(sds_gui_fi::PopUpService::TextField("info", oResolveVariableContent(Sds_ViewDB_oGetInfo(viewId), oHeaderValueMap).c_str(), sds_gui_fi::PopUpService::TextAttribute__NORMAL));
      }
   }
   return oHeaderString;
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vWriteHeadline(
   sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal,
   const std::map<std::string, std::string>& oScreenVariableData,
   std::string templateId,
   std::map<std::string, std::string> const& oHeaderValueMap,
   const std::string& viewId)
{
   std::string oHeaderString;
   switch (Sds_ViewDB_oGetHeadlineBehaviorId(viewId))
   {
      case Default:
      case Template:
      case StickyTemplate:
      {
         if (!templateId.empty())
         {
            oHeaderString = Sds_ViewDB_oGetTemplateString(templateId);
         }
         else
         {
            oHeaderString = Sds_ViewDB_oGetHeadline(viewId);
         }
         oHeaderString = oResolveVariableContent(oHeaderString, oHeaderValueMap);
      }
      break;
      case SplitTemplate:
      {
         if (!templateId.empty())
         {
            oHeaderString = Sds_ViewDB_oGetTemplateString(templateId);
         }
         oHeaderString = oResolveVariableContent(oHeaderString, oHeaderValueMap);
         oHeaderString = oSplitHeaderContent(oHeaderString, 0);
      }
      break;
      case Static:
      {
         oHeaderString = Sds_ViewDB_oGetHeadline(viewId);
         oHeaderString = oResolveVariableContent(oHeaderString, oHeaderValueMap);
      }
      break;
      default:
         break;
   }
   _lastViewId = viewId;
   viewPopupRequestSignal.setHeader(oHeaderString.c_str());
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vWriteHelpline(
   sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal,
   const std::map<std::string, std::string>& oScreenVariableData,
   std::string templateId,
   std::map<std::string, std::string> const& oHeaderValueMap,
   const std::string& viewId) const
{
   if (Sds_ViewDB_oGetHelpline(viewId).size())
   {
      viewPopupRequestSignal.getTextFieldsMutable().push_back(sds_gui_fi::PopUpService::TextField("help", Sds_ViewDB_oGetHelpline(viewId), sds_gui_fi::PopUpService::TextAttribute__NORMAL));
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vWriteInfo(
   sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal,
   const std::map<std::string, std::string>& oScreenVariableData,
   std::string templateId,
   std::map<std::string, std::string> const& oHeaderValueMap,
   const std::string& viewId) const
{
   std::string oInfoString;

   switch (Sds_ViewDB_oGetInfoBehaviorId(viewId))
   {
      case SplitTemplate:
      {
         if (!templateId.empty())
         {
            oInfoString = Sds_ViewDB_oGetTemplateString(templateId);
         }
         oInfoString = oResolveVariableContent(oInfoString, oScreenVariableData, false);
         oInfoString = oResolveVariableContent(oInfoString, oHeaderValueMap);
         oInfoString = oSplitHeaderContent(oInfoString, 1);
      }
      break;

      case Template:
      {
         if (!templateId.empty())
         {
            oInfoString = Sds_ViewDB_oGetTemplateString(templateId);
         }
         //oInfoString = oResolveVariableContent(oInfoString, oScreenVariableData, false);
         oInfoString = oResolveVariableContent(oInfoString, oHeaderValueMap);
         //oInfoString = oSplitHeaderContent(oInfoString, 1);
      }
      break;
      case Default:
      {
         oInfoString = Sds_ViewDB_oGetInfo(viewId);
         oInfoString = oResolveVariableContent(oInfoString, oScreenVariableData, false);
         oInfoString = oResolveVariableContent(oInfoString, oHeaderValueMap);
      }
      break;
      default:
         break;
   }
   viewPopupRequestSignal.getTextFieldsMutable().push_back(sds_gui_fi::PopUpService::TextField("info", oInfoString.c_str(), sds_gui_fi::PopUpService::TextAttribute__NORMAL));
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_View::oSplitHeaderContent(std::string& stringToSplit, size_t index) const
{
   std::string delimiter = "\n";
   std::string str = "";
   std::vector<std::string>tokens = StringUtils::split(stringToSplit, delimiter);

   try
   {
      str = tokens.at(index);
   }
   catch (...)
   {
      ETG_TRACE_ERR(("caught exception in split function"));// If an invalid index given
   }
   return str;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_View::oResolveVariableContent(
   std::string xmlStringContent,
   std::map<std::string, std::string> const& actualValue,
   bool isClearRequired) const
{
   std::map<std::string, std::string>::const_iterator iter;
   for (iter = actualValue.begin(); iter != actualValue.end(); ++iter)
   {
      std::string variable = "$(" + iter->first + ")";
      while (xmlStringContent.find(variable) != std::string::npos)
      {
         xmlStringContent.replace(xmlStringContent.find(variable), variable.size(), iter->second);
      }
   }
   xmlStringContent = clSDS_StringVarList::oResolveVariables(xmlStringContent);
   if (!isClearRequired)
   {
      return xmlStringContent;
   }
   clearUnresolvedVariables(xmlStringContent);
   removeUnnecessarySeparators(xmlStringContent);
   return oResolveNumberFormat(xmlStringContent);
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string& clSDS_View::clearUnresolvedVariables(std::string& str) const
{
   size_t startPos = str.find("$(");
   while (startPos != std::string::npos)
   {
      const size_t endPos = str.find(")", startPos);

      if (endPos != std::string::npos)
      {
         const size_t length = (endPos + 1) - startPos;

         ETG_TRACE_COMP(("removed unresolved template variable '%s'", str.substr(startPos, length).c_str()));
         str.erase(startPos, length);
      }
      else
      {
         // unmatched $(
         ETG_TRACE_ERR(("unterminated template variable in '%s'", str.c_str()));
         break;
      }
      startPos = str.find("$(", startPos);
   }
   return str;
}


/**
 * Replace all occurrences of 'old' in 'str' with 'new_'. The replacement is
 * done in-place, i.e. the input string is modified. In case 'old' does not
 * appear in 'str', 'str' remains unchanged.
 */
static std::string& replaceAll(std::string& str, const std::string& old, const std::string& new_)
{
   size_t pos = str.find(old);
   while (pos != std::string::npos)
   {
      str.replace(pos, old.length(), new_);
      pos = str.find(old, pos);
   }
   return str;
}


/**
 * Replace substring 'old' at position 'pos' in 'str' with 'new_'.
 * The replacement is done in-place, i.e. the input string is modified.
 * In case 'old' does not appear at 'pos' in 'str', 'str' remains unchanged.
 */
static std::string& replaceAt(std::string& str, size_t pos, const std::string& old, const std::string& new_)
{
   if ((pos < str.length()) && (str.substr(pos, old.length()) == old))
   {
      str.replace(pos, old.length(), new_);
   }
   return str;
}


/**
 * Remove any unnecessary separators from a string.
 * In case dynamic content variables resolve to an empty string,
 * a resolved template string may contain unnecessary comma or
 * space separators or trailing / leading whitespace.
 * ", , Hello  World\n, Foo, Bar, , " -> "Hello World\nFoo, Bar"
 */
std::string& clSDS_View::removeUnnecessarySeparators(std::string& str) const
{
   replaceAll(str, ", ,", ",");                 // remove empty separator sequences
   replaceAll(str, "\n, ", "\n");               // remove empty separator after linebreak
   replaceAt(str, 0, ", ", "");                 // remove leading empty separator
   replaceAt(str, str.length() - 2, ", ", "");  // remove trailing empty separator
   replaceAll(str, "  ", " ");                  // replace multiple spaces with single whitespace
   replaceAll(str, ", ?", "?");                 // remove trailing comma followed by question mark
   replaceAll(str, " ,", "");                   // remove space followed by comma
   return StringUtils::trim(str);
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_View::oResolveNumberFormat(const std::string& oTemplateString) const
{
   std::string startTag = "#FormatNumber(";
   std::string::size_type startTagPos = oTemplateString.find(startTag);
   if (startTagPos != std::string::npos)
   {
      std::string::size_type endTagPos = oTemplateString.find(")", startTagPos);
      if (endTagPos != std::string::npos)
      {
         std::string::size_type numStart = startTagPos + startTag.size();
         std::string::size_type numLen = endTagPos - (startTagPos + startTag.size());
         std::string oNumber = oTemplateString.substr(numStart, numLen);
         std::string oFormattedNumber = clSDS_PhoneNumberFormatter::oFormatNumber(oNumber);
         return oTemplateString.substr(0, startTagPos) + oFormattedNumber + oTemplateString.substr(endTagPos + 1);
      }
   }
   return oTemplateString;
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vWriteCommands(sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal, std::map<std::string, std::string> const& variables, const std::string& viewId, tU32 pageNumber)
{
   std::vector<SelectableText> oCommandList = Sds_ViewDB_oGetCommandList(variables, viewId);
   ::std::vector< sds_gui_fi::PopUpService::TextField >& oTextFields(viewPopupRequestSignal.getTextFieldsMutable());
   _commandListCount = (oCommandList.size() > _maxItemsPerPage) ? _maxItemsPerPage : oCommandList.size();
   ((Sds_ViewDB_getLayout(viewId) == "B") || (Sds_ViewDB_getLayout(viewId) == "A")) ? (pageNumber = 0) : pageNumber;
   const tU32 startIndex = pageNumber * _maxItemsPerPage;
   const tU32 endIndex = oCommandList.size() < startIndex + _maxItemsPerPage ? oCommandList.size() : startIndex + _maxItemsPerPage;

   for (tU32 u32Index = startIndex; u32Index < endIndex; u32Index++)
   {
      std::string resolved = clSDS_StringVarList::oResolveVariables(oCommandList[u32Index].text);
      std::string cmdTagName;

      // if (oCommandList[u32Index].enBehavior == Tab)
      if ((Sds_ViewDB_getLayout(viewId) == "B") || (Sds_ViewDB_getLayout(viewId) == "A"))
      {
         cmdTagName = std::string("tab_cmd") + StringUtils::toString(u32Index);
      }
      else
      {
         cmdTagName = std::string("cmd") + StringUtils::toString(u32Index);
      }

      clSDS_ListItems::tenColorofText attrib = clSDS_ListItems::COMMAND;
      if (!oCommandList[u32Index].isSelectable)
      {
         attrib = clSDS_ListItems::GREYED_OUT;
      }
      if (Sds_ViewDB_getIsSelected(viewId, u32Index, variables))
      {
         attrib = clSDS_ListItems::SELECTED;
      }
      sds_gui_fi::PopUpService::TextField otextField;
      otextField.setTagName(cmdTagName);
      otextField.setString(resolved);
      otextField.setAttrib(colorToTextAttribute(attrib));
      oTextFields.push_back(otextField);

      if (oCommandList[u32Index].enBehavior == Icon)
      {
         resolved = Sds_ViewDB_getGrammarId(viewId, u32Index, variables);
         cmdTagName = std::string("icon") + StringUtils::toString(u32Index);
         otextField.setTagName(cmdTagName);
         otextField.setString(resolved);
         otextField.setAttrib(colorToTextAttribute(clSDS_ListItems::NORMAL));
         oTextFields.push_back(otextField);
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vWriteSubCommands(sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal, std::map<std::string, std::string> const& variables, const std::string& viewId, tU32 cursorIndex) const
{
   std::vector<std::string> subCommandList = Sds_ViewDB_oGetSubCommandList(variables, viewId, cursorIndex);
   ::std::vector< sds_gui_fi::PopUpService::TextField >& textFields = viewPopupRequestSignal.getTextFieldsMutable();
   ETG_TRACE_USR1((" vWriteSubCommands 1"));
   for (size_t i = 0 ; i < subCommandList.size(); i++)
   {
      std::string resolved = clSDS_StringVarList::oResolveVariables(subCommandList[i]);
      std::string cmdTagName = std::string("subCmd") + StringUtils::toString(i);

      sds_gui_fi::PopUpService::TextField textField;
      textField.setTagName(cmdTagName);
      textField.setString(resolved);
      textField.setAttrib(colorToTextAttribute(clSDS_ListItems::NORMAL));
      textFields.push_back(textField);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vWriteSubCommands(sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal, std::map<std::string, std::string> const& variables, const std::string& viewId) const
{
   std::vector<std::string> subCommandList = Sds_ViewDB_oGetSubCommandList(variables, viewId);
   ::std::vector< sds_gui_fi::PopUpService::TextField >& textFields = viewPopupRequestSignal.getTextFieldsMutable();
   ETG_TRACE_USR1((" vWriteSubCommands 2"));
   ETG_TRACE_USR1((" vWriteSubCommands 2 %d:", subCommandList.size()));
   for (size_t i = 0 ; i < subCommandList.size() ; i++)
   {
      std::string resolved = clSDS_StringVarList::oResolveVariables(subCommandList[i]);
      std::string cmdTagName = std::string("subCmd") + StringUtils::toString(i);

      sds_gui_fi::PopUpService::TextField textField;
      textField.setTagName(cmdTagName);
      textField.setString(resolved);
      textField.setAttrib(colorToTextAttribute(clSDS_ListItems::NORMAL));
      textFields.push_back(textField);
      ETG_TRACE_USR1((" vWriteSubCommands 2 cmdTagName %s:", cmdTagName.c_str()));
      ETG_TRACE_USR1((" vWriteSubCommands 2 resolved %s:", resolved.c_str()));
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_View::vWriteExampleCommands(sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal, std::map<std::string, std::string> const& variables, const std::string& viewId)const
{
   std::vector<std::string> exampleCommandList = Sds_ViewDB_oGetExampleCommandList(variables, viewId);
   std::vector<std::string> subCommandList = Sds_ViewDB_oGetSubCommandList(variables, viewId);
   ::std::vector< sds_gui_fi::PopUpService::TextField >& textFields = viewPopupRequestSignal.getTextFieldsMutable();
   size_t exampleCommandListSize = exampleCommandList.size();
   for (size_t i = 0 ; i < exampleCommandListSize; i++)
   {
      std::string resolved = clSDS_StringVarList::oResolveVariables(exampleCommandList[i]);
      std::string cmdTagName = std::string("subCmd") + StringUtils::toString(i);
      sds_gui_fi::PopUpService::TextField textField;
      textField.setTagName(cmdTagName);
      textField.setString(resolved);
      textField.setAttrib(colorToTextAttribute(clSDS_ListItems::NORMAL));
      textFields.push_back(textField);
   }
   for (size_t j = 0, k = exampleCommandListSize; j < subCommandList.size(); j++, k++)
   {
      std::string resolved = clSDS_StringVarList::oResolveVariables(subCommandList[j]);
      std::string cmdTagName = std::string("subCmd") + StringUtils::toString(k);
      sds_gui_fi::PopUpService::TextField textField;
      textField.setTagName(cmdTagName);
      textField.setString(resolved);
      textField.setAttrib(colorToTextAttribute(clSDS_ListItems::NORMAL));
      textFields.push_back(textField);
   }
}


tVoid clSDS_View::vInitView(sds_gui_fi::PopUpService::PopupRequestSignal& viewPopupRequestSignal) const
{
   viewPopupRequestSignal.clearHeader();
   viewPopupRequestSignal.clearLayout();
   viewPopupRequestSignal.clearTextFields();
}


/***********************************************************************//**
*
***************************************************************************/
unsigned int clSDS_View::getCommandListCount() const
{
   return _commandListCount;
}
