/**************************************************************************//**
 * \file       clSDS_SessionControl.cpp
 *
 * SDS Session Control class implementation implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_SessionControl.h"
#include "application/clSDS_SDSStatus.h"
#include "application/clSDS_SdsControl.h"
#include "Sds2HmiServer/functions/clSDS_Property_PhoneStatus.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_SessionControl.cpp.trc.h"
#endif

using namespace sds_gui_fi::SdsGuiService;

#define ETG_ENABLED
#include "trace_interface.h"
#include "SdsAdapter_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_SessionControl.cpp.trc.h"
#endif
/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_SessionControl::~clSDS_SessionControl()
{
   _pSDSStatus = NULL;
   _pSdsControl = NULL;
   _pPhoneStatus = NULL;
   _pLanguageMediator = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_SessionControl::clSDS_SessionControl(
   clSDS_SDSStatus* pSDSStatus,
   clSDS_Property_PhoneStatus* pPhoneStatus,
   clSDS_SdsControl* pSdsControl,
   clSDS_LanguageMediator* pLanguageMediator)
   : _pSDSStatus(pSDSStatus)
   , _pPhoneStatus(pPhoneStatus)
   , _pSdsControl(pSdsControl)
   , _pLanguageMediator(pLanguageMediator)
   , _startupContext(sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_GLOBAL)
   , _dialedDigits("")
{
   _pSDSStatus->vRegisterObserver(this);
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_SessionControl::vPttLongPressed() const
{
   if (_pSdsControl)
   {
      _pSdsControl->vSendPTTLongPressEvent();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_SessionControl::vBackButtonPressed() const
{
   if (_pSdsControl)
   {
      _pSdsControl->vSendBackEvent();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_SessionControl::vStoreSessionContext(sds2hmi_fi_tcl_e8_SDS_EntryPoint::tenType startupContext)
{
   _startupContext = startupContext;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_SessionControl::vStopSession() const
{
   if (_pSdsControl)
   {
      _pSdsControl->sendCancelDialog();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_SessionControl::vAbortSession() const
{
   if (_pSdsControl)
   {
      _pSdsControl->vAbortDialog();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_SessionControl::vSwcPhoneEndPressed() const
{
   if (_pSdsControl)
   {
      _pSdsControl->vSendBackEvent();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_SessionControl::bIsPttPressAllowed() const
{
   // TODO jnd2hi: determine conditions for allowing PTT
   return TRUE;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_SessionControl::bIsSDSLanguageSupport() const
{
   return _pLanguageMediator->bIsSDSLanguageSupport();
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_SessionControl::vSDSStatusChanged()
{
   if (_pSDSStatus != NULL)
   {
      if (_pSDSStatus->getStatus() == clSDS_SDSStatus::EN_ACTIVE_PROMPTONLY)
      {
         startTimer();
      }
      else if (_timer.isActive())
      {
         stopTimer();
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
sds2hmi_fi_tcl_e8_SDS_EntryPoint::tenType clSDS_SessionControl::oGetStartupContext() const
{
   if (bMultipleCallDtmfSession() && (_startupContext == sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_GLOBAL))
   {
      return sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_PHONE_OMC;
   }
   if (bSingleCallDtmfSession() && (_startupContext == sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_GLOBAL))
   {
      return sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_PHONE_OAC;
   }
   return _startupContext;
}


/**************************************************************************//**
 *
 ******************************************************************************/
/*tBool clSDS_SessionControl::bSingleCallDtmfSession() const
{
   if ( HSI_PHONE_STS_IN_CALL == dataPool.u16GetValue(DPTELEPHONE__SDS_GETPHONESTATUS_STATUS) &&
         (dataPool.bGetValue(DPTELEPHONE__HANDS_FREE_STATE))
      )
   {
      return TRUE;
   }
   return FALSE;
}*/

/**************************************************************************//**
 *
 ******************************************************************************/
/*tBool clSDS_SessionControl::bMultipleCallDtmfSession() const
{
   if ( HSI_PHONE_STS_SWITCH_CALL == dataPool.u16GetValue(DPTELEPHONE__SDS_GETPHONESTATUS_STATUS) &&
         (dataPool.bGetValue(DPTELEPHONE__HANDS_FREE_STATE))
      )
   {
      return TRUE;
   }
   return FALSE;
}*/


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_SessionControl::vSendPttEvent()
{
   _startupContext = oGetStartupContext();
   if (_pSdsControl && (clSDS_ConfigurationFlags::getPhoneInCallFeatureKey() == true))
   {
      _pSdsControl->vSendPttEvent(tU32(_startupContext));
   }
   else if (_pSdsControl && (clSDS_ConfigurationFlags::getPhoneInCallFeatureKey() == false))
   {
      if ((_startupContext == sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_PHONE_OMC)
            || (_startupContext == sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_PHONE_OAC))
      {
         ETG_TRACE_USR4(("clSDS_SessionControl _startupContext inside RENAULT DTMF blocking mode"));
      }
      else
      {
         _pSdsControl->vSendPttEvent(tU32(_startupContext));
      }
   }
   _startupContext = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_GLOBAL;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_SessionControl::bDownloadFailed() const
{
   return _pSDSStatus->bIsError();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_SessionControl::bIsInitialising() const
{
   return _pSDSStatus->bIsLoading();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_SessionControl::bIsSessionActive() const
{
   return _pSDSStatus->bIsActive();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_SessionControl::bIsListening() const
{
   return _pSDSStatus->bIsListening();
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_SessionControl::bSingleCallDtmfSession() const
{
   if (_pPhoneStatus)
   {
      if ((_pPhoneStatus->getPhoneStatus() == sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_INCALL) ||
            (_pPhoneStatus->getPhoneStatus() == sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_DIALLING))
      {
         return TRUE;
      }
   }
   return FALSE;
}


tBool clSDS_SessionControl::bMultipleCallDtmfSession() const
{
   if (_pPhoneStatus)
   {
      if (_pPhoneStatus->getPhoneStatus() == sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_INCALL_HOLD)
      {
         return TRUE;
      }
   }
   return FALSE;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_SessionControl::vRequestAllSpeakers()
{
   if (_pSdsControl)
   {
      _pSdsControl->vRequestAllSpeakers();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_SessionControl::vSetSpeaker(tU16 u16SpeakerId)
{
   if (_pSdsControl)
   {
      _pSdsControl->vSetSpeaker(u16SpeakerId);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_SessionControl::sendStartSessionContext(ContextType contextId)
{
   _startupContext =  mapGui2SdsStartupContext(contextId);
   if (_pSdsControl)
   {
      if (_startupContext == sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_CALL_CONTACT)
      {
         if ((_pPhoneStatus) && (_pPhoneStatus->getPhoneStatus() == sds2hmi_fi_tcl_e8_PHN_Status::FI_EN_IDLE))
         {
            _pSdsControl->vSendPttEvent(tU32(_startupContext));
         }
         else
         {
            ETG_TRACE_USR1(("Ignore SDS_CONTEXT_SWC_TEL context: No Phone available"));
         }
      }
      else
      {
         _pSdsControl->vSendPttEvent(tU32(_startupContext));
      }
   }
   _startupContext = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_GLOBAL;
}


/**************************************************************************//**
 *
 ******************************************************************************/
sds2hmi_fi_tcl_e8_SDS_EntryPoint::tenType clSDS_SessionControl::mapGui2SdsStartupContext(ContextType startupContext) const
{
   switch (startupContext)
   {
      case ContextType__SDS_CONTEXT_CALL:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_CALL_CONTACT);

      case ContextType__SDS_CONTEXT_DIALNUM:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_DIAL_NUMBER);

      case ContextType__SDS_CONTEXT_CALLHIST:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_CALL_HISTORY);

      case ContextType__SDS_CONTEXT_QUICKDIAL:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_QUICK_DIAL);

      case ContextType__SDS_CONTEXT_READTEXT:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_READ_TEXT);

      case ContextType__SDS_CONTEXT_SENDTEXT:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_SEND_TEXT);

      case ContextType__SDS_CONTEXT_STREETADDR:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_ADDRESS);

      case ContextType__SDS_CONTEXT_POI:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI);

      case ContextType__SDS_CONTEXT_POICAT:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POICAT);

      case ContextType__SDS_CONTEXT_INTERSECTION:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_INTERSECTION);

      case ContextType__SDS_CONTEXT_CITYCENTER:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_CITYCENTER_CITY);

      case ContextType__SDS_CONTEXT_PLAYALBUM:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_SELECT_ALBUM);

      case ContextType__SDS_CONTEXT_PLAYARTIST:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_SELECT_ARTIST);

      case ContextType__SDS_CONTEXT_PLAYSONG:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_SELECT_SONG);

      case ContextType__SDS_CONTEXT_PLAYLIST:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_SELECT_PLAYLIST);

      default:
         return (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_GLOBAL);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_SessionControl::sendEnterPauseMode()
{
   if (_pSdsControl)
   {
      _pSdsControl->sendEnterPauseMode();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_SessionControl::sendExitPauseMode()
{
   if (_pSdsControl)
   {
      _pSdsControl->sendExitPauseMode();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_SessionControl::bIsTTSOnlySupport() const
{
   return _pSDSStatus->bIsTTSOnlySupport();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_SessionControl::sendEnterManualMode()
{
   if (_pSdsControl)
   {
      _pSdsControl->sendEnterManualMode();
      if (_timer.isActive())
      {
         startTimer();
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_SessionControl::onExpired(asf::core::Timer& /*timer*/, boost::shared_ptr<asf::core::TimerPayload> /*data*/)
{
   if (_pSdsControl)
   {
      _pSdsControl->sendCancelDialog();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_SessionControl::startTimer()
{
   _timer.start(*this, 60000);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_SessionControl::stopTimer()
{
   _timer.stop();
}
