/**************************************************************************//**
 * \file       clSDS_RecentCallsList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_RecentCallsList.h"
#include "application/clSDS_Iconizer.h"
#include "application/clSDS_PhoneNumberFormatter.h"
#include "application/clSDS_StringVarList.h"
#include "application/StringUtils.h"
#include "view_db/Sds_TextDB.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_RecentCallsList.cpp.trc.h"
#endif

using namespace MOST_PhonBk_FI;
using namespace most_PhonBk_fi_types;
using namespace most_BTSet_fi_types;
using namespace MOST_Tel_FI;
using namespace most_Tel_fi_types;


/**************************************************************************//**
 *Destructor
 ******************************************************************************/
clSDS_RecentCallsList::~clSDS_RecentCallsList()
{
   _pFormatTimeDate = NULL;
}


/**************************************************************************//**
 *Constructor
 ******************************************************************************/
clSDS_RecentCallsList::clSDS_RecentCallsList(::boost::shared_ptr< ::MOST_PhonBk_FI::MOST_PhonBk_FIProxy> pPhoneBookProxy,
      clSDS_FormatTimeDate* pFormatTimeDate,
      ::boost::shared_ptr< ::MOST_Tel_FI::MOST_Tel_FIProxy > telephoneProxy)
   : _recentCallList()
   , _phoneBookProxy(pPhoneBookProxy)
   , _pFormatTimeDate(pFormatTimeDate)
   , _downloadStatus(0)
   , _requestedCallHistoryType()
   , _telephoneProxy(telephoneProxy)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
tU32 clSDS_RecentCallsList::u32GetSize()
{
   return _recentCallList.size();
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::vector<clSDS_ListItems> clSDS_RecentCallsList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   if (_requestedCallHistoryType == T_e8_PhonBkCallHistoryType__e8CCH)
   {
      for (tU32 u32Index = u32StartIndex; u32Index < u32EndIndex; u32Index++)
      {
         clSDS_ListItems oListItem;
         oListItem.oDescription.szString = oGetItem(u32Index);
         oListItem.oIcon2.szString = oGetNumberType(u32Index);
         oListItem.oDistance.szString = oGetTime(u32Index);
         oListItem.oIcon.szString = NO_ICON;
         oListItems.push_back(oListItem);
      }
   }
   else
   {
      for (tU32 u32Index = u32StartIndex; u32Index < u32EndIndex; u32Index++)
      {
         clSDS_ListItems oListItem;
         oListItem.oDescription.szString = oGetItem(u32Index);
         oListItem.oIcon.szString = oGetNumberType(u32Index);
         oListItems.push_back(oListItem);
      }
   }
   return oListItems;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_RecentCallsList::oGetItem(tU32 u32Index)
{
   if (oGetName(u32Index) != "")
   {
      return oGetName(u32Index);
   }
   else if (oGetPhonenumber(u32Index) != "")
   {
      return oGetPhonenumber(u32Index);
   }
   else
   {
      return Sds_TextDB_vGetText(SDS_TEXT_PRIVATE_NUMBER);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_RecentCallsList::oGetTime(tU32 u32Index)
{
   if (u32Index < _recentCallList.size())
   {
      return _recentCallList[u32Index].formattedTimeDate;
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_RecentCallsList::oGetName(tU32 u32Index)
{
   if (u32Index < _recentCallList.size())
   {
      return _recentCallList[u32Index].Name;
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_RecentCallsList::oGetPhonenumber(tU32 u32Index)
{
   if (u32Index < _recentCallList.size())
   {
      return _recentCallList[u32Index].PhoneNumber;
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/
tU32 clSDS_RecentCallsList::u32GetNumberType(tU32 /*u32Index*/) const
{
   return 0;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_RecentCallsList::oGetNumberType(tU32 u32Index) const
{
   std::string callLogsListElement = " ";
   if (u32Index < _recentCallList.size())
   {
      tChar szTempString[3] = "";
      OSALUTIL_s32SaveNPrintFormat(szTempString, sizeof(szTempString), "%d", _recentCallList[u32Index].CallType);
      callLogsListElement = szTempString;
      ETG_TRACE_USR1(("clSDS_ReadSmsList::oListItem.smsListElement.szString: %s", callLogsListElement.c_str()));
   }
   return callLogsListElement;
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_RecentCallsList::bSelectElement(tU32 u32SelectedIndex)
{
   if (u32SelectedIndex > 0)
   {
      const std::string contactName = oGetName(u32SelectedIndex - 1);
      const std::string contactNumber = oGetPhonenumber(u32SelectedIndex - 1);

      if (contactName != "")
      {
         clSDS_StringVarList::vSetVariable("$(ContactName)", contactName);
      }
      else
      {
         clSDS_StringVarList::vSetVariable("$(ContactName)", contactNumber);
      }

      return TRUE;
   }
   return FALSE;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_RecentCallsList::oGetSelectedItem(tU32 u32Index)
{
   if (u32Index > 0)
   {
      return oGetPhonenumber(u32Index - 1);
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
uint8 clSDS_RecentCallsList::getPreferredSortOrderTypeForActiveDevice()
{
   uint8 preferredSortOrderType = T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_FIRSTNAME;
   if (_telephoneProxy->hasActivePhoneDevice() && _phoneBookProxy->hasPreferredSortOrderList())
   {
      const ::most_PhonBk_fi_types_Extended::T_PhonBkPreferredSortOrderList& preferredSortOrderList =
         _phoneBookProxy->getPreferredSortOrderList().getOPreferredSortOrderList();
      uint8 activePhoneDevice = _telephoneProxy->getActivePhoneDevice().getU8DeviceHandle();

      for (tU32 index = 0; index < preferredSortOrderList.size(); index++)
      {
         if (preferredSortOrderList[index].getU8DeviceHandle() == activePhoneDevice)
         {
            preferredSortOrderType = preferredSortOrderList[index].getE8PreferredPhoneBookSortOrder();
            break;
         }
      }
   }
   return preferredSortOrderType;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_RecentCallsList::setRecentCallList(const most_PhonBk_fi_types::T_PhonBkCallHistoryListSliceResult& callHistoryListSliceResult)
{
   most_PhonBk_fi_types::T_PhonBkCallHistoryListSliceResultItem listItem;
   RecentCall callEntry;
   _recentCallList.clear();

   uint8 preferredSortOrderType = getPreferredSortOrderTypeForActiveDevice();
   for (tU32 listCount = 0; listCount != callHistoryListSliceResult.size(); ++listCount)
   {
      listItem = callHistoryListSliceResult[listCount];
      std::string formattedString("");
      std::string contactName;
      if (preferredSortOrderType == T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_LASTNAME)
      {
         contactName = listItem.getSLastName() + " " + listItem.getSFirstName();
      }
      else
      {
         contactName = listItem.getSFirstName() + " " + listItem.getSLastName();
      }
      callEntry.Name = StringUtils::trim(contactName);

      callEntry.PhoneNumber = listItem.getSPhoneNumber();
      if (_requestedCallHistoryType == T_e8_PhonBkCallHistoryType__e8CCH)
      {
         callEntry.CallType = static_cast<uint8>(listItem.getE8CallHistoryType());
      }
      else
      {
         callEntry.CallType = static_cast<uint8>(listItem.getE8CallPhoneType());
         if (callEntry.CallType == T_e8_PhonBkCallPhoneType__e8Unknown)
         {
            callEntry.CallType =  static_cast<uint8>(T_e8_PhonBkCallPhoneType__e8Other);
         }
      }
      ETG_TRACE_USR1(("getAllCallHistoryList: %d ", callEntry.CallType));
      _pFormatTimeDate->formatTimeDate(listItem.getOCallDateTime().getOCallTime().getSHours(),
                                       listItem.getOCallDateTime().getOCallTime().getSMinutes(),
                                       listItem.getOCallDateTime().getOCallDate().getSCldrDay(),
                                       listItem.getOCallDateTime().getOCallDate().getSCldrMonth(),
                                       listItem.getOCallDateTime().getOCallDate().getSCldrYear(), formattedString);
      callEntry.formattedTimeDate = formattedString;
      _recentCallList.push_back(callEntry);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_RecentCallsList::getLastDialedNumber() const
{
   if (_recentCallList.size())
   {
      return _recentCallList[0].PhoneNumber;
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_RecentCallsList::getAllCallHistoryList()
{
   _requestedCallHistoryType = T_e8_PhonBkCallHistoryType__e8CCH;
   if ((_downloadStatus) && (_phoneBookProxy->isAvailable()) && (_telephoneProxy->hasActivePhoneDevice()))
   {
      ETG_TRACE_USR1(("getAllCallHistoryList: "));
      // create call history list with of type Outgoing calls sorted by Date and Time
      _phoneBookProxy->sendCreateCallHistoryListStart(*this
            , _telephoneProxy->getActivePhoneDevice().getU8DeviceHandle()
            , T_e8_PhonBkCallHistoryType__e8CCH, T_e8_PhonBkCallHistorySortType__e8CH_SORT_DATETIME);
   }
   else
   {
      notifyListObserver();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_RecentCallsList::getOutgoingCallHistoryList()
{
   _requestedCallHistoryType = T_e8_PhonBkCallHistoryType__e8OCH;
   if ((_downloadStatus) && (_phoneBookProxy->isAvailable()) && (_telephoneProxy->hasActivePhoneDevice()))
   {
      // create call history list with of type Outgoing calls sorted by Date and Time
      _phoneBookProxy->sendCreateCallHistoryListStart(*this
            , _telephoneProxy->getActivePhoneDevice().getU8DeviceHandle()
            , T_e8_PhonBkCallHistoryType__e8OCH, T_e8_PhonBkCallHistorySortType__e8CH_SORT_DATETIME);
   }
   else
   {
      notifyListObserver();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_RecentCallsList::getIncomingCallHistoryList()
{
   _requestedCallHistoryType = T_e8_PhonBkCallHistoryType__e8ICH;
   if ((_downloadStatus) && (_phoneBookProxy->isAvailable()) && (_telephoneProxy->hasActivePhoneDevice()))
   {
      // create call history list with of type Incoming calls sorted by Date and Time
      _phoneBookProxy->sendCreateCallHistoryListStart(*this
            , _telephoneProxy->getActivePhoneDevice().getU8DeviceHandle()
            , T_e8_PhonBkCallHistoryType__e8ICH, T_e8_PhonBkCallHistorySortType__e8CH_SORT_DATETIME);
   }
   else
   {
      notifyListObserver();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_RecentCallsList::getMissedCallHistoryList()
{
   _requestedCallHistoryType = T_e8_PhonBkCallHistoryType__e8MCH;
   if ((_downloadStatus) && (_phoneBookProxy->isAvailable()) && (_telephoneProxy->hasActivePhoneDevice()))
   {
      // create call history list with of type Missed calls sorted by Date and Time
      _phoneBookProxy->sendCreateCallHistoryListStart(*this
            , _telephoneProxy->getActivePhoneDevice().getU8DeviceHandle()
            , T_e8_PhonBkCallHistoryType__e8MCH, T_e8_PhonBkCallHistorySortType__e8CH_SORT_DATETIME);
   }
   else
   {
      notifyListObserver();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_RecentCallsList::updateHeadlineTagforRedial()
{
   bSelectElement(1);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_RecentCallsList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _phoneBookProxy)
   {
      _phoneBookProxy->sendDownloadStateUpReg(*this);
      _phoneBookProxy->sendPreferredSortOrderListUpReg(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_RecentCallsList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _phoneBookProxy)
   {
      _phoneBookProxy->sendDownloadStateRelUpRegAll();
      _phoneBookProxy->sendPreferredSortOrderListRelUpRegAll();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_RecentCallsList::onDownloadStateStatus(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DownloadStateStatus >& status)
{
   if (status->getODownloadStateStream().size())
   {
      const T_PhonBkDownloadStateStreamItem& oDownloadStatus  = status->getODownloadStateStream()[0];
      if (T_e8_PhonBkRecentCallListDownloadState__e8RCDS_COMPLETE == oDownloadStatus.getE8RecentCallListDownloadState())
      {
         _downloadStatus = 1;
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_RecentCallsList::onDownloadStateError(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DownloadStateError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_RecentCallsList::onCreateCallHistoryListResult(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< CreateCallHistoryListResult >& result)
{
   uint16 listHandle = result->getU16ListHandle();
   uint16 recentCallHistorySize = result->getU16ListLength();
   if (recentCallHistorySize)
   {
      _phoneBookProxy->sendRequestSliceCallHistoryListStart(*this, listHandle, 0, recentCallHistorySize);
   }
   else
   {
      _recentCallList.clear();
      notifyListObserver();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_RecentCallsList::onCreateCallHistoryListError(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< CreateCallHistoryListError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_RecentCallsList::onRequestSliceCallHistoryListResult(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestSliceCallHistoryListResult >& result)
{
   if (result->getOCallHistoryListSliceResult().size())
   {
      const T_PhonBkCallHistoryListSliceResult& callHistoryListSliceResult = result->getOCallHistoryListSliceResult();
      setRecentCallList(callHistoryListSliceResult);
      updateHeadlineTagforRedial();
      notifyListObserver();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_RecentCallsList::onRequestSliceCallHistoryListError(
   const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestSliceCallHistoryListError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_RecentCallsList::onPreferredSortOrderListError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredSortOrderListError >& /*error*/)
{
}


void clSDS_RecentCallsList::onPreferredSortOrderListStatus(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredSortOrderListStatus >& /*status*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_RecentCallsList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType enListType)
{
   switch (enListType)
   {
      case sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PHONE_MISSED_CALLS:
         getMissedCallHistoryList();
         break;

      case sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PHONE_ANSWERED_CALLS:
         getIncomingCallHistoryList();
         break;

      case sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PHONE_DIALLED_CALLS:
         getOutgoingCallHistoryList();
         break;

      case sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PHONE_ALL_CALLS:
         getAllCallHistoryList();
         break;

      default:
         notifyListObserver();
         break;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_RecentCallsList::phoneStatusChanged(uint8 /*deviceHandle*/, most_BTSet_fi_types_Extended::T_e8_BTConnectionStatus status)
{
   if (status  == most_BTSet_fi_types_Extended::T_e8_BTConnectionStatus__e8STATUS_DISCONNECTED)
   {
      _recentCallList.clear();
      _downloadStatus = false;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription> clSDS_RecentCallsList::getHmiElementDescription(unsigned int index)
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   if (oGetName(index - 1) != "")
   {
      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CONTACTNAME;
      hmiElementDescription.DescriptorValue = oGetName(index - 1).c_str();
      hmiElementDescriptionList.push_back(hmiElementDescription);
   }
   else if (oGetPhonenumber(index - 1) != "")
   {
      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PHONENUMBER;
      hmiElementDescription.DescriptorValue = oGetPhonenumber(index - 1).c_str();
      hmiElementDescriptionList.push_back(hmiElementDescription);
   }
   else
   {
      ETG_TRACE_USR1(("clSDS_RecentCallsList::getHmiElementDescription : no ContactInfo"));
   }
   return hmiElementDescriptionList;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> clSDS_RecentCallsList::getHmiListDescription()
{
   ETG_TRACE_USR1(("clSDS_RecentCallsList::getHmiListDescription"));
   std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescrptionList hmiElementList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElement;
   for (std::vector<RecentCall>::const_iterator itr = _recentCallList.begin(); itr != _recentCallList.end(); ++itr)
   {
      hmiElementList.DescriptionList.clear();
      if (itr->Name != "")
      {
         hmiElement.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CONTACTNAME;
         hmiElement.DescriptorValue = itr->Name.c_str();
         hmiElementList.DescriptionList.push_back(hmiElement);
      }
      else if (itr->PhoneNumber != "")
      {
         hmiElement.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PHONENUMBER;
         hmiElement.DescriptorValue = itr->PhoneNumber.c_str();
         hmiElementList.DescriptionList.push_back(hmiElement);
      }
      hmiElementDescriptionList.push_back(hmiElementList);
   }
   return hmiElementDescriptionList;
}
